# Directory is a collection of files. On the file system layer it is
# represented by a directory.
#
# This file is part of cms.sh.
#
# cms.sh is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# cms.sh is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# cms.sh. If not, see <http://www.gnu.org/licenses/>.
#
# (c) 2010-2011 Vitaly Minko <vitaly.minko@gmail.com> 

package CMSsh::StorageMode::Directory;

our @ISA = qw(CMSsh::Mode::PublicationCollection);

use strict;
use CMSsh::Mode::PublicationCollection;
use CMSsh::FileSystem::Directory;
use CMSsh::FileSystem::File;
use CMSsh::StorageMode::Config;
use CMSsh::StorageMode::Messages;

my $Cfg = CMSsh::StorageMode::Config->new;
my $Msg = CMSsh::StorageMode::Messages->new;

################################################################################
# Common methods

sub new ($$) {
    my ($class_name, $uri) = @_;

    my $self = CMSsh::Mode::PublicationCollection->new($uri);
    unless ($self) {
        warn $Msg->get('CANT_INST_PUB', $uri);
        return undef;
    }

    my $dir_path = $Cfg->get('ROOT_STORAGE') .
                       (!$self->is_root && $self->{_uri});
    $self->{_dir} = CMSsh::FileSystem::Directory->new($dir_path);
    unless ($self->{_dir}) {
        warn $Msg->get('CANT_INST_FSO', $dir_path);
        return undef;
    }

    bless $self, $class_name;
    return $self;
}

sub get_type ($) {
    my $self = shift;
    return $self->_extract_type(__PACKAGE__);
}

sub get_listing ($) {
    my $self = shift;
    my @result;

    my $listing = $self->{_dir}->get_listing;
    unless ($listing) {
        warn $Msg->get('CANT_LIST_DIR', $self->{_dir}->get_path);
        return undef;
    }

    my (@files, @dirs);

# Add files
    foreach my $file (@{$listing->{files}}) {
        my $file_name = $file->get_name;
            unless ($file_name) {
            warn $Msg->get('CANT_MAP_FSO', $file->get_path, $file->get_name);
            next;
        }

        my $file_uri = $self->{_uri} . '/' . $file_name;
        my $file = CMSsh::StorageMode::File->new($file_uri);
        unless ($file) {
            warn $Msg->get('CANT_INST_PUB', $file_uri);
            next;
        }

        push @files, $file;
    }

# Add directories
    foreach my $dir (@{$listing->{dirs}}) {
        my $dir_uri = $self->{_uri} . '/' . $dir->get_name;
        my $dir = CMSsh::StorageMode::Directory->new($dir_uri);
        unless ($dir) {
            warn $Msg->get('CANT_INST_PUB', $dir_uri);
            next;
        }

        push @dirs, $dir;
    }

    return { CMSsh::StorageMode::File->get_type => \@files,
             CMSsh::StorageMode::Directory->get_type => \@dirs };
}

# No CGI methods, because this mode does not have a view

################################################################################
# CMS methods

sub get_fs_dir ($) {
    my $self = shift;
    return $self->{_dir};
}

sub create_subfile ($$) {
    my ($self, $file) = @_;

    if ($file->get_name =~ /^\./) {
        warn $Msg->get('WRONG_SRC_NAME', $file->get_name);
        return undef;
    }

    my $desired_uri = $self->{_uri} . '/' . $file->get_name;
    if (CMSsh::StorageMode->find_publication($desired_uri)) {
        warn $Msg->get('OCCUPIED_URI', $desired_uri);
        return undef;
    }

    $file->move($self->{_dir})
    or do {
        warn $Msg->get('CANT_MOVE_FSO', $file->get_path, $self->{_dir});
        return undef;
    };

    return CMSsh::StorageMode::File->new($desired_uri);
}

sub create_subdir ($$) {
    my ($self, $name) = @_;

    if ($name =~ /\// || $name =~ /^\./) {
        warn $Msg->get('WRONG_NAME', $name);
        return undef;
    }

    my $desired_uri = $self->{_uri} . '/' . $name;
    if (CMSsh::StorageMode->find_publication($desired_uri)) {
        warn $Msg->get('OCCUPIED_URI', $desired_uri);
        return undef;
    }

    $self->{_dir}->create_subdir($name)
    or do {
        warn $Msg->get('CANT_MAKE_DIR', $name, $self->{_dir}->get_path);
        return undef;
    };

    return CMSsh::StorageMode::Directory->new($desired_uri);
}

sub move ($$) {
    my ($self, $dest_dir) = @_;
    
    my $new_uri = $dest_dir->get_uri . '/' . $self->{_name};
    if (CMSsh::StorageMode->find_publication($new_uri)) {
        warn $Msg->get('OCCUPIED_URI', $new_uri);
        return 0;
    }

    my $dest_fs_dir = $dest_dir->get_fs_dir;
    my $src_fs_dir = $self->{_dir};
    $src_fs_dir->move($dest_fs_dir)
    or do {
        warn $Msg->get('CANT_MOVE_FSO',
                       $src_fs_dir->get_path, $dest_fs_dir->get_path);
        return 0;
    };

    $self->_set_uri($dest_dir->get_uri . '/' . $self->{_name});

    return 1;
}

sub rename ($$) {
    my ($self, $new_name) = @_;

    if ($new_name =~ /\// || $new_name =~ /^\./) {
        warn $Msg->get('WRONG_NAME', $new_name);
        return 0;
    }

    my $new_uri = $self->get_parent_uri . '/' . $new_name;
    if (CMSsh::StorageMode->find_publication($new_uri)) {
        warn $Msg->get('OCCUPIED_URI', $new_uri);
        return 0;
    }

    $self->{_dir}->rename($new_name)
    or do {
        warn $Msg->get('CANT_RENAME_FSO', $self->{_dir}->get_path, $new_name);
        return 0;
    };

    $self->_set_uri($self->get_parent_uri . '/' . $new_name);
    $self->_update_name;

    return 1;
}

sub remove ($) {
    my $self = shift;

    if (@{$self->get_listing->{CMSsh::StorageMode::File->get_type}} ||
            @{$self->get_listing->{CMSsh::StorageMode::Directory->get_type}}) {
        warn $Msg->get('COLL_NOT_EMPTY', $self->{_uri});
        return 0;
    }

    $self->{_dir}->remove
    or do {
        warn $Msg->get('CANT_REMOVE_FSO', $self->{_dir}->get_path);
        return 0;
    };

    return 1;
}

1
