# Album is is a collection of images. On the file system layer it is
# represented by a directory containing images, a directory containg image
# thumbnails and a directory containing image previews.
#
# This file is part of cms.sh.
#
# cms.sh is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# cms.sh is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# cms.sh. If not, see <http://www.gnu.org/licenses/>.
#
# (c) 2010-2011 Vitaly Minko <vitaly.minko@gmail.com>

package CMSsh::GalleryMode::Album;

our @ISA = qw(CMSsh::Mode::PublicationCollection);

use strict;
use CMSsh::Mode::PublicationCollection;
use CMSsh::FileSystem::Directory;
use CMSsh::FileSystem::ImgFile;
use CMSsh::GalleryMode::Config;
use CMSsh::GalleryMode::Messages;

my $Cfg = CMSsh::GalleryMode::Config->new;
my $Msg = CMSsh::GalleryMode::Messages->new;

################################################################################
# Common methods

sub new ($$) {
    my ($class_name, $uri) = @_;

    my $self = CMSsh::Mode::PublicationCollection->new($uri);
    unless ($self) {
        warn $Msg->get('CANT_INST_PUB', $uri);
        return undef;
    }

    foreach (['_img_dir',     'ROOT_IMG'],
             ['_preview_dir', 'ROOT_PREVIEW'],
             ['_thumb_dir',   'ROOT_THUMB']) {
        my $dir_path = $Cfg->get($_->[1]) . (!$self->is_root && $self->{_uri});
        my $dir = $self->{$_->[0]}
                = CMSsh::FileSystem::Directory->new($dir_path);
        unless ($dir) {
            warn $Msg->get('CANT_INST_FSO', $dir_path);
            return undef;
        }
    }

    bless $self, $class_name;
    return $self;
}

sub get_type ($) {
    my $self = shift;
    return $self->_extract_type(__PACKAGE__);
}

sub get_listing ($) {
    my $self = shift;
    my @result;

    my $listing = $self->{_img_dir}->get_listing;
    unless ($listing) {
        warn $Msg->get('CANT_LIST_DIR', $self->{_img_dir}->get_path);
        return undef;
    }

    my (@imgs, @albs);

# Add single images
    foreach my $file (@{$listing->{files}}) {
        next if ($file->get_name =~ /^\./);
        my $img_file = CMSsh::FileSystem::ImgFile->new($file->get_path);
        unless ($img_file) {
            warn $Msg->get('CantInstFso', $file->get_path);
            next;
        }

        my $img_name = $img_file->get_basename;
        unless (defined $img_name) {
            warn $Msg->get('CANT_MAP_FSO',
                           $img_file->get_path, $img_file->get_name);
            next;
        }

        my $img_uri = $self->{_uri} . "/$img_name";
        my $img = CMSsh::GalleryMode::Image->new($img_uri);
        unless ($img) {
            warn $Msg->get('CANT_INST_PUB', $img_uri);
            next;
        }

        push @imgs, $img;
    }

# Add albums
    foreach my $dir (@{$listing->{dirs}}) {
        next if ($dir->get_name =~ /^\./);
        my $alb_uri =  $self->{_uri} . '/' . $dir->get_name;
        my $alb = CMSsh::GalleryMode::Album->new($alb_uri);
        unless ($alb) {
            warn $Msg->get('CANT_INST_PUB', $alb_uri);
            next;
        }

        push @albs, $alb;
    }

    sub by_name {$a->get_name cmp $b->get_name};
    @imgs = sort by_name @imgs;
    @albs = sort by_name @albs;

    return { CMSsh::GalleryMode::Image->get_type => \@imgs,
             CMSsh::GalleryMode::Album->get_type => \@albs };
}

sub get_img_dir ($) {
    my $self = shift;
    return $self->{_img_dir};
}

sub get_preview_dir ($) {
    my $self = shift;
    return $self->{_preview_dir};
}

sub get_thumb_dir ($) {
    my $self = shift;
    return $self->{_thumb_dir};
}

################################################################################
# Presentation methods

sub get_preview ($) {
    my $self = shift;
    return { name => $self->{_name}, path => $self->{_uri}, };
}

sub get_inner_preview ($) {
    my $self = shift;
    my @result;

    my $listing = $self->get_listing;
    unless ($listing) {
        warn $Msg->get('COLL_LISTING_ERR', $self->{_uri});
        return undef;
    }

    my (@imgs, @albs);

    foreach ([CMSsh::GalleryMode::Image->get_type, \@imgs],
             [CMSsh::GalleryMode::Album->get_type, \@albs]) {
        foreach my $publication (@{$listing->{$_->[0]}}) {
            my $item = $publication->get_preview;
            unless ($item) {
                warn $Msg->get('CANT_PREV_PUB', $publication->get_uri);
                next;
            }
            push @{$_->[1]}, $item;
        }
    }

    return { CMSsh::GalleryMode::Image->get_type => \@imgs,
             CMSsh::GalleryMode::Album->get_type => \@albs };
}

################################################################################
# Administration methods

sub create_subimg ($$) {
    my ($self, $img_file) = @_;

    if ($img_file->get_name =~ /^\./) {
        warn $Msg->get('WRONG_SRC_NAME', $img_file->get_name);
        return undef;
    }

    my $desired_uri = $self->{_uri} . '/' . $img_file->get_basename;
    if (CMSsh::GalleryMode->find_publication($desired_uri)) {
        warn $Msg->get('OCCUPIED_URI', $desired_uri);
        return undef;
    }

    $img_file->move($self->{_img_dir})
    or do {
        warn $Msg->get('CANT_MOVE_FSO',
                       $img_file->get_path, $self->{_img_dir}->get_path);
        return undef;
    };

    foreach ([$self->{_thumb_dir},   'THUMB_SIZE',    'THUMB_SIZE' ],
             [$self->{_preview_dir}, 'PREVIEW_WIDTH', 'PREVIEW_HEIGHT']) {
        $img_file->copy_resized($_->[0], $Cfg->get($_->[1]), $Cfg->get($_->[2]))
        or do {
            warn $Msg->get('CANT_CP_RES_IMG',
                           $img_file->get_path, $_->[0]->get_path);
            return undef;
        };
    }

    return CMSsh::GalleryMode::Image->new($desired_uri);
}

sub create_subalb ($$) {
    my ($self, $name) = @_;

    if ($name =~ /^\./) {
        warn $Msg->get('WRONG_NAME', $name);
        return undef;
    }

    my $desired_uri = $self->{_uri} . '/' . $name;
    if (CMSsh::GalleryMode->find_publication($desired_uri)) {
        warn $Msg->get('OCCUPIED_URI', $desired_uri);
        return undef;
    }

    foreach ($self->{_img_dir},
             $self->{_preview_dir},
             $self->{_thumb_dir}) {
        $_->create_subdir($name)
        or do {
            warn $Msg->get('CANT_MAKE_DIR', $name, $_->get_path);
            return undef;
        };
    }

    return CMSsh::GalleryMode::Album->new($desired_uri);
}

sub move ($$) {
    my ($self, $dest_alb) = @_;

    my $new_uri = $dest_alb->get_uri . '/' . $self->{_name};
    if (CMSsh::GalleryMode->find_publication($new_uri)) {
        warn $Msg->get('OCCUPIED_URI', $new_uri);
        return 0;
    }

    foreach (['_img_dir',     $dest_alb->get_img_dir],
             ['_preview_dir', $dest_alb->get_preview_dir],
             ['_thumb_dir',   $dest_alb->get_thumb_dir]) {
        my $src_dir = $self->{$_->[0]};
        $src_dir->move($_->[1])
        or do {
            warn $Msg->get('CANT_MOVE_FSO',
                           $src_dir->get_path, $_->[1]->get_path);
            return 0;
        };
    }

    $self->_set_uri($new_uri);

    return 1;
}

sub rename ($$) {
    my ($self, $new_name) = @_;

    if ($new_name =~ /^\./) {
        warn $Msg->get('WRONG_NAME', $new_name);
        return 0;
    }

    my $new_uri = $self->get_parent_uri . '/' . $new_name;
    if (CMSsh::GalleryMode->find_publication($new_uri)) {
        warn $Msg->get('OCCUPIED_URI', $new_uri);
        return 0;
    }

    foreach (qw(_img_dir _preview_dir _thumb_dir)) {
        $self->{$_}->rename($new_name)
        or do {
            warn $Msg->get('CANT_RENAME_FSO',
                           $self->{$_}->get_path, $new_name);
            return 0;
        };
    }

    $self->_set_uri($new_uri);
    $self->_update_name;

    return 1;
}

sub remove ($) {
    my $self = shift;

    if (@{$self->get_listing->{CMSsh::GalleryMode::Image->get_type}} ||
            @{$self->get_listing->{CMSsh::GalleryMode::Album->get_type}}) {
        warn $Msg->get('COLL_NOT_EMPTY', $self->{_uri});
        return 0;
    }

    foreach (qw(_img_dir _preview_dir _thumb_dir)) {
        $self->{$_}->remove
        or do {
            warn $Msg->get('CANT_REMOVE_FSO', $self->{$_}->get_path);
            return 0;
        };
    }

    return 1;
}

1
