;;; xetla-tips.el --- "Tip of the day" feature for XEtla.

;; Copyright (C) 2004  Free Software Foundation, Inc. (GPL)
;; Copyright (C) 2004 Steve Youngs (BSD)

;; Author:        Steve Youngs <steve@eicq.org>
;; Maintainer:    Steve Youngs <steve@eicq.org>
;; Created:       2004-11-25
;; Keywords:      arch archive tla

;; Based on xtla-tips.el by: Matthieu Moy <Matthieu.Moy@imag.fr>

;; This file is part of XEtla.

;; Redistribution and use in source and binary forms, with or without
;; modification, are permitted provided that the following conditions
;; are met:
;;
;; 1. Redistributions of source code must retain the above copyright
;;    notice, this list of conditions and the following disclaimer.
;;
;; 2. Redistributions in binary form must reproduce the above copyright
;;    notice, this list of conditions and the following disclaimer in the
;;    documentation and/or other materials provided with the distribution.
;;
;; 3. Neither the name of the author nor the names of any contributors
;;    may be used to endorse or promote products derived from this
;;    software without specific prior written permission.
;;
;; THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR
;; IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
;; WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
;; DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
;; FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
;; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
;; SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
;; BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
;; OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
;; IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


;;; Commentary:

;; To raise the learning curve for xetla.el users. Some commands can
;; (optionaly) pop-up a buffer with a tip. Currently, `xetla-commit'
;; does this.


;;; Code:

(eval-and-compile
  (require 'xetla-defs)
  (require 'xetla))

(defconst xetla-tips-array
  [
"Welcome to XEtla. I'm the tip buffer. I will appear from time to time
to show you interesting features that you may have missed! Disable me
by setting the variable `xetla-tips-enabled' to nil.

Press `q' to exit, `n' to view next tip, `p' to view previous tip."
   "For the available xetla commands Xetla provides a corresponding interactive
function.
Some examples:

M-x xetla-inventory   ... runs tla inventory
M-x xetla-undo        ... runs tla undo
M-x xetla-changes     ... runs tla changes

XEtla prompts for the needed parameters."
   "Most interesting commands are available through a global keyboard
shortcut. Try `C-x T C-h' to get a list"
   "XEtla provides several major modes for different buffers. Each mode
has its own keybindings. Get help with `\\[describe-mode]'"
   "When XEtla is loaded, `C-M-h' in a minibuffer prompt gives you help
about the command being run."
   "When you are prompted for an archive, category, branch, version or
revision name, lots of keybindings are available. Get a list with `C-h'."
   "XEtla allows you to manage a list of bookmarks. Try `\\[xetla-bookmarks]' and add
bookmarks from the menu.  You may also add bookmarks from an archives,
category, version or revision buffer as well as from the xetla-browse
buffer."
   "From the bookmark buffer, you can select some bookmarks and make
them partners with \\<xetla-bookmarks-mode-map>`\\[xetla-bookmarks-marked-are-partners]'. Afterwards, pressing `\\[xetla-bookmarks-missing]' on a bookmark will
show you the missing patches from his partners."
   "You can add changelog style comments to your commit log by `\\[xetla-add-log-entry]'."
   "You can enable ispell, flyspell or other useful mode for editing
log files by `M-x customize-variable RET xetla-log-edit-mode-hook RET'."
   "By default, XEtla caches any log file you retrieve with
`xetla-cat-log' or `xetla-cat-archive-log' in ~/.arch-log-library. This
speeds up many XEtla operations.

You can disable this by setting `xetla-log-library-greedy' to nil."
   "XEtla is highly customizable.
Start it by `M-x customize-group RET xetla RET'."
   "In an *xetla-changes* buffer you can quickly jump to the source file by
`RET', or view the source file in another window by `v', or start
an ediff session by `e' to inspect/reject parts of the changes."
   "In a *xetla-changes* buffer, you can quickly jump from the list of
files to the corresponding patch hunk, and come back with `j'"
   "After committing, you can review the last committed patch with
`M-x xetla-changes-last-revision RET'.

Usefull to review and fix a patch you've just merged without mixing
manual modifications and merge in the same patch."
   "After a merge, typing \\<xetla-log-edit-mode-map> `\\[xetla-log-edit-insert-log-for-merge-and-headers]' in the log buffer will generate
for you a summary line, keyword and body. This is highly
customizable."
   "Report bugs using `M-x xetla-submit-bug-report RET'"
   "You've got a nice, graphical, archive browser one 
`M-x xetla-browse RET' away."
   "In the bookmark buffer, pressing\\<xetla-bookmarks-mode-map> `\\[xetla-bookmarks-find-file] starts with the local
tree of the bookmark at point for the default directory."
   "`\\[xetla-file-ediff]' in a source file will open an ediff session with the
unmodified version of the file. From here, you can undo patch hunks
one by one with the key `b'"
   "In the *tree-lint* buffer, with your cursor on a message, most
commands will apply to all the files listed under this message."
   "Many XEtla commands accept one or more prefix arguments.  For example,
from the *xetla-log-edit* buffer:
\\<xetla-log-edit-mode-map>
            `\\[xetla-log-edit-done]' -- Save and kill the log buffer
        `\\[universal-argument] \\[xetla-log-edit-done]' -- Save and kill the log, plus commit the changes
    `\\[universal-argument] \\[universal-argument] \\[xetla-log-edit-done]' -- Save and kill the log, plus commit and \"seal\"
`\\[universal-argument] \\[universal-argument] \\[universal-argument] \\[xetla-log-edit-done]' -- Save and kill the log, plus commit and \"fix\"

All of the XEtla commands that accept prefix arguments are documented,
so use `\\[describe-function] xetla-[command] RET' to see more nifty
tricks."
   "Give a prefix argument to M-x xetla-changelog and you'll be prompted
for a version to display a ChangeLog for.  This is handy if you want to
see a ChangeLog for a previous version of a project."
   "In a *xetla-changes* buffer, hit RET with point somewhere in the diff
and you'll be taken to that exact spot in the source file."
   ]
  "List of tips. Add more !")

(defvar xetla-tips-number 0
  "Number of the last tip viewed.
Will be saved in state.el")

(defun xetla-tips-message-number (number)
  "Return the message number NUMBER, as a string."
  (let ((number (mod number (length xetla-tips-array))))
    (aref xetla-tips-array number)))

(define-derived-mode xetla-tips-mode fundamental-mode "xetla-tips"
  "Major mode for buffers displaying tip of the day.

Commands:
\\{xetla-tips-mode-map}"
  (toggle-read-only 1))


(defun xetla-tips-popup-number (number)
  "Pops up tip number NUMBER."
  (let ((message (xetla-tips-message-number number)))
    (switch-to-buffer (get-buffer-create "*xetla-tip*"))
    (xetla-tips-mode)
    (let ((inhibit-read-only t))
      (erase-buffer)
      (insert (xetla-face-add
               "*************************   Did you know?   *************************"
               'xetla-messages)
              "\n\n")
      (insert (substitute-command-keys message))
      (newline 2)
      (insert (xetla-face-add
               "*********************************************************************"
               'xetla-messages))
      (goto-char (point-min)))))

(defun xetla-tips-popup-maybe ()
  "Pop up a buffer with a tip if tips are enabled.

see `xetla-tips-enabled'."
  (when xetla-tips-enabled
    (xetla-tips-popup)))

(defun xetla-tips-popup (&optional direction)
  "Pop up a buffer with a tip message.

Don't use this function from XEtla. Use `xetla-tips-popup-maybe'
instead."
  (interactive)
  (xetla-load-state)
  (xetla-tips-popup-number xetla-tips-number)
  (setq xetla-tips-number
        (mod (+ xetla-tips-number (or direction 1)) (length xetla-tips-array)))
  (xetla-save-state))

(defun xetla-tips-next-tip ()
  "Show next tip."
  (interactive)
  (xetla-tips-popup 1))

(defun xetla-tips-previous-tip ()
  "Show previous tip."
  (interactive)
  (xetla-tips-popup -1))

(eval-when-compile
  (autoload 'customize-group "cus-edit" nil t))

(defun xetla-tips-customize ()
  "Run customize group for xetla-tips."
  (interactive)
  (customize-group 'xetla-tips))

(provide 'xetla-tips)
;;; xetla-tips.el ends here
