/*
* Copyright (c) 2005. IBM Corporation and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
*
* Contributors:
*     IBM Corporation - initial API and implementation
*/

package org.eclipse.wst.rdb.sqleditor.internal.utils;

import java.lang.reflect.InvocationTargetException;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.progress.IProgressService;
import org.eclipse.wst.rdb.connection.internal.ui.wizards.shared.UserIdentification;
import org.eclipse.wst.rdb.internal.core.connection.ConnectionInfo;
import org.eclipse.wst.rdb.internal.core.util.DatabaseProviderHelper;
import org.eclipse.wst.rdb.sqleditor.internal.SQLEditorResources;

/**
 * Provides database services for the editor.  This includes services for selecting
 * and re-establishing a database connection.
 * 
 * @author Hetty Dougherty
 * @author bgp
 */
public class SQLDBUtils {

    protected static int PROPOSAL_TYPE_INVALID = -1;
    protected static int PROPOSAL_TYPE_TABLES = 1;
    protected static int PROPOSAL_TYPE_COLUMNS = 2;

    /**
     * Creates a SQL Statement and execute the statement using JDBC executeQuery
     * method. This method uses a Statement object to execute the query. It does
     * not close the Statement exlicitly because the ResultSet object will be
     * closed if the Statement is closed.
     * 
     * @param conn the <code>Connection</code> object to use
     * @param sql the SQL statement to execute
     * @return The <code>ResultSet</code> resulting from running the query
     */
    public static ResultSet executeSQL( Connection conn, String sql )
            throws SQLException {
        ResultSet result = null;
        Statement statement = null;
        try {
            statement = conn.createStatement();
            result = statement.executeQuery( sql );
        }
        catch (SQLException ex) {
            if (statement != null)
                statement.close();
            throw ex;
        }
        return result;
    }

    /**
     * Determines if the DB connection described by the given <code>ConnectionInfo</code>
     * is actively connected 
     * 
     * @param connInfo the <code>ConnectionInfo</code> to check
     * @return true if the connection is active, otherwise false
     */
    public static boolean isConnected( ConnectionInfo connInfo ) {
        boolean connected = false;
        
        if (connInfo != null && connInfo.getSharedDatabase() != null) {
            connected = true;
        }
        return connected;
    }
   
    /**
     * Determines if the given <code>ConnectionInfo</code> is references the
     * default user ID.  This is done by checking if both the user ID and password 
     * are empty and non-null.
     * 
     * @param connInfo the <code>ConnectionInfo</code> object to check
     * @return true if the default user ID is being used, otherwise false
     */
    public static boolean isDefaultUser( ConnectionInfo connInfo ) {

        String username = null;
        String password = null;
        
        if (connInfo != null) {
            username = connInfo.getUserName();
            password = connInfo.getPassword();
        }

        return (username != null && username.length() == 0 && password != null && password.length() == 0);
    }

    /**
     * Determines if a user ID and password prompt is needed for the given
     * <code>ConnectionInfo</code> object.  This is done by checking if either 
     * the user ID or password is null or empty.
     * 
     * @param connInfo the <code>ConnectionInfo</code> to check
     * @param true when userid/password prompt is needed, otherwise false
     */
    public static boolean isPromptNeeded( ConnectionInfo connInfo ) {
        
        String username = null;
        String password = null;
        
        if (connInfo != null) {
            username = connInfo.getUserName();
            password = connInfo.getPassword();
        }
        
        return (((username == null || username.length() == 0 || password == null || password.length() == 0)) && !isDefaultUser( connInfo ));
    }

    /**
     * Prompts for the user ID and password and updates the given <code>ConnectionInfo</code>
     * object
     * 
     * @param connInfo the <code>ConnectionInfo</code> object to update
     * @param promptMessage a message to display at the top of the prompt
     * @return true if the user clicks OK, otherwise false
     */
    public static boolean promptIDPW( ConnectionInfo connInfo, String promptMessage ) {
        boolean ok = false;
        if (connInfo != null) {
            String userName = connInfo.getUserName();
            String connName = connInfo.getName();
            if (userName == null || userName.length() == 0) {
                userName = System.getProperty( "user.name" ); //$NON-NLS-1$
            }
            if (connName == null) {
                connName = "";
            }
            UserIdentification idDialog = new UserIdentification( userName, promptMessage, connName );
            if (idDialog.open() == Window.OK) {
                userName = idDialog.getUserNameInformation();
                String password = idDialog.getPasswordInformation();
                connInfo.setUserName( userName == null ? "" : userName ); //$NON-NLS-1$
                connInfo.setPassword( password == null ? "" : password ); //$NON-NLS-1$
                ok = true;
            }
        }
        return ok;
    }

    /**
     * Determines if we are able to connect, prompts the user if needed,
     * reconstructs connnection information if needed, and reconnects, if possible.
     * 
     * @param connInfo the <code>ConnectionInfo</code> to use to reestablish the connection
     * @return true if we are able to re-establish the connection, otherwise false
     */
    public static boolean reestablishConnection( ConnectionInfo connInfo ) {
        boolean reestablished = false;
        
        if (connInfo != null) {
            reestablished = true;
            
            if (isConnected( connInfo ) == false) {
                /* Prompt for the userid and password if needed. */
                if (isPromptNeeded( connInfo ) == true) {
                    if (promptIDPW( connInfo, null ) == true) {
                        try {
                            connInfo.saveConnectionInfo();
                        }
                        catch (Exception e) {
                            reestablished = false;
                        }
                    }
                    else {
                        /* The user cancelled from the userid/password prompt. */
                        reestablished = false;;
                    }
                }
            }

            /* If things are OK so far, test the connection. */
            if (reestablished == true) {
                StringBuffer msg = new StringBuffer();
                reestablished = testConnection( connInfo, msg );
                
                if (msg.length() > 0) {
                    MessageDialog.openError( Display.getCurrent().getActiveShell(),
                            SQLEditorResources.getString( "SQLEditor.connection.error.connectionFailed.dialogTitle" ), //$NON-NLS-1$
                            SQLEditorResources.getString( "SQLEditor.connection.error.connectionFailed.message", //$NON-NLS-1$
                                    new Object[] { msg.toString() } ) );
                }
            }
        }
        
        return reestablished;
    }    

    /**
     * Tests the connection indicated by the given <code>ConnectionInfo</code>
     * and returns whether or not the connection is valid.  Error messages, if any,
     * are returned in the given <code>StringBuffer</code>.
     * 
     * @param connInfo the <code>ConnectionInfo</code> for the connection to test
     * @param msgBuf a buffer for error messages if any are generated by the test
     * @return true if the connection is OK
     */
    public static boolean testConnection( final ConnectionInfo connInfo, final StringBuffer msgBuf ) {
        boolean connectedOK = false;
        
        if (connInfo != null) {
            /* Do the "test connection" task using a busy cursor, since it may 
             * take a while. */
            try {
                IProgressService progressService = PlatformUI.getWorkbench().getProgressService();
                progressService.busyCursorWhile(
                        new IRunnableWithProgress() {
                            public void run(IProgressMonitor monitor) {
                                Connection testConn = connInfo.getSharedConnection();
                                if (testConn == null) {
                                    try {
                                        testConn = connInfo.connect();
                                        if (testConn != null) {
                                            connInfo.setSharedConnection( testConn );
                                            new DatabaseProviderHelper().setDatabase( testConn,
                                                    connInfo, connInfo.getDatabaseName() );
                                            connInfo.saveConnectionInfo();
                                        }
                                    }
                                    catch (Exception e) {
                                        msgBuf.append( e.getMessage() );
                                    }
                                }
                            }
                        }
                );
            }
            catch(InterruptedException e) {
                /* do nothing */
            }
            catch(InvocationTargetException e) {
                /* do nothing */
            }
        }

        Connection conn = connInfo.getSharedConnection();
        if (conn != null) {
            connectedOK = true;
        }

        return connectedOK;
    }
    
}