# Set to 0 to turn off verbose logging.
set verbose 1

# The installation log file
set logfile $env(HOME)/.quirc-install-log


# log <message> [<message> ...]
#     Appends the given messages to the end of the log file.
proc log { args } {
    set fid [open $::logfile "a+"]
    foreach arg $args {
	puts $fid $arg
    }
    close $fid
}

# chmod <filename> <readable> <writeable> <executable>
#     Changes the user permissions on the filename adding the given bits as
#     requested.
proc chmod { filename readable writeable executable } {
    set perms [file attributes $filename -permissions]
    if { $readable } { set perms [expr $perms | (64*4)] }
    if { $writeable } { set perms [expr $perms | (64*2)] }
    if { $executable } { set perms [expr $perms | (64*1)] }
    if { $::verbose } {
	log "attributes $filename -permissions $perms"
    }
    file attributes $filename -permissions $perms
}

# mkdir <directory>
#     Makes a given directory and any parent directories.
proc mkdir { dir } {
    set path [file split $dir]
    for { set n 1 } { $n < [llength $path] } { incr n } {
	if { $::verbose } {
	    log "mkdir [eval "file join [lrange $path 0 $n]"]"
	}
	file mkdir [eval "file join [lrange $path 0 $n]"]
    }
}

# cp <src> <dest> [<ignore_if_not_exists>] [<overwrite>]
#     src is the source filename with optional path.
#     dest is the destination filename and path.  The path does not have to
#     exist beforehand.
#     If ignore_if_not_exists is 1, non-existant source files will be ignored.
#     If overwrite is 1, existing destination files will be overwritten.
proc cp { src dest args } {
    set ignore 0
    set overwrite 0
    if { [llength $args]>=1 } {
	set ignore [lindex $args 0]
    }
    if { [llength $args]==2 } {
	set overwrite [lindex $args 1]
    }
    mkdir [file dirname $dest]
    # If the file exists or we're not planning to ignore, attempt the copy.
    if { [file exists $src] || !$ignore } {
	if { !$overwrite } {
	    if { $::verbose } {
		log "copy $src $dest"
	    }
	    file copy $src $dest
	} else {
	    if { $::verbose } {
		log "copy -force $src $dest"
	    }
	    file copy -force $src $dest
	}
    }
    # Ensure copied files are readable
    if { [file exists $dest] } {
	chmod $dest 1 0 0
    }
}

# bgerror <message>
#     The background error handler.  This is called for basically any error.
proc bgerror { message } {
    set data "Error Information:\n$::errorInfo\n\nCode: $::errorCode\n\nAdditional Info:\n$message"
    catch { log $data }
    toplevel .error
    wm title .error "QuIRC Install Error"
    wm geometry .error 600x400
    pack [text .error.text -wrap word] -expand 1 -fill both
    .error.text insert end $data
    .error.text configure -state disabled
    pack [button .error.button -command fail -text "Exit"]
    bind . <KeyPress> fail
    bind . <Button> fail
    bind .error <KeyPress> fail
    bind .error <Down> {.error.text yview scroll 1 units}
    bind .error <Up> {.error.text yview scroll -1 units}
    raise .error
    # frame .top
    # pack [text .top.log -state disabled -yscrollcommand ".top.scroll set"] -side left
    # pack [scrollbar .top.scroll -command ".top.log yview"] -fill y -expand 1
    # pack .top
}


set quirc $env(HOME)/.quirc

set doinstall 0

if { $FOUNDI } { set doinstall 1 }

set mustexist {
    VERSION
    global.tcl
    qwidgets.tcl
    variables.tcl
    themes/default.tcl
}

foreach file $mustexist {
    if { !$doinstall && ![file exists $quirc/$file] } {
	log "--- $quirc/$file is missing and is needed for QuIRC init."
	set doinstall 1
    }
}

if { !$doinstall && [string compare [exec cat $quirc/VERSION] [version]] } {
    log "--- The contents of $quirc/VERSION do not match [version]."
    set doinstall 1
}

if { !$doinstall } { destroy .; return }

log "*** Commencing QuIRC install at [clock format [clock seconds]]."


# Set up the relatively pathed file list from the data dir.

set filelist ""
foreach file [glob $DATADIR/*.tcl] {
    lappend filelist [file tail $file]
}
foreach file [glob $DATADIR/themes/*.tcl] {
    lappend filelist themes/[file tail $file]
}
lappend filelist quedit
lappend filelist fontsel
lappend filelist servers
lappend filelist VERSION

# If ~/.quirc doesn't exist, just create it and install the data files.

if { ![file exists $quirc] } {
    # Copy files over
    foreach file $filelist {
	cp $DATADIR/$file $quirc/$file
	cp $DATADIR/$file $quirc/.originals/$file
    }
    # Set files executable
    chmod $quirc/quedit 0 0 1
    chmod $quirc/fontsel 0 0 1
    # Return
    destroy .; return
}

# ~/.quirc already exists.  Choose an installation type.

wm title . "Choose Installation Type"

pack [text .text -wrap word] -expand 1 -fill both
.text insert end {The installation routines are carefully designed to safegaurd your data.  A log of the install is kept in ~/.quirc-install-log should you need to analyze what has occured.

EASY
    - Renames existing ~/.quirc directory to ~/quirc-backup-DATE-TIME
    - No previous settings are transferred to the new version.
    - Gauranteed that QuIRC will work properly after this type of install.

NORMAL
    - Backs up files that will be replaced into ~/.quirc/backup-DATE-TIME
    - Replaces all the QuIRC provided script files.
    - Checks certain variables in dynamic.tcl for important changes.
    - Most likely will work properly, but in very rare cases may not.

ADVANCED
    - Not implemented yet.
    - Attempts to determine what needs upgraded and asks how you would like to
      deal with it.
}
.text configure -state disabled

pack [frame .buttons]
pack [button .buttons.easy -command {install easy} -text "Easy"] -side left
pack [button .buttons.normal -command {install normal} -text "Normal"] -side left
pack [button .buttons.advanced -command {install advanced} -text "Advanced"] -side left

bind . <Destroy> exit

# The install proc, it does all the work.

proc install { type } {
    # Kill main window contents so it may be reused.
    bind . <Destroy> {}
    destroy .text
    destroy .buttons
    bind . <Destroy> fail
    
    switch $type {
	"easy" {
	    set newname "quirc-backup-[clock format [clock seconds] -format {%m%d%y-%H%M%S}]"
	    log "rename $::env(HOME)/.quirc $::env(HOME)/$newname"
	    file rename $::env(HOME)/.quirc $::env(HOME)/$newname
	    # Copy files over
	    foreach file $::filelist {
		cp $::DATADIR/$file $::quirc/$file
		cp $::DATADIR/$file $::quirc/.originals/$file
	    }
	    # Set files executable
	    chmod $::quirc/quedit 0 0 1
	    chmod $::quirc/fontsel 0 0 1
	}
	"normal" {
	    # Back up existing files
	    set ::newname "backup-[clock format [clock seconds] -format {%m%d%y-%H%M%S}]"
	    foreach file $::filelist {
		cp $::quirc/$file $::quirc/$::newname/$file 1
	    }
	    # Remove old originals since we're replacing them
	    log "delete -force $::quirc/.originals"
	    file delete -force $::quirc/.originals
	    # Copy files over (overwrite)
	    foreach file $::filelist {
		cp $::DATADIR/$file $::quirc/$file 0 1
		cp $::DATADIR/$file $::quirc/.originals/$file
	    }
	    # Set files executable
	    chmod $::quirc/quedit 0 0 1
	    chmod $::quirc/fontsel 0 0 1
            # Check out dynamic.tcl.
	    check_dynamic
	}
	default {
	    puts "Install method not supported."
	    exit
	}
    }
    bind . <Destroy> {}
    destroy .; return
}

proc fail {} {
    log "!!! Install failed.  Removing ~/.quirc/VERSION to force install."
    if { [file exists $::quirc/VERSION] } {
	file delete $::quirc/VERSION
    }
    exit
}

proc escape { data } {
    return [string range [list \}$data] 2 end]
}

proc store_dynamic {} {
    if { [file exists $::env(HOME)/.quirc/dynamic.tcl] } {
	cp $::env(HOME)/.quirc/dynamic.tcl $::env(HOME)/.quirc/$::newname/dynamic.tcl
    }
    set of [open $::env(HOME)/.quirc/dynamic.tcl "WRONLY CREAT TRUNC"]
    foreach var [info vars ::dynamic::*] {
	if { ![string match ::dynamic::theme_* $var] } {
	    if { ![array exists $var] } {
		puts $of "set $var \"[escape [set $var]]\""
	    } else {
		puts $of "array set $var {[array get $var]}"
	    }
	}
    }
    foreach command [info commands ::dynamic::alias_*] {
	set procname [lindex [split $command ":"] 4]
	puts $of "proc $procname { [info args $command] } { [info body $command] }"
    }
    close $of
}

# Dummy procs
proc config {args} {}
proc doc {args} {}

proc check_dynamic {} {
    # We COULD check the originals dir here.
    set vars {
	default_scripts
	default_globalscripts
	theme
    }
    if { ![file exists $::quirc/dynamic.tcl] } { return }
    namespace eval ::dynamic {} 
    uplevel #0 {source $::DATADIR/variables.tcl}
    foreach var $vars {
	set ::default-$var [set ::dynamic::$var]
    }
    namespace eval ::dynamic { source $::quirc/dynamic.tcl }
    foreach var $vars {
	if { [set ::dynamic::$var]!=[set ::default-$var] } {
	    wm title . "Configuration for \"$var\""
	    pack [label .l1 -text "Your OLD settings from dynamic.tcl:"] -anchor w
	    pack [entry .e1] -anchor w -fill x
	    .e1 insert end [set ::dynamic::$var]
	    .e1 configure -state disabled
	    pack [label .l2 -text "The NEW QuIRC provided settings:"] -anchor w
	    pack [entry .e2] -anchor w -fill x
	    .e2 insert end [set ::default-$var]
	    .e2 configure -state disabled
	    pack [label .l3 -text "CUSTOMizable new settings:"] -anchor w
	    pack [entry .e3] -anchor w -fill x
	    .e3 insert end [set ::default-$var]
	    pack [frame .buttons]
	    pack [button .buttons.custom -text "Custom" -command {set ::dynamicvar [.e3 get]}] -side left
	    pack [button .buttons.new -text "New" -command {set ::dynamicvar [.e2 get]}] -side left
	    pack [button .buttons.old -text "Old" -command {set ::dynamicvar [.e1 get]}] -side left
	    update
	    wm geometry . 600x[winfo height .]
	    tkwait variable dynamicvar
	    set ::dynamic::$var $::dynamicvar
	    bind . <Destroy> {}
	    destroy .l1 .e1 .l2 .e2 .l3 .e3 .buttons
	    bind . <Destroy> fail
	}
    }
    store_dynamic
}

log "*** Install completed."

tkwait window .
return

############################### END ################################

proc cp { args } {
    # If there's no matching files in a glob, don't try to copy.  (Need to
    # catch glob results to do so)
    # This should pop up an error message if something bad happens.
    set source [lrange $args 0 [expr [llength $args]-2]]
    set dest [lindex $args end]
    file mkdir $dest
    foreach entry $source {
	if { ![catch {glob $entry}] } {
	    foreach file [glob $entry] {
		if { [file isdirectory $file] } {
		    cp $file/* $dest/[file tail $file]
		} else {
		    file copy -force $file $dest
		}
		exec chmod -f u+r $dest/[file tail $file]
	    }
	}
    }
}

#proc log { args } {
#    .top.log configure -state normal
#    foreach arg $args {
#	.top.log insert end $arg\n
#    }
#    .top.log configure -state disabled
#    .top.log see end
#}

# Set up the installation window.


#wm title . "QuIRC Installation System - Log"
#pack [button .ok -command {destroy .} -text "Ok"]
#bind . <Return> {destroy .}
#bind . <KP_Enter> {destroy .}

#log "Copied files from:"
#log "  $DATADIR"
#log "to:"
#log "  $quirc"
#log "and to:"
#log "  $quirc/.originals"

# Back up old files.
#file delete -force $quirc/backup
#log "Removed old (if any) $quirc/backup."
#log ""
#cp $quirc/*.tcl $quirc/quedit $quirc/fontsel $quirc/servers $quirc/VERSION $quirc/backup
#cp $quirc/themes/*.tcl $quirc/backup/themes
#log "Copied files from:"
#log "  $quirc"
#log "to:"
#log "  $quirc/backup"
#log ""

set choice [tk_dialog .type "Installation Type" "Please choose the type of installation you would like.  The EASY install will simply make a backup of your existing data files and install the new ones, this is gauranteed to work, although previous settings will not transfer to the new version.  The ADVANCED install will attempt to determine what needs upgraded and will ask you how you would like to deal with it.  It is more complex." "" 0 "Easy" "Normal" "Advanced"]

switch $choice {
    0 {
	# Easy
	set newname "quirc-backup-[clock format [clock seconds] -format {%m%d%y-%H%M%S}]"
	file rename $env(HOME)/.quirc $env(HOME)/$newname
	tk_messageBox -message "In the $env(HOME) dir, the .quirc dir has been renamed to $newname (quirc-DATE-TIME)." -title "QuIRC Backed Up"
    }
    1 {
	# Normal
	# Copy over new files
	cp $DATADIR/*.tcl $DATADIR/quedit $DATADIR/fontsel $DATADIR/servers $DATADIR/VERSION $quirc
	cp $DATADIR/themes/*.tcl $quirc/themes
	cp $DATADIR/*.tcl $DATADIR/quedit $DATADIR/fontsel $DATADIR/servers $DATADIR/VERSION $quirc/.originals
	cp $DATADIR/themes/*.tcl $quirc/.originals/themes
	exec chmod -f u+x $quirc/quedit $quirc/fontsel
	log "Copied files from:"
	log "  $DATADIR"
	log "to:"
	log "  $quirc"
	log "and to:"
	log "  $quirc/.originals"
	log ""
	log "***** It would be in your best interest to read the following:"
	log ""
	log "The original files have been placed in:"
	log "  $quirc/backup"
	log ""
	log "The easy install does not check for extraneous files in the:"
	log "  $quirc"
	log "directory.  You will need to either check for yourself, pretend"
	log "that no such files exist, or run the advanced install; all of"
	log "which are viable options."
    }
    2 {
	# Advanced
	# Options to see diff, and both files.
	
	# n is normal.  Not shown in list.
	# e is extraneous:
	#  Delete Extraneous
	#  ALL - Delete Extraneous
	# m is modified with no original:
	#  Overwrite
	#  ALL - Overwrite Modified Files With No Original Counterpart
	#  ALL - Overwrite
	# o is modified with original:
	#  Overwrite
	#  Merge
	#  Merge And Replace If No Conflicts Found
	#  ALL - Overwrite Modified Files That Have An Original Counterpart
	#  ALL - Overwrite
	#  ALL - Merge
	#  ALL - Merge And Replace If No Conflicts Found
	
	# Okay, so we want to merge.  They click on Merge.  Up pops a window with
	# the merged output.  The window allows editing and saving of the file, or
	# of killing of the editor.  If the safe option is chosen, it will not edit
	# files that merge okay, it will just replace them.
	
	# 	   -[M]erge The Files.  If the merge does not encounter any conflicts,
	#       the file will be named `basename $file`.merged.  If it does
	#       encounter conflicts, it will be called `basename $file`.fixme.
	# 	   -Merge All The [F]iles
	# 	   -Merge With [R]eplacement If Safe.  If the merge does not encounter
	#       any conflicts, the file in ~/.quirc will simply be replaced.
	#       Otherwise, the file will be left as `basename $file`.fixme in
	#       ~/.quirc just as the first merge option."
	# 	   -Merge And Replace All If [S]afe."
	#  
	foreach file $filelist {
	    set type($file) "n"
	    set diff($file) ""
	}
	foreach file [glob $env(HOME)/.quirc/*] {
	    if { ![file isdirectory $file] } {
		if { [lsearch -exact -- $filelist [file tail $file]]==-1 } {
		    set type([file tail $file]) "e"
		}
	    }
	}
	# Could display a list of files with status indicators beside them
	# indicating what the deal is.  Then buttons could be provided that would
	# manipulate the state of each file.
	# Don't forget about original's dir and checking for validation of files
	# that needed to be installed.
	# Possible states:
	#  -Does not exist in new version (But did exist in older version?)
	#         Could check diff and delete if the same.
	
	#  -Do not show correctly installed files
	#  -Copy over new files automatically and do not show them
	
	#  -Force fixing until all are correctly installed (or allow exit)
	
	
	# For each file have a few fields... "Extraneous," "Modified," and the diff
	
	# New steps:
	#  -Mark extraneous files and have "Delete" and "Delete All Extraneous."
	#  -Make sure new directories exist
	#  -Copy over new files to ~/.quirc
	#  -Does orignals dir exist?
	#    Yes:
	#     -Copy over new files to originals.
	#     -Remove new files from originals.
	#     -If the files are not the same between the originals and quirc as
	#      well as the data and quirc, then mark it as different and store the
	#      diff.
	#    No:
	#     -Check for differences between data and quirc.  If different, 
	
	# Old steps:
	#  -Make backup
	#  -Offer to delete extraneous files
	#  -Copy over new files to ~/.quirc
	#  -Copy over new files to ~/.quirc/.originals (if originals dir exists)
	#  -Does the orignals dir exist?
	#   Yes:
	#    -See if the file in the quirc and original files differ.  If not, just
	#     copy over.
	#    -If so, check if different between data and quirc.  Display diff.
	#    -Display options like:
	# 	   -[L]eave It Alone
	# 	   -Leave Them [A]ll Alone
	# 	   -[O]verwrite It
	# 	   -Overwrite All Of [T]hem
	# 	   -[M]erge The Files.  If the merge does not encounter any conflicts,
	#       the file will be named `basename $file`.merged.  If it does
	#       encounter conflicts, it will be called `basename $file`.fixme.
	# 	   -Merge All The [F]iles
	# 	   -Merge With [R]eplacement If Safe.  If the merge does not encounter
	#       any conflicts, the file in ~/.quirc will simply be replaced.
	#       Otherwise, the file will be left as `basename $file`.fixme in
	#       ~/.quirc just as the first merge option."
	# 	   -Merge And Replace All If [S]afe."
	#   No:
	#    -See if the data files and the quirc files are different
	#    -If so, display diff and allow overwrite.
	#  -Replace the original files
	#  -chmod the quirc files
	log "Note that the advanced install does not perform checks to make sure"
	log "that all the necessary files have been upgraded.  You may need to"
	log "rerun the install using quirc -i, or edit some files manually if you"
	log "did not upgrade all the ncessary files."
    }
}

log ""
log "If QuIRC seems to malfunction after the install, it may be because your"
log "default_scripts or default_globalscripts variables are not loading the"
log "correct scripts.  You should either delete those lines from"
log "~/.quirc/dynamic.tcl, remove the file entirely, or make the scripts match"
log "up appropriately with those listed in ~/.quirc/variables.tcl."

# Install script here using $DATADIR

# echo "This script is fairly new, and relatively untested, so please let me"
# echo "know about any problems you may have with it.  Thanks."
# echo ""
# echo "This script uses four different directories.  Their names and locations"
# echo "are as follows:"
# echo "     Data = $DATADIR"
# echo "     Backup = $HOME/.quirc/.backup"
# echo "     Originals = $HOME/.quirc/.originals"
# echo "     ~/.quirc = $HOME/.quirc"
# echo "Data contains the unmodified versions of the data files of the latest"
# echo "installed version of QuIRC."
# echo "Backup is created and used by the script as mentioned later."
# echo "Originals contains files that are used to attempt automatic merging of"
# echo "script files upon upgrading."
# echo "~/.quirc is the dir where the currently in use scripts for QuIRC are stored."
# echo ""
# echo "This script essentially copies the files from the data dir into ~/.quirc.  It"
# echo "does cleaning and merging as well, if requested.  The first thing it does is"
# echo "make a back-up copy of the files in ~/.quirc into the backup dir.  If the script"
# echo "fails for some reason, it is suggested that you recover your files, if"
# echo "necessary, from that directory, before running quirc.inst again."
# echo ""
# echo "Press Enter to continue..."
# read input

# echo "Making backup copy of ~/.quirc/* in ~/.quirc/.backup (Ignore directory errors)"
# echo ""
# rm -rf $HOME/.quirc/.backup
# mkdir $HOME/.quirc/.backup
# cp -f $HOME/.quirc/* $HOME/.quirc/.backup
# echo ""

# echo "Checking through each file in ~/.quirc.  If there is not a corresponding"
# echo "file in the data directory, offer to delete it."
# echo ""
# deleteall="n"
# for file in $HOME/.quirc/*; do
#     if [ ! -d $file ]; then
# 	if [ ! -f $DATADIR/`basename $file` ]; then
# 	    if test "$deleteall" = "y"; then
# 		echo "Deleting $file."
# 		rm $file
# 		continue
# 	    fi
# 	    echo "`basename $file` does not appear in the original data directory."
# 	    echo -n "Would you like to delete it ([Y]es/[N]o/Yes [T]o All/No To [A]ll)? "
# 	    read input
# 	    if test "$input" = "a" || test "$input" = "A"; then
# 		break
# 	    fi
# 	    if test "$input" = "t" || test "$input" = "T"; then
# 		echo "Deleting $file."
# 		rm $file
# 		deleteall="y"
# 	    fi
# 	    if test "$input" = "y" || test "$input" = "Y"; then
# 		echo "Deleting $file."
# 		rm $file
# 	    fi
# 	fi
#     fi
# done
# echo ""

# echo "Checking for files that are in the data dir but not in ~/.quirc, and"
# echo "copying them over."
# echo ""
# for file in $DATADIR/*; do
#     if [ ! -f $HOME/.quirc/`basename $file` ]; then
#         echo "Copying $file to ~/.quirc/`basename $file`"
# 	cp $file $HOME/.quirc/`basename $file`
#     fi
# done
# echo ""

# echo "Checking for new data files that don't appear in the original files"
# echo "directory, and copying them over."
# echo ""
# for file in $DATADIR/*; do
#     if [ ! -f $HOME/.quirc/.originals/`basename $file` ]; then
# 	echo "Copying $file to ~/.quirc/.originals"
# 	cp $file $HOME/.quirc/.originals
#     fi
# done
# echo ""

# if [ ! -d $HOME/.quirc/.originals ]; then
#     echo "~/.quirc/.originals doesn't exist.  Going through the data files"
#     echo "from the data dir, and seeing if they exist in ~/.quirc and are"
#     echo "different from each other. If so, offer to overwrite.  The unified"
#     echo "diff will be displayed for any files that are different."
#     echo ""
#     echo "Press Enter to continue..."
#     read input
#     overwriteall="n"
#     for file in $DATADIR/*; do
# 	if ! diff -c $HOME/.quirc/`basename $file` $file > /dev/null; then
# 	    if test "$overwriteall" = "y"; then
# 		echo "Overwriting ~/.quirc/`basename $file` with $file."
# 		cp -f $file $HOME/.quirc
# 		continue
# 	    fi
# 	    if ! diff -c $HOME/.quirc/`basename $file` $file; then
# 		echo "~/.quirc/`basename $file` is different from $file."
# 		echo -n "Overwrite it ([Y]es/[N]o/Yes [T]o All/No To [A]ll)? "
# 		read input
# 		if test "$input" = "a" || test "$input" = "A"; then
# 		    break
# 		fi
# 		if test "$input" = "t" || test "$input" = "T"; then
# 		    echo "Overwriting ~/.quirc/`basename $file` with $file."
# 		    cp -f $file $HOME/.quirc
# 		    overwriteall="y"
# 		fi
# 		if test "$input" = "y" || test "$input" = "Y"; then
# 		    echo "Overwriting ~/.quirc/`basename $file` with $file."
# 		    cp -f $file $HOME/.quirc
# 		fi
# 	    fi
# 	fi
#     done
#     echo ""
# else
#     overwriteall="n"
#     mergeall="n"
#     safemergeall="n"
#     echo "For each file in the data dir, check if the corresponding file in"
#     echo "the ~/.quirc dir is different from the corresponding file in the"
#     echo "originals dir.  If it isn't, overwrite it with the ne data file.  If"
#     echo "it is, check for differences between the old file in ~/.quirc and"
#     echo "the new data file.  Unified diff results will be displayed if so."
#     echo ""
#     # Go through the data files.
#     for file in $DATADIR/*; do
# 	# If the original file is the same as the ~/.quirc file, copy the data
#         # file to ~/.quirc.
# 	if diff -c $HOME/.quirc/`basename $file` $HOME/.quirc/.originals/`basename $file` > /dev/null; then
# 	    echo "Copying $file to $HOME/.quirc."
# 	    cp -f $file $HOME/.quirc
# 	    continue
# 	fi
# 	# If the files in ~/.quirc and the data dir are different, do the "all"
#         # options presently set.
# 	if ! diff -c $HOME/.quirc/`basename $file` $file > /dev/null; then
# 	    if test "$overwriteall" = "y"; then
# 		echo "Overwriting ~/.quirc/`basename $file` with $file."
# 		cp -f $file $HOME/.quirc
# 		continue
# 	    fi
# 	    if test "$mergeall" = "y"; then
# 		if diff3 -m $HOME/.quirc/`basename $file` $HOME/.quirc/.originals/`basename $file` $file > $HOME/.quirc/`basename $file`.fixme; then
# 		    mv -f $HOME/.quirc/`basename $file`.fixme $HOME/.quirc/`basename $file`.merged
# 		    echo "Merged to ~/.quirc/`basename $file`.merged"
# 		else
# 		    echo "Merged to ~/.quirc/`basename $file`.fixme"
# 		fi
# 		continue
# 	    fi
# 	    if test "$safemergeall" = "y"; then
# 		if diff3 -m $HOME/.quirc/`basename $file` $HOME/.quirc/.originals/`basename $file` $file > $HOME/.quirc/`basename $file`.fixme; then
# 		    mv -f $HOME/.quirc/`basename $file`.fixme $HOME/.quirc/`basename $file`
# 		    echo "Merged to ~/.quirc/`basename $file`"
# 		else
# 		    echo "Merged to ~/.quirc/`basename $file`.fixme"
# 		fi
# 		continue
# 	    fi
# 	fi
# 	# Do a visible diff of the files in ~/.quirc and the data dir.  If
#         # different, offer alternatives.
# 	if ! diff -c $file $HOME/.quirc/`basename $file`; then
# 	    echo ""
# 	    echo "`basename $file` in the originals dir is different from that in the data dir."
# 	    echo "What would you like to do with the existing file in ~/.quirc?"
# 	    echo "  [L]eave It Alone"
# 	    echo "  Leave Them [A]ll Alone"
# 	    echo "  [O]verwrite It"
# 	    echo "  Overwrite All Of [T]hem"
# 	    echo "  [M]erge The Files.  If the merge does not encounter any conflicts, the"
# 	    echo "    file will be named `basename $file`.merged.  If it does encounter conflicts,"
# 	    echo "    it will be called `basename $file`.fixme."
# 	    echo "  Merge All The [F]iles"
# 	    echo "  Merge With [R]eplacement If Safe.  If the merge does not encounter any"
# 	    echo "    conflicts, the file in ~/.quirc will simply be replaced.  Otherwise, the"
# 	    echo "    file will be left as `basename $file`.fixme in ~/.quirc just as the first"
# 	    echo "    merge option."
# 	    echo "  Merge And Replace All If [S]afe."
# 	    echo -n "> "
# 	    read input
# 	    if test "$input" = "a" || test "$input" = "A"; then
# 		break
# 	    fi
# 	    if test "$input" = "o" || test "$input" = "O"; then
# 		echo "Overwriting ~/.quirc/`basename $file` with $file."
# 		cp -f $file $HOME/.quirc
# 	    fi
# 	    if test "$input" = "t" || test "$input" = "T"; then
# 		overwriteall="y"
# 		echo -n "Overwriting ~/.quirc/`basename $file` with $file."
# 		cp -f $file $HOME/.quirc
# 	    fi
# 	    if test "$input" = "m" || test "$input" = "M"; then
# 		if diff3 -m $HOME/.quirc/`basename $file` $HOME/.quirc/.originals/`basename $file` $file > $HOME/.quirc/`basename $file`.fixme; then
# 		    mv -f $HOME/.quirc/`basename $file`.fixme $HOME/.quirc/`basename $file`.merged
# 		    echo "Merged to ~/.quirc/`basename $file`.merged"
# 		else
# 		    echo "Merged to ~/.quirc/`basename $file`.fixme"
# 		fi
# 	    fi
# 	    if test "$input" = "f" || test "$input" = "F"; then
# 		if diff3 -m $HOME/.quirc/`basename $file` $HOME/.quirc/.originals/`basename $file` $file > $HOME/.quirc/`basename $file`.fixme; then
# 		    mv -f $HOME/.quirc/`basename $file`.fixme $HOME/.quirc/`basename $file`.merged
# 		    echo -n "Merged to ~/.quirc/`basename $file`.merged"
# 		else
# 		    echo -n "Merged to ~/.quirc/`basename $file`.fixme"
# 		fi
# 		mergeall="y"
# 	    fi
# 	    if test "$input" = "r" || test "$input" = "R"; then
# 		if diff3 -m $HOME/.quirc/`basename $file` $HOME/.quirc/.originals/`basename $file` $file > $HOME/.quirc/`basename $file`.fixme; then
# 		    mv -f $HOME/.quirc/`basename $file`.fixme $HOME/.quirc/`basename $file`
# 		    echo "Merged to ~/.quirc/`basename $file`"
# 		else
# 		    echo "Merged to ~/.quirc/`basename $file`.fixme"
# 		fi
# 	    fi
# 	    if test "$input" = "s" || test "$input" = "S"; then
# 		if diff3 -m $HOME/.quirc/`basename $file` $HOME/.quirc/.originals/`basename $file` $file > $HOME/.quirc/`basename $file`.fixme; then
# 		    mv -f $HOME/.quirc/`basename $file`.fixme $HOME/.quirc/`basename $file`
# 		    echo -n "Merged to ~/.quirc/`basename $file`"
# 		else
# 		    echo -n "Merged to ~/.quirc/`basename $file`.fixme"
# 		fi
# 		safemergeall="y"
# 	    fi
# 	    echo ""
# 	else
# 	    echo "Copying $file to ~/.quirc"
# 	    cp $file $HOME/.quirc
# 	fi
#     done
#     echo ""
# fi

# echo "Replace the files in ~/.quirc/.originals with the data files for the"
# echo -n "version which you are installing ([Y]es/[N]o)? "
# read input
# if test "$input" = "y" || test "$input" = "Y"; then
#     if [ ! -d $HOME/.quirc/.originals ]; then
# 	mkdir $HOME/.quirc/.originals
#     fi
#     rm $HOME/.quirc/.originals/*
#     cp $DATADIR/* $HOME/.quirc/.originals
# fi
# echo ""

# #diff3 MINE OLDER YOURS
# #diff3 -m file.custom file.quirc.older file.quirc.newer
# #$? is 0 if succesful, 1 if not.

# echo "Running chmod on the necessary data files in ~/.quirc in order to"
# echo "ensure that they are useable."
# for file in $DATADIR/*; do
#   chmod -f u+r $HOME/.quirc/`basename $file`
# done
# chmod -f u+x $HOME/.quirc/quedit
# echo ""

# echo "If you got any files that were merged to .fixme files, you will need to"
# echo "edit those files, locate the conflicts, and resolve those manually."
# echo "For more information about the merging process, see your documentation"
# echo "for diff3."
# echo ""
# echo "To start QuIRC, enter quirc at the command line."

# #    echo "Checking for original files that aren't included in the new set of"
# #    echo "data files, and deleting them."
# #    echo ""
# #    for file in $HOME/.quirc/.originals/*; do
# #	if [ ! -f $DATADIR/`basename $file` ]; then
# #	    echo "Deleting $file."
# #	    rm $file
# #	fi
# #    done
# #    echo ""
