/*****************************************************************************
 * $Id: audio-alsa.c,v 1.7 2004/09/19 16:20:31 alainjj Exp $
 * Program under GNU General Public License (see ../COPYING)
 *****************************************************************************/

#include "config.h"
#ifdef HAVE_ALSA
//#define ALSA_PCM_OLD_HW_PARAMS_API
//#define ALSA_PCM_OLD_SW_PARAMS_API
# if defined(HAVE_ALSA_ASOUNDLIB_H)
#   include <alsa/asoundlib.h>
# elif defined (HAVE_SYS_ASOUNDLIB_H)
#   include <sys/asoundlib.h>
# endif

#if defined(SND_LIB_MAJOR) && SND_LIB_MAJOR >= 1
#define ALSA_NEW_PARAMS
#else
#undef ALSA_NEW_PARAMS
#endif

#include <stdio.h>
#include "audio.h"
extern int debug;

/* "default" (in general equivalent to "hw:0,0")
   depends on /usr/share/alsa/alsa.conf:/etc/asound.conf:~/.asoundrc
   "hw:m,n"  means the n-th subdevice of the m-th card
*/
static char *card = "default";
#define CHECKALSA(e,msg) \
 if((e)<0) { \
   fprintf(stderr, "%s: %s\n", msg, snd_strerror (err)); \
   return -1; \
 }

static snd_pcm_t *handle_in = NULL, *handle_out = NULL;
static int sizefmt;
static int alsa_check(void) {
  int err;
  snd_pcm_t *handle;
  if(audio_dev) card=audio_dev;
  err=snd_pcm_open (&handle, card, SND_PCM_STREAM_PLAYBACK, SND_PCM_NONBLOCK);
  if(err<0)
    err= snd_pcm_open (&handle, card, SND_PCM_STREAM_CAPTURE, SND_PCM_NONBLOCK);
  if(err<0) return 0;
  snd_pcm_close (handle);
  return 1;
}

static int alsa_configure(snd_pcm_t *handle,unsigned int *rate,
			  int fmtsample, int nchannels, int nfrags, int fragsize) {
  snd_pcm_hw_params_t *hw_params = NULL;
  int err;
  err = snd_pcm_nonblock(handle, 0);
  CHECKALSA(err,"snd_pcm_nonblock");
  snd_pcm_hw_params_alloca(&hw_params);
  err = snd_pcm_hw_params_any(handle, hw_params);
  CHECKALSA(err,"snd_pcm_hw_params_any");
  err = snd_pcm_hw_params_set_format
    (handle, hw_params, 
     fmtsample==AUDIO_U8 ? SND_PCM_FORMAT_U8 : SND_PCM_FORMAT_S16_LE);
  CHECKALSA(err,"snd_pcm_hw_params_set_format");
  err = snd_pcm_hw_params_set_channels(handle, hw_params, nchannels);
  CHECKALSA(err,"snd_pcm_hw_params_set_channels");
#ifdef ALSA_NEW_PARAMS
  err = snd_pcm_hw_params_set_rate_near(handle, hw_params, rate, 0);
#else
  err = snd_pcm_hw_params_set_rate_near(handle, hw_params, *rate, 0);
#endif
  CHECKALSA(err,"snd_pcm_hw_params_set_rate_near");
  err = snd_pcm_hw_params_set_access(handle, hw_params,SND_PCM_ACCESS_RW_INTERLEAVED);
  CHECKALSA(err,"snd_pcm_hw_params_set_access");
#ifdef ALSA_NEW_PARAMS
  err = snd_pcm_hw_params_set_period_size_near(handle, hw_params,(snd_pcm_uframes_t*)&fragsize, 0);
#else
  err = snd_pcm_hw_params_set_period_size_near(handle, hw_params, fragsize, 0);
#endif
  CHECKALSA(err,"snd_pcm_hw_params_set_period_size_near");
#ifdef ALSA_NEW_PARAMS
  err = snd_pcm_hw_params_set_periods_near(handle, hw_params, &nfrags, 0);
#else
  err = snd_pcm_hw_params_set_periods_near(handle, hw_params, nfrags, 0);
#endif
  CHECKALSA(err,"snd_pcm_hw_params_set_periods_near");
  err = snd_pcm_hw_params(handle, hw_params);
  CHECKALSA(err,"snd_pcm_hw_params");
#ifndef ALSA_NEW_PARAMS
  *rate = snd_pcm_hw_params_get_rate(hw_params, 0);
#endif
  //snd_pcm_hw_params_free(hw_params); //?
  return 0;
}

static int alsa_open(int mode, int fmtsample, int nchannels, int freq,
		     int nfrags, int fragsize) {
  int err, rate_in = freq, rate_out = freq;
  if(audio_dev) card=audio_dev;
  if(mode&AUDIO_WO) {
    err = snd_pcm_open (&handle_out, card, SND_PCM_STREAM_PLAYBACK, 0);
    CHECKALSA(err, "snd_pcm_open out");
    if(alsa_configure(handle_out, &rate_out, fmtsample, nchannels, nfrags, fragsize )<0)
      return -1;
  }
  if(mode&AUDIO_RO) {
    err = snd_pcm_open (&handle_in, card, SND_PCM_STREAM_CAPTURE, 0);
    CHECKALSA(err, "snd_pcm_open in");
    if(alsa_configure(handle_in, &rate_in, fmtsample, nchannels, nfrags, fragsize)<0)
      return -1;
  }
  if(mode==AUDIO_RW) {
    if(rate_in != rate_out) {
      if(debug)
	fprintf(stderr, "ALSA RATE_IN=%d RATE_OUT=%d\n",rate_in, rate_out);
      if(rate_out != freq) {
	rate_in = rate_out;
	alsa_configure(handle_in, &rate_in, fmtsample, nchannels, nfrags, fragsize);
      } else {
	rate_out = rate_in;
	alsa_configure(handle_out, &rate_out, fmtsample, nchannels, nfrags, fragsize);
      }
      if(debug)
	fprintf(stderr, "ALSA RATE_IN=%d RATE_OUT=%d\n",rate_in, rate_out);
    }
  }
  if(mode&AUDIO_WO) {
    err = snd_pcm_prepare(handle_out);
    CHECKALSA(err, "snd_pcm_prepare out");
  }
  if(mode&AUDIO_RO) {
    err = snd_pcm_prepare(handle_in);
    CHECKALSA(err, "snd_pcm_prepare in");
  }
  if(fmtsample==AUDIO_U8) sizefmt=nchannels; else sizefmt=2*nchannels;
  if(debug) {
    snd_output_t *log;
    snd_output_stdio_attach(&log, stderr, 0);
    if(mode&AUDIO_RO) {
      snd_pcm_dump_hw_setup(handle_in, log);
      snd_pcm_dump_sw_setup(handle_in, log);
    }
    if(mode&AUDIO_WO) {
      snd_pcm_dump_hw_setup(handle_out, log);
      snd_pcm_dump_sw_setup(handle_out, log);
    }
    snd_output_close(log);
  }
  return 0;
}

static int alsa_close(void) {
  if(handle_in) snd_pcm_close (handle_in);
  if(handle_out) snd_pcm_close (handle_out);
  handle_in = handle_out = NULL;
  return 0;
}

static int alsa_read(void *buf, int n) {
  int ret;
  int nbframes = n/sizefmt;
  while (1) {
    snd_pcm_state_t  state = snd_pcm_state(handle_in);
    if(debug>=2) fprintf(stderr,"state read=%d\n", state);
    if (state == SND_PCM_STATE_XRUN || 
	state == SND_PCM_STATE_SUSPENDED) {
      if(debug) fprintf(stderr, "alsa_read: snd_pcm_prepare\n");
      ret = snd_pcm_prepare(handle_in);
      if(debug>=2) fprintf(stderr, "ret_prepare ret=%d\n", ret);
      if (ret < 0)
	break;
    }
    ret = snd_pcm_readi (handle_in, buf, nbframes);
    if(debug>=2) fprintf(stderr,"ret_readi=%d\n",ret);
    if (ret != -EPIPE && ret != -ESTRPIPE)
      break;
  }
  if(ret>0) ret*=sizefmt;
  return ret;
}

static int alsa_write(void *buf, int n) {
  int ret;
  int nbframes = n/sizefmt;
  while (1) {
    snd_pcm_state_t  state = snd_pcm_state(handle_out);
    if(debug>=2) fprintf(stderr,"state write=%d\n", state);
    if (state == SND_PCM_STATE_XRUN ||
	state == SND_PCM_STATE_SUSPENDED) {
      if(debug) fprintf(stderr, "alsa_write: snd_pcm_prepare\n");
      ret = snd_pcm_prepare(handle_out);
      if(debug>=2) fprintf(stderr, "ret_prepare write=%d\n",ret);
      if (ret < 0)
	break;
    }
    ret = snd_pcm_writei(handle_out, buf, nbframes);
    if(debug>=2) fprintf(stderr, "ret_writei=%d\n",ret);
    if (ret != -EPIPE && ret != -ESTRPIPE)
      break;
  }
  if(ret>0) ret*=sizefmt;
  return ret;
}


struct AUDRIVER audio_alsa = {
  "alsa",
  alsa_check,
  alsa_open,
  alsa_close,
  alsa_read,
  alsa_write,
};
#endif
