/*
   NAND flash resize utility for Sharp Zaurus SL-C7x0/860 and SL-5600
   version 1.0
   Copyright 2004 Alexander Chukov <sash@cacko.biz>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 */

#define CONFIG_MTD_NAND_LOGICAL_ADDRESS_ACCESS

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>
#include <string.h>

#include <asm/types.h>
#include <linux/mtd/mtd.h>

static int open_mtd(void)
{
	int fd;
	mtd_info_t meminfo;

	/* Open the device */
	if((fd = open("/dev/mtd1", O_RDWR)) == -1) {
	    if((fd = open("/dev/mtd/1", O_RDWR)) == -1) {
		perror("open flash");
		exit(1);
	    }
	}

	/* Fill in MTD device capability structure */  
	if(ioctl(fd, MEMGETINFO, &meminfo) != 0) {
		perror("MEMGETINFO");
		close(fd);
		exit(1);
	}

	/* Make sure device page sizes are valid */
	if( !(meminfo.oobsize == 64 && meminfo.oobblock == 2048) &&
	    !(meminfo.oobsize == 16 && meminfo.oobblock == 512 ) &&
	    !(meminfo.oobsize ==  8 && meminfo.oobblock == 256 ) ) {
		printf("Unknown flash (not normal NAND)\n");
		close(fd);
		exit(1);
	}

	return fd;
}

static void read_mtd(int mfd, loff_t offset, size_t len, void *buf)
{
	struct read_laddr_info_user rinfo;
	
	rinfo.from = offset;
	rinfo.len = len;
	rinfo.buf = (u_char *) buf;
	if (ioctl(mfd, MEMREADLADDR, &rinfo) != 0) {
		perror("ioctl(MEMREADLADDR)");
		close(mfd);
		exit(1);
	}
}

static void write_mtd(int mfd, loff_t offset, size_t len, void *buf)
{
	struct write_laddr_info_user winfo;
	
	winfo.to = offset;
	winfo.len = len;
	winfo.buf = (u_char *) buf;
	if (ioctl(mfd, MEMWRITELADDR, &winfo) != 0) {
		perror("ioctl(MEMWRITELADDR)");
		close(mfd);
		exit(1);
	}
}

int main(int argc, char *argv[])
{
	int mfd;
	unsigned int boot_partition;
	unsigned int fsro_partition;
	unsigned int fsrw_partition;

	mfd = open_mtd();
	read_mtd(mfd, 0x60004, 4, &boot_partition);
	read_mtd(mfd, 0x60014, 4, &fsro_partition);
	read_mtd(mfd, 0x60024, 4, &fsrw_partition);
	close(mfd);

	if (argc > 1) {
		int new_fsro_size, new_fsro_partition;
		
		if (strcmp(argv[1],"--h") == 0) {
			printf("Usage:\n%s [new root partition size]\n", argv[0]);
			return 0;
		}
		
		new_fsro_size = atoi(argv[1]);
		if (new_fsro_size < 1) {
		    printf("value must be > 0\n");
		    return 1;
		}
		
		new_fsro_partition = new_fsro_size * 1024 * 1024 + boot_partition;

		mfd = open_mtd();
		write_mtd(mfd, 0x60014, 4, &new_fsro_partition);
		write_mtd(mfd, 0x60020, 4, &new_fsro_partition);
		write_mtd(mfd, 0x64014, 4, &new_fsro_partition);
		write_mtd(mfd, 0x64020, 4, &new_fsro_partition);
		close(mfd);

		printf("old FSRO size: %d MB\n", (fsro_partition - boot_partition)/(1024*1024));
		printf("new FSRO size: %d MB\n", (new_fsro_partition - boot_partition)/(1024*1024));
		printf("Partition resized, please reboot...\n");
	} else {
/*
		printf("ROOT end: %08x\n", boot_partition);
		printf("FSRO end: %08x\n", fsro_partition);
		printf("FSRW end: %08x (for SL-C700/750 and SL-5600 only)\n", fsrw_partition);
 */
		printf("FSRO size: %d MB\n", (fsro_partition - boot_partition)/(1024*1024));
	}

	return 0;
}
