/** @file scim_rawcode_imengine.cpp
 * implementation of class RawCodeInstance.
 */

/*
 * Smart Common Input Method
 * 
 * Copyright (c) 2002-2005 James Su <suzhe@tsinghua.org.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_rawcode_imengine.cpp,v 1.6 2005/03/16 15:33:06 suzhe Exp $
 *
 */

#define Uses_SCIM_IMENGINE
#define Uses_SCIM_ICONV
#define Uses_SCIM_CONFIG_BASE
#define Uses_SCIM_CONFIG_PATH
#include "scim_private.h"
#include "scim.h"
#include "scim_rawcode_imengine.h"

#define scim_module_init rawcode_LTX_scim_module_init
#define scim_module_exit rawcode_LTX_scim_module_exit
#define scim_imengine_module_init rawcode_LTX_scim_imengine_module_init
#define scim_imengine_module_create_factory rawcode_LTX_scim_imengine_module_create_factory

#define SCIM_CONFIG_IMENGINE_RAWCODE_LANGUAGES "/IMEngine/RawCode/Languages"

#define SCIM_PROP_STATUS                       "/IMEngine/RawCode/Status"

#define SCIM_RAWCODE_ICON_FILE                 (SCIM_ICONDIR "/rawcode.png")

using namespace scim;

static Pointer <RawCodeFactory> _scim_rawcode_factory;
static ConfigPointer _scim_config;

static const char * _DEFAULT_LANGUAGES = N_(
    "zh_CN,zh_TW,zh_HK,zh_SG,ja_JP,ko_KR");

extern "C" {
    void scim_module_init (void)
    {
    }

    void scim_module_exit (void)
    {
        _scim_rawcode_factory.reset ();
        _scim_config.reset ();
    }

    unsigned int scim_imengine_module_init (const ConfigPointer &config)
    {
        _scim_config = config;
        return 1;
    }

    IMEngineFactoryPointer scim_imengine_module_create_factory (unsigned int factory)
    {
        String languages;

        if (factory != 0) return NULL;

        if (!_scim_config.null ()) {
            languages = _scim_config->read (
                        String (SCIM_CONFIG_IMENGINE_RAWCODE_LANGUAGES),
                        String ("default"));
        } else {
            languages = String ("default");
        }

        if (_scim_rawcode_factory.null ()) {
            _scim_rawcode_factory =
                new RawCodeFactory (utf8_mbstowcs (String (_("RAW CODE"))), languages);
        }
        return _scim_rawcode_factory;
    }
}

// implementation of RawCode
RawCodeFactory::RawCodeFactory ()
{
    m_name = utf8_mbstowcs (_("RAW CODE"));
    set_languages (String (_(_DEFAULT_LANGUAGES)));
}

RawCodeFactory::RawCodeFactory (const WideString& name,
                                const String& languages)
{
    // do not allow too long name
    if (name.length () <= 8)
        m_name = name;
    else
        m_name.assign (name.begin (), name.begin () + 8);

    if (languages == String ("default"))
        set_languages (String (_(_DEFAULT_LANGUAGES)));
    else
        set_languages (languages);
}

RawCodeFactory::~RawCodeFactory ()
{
}

WideString
RawCodeFactory::get_name () const
{
    return m_name;
}

WideString
RawCodeFactory::get_authors () const
{
    return utf8_mbstowcs (String (
                _("(C) 2002-2005 James Su <suzhe@tsinghua.org.cn>")));
}

WideString
RawCodeFactory::get_credits () const
{
    return WideString ();
}

WideString
RawCodeFactory::get_help () const
{
    return utf8_mbstowcs (String (_(
                            "Hot Keys:\n\n"
                            "  Control+u:\n"
                            "    switch between Multibyte encoding and Unicode.\n\n"
                            "  Control+comma:\n"
                            "    switch between full/half width punctuation mode.\n\n"
                            "  Shift+space:\n"
                            "    switch between full/half width letter mode.\n\n"
                            "  Esc:\n"
                            "    reset the input method.\n")));
}

String
RawCodeFactory::get_uuid () const
{
    return String ("6e029d75-ef65-42a8-848e-332e63d70f9c");
}

String
RawCodeFactory::get_icon_file () const
{
    return String (SCIM_RAWCODE_ICON_FILE);
}

String
RawCodeFactory::get_language () const
{
    return scim_validate_language ("other");
}

IMEngineInstancePointer
RawCodeFactory::create_instance (const String& encoding, int id)
{
    return new RawCodeInstance (this, encoding, id);
}

int
RawCodeFactory::get_maxlen (const String &encoding)
{
    std::vector <String> locales;

    scim_split_string_list (locales, get_locales ());

    for (unsigned int i=0; i<locales.size (); ++i)
        if (scim_get_locale_encoding (locales [i]) == encoding)
            return scim_get_locale_maxlen (locales [i]);
    return 1;
}

// implementation of RawCodeInstance
RawCodeInstance::RawCodeInstance (RawCodeFactory *factory,
                                              const String& encoding,
                                              int id)
    : IMEngineInstanceBase (factory, encoding, id),
      m_factory (factory),
      m_status_property (SCIM_PROP_STATUS, _("Unicode"), "", _("The status of the current input method. Click to change it.")),
      m_unicode (true),
      m_forward (false),
      m_focused (false),
      m_max_preedit_len (4),
      m_iconv (encoding)
{
}

RawCodeInstance::~RawCodeInstance ()
{
}

bool
RawCodeInstance::process_key_event (const KeyEvent& key)
{
    if (!m_focused) return false;

    // capture the state switch key events
    if ( ((key.code == SCIM_KEY_Alt_L || key.code == SCIM_KEY_Alt_R) &&
            key.is_shift_down ()) ||
         ((key.code == SCIM_KEY_Shift_L || key.code == SCIM_KEY_Shift_R) && 
            (key.is_alt_down () || key.is_control_down ())) &&
         key.is_key_press ()) {
        m_forward = !m_forward;
        refresh_status_property ();
        reset ();
        return true;
    }

    if (key.is_key_release ()) return true;

    if (!m_forward) {
        // switch between unicode and native code mode
        if ((key.code == SCIM_KEY_u || key.code == SCIM_KEY_U) &&
             key.is_control_down ()) {
            m_unicode = !m_unicode;
            refresh_status_property ();
            reset ();
            return true;
        }

        //reset key
        if (key.code == SCIM_KEY_Escape && key.mask == 0) {
            reset ();
            return true;
        }

        //delete key
        if (key.code == SCIM_KEY_BackSpace && key.mask == 0 &&
            m_preedit_string.size () != 0) {
            m_preedit_string.erase (m_preedit_string.length () - 1, 1);
            update_preedit_string (m_preedit_string);
            update_preedit_caret (m_preedit_string.length ());
            process_preedit_string ();
            return true;
        }

        // valid keys
        if (((key.code >= SCIM_KEY_0 && key.code <= SCIM_KEY_9) ||
             (key.code >= SCIM_KEY_A && key.code <= SCIM_KEY_F) ||
             (key.code >= SCIM_KEY_a && key.code <= SCIM_KEY_f)) &&
            (key.mask == 0 || key.is_shift_down ()) &&
            m_preedit_string.length () < m_max_preedit_len) {

            if (m_preedit_string.length () == 0)
                show_preedit_string ();

            ucs4_t ascii = (ucs4_t) tolower (key.get_ascii_code ());
            m_preedit_string.push_back (ascii);
            update_preedit_string (m_preedit_string);
            update_preedit_caret (m_preedit_string.length ());
            process_preedit_string ();
            return true;
        }

        //page up key.
        if (key.code == SCIM_KEY_comma && key.mask == 0 &&
            m_preedit_string.length () && m_lookup_table.number_of_candidates ()) {
            m_lookup_table.page_up ();
            if (create_lookup_table (m_lookup_table.get_current_page_start ()) > 0)
                update_lookup_table (m_lookup_table);
            else
                hide_lookup_table ();
        }

        //page down key.
        if (key.code == SCIM_KEY_period && key.mask == 0 &&
            m_preedit_string.length () && m_lookup_table.number_of_candidates ()) {
            m_lookup_table.page_down ();
            if (create_lookup_table (m_lookup_table.get_current_page_start ()) > 0)
                update_lookup_table (m_lookup_table);
            else
                hide_lookup_table ();
        }
    }

    //other keys is not allowed when preediting
    if (m_preedit_string.length ())
        return true;

    return false;
}

void
RawCodeInstance::select_candidate (unsigned int item)
{
    WideString label = m_lookup_table.get_candidate_label (item);
    KeyEvent key ((int) label [0], 0);
    process_key_event (key);
}

void
RawCodeInstance::update_lookup_table_page_size (unsigned int page_size)
{
    if (page_size > 0)
        m_lookup_table.set_page_size (page_size);
}

void
RawCodeInstance::lookup_table_page_up ()
{
    if (m_preedit_string.length () && m_lookup_table.number_of_candidates ()) {
        m_lookup_table.page_up ();
        if (create_lookup_table (m_lookup_table.get_current_page_start ()) > 0) {
            update_lookup_table (m_lookup_table);
            show_lookup_table ();
        } else {
            hide_lookup_table ();
        }
    }
}

void
RawCodeInstance::lookup_table_page_down ()
{
    if (m_preedit_string.length () && m_lookup_table.number_of_candidates ()) {
        m_lookup_table.page_down ();
        if (create_lookup_table (m_lookup_table.get_current_page_start ()) > 0) {
            update_lookup_table (m_lookup_table);
            show_lookup_table ();
        } else {
            hide_lookup_table ();
        }
    }
}

void
RawCodeInstance::move_preedit_caret (unsigned int /*pos*/)
{
}

void
RawCodeInstance::reset ()
{
    m_preedit_string = WideString ();

    if (m_unicode) m_max_preedit_len = 4;
    else if (m_factory)
        m_max_preedit_len = m_factory->get_maxlen (get_encoding ()) * 2;

    m_iconv.set_encoding (get_encoding ());
    m_lookup_table.clear ();

    hide_lookup_table ();
    hide_preedit_string ();
}

void
RawCodeInstance::focus_in ()
{
    m_focused = true;

    initialize_properties ();

    if (m_preedit_string.length ()) {
        update_preedit_string (m_preedit_string);
        update_preedit_caret (m_preedit_string.length ());
        show_preedit_string ();
        if (m_lookup_table.number_of_candidates ()) {
            update_lookup_table (m_lookup_table);
            show_lookup_table ();
        }
    }
}

void
RawCodeInstance::focus_out ()
{
    m_focused = false;
}

void
RawCodeInstance::trigger_property (const String &property)
{
    if (property == SCIM_PROP_STATUS) {
        if (m_forward)
            m_forward = false;
        else if (m_unicode)
            m_unicode = false;
        else
            m_unicode = m_forward = true;

        refresh_status_property ();
    }
}

void
RawCodeInstance::initialize_properties ()
{
    PropertyList proplist;

    proplist.push_back (m_status_property);

    register_properties (proplist);
}

void
RawCodeInstance::refresh_status_property ()
{
    if (m_focused) {
        if (m_forward)
            m_status_property.set_label (_("En"));
        else if (m_unicode)
            m_status_property.set_label (_("Unicode"));
        else
            m_status_property.set_label (get_encoding ());

        update_property (m_status_property);
    }
}

void
RawCodeInstance::process_preedit_string ()
{
    if (m_preedit_string.length () == 0) {
        hide_preedit_string ();
        hide_lookup_table ();
        return;
    }

    if (m_unicode) {
        if (m_preedit_string.length () == 3 &&
            create_lookup_table () > 0) {
            update_lookup_table (m_lookup_table);
        } else if (m_preedit_string.length () == 4) {
            WideString str;
            ucs4_t code = get_unicode_value (m_preedit_string);

            // If code is valid under current encoding,
            // then commit it.
            if (m_iconv.test_convert (&code, 1)) {
                str.push_back (code);
                m_preedit_string = WideString ();
                m_lookup_table.clear ();

                hide_preedit_string ();
                commit_string (str);
            }
        } else if (m_lookup_table.number_of_candidates ()){
            m_lookup_table.clear ();
        }
    } else {
        String code = get_multibyte_string (m_preedit_string);
        WideString str;

        // convert ok, then commit.
        if (m_iconv.convert (str, code) && str.length () > 0 && str [0] >= 128) {
            m_preedit_string = WideString ();
            m_lookup_table.clear ();

            hide_preedit_string ();
            commit_string (str);
        } else if (create_lookup_table () > 0) {
            update_lookup_table (m_lookup_table);
        }
    }

    if (m_lookup_table.number_of_candidates ())
        show_lookup_table ();
    else
        hide_lookup_table ();
}

inline static int ascii_to_hex (int ascii)
{
    if (ascii >= '0' && ascii <= '9')
        return ascii - '0';
    else if (ascii >= 'a' && ascii <= 'f')
        return ascii - 'a' + 10;
    else if (ascii >= 'A' && ascii <= 'F')
        return ascii - 'A' + 10;
    return 0;
}
inline static int hex_to_ascii (int hex)
{
    hex %= 16;

    if (hex >= 0 && hex <= 9)
        return hex + '0';

    return hex - 10 + 'a';
}

String
RawCodeInstance::get_multibyte_string (const WideString & preedit)
{
    String str;
    char ch = 0;

    if (preedit.length () == 0)
        return str;

    for (unsigned int i=0; i<preedit.length (); ++i) {
        if (i % 2 == 0)
            ch = (char) ascii_to_hex ((int) preedit [i]) & 0x0f;
        else {
            ch = (ch << 4) | ((char) ascii_to_hex ((int) preedit [i]) & 0x0f);
            str.push_back (ch);
            ch = 0;
        }
    }

    if (ch != 0)
        str.push_back (ch);

    return str;
}

ucs4_t
RawCodeInstance::get_unicode_value (const WideString &preedit)
{
    ucs4_t code = 0;
    for (unsigned int i=0; i<preedit.length (); ++i) {
        code = (code << 4) | (ascii_to_hex ((int) preedit [i]) & 0x0f);
    }
    return code;
}

int
RawCodeInstance::create_lookup_table (int start)
{
    std::vector<WideString> labels;
    String mbs_code;
    ucs4_t ucs_code;
    WideString trail;
    WideString str;

    m_lookup_table.clear ();

    trail.push_back (0);

    for (int i=start; i<16; ++i) {
        trail [0] = (ucs4_t) hex_to_ascii (i);
        if (m_unicode) {
            ucs_code = get_unicode_value (m_preedit_string + trail);
            if (m_iconv.test_convert (&ucs_code, 1)) {
                labels.push_back (trail);
                m_lookup_table.append_candidate (ucs_code);
            }
        } else {
            mbs_code = get_multibyte_string (m_preedit_string + trail);
            if (m_iconv.convert (str, mbs_code) && str.length () > 0 && str [0] >= 128) {
                labels.push_back (trail);
                m_lookup_table.append_candidate (str [0]);
            }
        }
    }

    m_lookup_table.set_page_size (labels.size ());
    m_lookup_table.set_candidate_labels (labels);

    return labels.size ();
}

/*
vi:ts=4:nowrap:ai:expandtab
*/
