/*
  qpegps is a program for displaying a map centered at the current longitude/
  latitude as read from a gps receiver.

  Copyright (C) 2002 Ralf Haselmeier <Ralf.Haselmeier@gmx.de>
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 
*/
#include "mapdisp.h"

MapDisp::MapDisp(Qpegps *appl, QSortedList<MapBase> *mapList, QWidget *parent,
                 const char *name, WFlags fl):
    QWidget (parent, name, fl)
{
    application = appl;
    gpsData = application->gpsData;

    mapdisp = new QPixmap;
    map = new QImage;

    maps = mapList;

    actmap = 0;
    createMap();
    setBackgroundMode(NoBackground);

    selectedScale = 0;
    actMapList.setAutoDelete(FALSE);

    connect( this, SIGNAL(lessDetail()), SLOT(chooseLessDetailedMap()) );
    connect( this, SIGNAL(moreDetail()), SLOT(chooseMoreDetailedMap()) );
    connect( this, SIGNAL(debugMaps()), SLOT(showAvailableMaps()) );

    setFocusPolicy(QWidget::StrongFocus);
}


MapDisp::~MapDisp()
{
}

double MapDisp::coverage(MapBase *aMap, double x, double y, int width, int height)
{
    // determines the area on the display, which is not covered by aMap
    // 0 = map fills the display
    // <0 = -1 * pixels without map

    double coverX, coverY;

    if(x > aMap->mapSizeX - x)
	coverX = aMap->mapSizeX - x;
    else
	coverX = x;
    if(y > aMap->mapSizeY - y)
	coverY = aMap->mapSizeY - y;
    else
	coverY = y;
    coverX -= width/2;
    coverY -= height/2;
    if(coverX > 0)
	coverX = 0;
    else
	coverX *= height;
    if(coverY > 0)
	coverY = 0;
    else
	coverY *= width;

    return coverX + coverY;
}

void MapDisp::chooseLessDetailedMap()
{
    MapBase *aMap;
   
    if(actmap)
    {
	if(actmap->scale < actMapList.last()->scale)
	{
	    aMap=actMapList.first();
	    while(aMap && (aMap->scale < /*actmap->scale*/selectedScale + 1))
	    {
		aMap = actMapList.next();
	    }
	    if(aMap)
		selectedScale = aMap->scale;
	    else
		selectedScale = actmap->scale;
	}
	else
	    selectedScale = actmap->scale;
    }
}

void MapDisp::chooseMoreDetailedMap()
{
    MapBase *aMap;

    if(actmap && actmap->scale > actMapList.first()->scale)
    {
	aMap = actMapList.last();
	while(aMap && (aMap->scale > /*actmap->scale*/selectedScale - 1))
	    aMap = actMapList.prev();
	if(aMap)
	    selectedScale = aMap->scale;
    }
    else
	selectedScale = 0;
}

void MapDisp::showAvailableMaps()
{
    MapBase *aMap;

    qDebug(tr("Maps at current position:"));
    aMap = actMapList.first();
    while(aMap)
    {
	qDebug(tr("%1 %2").arg(aMap->scale).arg(aMap->name));
	aMap = actMapList.next();
    }
}

void MapDisp::clearActMapList()
{
    actMapList.clear();
    actmap = 0;
}

void MapDisp::createMap()
{
    QRect tempRect;
    QPainter painter;
    bool mapavailable=FALSE, mapchanged=FALSE;
    double lg,lt;
    double xwp=0,ywp=0;
    int tx,ty,tw,th;
    double xc,yc, cover, scale,tcover;
    MapBase *aMap,*theMap;
    lg = gpsData->currPos.longitude * M_PI / 180.0;
    lt = gpsData->currPos.latitude * M_PI / 180.0;

    mapdisp->resize(geometry().size());

    // find map
    // and get position in map coordinates

    // check if the actual maps are still valid
    aMap = actMapList.first();
    while(aMap)
    {
        if(!aMap->calcxy(&xc, &yc, lg, lt))
        {
	    actMapList.take();
	    aMap = actMapList.current(); //next item
	}
	else
	    aMap = actMapList.next();
    }
    // check all maps, if actMapList is empty !
    if(actMapList.isEmpty())
    {
	aMap = maps->first ();
	while(aMap)
	{
	    if(aMap->calcxy(&xc, &yc, lg, lt))
		actMapList.append(aMap);
	    aMap = maps->next ();
	}
	aMap = maps->first(); // set to start, for following checks
    }
    else
    {
    	// check several maps (not all!) from global list, if they would fit the current position
    	unsigned int numberOfChecks;
    	numberOfChecks = maps->count()/10;
    	if(numberOfChecks > 20)
	    numberOfChecks = 20;
    	else if(numberOfChecks < 1 && !maps->isEmpty())
	    numberOfChecks = 1;
	aMap = maps->current();
	bool mapAppended=FALSE;
    	while(numberOfChecks)
    	{
	    if(aMap && !actMapList.containsRef(aMap) && aMap->calcxy(&xc, &yc, lg, lt))
	    {
		mapAppended = TRUE;
		actMapList.append(aMap);
	    }
	    if(aMap)
		aMap = maps->next();
	    if(!aMap)
		aMap = maps->first();
	    numberOfChecks--;
    	}
	if(mapAppended)
	    actMapList.sort();
    }

    if (!actMapList.isEmpty())
    {
        mapavailable = TRUE;
	// get Map with best scale + coverage
	// list is still sorted by scale
 	// scroll to the map, with appropriate scale
	aMap = actMapList.first();
	scale = aMap->scale;
        while( aMap && (aMap->scale < 0.9*selectedScale))
	{
	    aMap = actMapList.next();
	};
	// all maps have a lower scale => take map with highest scale
	if(!aMap)
	    aMap = actMapList.last();

        theMap = aMap;
 	scale = aMap->scale;
	// get map with best coverage with approx. the same scale
        aMap->calcxy(&xc, &yc, lg, lt);
 	cover = coverage(aMap, xc, yc,mapdisp->width(),mapdisp->height());
	aMap = actMapList.next();
        while( aMap && (cover < -1) && (2.1 * scale > aMap->scale))
	{
	    aMap->calcxy(&xc, &yc, lg, lt);
	    tcover = coverage(aMap,xc,yc,mapdisp->width(),mapdisp->height());
	    if(tcover > cover)
	    {
		theMap=aMap;
		cover = tcover;
	    }
	    aMap = actMapList.next();
	};

        if(theMap != actmap)
        {
	    mapchanged = TRUE;
	    actmap = theMap;
        }
    }

    // get map data !!!!

    if(mapavailable)
    {
        actmap->calcxy(&xc, &yc, lg, lt);
	xc = rint(xc);
	yc = rint(yc);
	if(mapchanged)
	{
	    //if(map) delete(map);
	    delete map;
	    QString mapfilename = gpsData->mapPathStr;
	    mapfilename.append("/");
	    mapfilename.append(actmap->name);
	    map = new QImage(mapfilename);
            if(!map) 
	    {
		mapavailable = FALSE;
		qWarning(tr("Couldn't open mapfile %1").arg(mapfilename));
	    }
	}
	if(map)
	{
	    // get waypoint in display coordinates
	    lg = gpsData->wpPos.longitude * M_PI / 180.0;
	    lt = gpsData->wpPos.latitude * M_PI / 180.0;
	    actmap->calcxy(&xwp, &ywp, lg, lt);
	    xwp = rint(xwp - xc);
	    ywp = rint(ywp - yc);

	    //  get part of map, with center xc,yc
	    QImage tempImage(mapdisp->width(), mapdisp->height(), 8);
	    tempImage = map->copy((int)xc - mapdisp->width()/2, (int)yc - mapdisp->height()/2,
				  mapdisp->width(), mapdisp->height());

	    mapdisp->convertFromImage(tempImage);
	}
	else
	    map = new QImage();
    }
    else
    {
        actmap = 0;
        mapdisp->fill();
    }
    tempRect = mapdisp->rect();
    painter.begin(mapdisp);
    QFont f=painter.font();
    f.setPointSize(gpsData->textSize);
    painter.setFont(f);
    QPen p=painter.pen();
    p.setWidth(1);
    if(gpsData->status)
        statColor = *gpsData->statusOkColor;
    else
        statColor = *gpsData->statusNoFixColor;
    p.setColor(statColor);
    painter.setPen(p);
    if(gpsData->statusStr->length() > 0)
    {
        painter.drawText( tempRect, AlignVCenter | AlignHCenter, *gpsData->statusStr );
    }
    else
    {
        int xcenter = mapdisp->width()/2;
        int ycenter = mapdisp->height()/2;
        int lgth;
        if(xcenter < ycenter)
            lgth = xcenter/6;
        else
            lgth = ycenter/6;
        //cross
        painter.drawLine(xcenter, ycenter, xcenter, ycenter + lgth );
        painter.drawLine(xcenter - lgth, ycenter , xcenter + lgth, ycenter );
        p.setWidth(2);
        painter.setPen(p);
        painter.drawLine(xcenter, ycenter - lgth , xcenter, ycenter);
        // bearing line
        if(gpsData->bearing.show)
        {
            if( !mapavailable || ((xwp + xcenter > mapdisp->width() - lgth/2.0)
                                  || (xwp + xcenter < lgth/2.0)
                                  || (ywp + ycenter > mapdisp->height() - lgth/2.0)
                                  || (ywp + ycenter < lgth/2.0)))
            {
                tx = xcenter;
                ty = ycenter;
                tw = xcenter + (int)rint((double)(sin(gpsData->bearing.angle*M_PI/180.0)
						  *(double)lgth*3.0));
                th = ycenter - (int)rint((double)(cos(gpsData->bearing.angle*M_PI/180.0)
						  *(double)lgth*3.0));
                p.setWidth(2);
                p.setColor(*gpsData->bearColor);
                painter.setPen(p);
                painter.drawLine( tx,ty,tw,th);
            }
            else
            {
                tx = xcenter;
                ty = ycenter;
                tw = xcenter + (int)xwp;
                th = ycenter + (int)ywp;
                p.setWidth(2);
                p.setColor(*gpsData->bearColor);
                painter.setPen(p);
                painter.drawLine( tx,ty,tw,th);

                tx = xcenter + (int)(xwp - lgth/4);
                ty = ycenter + (int)(ywp - lgth/4);
                tw = (int)lgth/2;
                th = (int)lgth/2;
                painter.drawRect( tx,ty,tw,th);
            }
        }
        // heading line
        if(gpsData->heading.show)
        {
            tx = xcenter;
            ty = ycenter;
            tw = xcenter + (int)((double)(sin(gpsData->heading.angle*M_PI/180.0)
                                          *(double)lgth*2.0));
            th = ycenter - (int)((double)(cos(gpsData->heading.angle*M_PI/180.0)
                                          *(double)lgth*2.0));
            p.setColor(*gpsData->headColor);
            p.setWidth(3);
            painter.setPen(p);
            painter.drawLine( tx,ty,tw,th);
        }
        if(application->track)
           application->track->drawTrack(&painter,actmap,
                                      (int)(xc-xcenter),(int)(yc-ycenter),(int)(xc+xcenter),(int)(yc+ycenter));
        p.setColor(statColor);
        painter.setPen(p);
        if(!mapavailable)
        {
            painter.drawText( tempRect, AlignVCenter | AlignLeft,
                              gpsData->currPos.latToString() );
            painter.drawText( tempRect, AlignVCenter | AlignRight,
                              gpsData->currPos.longToString() );
        }
	QString mapSelectString;
	if(selectedScale > 1) // !=0
        {
	    if(actmap && actmap->scale < actMapList.last()->scale)
		mapSelectString.append(tr("-/"));
	    else
		mapSelectString.append(tr(" /"));
	    if(actmap && actmap->scale > actMapList.first()->scale)
		mapSelectString.append(tr("+"));
	    else
		mapSelectString.append(tr("D"));
        }
	//if(actmap) // only for debugging, don't forget to remove this !!!!
	//QTextOStream(&mapSelectString) << mapSelectString << actmap->scale << " " << selectedScale;

	painter.drawText( tempRect, AlignTop | AlignLeft,
                          mapSelectString);
        painter.drawText( tempRect, AlignTop | AlignHCenter,
                          gpsData->altitude.toString() );
        painter.drawText( tempRect, AlignBottom | AlignHCenter,
                          tr("%1 %2\n%3 %4 %5").arg(gpsData->heading.toString())
			  .arg(gpsData->speed.toString())
                          .arg(gpsData->bearing.toString())
                          .arg(gpsData->wpDistance.toString())
			  .arg(gpsData->timeToString()) );
    }
    painter.end();
}


void MapDisp::paintEvent( QPaintEvent * )
{
    createMap();
    bitBlt(this, 0, 0, mapdisp);
}

void MapDisp::mousePressEvent(QMouseEvent *)
{
    //emit mouseClick(e->x(), e->y());
    emit mouseClick(this);
}

void MapDisp::keyPressEvent(QKeyEvent* event)
{
    switch(event->key())
    {
	case Qt::Key_Space: // Select
	case Qt::Key_Return:
	case Qt::Key_F33:
	    emit mouseClick(this);
	    break;

	case Qt::Key_Down:
	case Qt::Key_Right:
     	    emit lessDetail();
	    break;

	case Qt::Key_Up:
	case Qt::Key_Left:
	    emit moreDetail();
	    break;

	    //case Qt::Key_Left:
	    //emit debugMaps(); // only for debugging, don't forget to remove !!!!
	    //break;
	default:
	    event->ignore();
	    break;
    }
}

