/*
  qpegps is a program for displaying a map centered at the current longitude/
  latitude as read from a gps receiver.

  Copyright (C) 2002 Ralf Haselmeier <Ralf.Haselmeier@gmx.de>
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 
*/


#include "settings.h"

Settings::Settings(GpsData *gData, QWidget *parent, const char *name, WFlags fl):
        QScrollView (parent, name, fl)
{
    gpsData = gData;

#ifndef DESKTOP
    setHScrollBarMode(AlwaysOff);
    setVScrollBarMode(Auto);
    mainBox = new QVBox(this);
    addChild(mainBox);
    setResizePolicy(AutoOneFit);

    horbox3 = new QHBox(mainBox);
    mapDir = new QLabel("Map directory:",horbox3);
    mapPath = new QLineEdit(horbox3);
    // FIXME mapPath->setReadOnly(TRUE);
    mapPathB = new QPushButton("search",horbox3);
    //mapPathB->setDisabled(TRUE);

    horbox6 = new QHBox(mainBox);
    altL = new QLabel("Altitude", horbox6);
    altL->setAlignment(AlignCenter | AlignVCenter | ExpandTabs);
    altCB = new QComboBox(FALSE, horbox6, "Altitude");
    altCB->insertItem("none");
    altCB->insertItem("m");
    altCB->insertItem("feet");
    altCB->insertItem("FL");
    horbox8 = new QHBox(mainBox);
    speedL = new QLabel("Speed", horbox8);
    speedL->setAlignment(AlignCenter | AlignVCenter | ExpandTabs);
    speedCB = new QComboBox(FALSE, horbox8, "Speed");
    speedCB->insertItem("none");
    speedCB->insertItem("kmh");
    speedCB->insertItem("kn");
    speedCB->insertItem("mph");

    horbox7 = new QHBox(mainBox);
    distL = new QLabel("Distance", horbox7);
    distL->setAlignment(AlignCenter | AlignVCenter | ExpandTabs);
    distCB = new QComboBox(FALSE, horbox7, "Distance");
    distCB->insertItem("none");
    distCB->insertItem("km");
    distCB->insertItem("nmi");
    distCB->insertItem("mi");

    horbox9 = new QHBox(mainBox);
    posL = new QLabel("Position", horbox9);
    posL->setAlignment(AlignCenter | AlignVCenter | ExpandTabs);
    posCB = new QComboBox(FALSE, horbox9, "Position");
    posCB->insertItem("DD,d");
    posCB->insertItem("DDMM,m'");
    posCB->insertItem("DDMM'SS,s''");

    horbox10 = new QHBox(mainBox);
    textSizeL = new QLabel("Text Size", horbox10);
    textSizeL->setAlignment(AlignCenter | AlignVCenter | ExpandTabs);
    textSizeSB = new QSpinBox(4, 30, 1, horbox10, "Text Size");

    horbox1 = new QHBox(mainBox);
    checkLabel = new QLabel("display:", horbox1 );
    bearTB = new QCheckBox(horbox1);
    bearTB->setText("Bearing");
    headTB = new QCheckBox(horbox1);
    headTB->setText("Heading");
    timeTB = new QCheckBox(horbox1);
    timeTB->setText("Time");


    colorList << "black" << "white" << "darkGray" << "gray" << "lightGray" << "red" << "green"
    << "blue" << "cyan" << "magenta" << "yellow" << "darkRed" << "darkGreen" << "darkBlue"
    << "darkCyan" << "darkMagenta" << "darkYellow" << "color0" << "color1";
    qColorPtrList.append(new QColor(Qt::black));
    qColorPtrList.append(new QColor(Qt::white));
    qColorPtrList.append(new QColor(Qt::darkGray));
    qColorPtrList.append(new QColor(Qt::gray));
    qColorPtrList.append(new QColor(Qt::lightGray));
    qColorPtrList.append(new QColor(Qt::red));
    qColorPtrList.append(new QColor(Qt::green));
    qColorPtrList.append(new QColor(Qt::blue));
    qColorPtrList.append(new QColor(Qt::cyan));
    qColorPtrList.append(new QColor(Qt::magenta));
    qColorPtrList.append(new QColor(Qt::yellow));
    qColorPtrList.append(new QColor(Qt::darkRed));
    qColorPtrList.append(new QColor(Qt::darkGreen));
    qColorPtrList.append(new QColor(Qt::darkBlue));
    qColorPtrList.append(new QColor(Qt::darkCyan));
    qColorPtrList.append(new QColor(Qt::darkMagenta));
    qColorPtrList.append(new QColor(Qt::darkYellow));
    qColorPtrList.append(new QColor(Qt::color0));
    qColorPtrList.append(new QColor(Qt::color1));
    horbox4 = new QHBox(mainBox);
    okLabel = new QLabel("GPS-OK",horbox4);
    okLabel->setAlignment(AlignCenter | AlignVCenter | ExpandTabs);
    okColorCB = new MenuButton(colorList, horbox4);
    noFixLabel = new QLabel("GPS-noFix",horbox4);
    noFixLabel->setAlignment(AlignCenter | AlignVCenter | ExpandTabs);
    noFixColorCB = new MenuButton(colorList, horbox4);
    horbox5 = new QHBox(mainBox);
    headColorLabel = new QLabel("heading",horbox5);
    headColorLabel->setAlignment(AlignCenter | AlignVCenter | ExpandTabs);
    headColorCB = new MenuButton(colorList, horbox5);
    bearColorLabel = new QLabel("bearing",horbox5);
    bearColorLabel->setAlignment(AlignCenter | AlignVCenter | ExpandTabs);
    bearColorCB = new MenuButton(colorList, horbox5);

    //read config file
    qpegpsConfig = new Config("qpegps");
    qpegpsConfig->setGroup("units");
    gpsData->altitude.altUnit = (Altitude::Alt) qpegpsConfig->readNumEntry("altitude", Altitude::Feet);
    gpsData->speed.speedUnit = (Speed::Sp) qpegpsConfig->readNumEntry("speed", Speed::Knots);
    gpsData->wpDistance.distUnit = (Distance::Dist) qpegpsConfig->readNumEntry("distance", Distance::Naut);
    gpsData->currPos.posUnit = (Position::Pos) qpegpsConfig->readNumEntry("position", Position::DegMin);

    qpegpsConfig->setGroup("show");
    gpsData->bearing.show = qpegpsConfig->readBoolEntry("bearing", TRUE);
    gpsData->heading.show = qpegpsConfig->readBoolEntry("heading", TRUE);
    gpsData->showTime = qpegpsConfig->readBoolEntry("time", TRUE);

    qpegpsConfig->setGroup("gps");
    gpsData->gpsdArgStr = qpegpsConfig->readEntry("gpsd",gpsdDefaultArg);
    gpsData->host = qpegpsConfig->readEntry("host",gpsdDefaultHost);
    gpsData->port = qpegpsConfig->readNumEntry("port",gpsdDefaultPort);

    qpegpsConfig->setGroup("map");
    gpsData->mapPathStr = qpegpsConfig->readEntry("path",gpsData->qpedir+"/qpegps/maps");

    QDir md(gpsData->mapPathStr);
    
    gpsData->mapPathStr = md.canonicalPath();

    qpegpsConfig->setGroup("icons");
    gpsData->iconsPathStr = qpegpsConfig->readEntry("path",gpsData->qpedir+"/qpegps/icons");
    QDir md2(gpsData->iconsPathStr);
    gpsData->iconsPathStr = md2.canonicalPath();


    qpegpsConfig->setGroup("color");
    okColorName = qpegpsConfig->readEntry("ok","black");
    noFixColorName = qpegpsConfig->readEntry("noFix","yellow");
    headColorName = qpegpsConfig->readEntry("heading","green");
    bearColorName = qpegpsConfig->readEntry("bearing","red");
    gpsData->textSize = qpegpsConfig->readEntry("textSize","15").toInt();

    qpegpsConfig->setGroup("track");
    gpsData->trackPathStr = qpegpsConfig->readEntry("path",
						    gpsData->qpedir+"/qpegps/tracks");
    QDir md3(gpsData->trackPathStr);
    gpsData->trackPathStr = md3.canonicalPath();
    gpsData->updt_freq = qpegpsConfig->readEntry("updt_freq","10").toInt();
    gpsData->track_thick = qpegpsConfig->readEntry("track_thick","2").toInt();

    // create config file
    writeConfig();

    // set buttons...
    altCB->setCurrentItem((int)gpsData->altitude.altUnit);
    speedCB->setCurrentItem((int)gpsData->speed.speedUnit);
    distCB->setCurrentItem((int)gpsData->wpDistance.distUnit);
    posCB->setCurrentItem((int)gpsData->currPos.posUnit);
    bearTB->setChecked(gpsData->bearing.show);
    headTB->setChecked(gpsData->heading.show);
    timeTB->setChecked(gpsData->showTime);
    mapPath->setText(gpsData->mapPathStr);
    textSizeSB->setValue(gpsData->textSize);


    connect( altCB, SIGNAL(activated(int)),
             SLOT(setAlt(int)) );
    connect( speedCB, SIGNAL(activated(int)),
             SLOT(setSpeed(int)) );
    connect( distCB, SIGNAL(activated(int)),
             SLOT(setDist(int)) );
    connect( posCB, SIGNAL(activated(int)),
             SLOT(setPos(int)) );
    connect( bearTB, SIGNAL(toggled(bool)),
             SLOT(setBear(bool)) );
    connect( headTB, SIGNAL(toggled(bool)),
             SLOT(setHead(bool)) );
    connect( timeTB, SIGNAL(toggled(bool)),
             SLOT(setTime(bool)) );
    connect( mapPathB, SIGNAL(pressed()),
             SLOT(setMapPath()) );
    connect( mapPath, SIGNAL(returnPressed()),
             SLOT(mapPathLEChanged()) );
    connect( okColorCB , SIGNAL(selected(int)),
             SLOT(okColorChanged(int)) );
    connect( noFixColorCB , SIGNAL(selected(int)),
             SLOT(noFixColorChanged(int)) );
    connect( headColorCB , SIGNAL(selected(int)),
             SLOT(headColorChanged(int)) );
    connect( bearColorCB , SIGNAL(selected(int)),
             SLOT(bearColorChanged(int)) );
    connect( textSizeSB , SIGNAL(valueChanged(int)),
             SLOT(textSizeChanged(int)) );

    okColorCB->select(okColorName);
    noFixColorCB->select(noFixColorName);
    headColorCB->select(headColorName);
    bearColorCB->select(bearColorName);

#else
    //read config file
    gpsData->altitude.altUnit =  Altitude::Feet;
    gpsData->speed.speedUnit =  Speed::Knots;
    gpsData->wpDistance.distUnit = Distance::Naut;
    gpsData->currPos.posUnit = Position::DegMin;

    gpsData->bearing.show = TRUE;
    gpsData->heading.show = TRUE;
    gpsData->showTime = TRUE;

    gpsData->gpsdArgStr = gpsdDefaultArg;
    gpsData->host = gpsdDefaultHost;
    gpsData->port = gpsdDefaultPort;

    gpsData->mapPathStr = gpsData->qpedir+"/maps";
    QDir md(gpsData->mapPathStr);
    gpsData->mapPathStr = md.canonicalPath();

    gpsData->iconsPathStr = gpsData->qpedir+"/icons";
    QDir md2(gpsData->iconsPathStr);
    gpsData->iconsPathStr = md2.canonicalPath();

    okColorName = "black";
    noFixColorName = "yellow";
    headColorName = "green";
    bearColorName = "red";

    gpsData->statusOkColor = new QColor(Qt::black);
    gpsData->statusNoFixColor = new QColor(Qt::yellow);
    gpsData->headColor = new QColor(Qt::green);
    gpsData->bearColor = new QColor(Qt::red);

    gpsData->trackPathStr = gpsData->qpedir+"/tracks";
    gpsData->updt_freq = 10;
    gpsData->track_thick = 2;

    gpsData->textSize = 12;

#endif

}

Settings::~Settings()
{
}

void Settings::writeConfig()
{
#ifndef DESKTOP
    qpegpsConfig->setGroup("units");
    qpegpsConfig->writeEntry("altitude",(int)gpsData->altitude.altUnit);
    qpegpsConfig->writeEntry("speed",(int)gpsData->speed.speedUnit);
    qpegpsConfig->writeEntry("distance",(int)gpsData->wpDistance.distUnit);
    qpegpsConfig->writeEntry("position",(int)gpsData->currPos.posUnit);
    qpegpsConfig->setGroup("show");
    qpegpsConfig->writeEntry("bearing",gpsData->bearing.show);
    qpegpsConfig->writeEntry("heading",gpsData->heading.show);
    qpegpsConfig->writeEntry("time",gpsData->showTime);
    qpegpsConfig->setGroup("gps");
    qpegpsConfig->writeEntry("gpsd", gpsData->gpsdArgStr);
    qpegpsConfig->writeEntry("host", gpsData->host);
    qpegpsConfig->writeEntry("port", gpsData->port);
    qpegpsConfig->setGroup("map");
    qpegpsConfig->writeEntry("path", gpsData->mapPathStr);
    qpegpsConfig->setGroup("color");
    qpegpsConfig->writeEntry("ok", okColorName);
    qpegpsConfig->writeEntry("noFix", noFixColorName);
    qpegpsConfig->writeEntry("heading", headColorName);
    qpegpsConfig->writeEntry("bearing", bearColorName);
    qpegpsConfig->writeEntry("textSize", gpsData->textSize);
    qpegpsConfig->setGroup("track");
    qpegpsConfig->writeEntry("path", gpsData->trackPathStr);
    qpegpsConfig->writeEntry("updt_freq", gpsData->updt_freq);
    qpegpsConfig->writeEntry("track_thick", gpsData->track_thick);
    qpegpsConfig->~Config();
    qpegpsConfig = new Config("qpegps"); /* uggly, but works ... */

#endif
}

void Settings::setAlt(int id)
{
    gpsData->altitude.altUnit = (Altitude::Alt)id;
    writeConfig();
}
void Settings::setSpeed(int id)
{
    gpsData->speed.speedUnit = (Speed::Sp)id;
    writeConfig();
}
void Settings::setDist(int id)
{
    gpsData->wpDistance.distUnit = (Distance::Dist)id;
    writeConfig();
}
void Settings::setPos(int id)
{
    gpsData->currPos.posUnit = (Position::Pos)id;
    writeConfig();
}

void Settings::setBear(bool state)
{
    gpsData->bearing.show = state;
    writeConfig();
}
void Settings::setHead(bool state)
{
    gpsData->heading.show = state;
    writeConfig();
}
void Settings::setTime(bool state)
{
    gpsData->showTime = state;
    writeConfig();
}

void Settings::mapPathLEChanged()
{
    gpsData->mapPathStr = mapPath->text();
    QDir md(gpsData->mapPathStr);

    gpsData->mapPathStr = md.canonicalPath();
    writeConfig();
    emit mapPathChanged();
}

void Settings::setMapPath()
{
    // FIXME *gpsData->mapPathStr = QFileDialog::getExistingDirectory(*gpsData->mapPathStr);
    DirDialog getDirDialog(this, 0, TRUE, 0);
    getDirDialog.setCaption("select map directory");
    getDirDialog.exec();
    if(getDirDialog.result()==QDialog::Accepted)
    {
        gpsData->mapPathStr = getDirDialog.selectedPath;
        QDir md(gpsData->mapPathStr);

        gpsData->mapPathStr = md.canonicalPath();
        mapPath->setText(gpsData->mapPathStr);
        writeConfig();
        emit mapPathChanged();
    }
}

void Settings::okColorChanged(int idx)
{
#ifndef DESKTOP
    okColorName = okColorCB->currentText();
    gpsData->statusOkColor = qColorPtrList.at(idx);
    writeConfig();
#endif
}

void Settings::noFixColorChanged(int idx)
{
#ifndef DESKTOP
    noFixColorName = noFixColorCB->currentText();
    gpsData->statusNoFixColor = qColorPtrList.at(idx);
    writeConfig();
#endif
}

void Settings::headColorChanged(int idx)
{
#ifndef DESKTOP
    headColorName = headColorCB->currentText();
    gpsData->headColor = qColorPtrList.at(idx);
    writeConfig();
#endif
}

void Settings::bearColorChanged(int idx)
{
#ifndef DESKTOP
    bearColorName = bearColorCB->currentText();
    gpsData->bearColor = qColorPtrList.at(idx);
    writeConfig();
#endif
}

void Settings::textSizeChanged(int idx)
{
#ifndef DESKTOP
    gpsData->textSize = idx;
    writeConfig();
#endif
}


