/*
 * Copyright (c) 2004 Nokia. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the
 * distribution.
 *
 * Neither the name of Nokia nor the names of its contributors may be
 * used to endorse or promote products derived from this software
 * without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include "KWQResourceLoader.h"
#include "KWIQResponse.h"
#include "KWQLogging.h"
#include "KWQKJobClasses.h"
#include "loader.h"

using khtml::Loader;
using KIO::TransferJob;

KWQResourceLoader::KWQResourceLoader(KIO::TransferJob *job)
    : m_job(job)
    , m_handle(0)
{
    ASSERT(job);
    
    job->setLoader(this);
    job->addMetaData("expire-date","0");
}

KWQResourceLoader::~KWQResourceLoader()
{
    finishJobAndHandle();
}

void KWQResourceLoader::setHandle(WebCoreResourceHandle* handle)
{
    ASSERT(!m_handle);
    m_handle = handle;
}

void KWQResourceLoader::receivedResponse(WebCoreResponse* response)
{
    ASSERT(response);
    ASSERT(m_job);
    m_job->emitReceivedResponse(response);
}

void KWQResourceLoader::redirectedToURL(const gchar *URL)
{
    ASSERT(m_job);
    m_job->emitRedirection(KURL(URL));
}

void KWQResourceLoader::addData(const void *data, unsigned int length)
{
    ASSERT(data);
    ASSERT(m_job);
    m_job->emitData((const char *)data, length);
}

void KWQResourceLoader::jobWillBeDeallocated(void)
{
    WebCoreResourceHandle* handle = m_handle;	
    m_handle = 0;
    m_job = 0;

    if (handle) {
		handle->cancel();
    }
}

void KWQResourceLoader::finishJobAndHandle(void)
{
    TransferJob *job = m_job;
    WebCoreResourceHandle *handle = m_handle;

    m_job = 0;
    m_handle = 0;
    
    if (job) {
	job->emitResult();
    }
    
    delete job; // will call jobWillBeDeallocated
}

/** called when resource loading is cancelled. */
void KWQResourceLoader::cancel(void)
{
	ASSERT(m_job);
	ASSERT(m_handle);
    if (m_job) {
        m_job->setError(1);
    }
    finishJobAndHandle();
}

/** called when resource loading has an error. */
void KWQResourceLoader::reportError(void)
{
    ASSERT(m_job);
    ASSERT(m_handle);
    m_job->setError(1);
    finishJobAndHandle();
}

/** called when resource loading is done successfully */
void KWQResourceLoader::finish(void)
{
    ASSERT(m_job);
    ASSERT(m_handle);
    finishJobAndHandle();
}
