/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2003 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rigths reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#ifdef __cplusplus
extern "C" {
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/utsname.h>

#include "NX.h"

#define PANIC
#define WARNING
#undef  TEST

#ifndef __APPLE__

#include <dlfcn.h>

#else

void *dlopen(const char *filename, int flag);
const char *dlerror(void);
void *dlsym(void *handle, char *symbol);
int dlclose(void *handle);

#define RTLD_LAZY       0x00001
#define RTLD_NOW        0x00002
#define RTLD_GLOBAL     0x00100

#endif

static void *handle = NULL;

/*
 * Get requested NX library version
 * from command line parameters.
 */

static char *GetVersion(int argc, const char **argv);

/*
 * Load the requested version of NX
 * the NX transport library.
 */

static int LoadVersion(char *);

/*
 * Borrow some code from NX library.
 */

static int (*___NXProxy)(int fd, int mode, const char* display);

static void (*___NXExit)(int code);

static int (*___ParseCommandLineOptions)(int argc, const char **argv);
static int (*___ParseEnvironmentOptions)(const char *env, int force);

static void (*___HandleCleanup)(void) __attribute__((noreturn));

/*
 * Name of symbols.
 */

static char *___NXProxyName = "NXProxy";

static char *___NXExitName = "NXExit";

static char *___ParseCommandLineOptionsName = "ParseCommandLineOptions";
static char *___ParseEnvironmentOptionsName = "ParseEnvironmentOptions";

static char *___HandleCleanupName = "HandleCleanup";

/*
 * Entry point when running nxproxy stand-alone.
 */

int main(int argc, const char **argv)
{
  int result;

  char *options;
  char *version;

  version = GetVersion(argc, argv);

  if (version == NULL)
  {
    version = getenv("NX_VERSION");
  }

  if (version != NULL)
  {
    #ifdef TEST
    fprintf(stderr, "main: Loading NX transport library version '%s'.\n",
                version);
    #endif
  }
  else
  {
    version = VERSION;

    if (version == NULL)
    {
      fprintf(stderr, "Error: No NX library version specified at compile time.\n");

      exit(1);
    }
  }

  if (LoadVersion(version) < 0)
  {
    exit(1);
  }

  //
  // Get NX transport options from the environment.
  //

  options = getenv("NX_DISPLAY");

  if ((*___ParseEnvironmentOptions)(options, 0) < 0)
  {
    (*___HandleCleanup)();
  }

  if ((*___ParseCommandLineOptions)(argc, argv) < 0)
  {
    (*___HandleCleanup)();
  }

  #ifdef TEST
  fprintf(stderr, "main: Yielding control to NX entry point.\n");
  #endif

  /*
   * This usually doesn't return...
   */

  result = (*___NXProxy)(NX_FD_ANY, NX_MODE_ANY, NX_DISPLAY_ANY);

  /*
   * ...So this is usually not called.
   */

  if (handle != NULL)
  {
    dlclose(handle);
  }

  (*___NXExit)(result);

  return 0;
}

char *GetVersion(int argc, const char **argv)
{
  /*
   * Shall use getopt here.
   */

  int argi;

  for (argi = 1; argi < argc; argi++)
  {
    const char *nextArg = argv[argi];

    if (strcmp(nextArg, "-V") == 0)
    {
      const char *arg = (argi + 1 < argc ? argv[argi + 1] : NULL);

      if (arg == NULL)
      {
        fprintf(stderr, "Error: No version specified in option '-V'.\n");

        exit(1);
      }
      else
      {
        return (char *) arg;
      }
    }
  }

  return NULL;
}

int LoadVersion(char *version)
{
  const char *error = NULL;

  char file[256] = { 0 };

  #ifdef __APPLE__

  struct utsname buffer;

  #endif

  /*
   * All platforms use the same library name at the moment.
   */

  #ifndef __CYGWIN32__

  strcpy(file, "libXcomp.so.");

  #else

  strcpy(file, "libXcomp.so.");

  #endif 

  if (strlen(version) + strlen(file) > 255)
  {
    fprintf(stderr, "Error: NX library name cannot exceed 255 characters.\n");

    return -1;
  }

  strcat(file, version);

  /*
   * Open the library and get pointers to functions.
   */

  #ifdef TEST
  fprintf(stderr, "main: Opening NX transport library file '%s'.\n",
              file);
  #endif

  handle = dlopen(file, RTLD_NOW | RTLD_GLOBAL);

  if (handle == NULL)
  {
    error = dlerror();

    fprintf(stderr, "Error: Cannot open NX transport library '%s'.", file);

    if (error != NULL && *error != '\0')
    {
      fprintf(stderr, " Error is '%s'.", error);
    }

    fprintf(stderr, "\n");

    return -1;
  }

  #ifdef TEST
  fprintf(stderr, "main: Transport library file '%s' opened.\n",
              file);
  #endif

  /*
   * Get names of symbols. If this is a MacOS/X find
   * out the OS version because on 10.2.8 we have to
   * specify function names prepended by underscore.
   */

  ___NXProxyName                 = "NXProxy";
  ___NXExitName                  = "NXExit";
  ___ParseCommandLineOptionsName = "ParseCommandLineOptions";
  ___ParseEnvironmentOptionsName = "ParseEnvironmentOptions";
  ___HandleCleanupName           = "HandleCleanup";

  #ifdef __APPLE__

  if (uname(&buffer) < 0)
  {
    fprintf(stderr, "Error: Failed to get system info.\n");

    return -1;
  }
  else
  {
    #ifdef TEST
    fprintf(stderr, "main: System is '%s' nodename '%s' release '%s'.\n",
                buffer.sysname, buffer.nodename, buffer.release);

    fprintf(stderr, "main: Version is '%s' machine '%s'.\n",
                buffer.version, buffer.machine);
    #endif

    if (strncmp(buffer.release, "6.", 2) == 0)
    {
      ___NXProxyName                 = "_NXProxy";
      ___NXExitName                  = "_NXExit";
      ___ParseCommandLineOptionsName = "_ParseCommandLineOptions";
      ___ParseEnvironmentOptionsName = "_ParseEnvironmentOptions";
      ___HandleCleanupName           = "_HandleCleanup";
    }
  }

  #endif /* #ifdef __APPLE__ */

  /*
   * Load the symbols.
   */

  ___NXProxy = dlsym(handle, ___NXProxyName);

  if (error == NULL)
  {
    error = dlerror();
  }

  ___NXExit  = dlsym(handle, ___NXExitName);

  if (error == NULL)
  {
    error = dlerror();
  }

  ___ParseCommandLineOptions = dlsym(handle, ___ParseCommandLineOptionsName);

  if (error == NULL)
  {
    error = dlerror();
  }

  ___ParseEnvironmentOptions = dlsym(handle, ___ParseEnvironmentOptionsName);

  if (error == NULL)
  {
    error = dlerror();
  }

  ___HandleCleanup = dlsym(handle, ___HandleCleanupName);

  if (error == NULL)
  {
    error = dlerror();
  }

  /*
   * These symbols cannot be NULL.
   */

  if (error != NULL || ___NXProxy == NULL ||
          ___NXExit == NULL || ___ParseCommandLineOptions == NULL ||
              ___ParseEnvironmentOptions == NULL ||
                  ___HandleCleanup == NULL)
  {
    fprintf(stderr, "Error: Cannot load symbols from NX library '%s'.", file);

    if (error != NULL && *error != '\0')
    {
      fprintf(stderr, " Error is '%s'.", error);
    }

    fprintf(stderr, "\n");

    return -1;
  }

  return 1;
}

#ifdef __cplusplus
}
#endif

