/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2002 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rigths reserved.                                                   */
/*                                                                        */
/**************************************************************************/


#include "Protocol.h"
#include "Session.h"
#include "TransportSSH.h"
#include "ParametersList.h"
#include "StringUtilities.h"
#include "Utilities.h"

#include "NXParameters.h"
#include "NXErrors.h"
#include "NXStates.h"
#include "NXCompSh.h"

#include <ctype.h>
#include <string>
using namespace std;


#undef NX_PROTOCOL_DEBUG


#if defined( NX_DEBUG ) || defined( NX_PROTOCOL_DEBUG )
#include <iostream>
#endif



namespace NX
{


string Protocol::m_version = NX_PROTOCOL_VERSION;

const unsigned int SSHErrorsNumber = 8;
static char * SSHErrors[SSHErrorsNumber] =
{
  "command not found",
  "No such file or directory",
  "Connection refused",
  "Attempt to connect timed out without establishing a connection",
  "no address associated with hostname.",
  "Name or service not known",
  "No route to host",
  "Host key verification failed"
};


Protocol::Protocol()
{}

Protocol::~Protocol()
{}

static unsigned char toHex( const unsigned char &x )
{
  return x > 9 ? x + 55: x + 48;
}

string Protocol::EncodeUrl( const string& sIn )
{
  string sOut = "";
  char c;
  for( string::const_iterator it = sIn.begin(); it != sIn.end(); it++ )
  {
    c = *it;
    if(!isalnum(c) )
    {
      if( isspace(c) )
        sOut += '+';
      else
      {
        sOut += '%';
        sOut += toHex(c>>4);
        sOut += toHex(c%16);
      }
    }
    else
      sOut += c;
  }

  return sOut;
}

string Protocol::EncodeUrl( const char* pString )
{
  string sIn( pString );
  return EncodeUrl( sIn );
}

string Protocol::DecodeUrl( const string& sIn )
{
  // TODO
  string sOut = sIn;
  return sOut;
}

string Protocol::GetServerOptions( const ParametersList& pl, bool force_creation )
{
  string sRet = "";

  sRet = pl.GetString( NX_ServerOptions, "" );
  if( !force_creation && !sRet.empty() )
  {
#if defined( NX_DEBUG )
    cout << "NXProtocol: getting options for server connection." << endl << flush;
#endif

#if defined( NX_PROTOCOL_DEBUG )
    cout << "NXProtocol: options for server = '" << sRet << "'." << endl << flush;
#endif

    return sRet;
  }

#if defined( NX_DEBUG )
  cout << "NXProtocol: creating options for server connection." << endl << flush;
#endif


  sRet = GetOptions( pl, "&", NX_Server_FirstParameter, NX_Server_LastParameter, true );

#if defined( NX_PROTOCOL_DEBUG )
  cout << "NXProtocol: options for server = '" << sRet << "'." << endl << flush;
#endif

  return sRet;
}

string Protocol::GetOptions( const ParametersList& pl, string separator, int first_parameter, int last_parameter , bool encode_value )
{
  string sRet = "";
  string sParameterKey = "";
  string sParameterValor = "";
  bool addSeparator = false;

  for( int i = first_parameter; i <= last_parameter; i++ )
  {
    sParameterKey = ParameterToString( i );
    sParameterValor = pl.GetString( i, "" );

    if( !sParameterKey.empty() && !sParameterValor.empty() )
    {
#if defined( NX_PROTOCOL_DEBUG )
      cout << "NXProtocol: set option '" << sParameterKey
           << "' to '" << sParameterValor << "'." << endl << flush;
#endif
      if( addSeparator )
        sRet += separator;
      else
        addSeparator = true;

      sRet += sParameterKey;
      sRet += "=";
      StringUtilities::ReplaceAll( sParameterValor, " ", "_" );
      if( encode_value )
        sRet += EncodeUrl( sParameterValor );
      else
        sRet += sParameterValor;
    }
  }

  return sRet;
}

string Protocol::ParameterToString( int parameter_id )
{
  switch( parameter_id )
  {
  /* Server */
  case NX_SessionName:              return "session";
  case NX_SessionCookie:            return "cookie";
  case NX_SessionType:              return "type";
  case NX_ApplicationPath:          return "application";
  case NX_CacheSizeInMemory:        return "cache";
  case NX_LinkSpeed:                return "link";
  case NX_EnableBackingstore:       return "backingstore";
  case NX_DesktopGeometry:          return "geometry";
  case NX_Keyboard:                 return "keyboard";
  case NX_EnableEncryption:         return "encryption";
  case NX_RestoreCache:             return "cache";
  case NX_EnableTcpNoDelay:         return "nodelay";
  case NX_EnableStreamCompression:  return "stream";
  case NX_EnableMedia:              return "media";
  case NX_MediaDriver:              return "mediahelper";
  case NX_EnableSamba:              return "samba";
  case NX_AgentServer:              return "agent_server";
  case NX_AgentUser:                return "agent_user";
  case NX_AgentPassword:            return "agent_password";
  case NX_ImageCompressionType:     return "imagecompressionmethod";
  case NX_ImageCompressionLevel:    return "imagecompressionlevel";
  case NX_EnableLowBandwidth:       return "lowbandwidth";
  case NX_EnableVirtualDesktop:     return "virtualdesktop";
  case NX_EnableTaint:              return "taint";
  case NX_EnableRender:             return "render";
  case NX_UseSessionId:             return "sessionid";
  case NX_Keymap:                   return "keymap";
  case NX_CacheSizeOnDisk:          return "images";
  /* Proxy */
  case NX_ProxySessionName:         return "session";
  case NX_ProxyConnect:             return "connect";
  case NX_ProxyCookie:              return "cookie";
  case NX_ProxyRoot:                return "root";
  case NX_ProxySessionId:           return "id";
  //case NX_ProxyCacheType:           return "type"; not useful since proxy 1.2.2
  case NX_ProxyStream:              return "stream";
  case NX_ProxyNoDelay:             return "nodelay";
  case NX_ProxyRender:              return "render";
  case NX_ProxyTaint:               return "taint";
  case NX_ProxySamba:               return "samba";
  case NX_ProxyMediaPort:           return "media";

  default:
    return "";
  }
}

bool Protocol::ParseHello( const string& HelloLine, int& NXCode, string& NXString )
{
  string ServerHello = "HELLO NXSERVER - Version ";
  int iHelloPos1 = HelloLine.find( ServerHello );
  if( iHelloPos1 != string::npos )
  {
    int iHelloPos2 = HelloLine.find( " - ", ServerHello.size() );
    if( iHelloPos2 == string::npos )
      iHelloPos2 = HelloLine.size() - 1;
    NXCode = 100;
    NXString = HelloLine.substr( ServerHello.size(), iHelloPos2 - ServerHello.size() );
    iHelloPos2 = NXString.find( "-" );
    if( iHelloPos2 != string::npos )
      NXString = NXString.substr( 0, iHelloPos2 );
    return true;
  }

  return false;
}

bool Protocol::ParseLineFromServer( const string& NXResponse, int& NXCode, string& NXString )
{
  if( ParseHello( NXResponse, NXCode, NXString ) )
    return true;

  int iPtr = 0;
  int iNXPos = NXResponse.find( "NX> " );
  NXCode = -1;
  NXString = NXResponse;

  if( iNXPos == string::npos )
  {
#if defined( NX_PROTOCOL_DEBUG )
    cout << "NXProtocol: line does not contain token NX." << endl << flush;
#endif
    return false;
  }

  iPtr = iNXPos + 4;  // skip NX>
  string sNXCode = "";
  for( unsigned int i = iPtr; i < NXResponse.size(); i++ )
  {
    if( isspace( NXResponse[i] ) )
    {
      iPtr = i + 1; // skip space
      break;
    }
    else
      sNXCode += NXResponse[i];
  }

  try
  {
    NXCode = StringUtilities::StringToNumber( sNXCode );
  }
  catch( XInvalidNumber )
  {
#if defined( NX_DEBUG )
    cout << "NXProtocol: '" << NXResponse << "' is invalid response." << endl << flush;
#endif
    return false;
  }

  NXString = NXResponse.substr( iPtr, NXResponse.size() -1 );

  StringUtilities::ReplaceAll( NXString, "\n", " " );
  StringUtilities::ReplaceAll( NXString, "\r", " " );

#if defined( NX_PROTOCOL_DEBUG )
  cout << "NXProtocol: parsed line has code '" << NXCode
       << "' and string '" << NXString << "'." << endl << flush;
#endif
  return true;
}

bool Protocol::ParseResponseFromServer( Session* pSession, const string& NXResponse,
                                          int& NXCode, string& NXString )
{
  NXCode = -1;
  NXString = "";

  if( NXResponse.empty() )
  {
#if defined( NX_PROTOCOL_DEBUG )
    cout << "NXProtocol: response is empty." << endl << flush;
#endif
    return false;
  }

  list<string> lines;
  StringUtilities::SplitString( NXResponse, '\n', lines );

  bool error_found = false;

  for( list<string>::const_iterator it = lines.begin(); it != lines.end(); ++it )
  {
    if( ParseLineFromServer( *it, NXCode, NXString ) )
      SetConnectionStatus( pSession, NXCode, NXString );

    if( ConnectionErrorInLine( pSession, NXCode, NXString ) )
      return false;
  }

  return true;
}

int Protocol::CheckConnectionStatus( const string& buffer, Session* pSession, Transport* pTransport )
{
  int connection_state = pSession->GetState();

  if( connection_state == NX_Exception || connection_state == NX_SessionAccepted )
    return connection_state;  // nothing to do...

  if( buffer.empty() )
    return connection_state;

  int iCode = 0;
  string sString = "";

  if( !ParseResponseFromServer( pSession, buffer, iCode, sString ) )
    return pSession->GetState();

  bool connectionOk = AdvanceConnection( pSession, pTransport );
#if defined( NX_DEBUG )
  if( !connectionOk )
    cout << "NXProtocol: error occurred during advance connection." << endl << flush;
#endif

  return pSession->GetState();
}

void Protocol::SetConnectionStatus( Session* pSession, int NXCode, string NXString )
{
  int connection_state = pSession->GetState();

  switch( NXCode )
  {
  case 203:
    connection_state = NX_Connecting;
    break;

  case 200:
    connection_state = NX_Connecting;
#if defined( NX_DEBUG )
    cout << "NXProtocol: connecting..." << endl << flush;
#endif
    break;

  case 202:
    connection_state = NX_Connected;
#if defined( NX_DEBUG )
    cout << "NXProtocol: connected." << endl << flush;
#endif
    break;

  case 100: // FIXME: maybe 100 is not a valid code
    connection_state = NX_SendingVersion;
    pSession->GetParameters().SetString( NX_ServerVersion, NXString );
#if defined( NX_DEBUG )
    cout << "NXProtocol: set server version to '" << NXString << "'." << endl << flush;
    cout << "NXProtocol: sending version." << endl << flush;
#endif
    break;

  case 134:
    connection_state = NX_VersionAccepted;
#if defined( NX_DEBUG )
    cout << "NXProtocol: version accepted." << endl << flush;
#endif
    break;

  case 101:
    connection_state = NX_SendingAuth;
    break;

  case 105:
    break;

  case 102:
  case 109:
    connection_state = NX_Authenticating;
#if defined( NX_DEBUG )
    cout << "NXProtocol: authenticating..." << endl << flush;
#endif
    break;

  case 103:
    connection_state = NX_Authenticated;
#if defined( NX_DEBUG )
    cout << "NXProtocol: authenticated." << endl << flush;
#endif
    break;

  case 106:
    connection_state = NX_SendingParameters;
    break;

  case 710:
    connection_state = NX_SessionAccepted;
#if defined( NX_DEBUG )
    cout << "NXProtocol: session accepted (proxy needed)." << endl << flush;
#endif
    pSession->NeedProxy( true );
    break;

  case 715:
    connection_state = NX_SessionAccepted;
#if defined( NX_DEBUG )
    cout << "NXProtocol: session accepted (proxy not needed)." << endl << flush;
#endif
    pSession->NeedProxy( false );
    break;

  case 999:
    connection_state = NX_ConnectionEnd;
#if defined( NX_DEBUG )
    cout << "NXProtocol: connection closed." << endl << flush;
#endif
    break;

  default:
#if defined( NX_PROTOCOL_DEBUG )
    cout << "NXProtocol: skip NX code '" << NXCode << "'" << endl << flush;
#endif
    break;
  }

  pSession->SetState( connection_state );
}

bool Protocol::AdvanceConnection( Session *pSession, Transport *pTransport )
{
  switch( pSession->GetState() )
  {
  case NX_SendingVersion:
    return SendHello( pSession, pTransport );
    break;

  case NX_VersionAccepted:
    return StartAuth( pSession, pTransport );
    break;
  case NX_SendingAuth:
    return SendUsernameAndPassword( pSession, pTransport );
    break;

  case NX_Authenticated:
    return StartSession( pSession, pTransport );
    break;

  case NX_SendingParameters:
    return SendParameters( pSession, pTransport );
    break;

  case NX_SessionAccepted:
    return SetParametersFromServer( pSession, pTransport );
    break;

  case NX_ConnectionEnd:
    return !CheckConnectionError( pSession, pTransport );
    break;

  default:
    return true;
  }
}

bool Protocol::ConnectionErrorInLine( Session* pSession, int NXCode, string NXString )
{
  bool connection_error = false;

  if( NXCode > 0 )
  {
    if( NXCode == 999 )
      pSession->SetState( NX_ConnectionEnd );

    if( NXCode >= 300 && NXCode <= 699 )
      connection_error = true;
  }


  if( !connection_error )
  {
    for( unsigned int i = 0 ; i < SSHErrorsNumber; i++ )
    {
      if( NXString.find( SSHErrors[i] ) != string::npos )
      {
        NXCode = NX_SSHError;
        connection_error = true;
        break;
      }
    }
  }

  if( connection_error  )
  {
#if defined( NX_DEBUG )
    cout << "NXProtocol: error found." << endl;
    cout << "NXProtocol: error code " << NXCode << " = "
         << NXString << "." << endl << flush;
#endif
    pSession->SetException( NXCode, NXString );
  }

  return connection_error;
}


bool Protocol::CheckConnectionError( Session* pSession, Transport* pTransport )
{
#if defined( NX_DEBUG )
  cout << "NXProtocol: searching for error... " << endl << flush;
#endif

  string sBuffer = DecodeUrl( pTransport->GetBuffer() );
  StringUtilities::ReplaceAll( sBuffer, "\r", "" );
  list<string> lines;
  int NXCode = -1;
  string NXString = "";

  StringUtilities::SplitString( sBuffer, '\n', lines );

  for( list<string>::const_iterator it = lines.begin(); it != lines.end() ; ++it )
  {
    ParseLineFromServer( *it, NXCode, NXString );

    if( ConnectionErrorInLine( pSession, NXCode, NXString ) )
      return true;
  }

#if defined( NX_DEBUG )
  cout << "NXProtocol: error not found." << endl << flush;
#endif

  return false;
}

bool Protocol::SetParametersFromServer( Session* pSession, Transport* pTransport )
{
#if defined( NX_DEBUG )
  cout << "NXProtocol: setting parameters negotiated with server." << endl << flush;
#endif

  string sBuffer = DecodeUrl( pTransport->GetBuffer() );
  StringUtilities::ReplaceAll( sBuffer, "\r", "" );
  list<string> lines;
  int NXCode = 0;
  string NXString = "";
  bool sRet = false;

  StringUtilities::SplitString( sBuffer, '\n', lines );

  for( list<string>::const_iterator it = lines.begin(); it != lines.end() ; ++it )
  {
    if( ParseLineFromServer( *it, NXCode, NXString ) )
    {
      if( SetParameterFromServer( pSession, NXCode, NXString ) )
        sRet = true;
    }
  }

#if defined( NX_DEBUG )
  cout << "NXProtocol: parameters stored." << endl << flush;
#endif
  return sRet;
}

string Protocol::GetParameterFromResponse( string NXString )
{
  list<string> tokens;
  StringUtilities::SplitString( NXString, ':', tokens );
  string sRet = tokens.back();
  StringUtilities::ReplaceAll( sRet, " ", "" );
  return sRet;
}

bool Protocol::SetParameterFromServer( Session* pSession, int NXCode, string NXString )
{
  string sTmp = GetParameterFromResponse( NXString );

  switch( NXCode )
  {
  case 700:  // SessionId
#if defined( NX_PROTOCOL_DEBUG )
    cout << "NXProtocol: set ID = " << sTmp << "." << endl << flush;
#endif
    pSession->GetParameters().SetString( NX_SessionId, sTmp );
    break;
  case 701:  // Proxy Cookie
#if defined( NX_PROTOCOL_DEBUG )
    cout << "NXProtocol: set Cookie = " << sTmp << "." << endl << flush;
#endif
    pSession->GetParameters().SetString( NX_ProxyCookie, sTmp );
    break;
  case 702:  // Proxy accept connection from IP
  /*
#if defined( NX_DEBUG )
    cout << "NXProtocol: set Server = " << sTmp << "." << endl << flush;
#endif
    pSession->GetParameters().SetString( NX_ProxyServer, sTmp );
    */
    break;
  case 703:  // Session Type
#if defined( NX_PROTOCOL_DEBUG )
    cout << "NXProtocol: set Session Type = " << sTmp << "." << endl << flush;
#endif
    pSession->GetParameters().SetString( NX_SessionType, sTmp );
    break;
  case 704:  // Not useful since proxy 1.2.2
  /*
#if defined( NX_PROTOCOL_DEBUG )
    cout << "NXProtocol: set Cache Type = " << sTmp << "." << endl << flush;
#endif
    pSession->GetParameters().SetString( NX_ProxyCacheType, sTmp );
 */
    break;
  case 705:  // Session display port
#if defined( NX_PROTOCOL_DEBUG )
    cout << "NXProtocol: set Port = " << sTmp << "." << endl << flush;
#endif
    pSession->GetParameters().SetString( NX_ProxyPort, sTmp );
    break;
  case 706:  // Agent Cookie
#if defined( NX_PROTOCOL_DEBUG )
    cout << "NXProtocol: set Agent Cookie = " << sTmp << "." << endl << flush;
#endif
    pSession->GetParameters().SetString( NX_AgentCookie, sTmp );
    break;
  case 707:  // SSL tunneling
#if defined( NX_PROTOCOL_DEBUG )
    cout << "NXProtocol: set SSL Tunneling = " << sTmp << "." << endl << flush;
#endif
    pSession->GetParameters().SetString( NX_EnableEncryption, sTmp );
    break;
  default:
    return false;
  }
  return true;
}

bool Protocol::SendHello( Session *pSession, Transport *pTransport )
{
  ParametersList& parameters = pSession->GetParameters();

  string sToSend = parameters.GetString( NX_Hello, "" );
  if( sToSend.empty() )
  {
#if defined( NX_PROTOCOL_DEBUG )
    cout << "NXProtocol: standard 'Hello' created " << endl << flush;
#endif
    sToSend = "Hello NXCLIENT - Version ";
    sToSend += Protocol::GetVersion();
    parameters.SetString( NX_Hello, sToSend );
  }

#if defined( NX_DEBUG )
  cout << "NXProtocol: sending Hello... " << flush;
#endif

  sToSend += string( "\n" );

  if( !pTransport->Write( sToSend ) )
  {
#if defined( NX_DEBUG )
    cout << "failed!" << endl << flush;
#endif
    return false;
  }

#if defined( NX_DEBUG )
  cout << "success!" << endl << flush;
#endif

#if defined( NX_PROTOCOL_DEBUG )
  StringUtilities::ReplaceAll( sToSend, "\n", "" );
  cout << "NXProtocol: " << sToSend << endl << flush;
#endif

  pSession->SetState( NX_AcceptingVersion );
  return true;
}

bool Protocol::StartAuth( Session *pSession, Transport *pTransport )
{
#if defined( NX_DEBUG )
  cout << "NXProtocol: starting authentication... " << flush;
#endif
  string sToSend( "login\n" );

  if( !pTransport->Write( sToSend ) )
  {
#if defined( NX_DEBUG )
    cout << "failed!" << endl << flush;
#endif
    return false;
  }

#if defined( NX_DEBUG )
  cout << "success!" << endl << flush;
#endif
  pSession->SetState( NX_WaitingForAuth );
  return true;
}

bool Protocol::SendUsernameAndPassword( Session *pSession, Transport *pTransport )
{
#if defined( NX_DEBUG )
  cout << "NXProtocol: sending authentication... " << flush;
#endif

  const ParametersList& parameters = pSession->GetParameters();

  string sToSend = parameters.GetString( NX_Username, "" );
  sToSend += string( "\n" );

  string sPassword = parameters.GetString( NX_Password, "" );
  if( sPassword.empty() )
  {
    sToSend += string( "\n" );
    sToSend += parameters.GetString( NX_PasswordMD5, "" );
  }
  else
    sToSend += sPassword;

  sToSend += string( "\n" );

  if( !pTransport->Write( sToSend ) )
  {
#if defined( NX_DEBUG )
    cout << "failed!" << endl << flush;
#endif
    return false;
  }

#if defined( NX_DEBUG )
  cout << "success!" << endl << flush;
#endif
  pSession->SetState( NX_Authenticating );
  return true;
}

bool Protocol::StartSession( Session *pSession, Transport *pTransport )
{
#if defined( NX_DEBUG )
  cout << "NXProtocol: starting session... " << flush;
#endif

  string sToSend( "startsession\n" );

  if( !pTransport->Write( sToSend ) )
  {
#if defined( NX_DEBUG )
    cout << "failed!" << endl << flush;
#endif
    return false;
  }

#if defined( NX_DEBUG )
  cout << "success!" << endl << flush;
#endif
  pSession->SetState( NX_StartingSession );
  return true;
}

bool Protocol::SendParameters( Session *pSession, Transport *pTransport )
{
  string sToSend = GetServerOptions( pSession->GetParameters() );

  sToSend += string( "\n" );

#if defined( NX_DEBUG )
  cout << "NXProtocol: sending parameters..." << endl << flush;
#endif

  if( !pTransport->Write( sToSend ) )
  {
#if defined( NX_DEBUG )
    cout << "NXProtocol: sending parameters... failed!" << endl << flush;
#endif
    return false;
  }

#if defined( NX_PROTOCOL_DEBUG )
  StringUtilities::ReplaceAll( sToSend, "\n", "" );
  cout << "NXProtocol: parameters sent '" << sToSend << "'." << endl << flush;
#endif
  pSession->SetState( NX_Negotiating );
  return true;
}

bool Protocol::CloseConnection( Session* pSession, Transport* pTransport )
{
#if defined( NX_DEBUG )
  cout << "NXProtocol: closing connection... " << flush;
#endif

  string sToSend( "exit\n" );

  if( !pTransport->Write( sToSend ) )
  {
#if defined( NX_DEBUG )
    cout << "failed!" << endl << flush;
#endif
    return false;
  }

#if defined( NX_DEBUG )
  cout << "success!" << endl << flush;
#endif

  if( pSession->GetState() != NX_SessionAccepted && pSession->GetState() != NX_Exception )
    pSession->SetState( NX_ConnectionEnd );

  return true;
}

bool Protocol::SetStandardSessionLogPath( ParametersList& parameters )
{
  string log_path = "";
  string sTmp = "";

  sTmp = parameters.GetString( NX_SessionDirectory, "" );
  if( sTmp.empty() )
  {
    if( !SetStandardSessionDirectory( parameters ) )
    {
#if defined( NX_DEBUG )
      cout << "NXProtocol: cannot create standard session directory." << endl << flush;
#endif
      return false;
    }
    else
      sTmp = parameters.GetString( NX_SessionDirectory, "" );
  }

  log_path += sTmp;
  log_path += string( "/session" );

  Utilities::ConvertSeparatorsInPath( log_path );

  parameters.SetString( NX_ProxyLogPath, log_path );

  return true;
}

bool Protocol::SetStandardProxyOptionsPath( ParametersList& parameters )
{
  string options_path = "";
  string sTmp = "";

  sTmp = parameters.GetString( NX_SessionDirectory, "" );
  if( sTmp.empty() )
  {
    if( !SetStandardSessionDirectory( parameters ) )
    {
#if defined( NX_DEBUG )
      cout << "NXProtocol: cannot create standard session directory." << endl << flush;
#endif
      return false;
    }
    else
      sTmp = parameters.GetString( NX_SessionDirectory, "" );
  }

  options_path += sTmp;
  options_path += string( "/options" );

  Utilities::ConvertSeparatorsInPath( options_path );

  parameters.SetString( NX_ProxyOptionsPath, options_path );

  return true;
}

bool Protocol::SetStandardSessionDirectory( ParametersList& parameters )
{
  string dir_path = "";
  string sTmp = "";

  sTmp = parameters.GetString( NX_PersonalDirectory, "" );
  if( sTmp.empty() )
  {
#if defined( NX_DEBUG )
    cout << "NXProtocol: parameter 'NX_PersonalDirectory' is empty." << endl << flush;
#endif
    return false;
  }

  dir_path += sTmp;
  dir_path += string( "/" );

  sTmp = parameters.GetString( NX_ProxyMode, "" );
  if( sTmp.empty() )
  {
#if defined( NX_DEBUG )
    cout << "NXProtocol: parameter 'NX_ProxyMode' is empty." << endl << flush;
#endif
    return false;
  }

  dir_path += sTmp;
  dir_path += string( "-" );

  sTmp = parameters.GetString( NX_ProxySessionId, "" );
  if( sTmp.empty() )
  {
#if defined( NX_PROTOCOL_DEBUG )
    cout << "NXProtocol: parameter 'NX_ProxySessionId' is empty." << endl << flush;
#endif
    sTmp = parameters.GetString( NX_SessionId, "" );
    if( sTmp.empty() )
    {
#if defined( NX_DEBUG )
      cout << "NXProtocol: parameter 'NX_SessionId' is empty." << endl << flush;
#endif
      return false;
    }
  }

  dir_path += sTmp;

  Utilities::ConvertSeparatorsInPath( dir_path );

  parameters.SetString( NX_SessionDirectory, dir_path );

  return true;
}

string Protocol::GetProxyOptions( const ParametersList& pl, bool force_creation )
{
  string sRet = "";

  sRet = pl.GetString( NX_ProxyOptions, "" );
  if( !force_creation && !sRet.empty() )
  {
#if defined( NX_DEBUG )
    cout << "NXProtocol: getting preset options for proxy connection." << endl << flush;
#endif

#if defined( NX_PROTOCOL_DEBUG )
    cout << "NXProtocol: options for proxy = '" << sRet << "'." << endl << flush;
#endif
    return sRet;
  }
#if defined( NX_DEBUG )
  cout << "NXProtocol: creating options for proxy connection." << endl << flush;
#endif

  string sTmp = GetOptions( pl, ",", NX_Proxy_FirstParameter, NX_Proxy_LastParameter, false );

  if( !sTmp.empty() )
  {
    sRet = "nx,";
    sRet += sTmp;
  }
  else
    sRet = "nx";

#if defined( NX_PROTOCOL_DEBUG )
  cout << "NXProtocol: options for proxy = '" << sRet << "'." << endl << flush;
#endif

  return sRet;
}

bool Protocol::ParseLineFromProxy( Session* pSession, const string& NXResponse )
{
  string ProxyErrorToken = "Error: ";
  int iErrorPos = NXResponse.find( ProxyErrorToken );
  if( iErrorPos != string::npos )
  {
    string NXError = NXResponse.substr( iErrorPos + ProxyErrorToken.size(), NXResponse.size() -1 );
    StringUtilities::ReplaceAll( NXError, "\n", " " );
    StringUtilities::ReplaceAll( NXError, "\r", " " );
#if defined( NX_DEBUG )
    cout << "NXProtocol: proxy error '" << NXError << "' found." << endl << flush;
#endif
    pSession->SetException( NX_ProxyError, NXError );
    return false;
  }

#if defined( NX_PROTOCOL_DEBUG )
  string ProxyInfoToken = "Info: ";
  int iInfoPos = NXResponse.find( ProxyInfoToken );
  if( iInfoPos != string::npos )
  {
    string NXInfo = NXResponse.substr( iInfoPos + ProxyInfoToken.size(), NXResponse.size() -1 );
    StringUtilities::ReplaceAll( NXInfo, "\n", " " );
    StringUtilities::ReplaceAll( NXInfo, "\r", " " );
    cout << "NXProtocol: " << NXInfo << endl << flush;
  }
  else
    cout << "NXProtocol: " << NXResponse << endl << flush;
#endif

  if( NXResponse.find( "Established X server connection" ) != string::npos )
    pSession->SetState( NX_ProxyConnected );


  return true;
}

bool Protocol::ParseResponseFromProxy( Session* pSession, const string& NXResponse )
{
  if( !NXResponse.empty() )
  {
    list<string> lines;
    StringUtilities::SplitString( NXResponse, '\n', lines );

    for( list<string>::const_iterator it = lines.begin(); it != lines.end(); ++it )
    {
      if( !ParseLineFromProxy( pSession, *it ) )
        return false;
    }
  }

  return true;
}

int Protocol::CheckProxyStatus( const string& buffer, Session* pSession, Proxy* pProxy )
{
  int connection_state = pSession->GetState();

  if( connection_state == NX_Exception || connection_state == NX_ProxyConnected )
    return connection_state;  // nothing to do...

  if( buffer.empty() )
    return connection_state;

  pSession->SetState( NX_ProxyNegotiating );

  ParseResponseFromProxy( pSession, buffer );

  return pSession->GetState();
}

} /* NX */

