/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2006 NoMachine, http://www.nomachine.com/.         */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include <X11/Xmd.h>

#include "Misc.h"
#include "Unpack.h"

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG

//
// Defines used for Hextile encoding
//

#ifndef True
#define True  (1)
#define False (0)
#endif

#define rfbHextileRaw                  (1 << 0)
#define rfbHextileBackgroundSpecified  (1 << 1)
#define rfbHextileForegroundSpecified  (1 << 2)
#define rfbHextileAnySubrects          (1 << 3)
#define rfbHextileSubrectsColoured     (1 << 4)

#define rfbHextileExtractX(byte) ((byte) >> 4)
#define rfbHextileExtractY(byte) ((byte) & 0xf)
#define rfbHextileExtractW(byte) (((byte) >> 4) + 1)
#define rfbHextileExtractH(byte) (((byte) & 0xf) + 1)

#define GET_PIXEL8(pix, ptr) ((pix) = *(ptr)++)

#define GET_PIXEL16(pix, ptr) (((CARD8*)&(pix))[0] = *(ptr)++, \
                              ((CARD8*)&(pix))[1] = *(ptr)++)

#define GET_PIXEL32(pix, ptr) (((CARD8*)&(pix))[0] = *(ptr)++, \
                              ((CARD8*)&(pix))[1] = *(ptr)++, \
                              ((CARD8*)&(pix))[2] = *(ptr)++, \
                              ((CARD8*)&(pix))[3] = *(ptr)++)

static CARD32 bg32, fg32;
static CARD16 bg16, fg16;
static CARD8  bg8 , fg8;

//
// The following are from the VNC client software.
//

int UnpackHextileTo8(int src_width, int src_height, unsigned char *src_data,
                         int src_size, int dst_depth, int dst_width, int dst_height,
                             unsigned char *dst_data, int dst_size)
{
    int i, j, k, x, y, w, h, sx, sy, sw, sh;
    CARD8 *ptr;
    CARD8 subencoding;
    CARD8 nSubrects;
    CARD8 bg, fg;
    unsigned int bytesPerPixel, bytesPerLine;

    unsigned char *compData = src_data;
    unsigned char *decompData = dst_data;

    if (dst_height == 0)
    {
      #ifdef TEST
      *logofs << "UnpackHextile8: WARNING! Spurious hextile "
              << "packet with dst_height " << dst_height
              << " and dst_size " << dst_size << ".\n"
              << logofs_flush;
      #endif

      bytesPerLine = 0;
    }
    else
    {
      bytesPerLine = dst_size / dst_height;
    }

    bytesPerPixel = dst_depth / 8;

    bg = bg8;
    fg = fg8;

    for (y = 0; y < dst_height; y += 16)
    {
        for (x = 0; x < dst_width; x += 16)
        {
            w = h = 16;
            if (dst_width - x < 16)
                w = dst_width - x;
            if (dst_height - y < 16)
                h = dst_height - y;

            // read the subencoding
            subencoding = *(CARD8 *)compData;
            compData ++;

            // raw Hextile, just copy everything in the right place
            if (subencoding & rfbHextileRaw)
            {
                decompData = dst_data + y * bytesPerLine + x * bytesPerPixel;
                for (j = 0; j < h; j++)
                {
                    if ((int) ((decompData - dst_data) + w * bytesPerPixel) > dst_size)
                    {
                        #ifdef TEST
                        *logofs << "UnpackHextile8: PANIC! Can't write at pixel "
                                << (decompData - dst_data) + bytesPerPixel
                                << " with size " << dst_size << " in context "
                                << "[A].\n" << logofs_flush;
                        #endif

                        continue;
                    }

                    memcpy(decompData, compData, w * bytesPerPixel);
                    decompData += bytesPerLine;
                    compData += w * bytesPerPixel;
                }

                continue;
            }

            // backgroundSpecified Hextile, read the new background
            if (subencoding & rfbHextileBackgroundSpecified)
            {
                    bg = *(CARD8 *)compData;
                    compData += bytesPerPixel;
            }

            // fill the tile with the background color
            decompData = dst_data + y * bytesPerLine + x * bytesPerPixel;
            for (j = 0; j < h; j++)
            {
                for (k = 0; k < w; k++)
                {
                    if ((int) ((decompData - dst_data) + bytesPerPixel) > dst_size)
                    {
                        #ifdef TEST
                        *logofs << "UnpackHextile8: PANIC! Can't write at pixel "
                                << (decompData - dst_data) + bytesPerPixel
                                << " with size " << dst_size << " in context "
                                << "[B].\n" << logofs_flush;
                        #endif

                        continue;
                    }

                    *(CARD8 *)decompData = bg;
                    decompData += bytesPerPixel;
                }
                decompData += bytesPerLine - w * bytesPerPixel;
            }

            // foregroundSpecified Hextile, read the new foreground
            if (subencoding & rfbHextileForegroundSpecified)
            {
                    fg = *(CARD8 *)compData;
                    compData += bytesPerPixel;
            }

            if (!(subencoding & rfbHextileAnySubrects))
            {
                continue;
            }

            nSubrects = *(CARD8 *)compData;
            compData++;
            ptr = (CARD8 *)compData;

            // handle the subrectangles of the tile
            if (subencoding &rfbHextileSubrectsColoured)
            {
                for (i = 0; i < nSubrects; i++)
                {
                    GET_PIXEL8(fg, ptr);
                    sx = rfbHextileExtractX(*ptr);
                    sy = rfbHextileExtractY(*ptr);
                    ptr++;
                    sw = rfbHextileExtractW(*ptr);
                    sh = rfbHextileExtractH(*ptr);
                    ptr++;

                    decompData = dst_data + (y + sy) * bytesPerLine + (x + sx) * bytesPerPixel;
                    for (j = 0; j < sh; j++)
                    {
                        for (k = 0; k < sw; k++)
                        {
                            if ((int) ((decompData - dst_data) + bytesPerPixel) > dst_size)
                            {
                                #ifdef TEST
                                *logofs << "UnpackHextile8: PANIC! Can't write at pixel "
                                        << (decompData - dst_data) + bytesPerPixel
                                        << " with size " << dst_size << " in context "
                                        << "[C].\n" << logofs_flush;
                                #endif

                                continue;
                            }

                            *(CARD8 *)decompData = fg;
                            decompData += bytesPerPixel;
                        }
                        decompData += bytesPerLine - sw * bytesPerPixel;
                    }
                }
            }
            else
            {
                for (i = 0; i < nSubrects; i++)
                {
                    sx = rfbHextileExtractX(*ptr);
                    sy = rfbHextileExtractY(*ptr);
                    ptr++;
                    sw = rfbHextileExtractW(*ptr);
                    sh = rfbHextileExtractH(*ptr);
                    ptr++;

                    decompData = dst_data + (y + sy) * bytesPerLine + (x + sx) * bytesPerPixel;
                    for (j = 0; j < sh; j++)
                    {
                        for (k = 0; k < sw; k++)
                        {
                            if ((int) ((decompData - dst_data) + bytesPerPixel) > dst_size)
                            {
                                #ifdef TEST
                                *logofs << "UnpackHextile8: PANIC! Can't write at pixel "
                                        << (decompData - dst_data) + bytesPerPixel
                                        << " with size " << dst_size << " in context "
                                        << "[D].\n" << logofs_flush;
                                #endif

                                continue;
                            }

                            *(CARD8 *)decompData = fg;
                            decompData += bytesPerPixel;
                        }
                        decompData += bytesPerLine - sw * bytesPerPixel;
                    }
                }
                
            }
            compData = ptr;
        }
    }

    bg8 = bg;
    fg8 = fg;

    #ifdef DEBUG
    *logofs << "UnpackHextileTo8. Incoming " << src_size << " run "
            << compData - src_data << " outgoing " << dst_size
            << " next offset " << decompData - dst_data << ".\n"
            << logofs_flush;
    #endif

    return True;
}

int UnpackHextileTo16(int src_width, int src_height, unsigned char *src_data,
                          int src_size, int dst_depth, int dst_width, int dst_height,
                              unsigned char *dst_data, int dst_size)
{
    int i, j, k, x, y, w, h, sx, sy, sw, sh;
    CARD8 *ptr;
    CARD8 subencoding;
    CARD8 nSubrects;
    CARD16 bg, fg;
    unsigned int bytesPerPixel, bytesPerLine;

    unsigned char *compData = src_data;
    unsigned char *decompData = dst_data;

    if (dst_height == 0)
    {
      #ifdef TEST
      *logofs << "UnpackHextile16: WARNING! Spurious hextile "
              << "packet with dst_height " << dst_height
              << " and dst_size " << dst_size << ".\n"
              << logofs_flush;
      #endif

      bytesPerLine = 0;
    }
    else
    {
      bytesPerLine = dst_size / dst_height;
    }

    bytesPerPixel = dst_depth / 8;

    bg = bg16;
    fg = fg16;

    for (y = 0; y < dst_height; y += 16)
    {
        for (x = 0; x < dst_width; x += 16)
        {
            w = h = 16;
            if (dst_width - x < 16)
                w = dst_width - x;
            if (dst_height - y < 16)
                h = dst_height - y;

            // read the subencoding
            subencoding = *(CARD8 *)compData;
            compData++;

            // raw Hextile, just copy everything in the right place
            if (subencoding & rfbHextileRaw)
            {
                decompData = dst_data + y * bytesPerLine + x * bytesPerPixel;
                for (j = 0; j < h; j++)
                {
                    if ((int) ((decompData - dst_data) + w * bytesPerPixel) > dst_size)
                    {
                        #ifdef TEST
                        *logofs << "UnpackHextile16: PANIC! Can't write at pixel "
                                << (decompData - dst_data) + bytesPerPixel
                                << " with size " << dst_size << " in context "
                                << "[A].\n" << logofs_flush;
                        #endif

                        continue;
                    }

                    memcpy(decompData, compData, w * bytesPerPixel);
                    decompData += bytesPerLine;
                    compData += w * bytesPerPixel;
                }

                continue;
            }

            // backgroundSpecified Hextile, read the new background
            if (subencoding & rfbHextileBackgroundSpecified)
            {
                    bg = *(CARD16 *)compData;
                    compData += bytesPerPixel;
            }

            // fill the tile with the background color
            decompData = dst_data + y * bytesPerLine + x * bytesPerPixel;
            for (j = 0; j < h; j++)
            {
                for (k = 0; k < w; k++)
                {
                    if ((int) ((decompData - dst_data) + bytesPerPixel) > dst_size)
                    {
                        #ifdef TEST
                        *logofs << "UnpackHextile16: PANIC! Can't write at pixel "
                                << (decompData - dst_data) + bytesPerPixel
                                << " with size " << dst_size << " in context "
                                << "[B].\n" << logofs_flush;
                        #endif

                        continue;
                    }

                    *(CARD16 *)decompData = bg;
                    decompData += bytesPerPixel;
                }
                decompData += bytesPerLine - w * bytesPerPixel;
            }

            // foregroundSpecified Hextile, read the new foreground
            if (subencoding & rfbHextileForegroundSpecified)
            {
                    fg = *(CARD16 *)compData;
                    compData += bytesPerPixel;
            }

            if (!(subencoding & rfbHextileAnySubrects))
            {
                continue;
            }

            nSubrects = *(CARD8 *)compData;
            compData++;
            ptr = (CARD8 *)compData;

            // handle the subrectangles of the tile
            if (subencoding &rfbHextileSubrectsColoured)
            {
                for (i = 0; i < nSubrects; i++)
                {
                    GET_PIXEL16(fg, ptr);
                    sx = rfbHextileExtractX(*ptr);
                    sy = rfbHextileExtractY(*ptr);
                    ptr++;
                    sw = rfbHextileExtractW(*ptr);
                    sh = rfbHextileExtractH(*ptr);
                    ptr++;

                    decompData = dst_data + (y + sy) * bytesPerLine + (x + sx) * bytesPerPixel;
                    for (j = 0; j < sh; j++)
                    {
                        for (k = 0; k < sw; k++)
                        {
                            if ((int) ((decompData - dst_data) + bytesPerPixel) > dst_size)
                            {
                                #ifdef TEST
                                *logofs << "UnpackHextile16: PANIC! Can't write at pixel "
                                        << (decompData - dst_data) + bytesPerPixel
                                        << " with size " << dst_size << " in context "
                                        << "[C].\n" << logofs_flush;
                                #endif

                                continue;
                            }

                            *(CARD16 *)decompData = fg;
                            decompData += bytesPerPixel;
                        }
                        decompData += bytesPerLine - sw * bytesPerPixel;
                    }
                }
            }
            else
            {
                for (i = 0; i < nSubrects; i++)
                {
                    sx = rfbHextileExtractX(*ptr);
                    sy = rfbHextileExtractY(*ptr);
                    ptr++;
                    sw = rfbHextileExtractW(*ptr);
                    sh = rfbHextileExtractH(*ptr);
                    ptr++;

                    decompData = dst_data + (y + sy) * bytesPerLine + (x + sx) * bytesPerPixel;
                    for (j = 0; j < sh; j++)
                    {
                        for (k = 0; k < sw; k++)
                        {
                            if ((int) ((decompData - dst_data) + bytesPerPixel) > dst_size)
                            {
                                #ifdef TEST
                                *logofs << "UnpackHextile16: PANIC! Can't write at pixel "
                                        << (decompData - dst_data) + bytesPerPixel
                                        << " with size " << dst_size << " in context "
                                        << "[D].\n" << logofs_flush;
                                #endif

                                continue;
                            }

                            *(CARD16 *)decompData = fg;
                            decompData += bytesPerPixel;
                        }
                        decompData += bytesPerLine - sw * bytesPerPixel;
                    }
                }
                
            }
            compData = ptr;
        }
    }

    bg16 = bg;
    fg16 = fg;

    #ifdef DEBUG
    *logofs << "UnpackHextileTo16. Incoming " << src_size << " run "
            << compData - src_data << " outgoing " << dst_size
            << " next offset " << decompData - dst_data << ".\n"
            << logofs_flush;
    #endif

    return True;
}

int UnpackHextileTo32(int src_width, int src_height, unsigned char *src_data,
                          int src_size, int dst_depth, int dst_width, int dst_height,
                              unsigned char *dst_data, int dst_size)
{
    int i, j, k, x, y, w, h, sx, sy, sw, sh;
    CARD8 *ptr;
    CARD8 subencoding;
    CARD8 nSubrects;
    CARD32 bg, fg;
    unsigned int bytesPerPixel, bytesPerLine;

    unsigned char *compData = src_data;
    unsigned char *decompData = dst_data;

    if (dst_height == 0)
    {
      #ifdef TEST
      *logofs << "UnpackHextile32: WARNING! Spurious hextile "
              << "packet with dst_height " << dst_height
              << " and dst_size " << dst_size << ".\n"
              << logofs_flush;
      #endif

      bytesPerLine = 0;
    }
    else
    {
      bytesPerLine = dst_size / dst_height;
    }

    bytesPerPixel = dst_depth / 8;

    bg = bg32;
    fg = fg32;

    for (y = 0; y < dst_height; y += 16)
    {
        for (x = 0; x < dst_width; x += 16)
        {
            w = h = 16;
            if (dst_width - x < 16)
                w = dst_width - x;
            if (dst_height - y < 16)
                h = dst_height - y;

            // read the subencoding
            subencoding = *(CARD8 *)compData;
            compData ++;

            // raw Hextile, just copy everything in the right place
            if (subencoding & rfbHextileRaw)
            {
                decompData = dst_data + y * bytesPerLine + x * bytesPerPixel;
                for (j = 0; j < h; j++)
                {
                    if ((int) ((decompData - dst_data) + w * bytesPerPixel) > dst_size)
                    {
                        #ifdef TEST
                        *logofs << "UnpackHextile32: PANIC! Can't write at pixel "
                                << (decompData - dst_data) + bytesPerPixel
                                << " with size " << dst_size << " in context "
                                << "[A].\n" << logofs_flush;
                        #endif

                        continue;
                    }

                    memcpy(decompData, compData, w * bytesPerPixel);
                    decompData += bytesPerLine;
                    compData += w * bytesPerPixel;
                }

                continue;
            }

            // backgroundSpecified Hextile, read the new background
            if (subencoding & rfbHextileBackgroundSpecified)
            {
                    bg = *(CARD32 *)compData;
                    compData += bytesPerPixel;
            }

            // fill the tile with the background color
            decompData = dst_data + y * bytesPerLine + x * bytesPerPixel;
            for (j = 0; j < h; j++)
            {
                for (k = 0; k < w; k++)
                {
                    if ((int) ((decompData - dst_data) + bytesPerPixel) > dst_size)
                    {
                        #ifdef TEST
                        *logofs << "UnpackHextile32: PANIC! Can't write at pixel "
                                << (decompData - dst_data) + bytesPerPixel
                                << " with size " << dst_size << " in context "
                                << "[B].\n" << logofs_flush;
                        #endif

                        continue;
                    }

                    *(CARD32 *)decompData = bg;
                    decompData += bytesPerPixel;
                }
                decompData += bytesPerLine - w * bytesPerPixel;
            }

            // foregroundSpecified Hextile, read the new foreground
            if (subencoding & rfbHextileForegroundSpecified)
            {
                    fg = *(CARD32 *)compData;
                    compData += bytesPerPixel;
            }

            if (!(subencoding & rfbHextileAnySubrects))
            {
                continue;
            }

            nSubrects = *(CARD8 *)compData;
            compData++;
            ptr = (CARD8 *)compData;

            // handle the subrectangles of the tile
            if (subencoding &rfbHextileSubrectsColoured)
            {
                for (i = 0; i < nSubrects; i++)
                {
                    GET_PIXEL32(fg, ptr);
                    sx = rfbHextileExtractX(*ptr);
                    sy = rfbHextileExtractY(*ptr);
                    ptr++;
                    sw = rfbHextileExtractW(*ptr);
                    sh = rfbHextileExtractH(*ptr);
                    ptr++;

                    decompData = dst_data + (y + sy) * bytesPerLine + (x + sx) * bytesPerPixel;
                    for (j = 0; j < sh; j++)
                    {
                        for (k = 0; k < sw; k++)
                        {
                            if ((int) ((decompData - dst_data) + bytesPerPixel) > dst_size)
                            {
                                #ifdef TEST
                                *logofs << "UnpackHextile32: PANIC! Can't write at pixel "
                                        << (decompData - dst_data) + bytesPerPixel
                                        << " with size " << dst_size << " in context "
                                        << "[C].\n" << logofs_flush;
                                #endif

                                continue;
                            }

                            *(CARD32 *)decompData = fg;
                            decompData += bytesPerPixel;
                        }
                        decompData += bytesPerLine - sw * bytesPerPixel;
                    }
                }
            }
            else
            {
                for (i = 0; i < nSubrects; i++)
                {
                    sx = rfbHextileExtractX(*ptr);
                    sy = rfbHextileExtractY(*ptr);
                    ptr++;
                    sw = rfbHextileExtractW(*ptr);
                    sh = rfbHextileExtractH(*ptr);
                    ptr++;

                    decompData = dst_data + (y + sy) * bytesPerLine + (x + sx) * bytesPerPixel;
                    for (j = 0; j < sh; j++)
                    {
                        for (k = 0; k < sw; k++)
                        {
                            if ((int) ((decompData - dst_data) + bytesPerPixel) > dst_size)
                            {
                                #ifdef TEST
                                *logofs << "UnpackHextile32: PANIC! Can't write at pixel "
                                        << (decompData - dst_data) + bytesPerPixel
                                        << " with size " << dst_size << " in context "
                                        << "[D].\n" << logofs_flush;
                                #endif

                                continue;
                            }

                            *(CARD32 *)decompData = fg;
                            decompData += bytesPerPixel;
                        }
                        decompData += bytesPerLine - sw * bytesPerPixel;
                    }
                }
                
            }
            compData = ptr;
        }
    }

    bg32 = bg;
    fg32 = fg;

    #ifdef DEBUG
    *logofs << "UnpackHextileTo32. Incoming " << src_size << " run "
            << compData - src_data << " outgoing " << dst_size
            << " next offset " << decompData - dst_data << ".\n"
            << logofs_flush;
    #endif

    return True;
}

int UnpackHextile(int src_width, int src_height, unsigned char *src_data,
                      int src_size, int dst_depth, int dst_width, int dst_height,
                          unsigned char *dst_data, int dst_size)
{
  switch (dst_depth)
  {
    case 8:
    {
      return UnpackHextileTo8(src_width, src_height, src_data, src_size, dst_depth,
                                  dst_width, dst_height, dst_data, dst_size);
    }
    case 16:
    {
      return UnpackHextileTo16(src_width, src_height, src_data, src_size, dst_depth,
                                   dst_width, dst_height, dst_data, dst_size);
    }
    case 32:
    {
      return UnpackHextileTo32(src_width, src_height, src_data, src_size, dst_depth,
                                   dst_width, dst_height, dst_data, dst_size);
    }
    default:
    {
      #ifdef PANIC
      *logofs << "UnpackHextile: Unsupported depth " << dst_depth
              << " for Hextile encoding\n" << logofs_flush;
      #endif

      return -1;
    }
  }
}
