/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2003 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/socket.h>

#if defined(__CYGWIN32__) || defined(__APPLE__) || defined(__FreeBSD__)
#include <netinet/in_systm.h>
#endif

#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/tcp.h>

#include <unistd.h>
#include <stdlib.h>

#include "Socket.h"
#include "Alerts.h"

#include "ServerProxy.h"

#include "ServerChannel.h"
#include "GenericChannel.h"

//
// Set the verbosity level.
//

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG

ServerProxy::ServerProxy(int proxyFd, const char *xServerDisplay, int xServerAddrFamily,
                             sockaddr *xServerAddr, unsigned int xServerAddrLength,
                                 int syncServerPort, int keybdServerPort,
                                      int sambaServerPort, int mediaServerPort)

  : Proxy(proxyFd), xServerAddrFamily_(xServerAddrFamily),
        xServerAddr_(xServerAddr), xServerAddrLength_(xServerAddrLength),
            syncServerPort_(syncServerPort), keybdServerPort_(keybdServerPort),
                sambaServerPort_(sambaServerPort), mediaServerPort_(mediaServerPort)
{
  for (int i = 0; i < CONNECTIONS_LIMIT; i++)
  {
    channelMap_[i] = -1;
    fdMap_[i] = -1;
  }

  xServerDisplay_ = new char[strlen(xServerDisplay) + 1];

  strcpy(xServerDisplay_, xServerDisplay);

  #ifdef DEBUG
  *logofs << "ServerProxy: Created new object at " << this
          << ".\n" << logofs_flush;
  #endif
}

ServerProxy::~ServerProxy()
{
  delete [] xServerDisplay_;

  delete xServerAddr_;

  #ifdef DEBUG
  *logofs << "ServerProxy: Deleted object at " << this
          << ".\n" << logofs_flush;
  #endif
}

int ServerProxy::needMotion() const
{
  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] != NULL &&
            channels_[channelId] -> needMotion() > 0)
    {
      return getFd(channelId);
    }
  }

  return -1;
}

int ServerProxy::handleNewXConnection(int clientFd)
{
  #ifdef PANIC
  *logofs << "ServerProxy: PANIC! Function handleNewXConnection() "
          << "shouldn't be called at server side.\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Function handleNewXConnection() "
       << "shouldn't be called at server side.\n";

  HandleCleanup();

  return -1;
}

int ServerProxy::handleNewXConnectionFromProxy(int channelId)
{
  if (channelId >= CONNECTIONS_LIMIT)
  {
    #ifdef PANIC
    *logofs << "ServerProxy: PANIC! Maximum mumber of available "
            << "channels exceeded.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Maximum mumber of available "
         << "channels exceeded.\n";

    return -1;
  }
  else if (channels_[channelId] != NULL)
  {
    #ifdef PANIC
    *logofs << "ServerProxy: PANIC! Trying to open a new connection "
            << "over an existing channel ID#" << channelId
            << " with FD#" << getFd(channelId) << ".\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Trying to open a new connection "
         << "over an existing channel ID#" << channelId
         << " with FD#" << getFd(channelId) << ".\n";

    return -1;
  }

  //
  // Connect to the real X server.
  //

  int retryConnect = control -> OptionServerRetryConnect;

  int xServerFd;

  for (;;)
  {
    xServerFd = socket(xServerAddrFamily_, SOCK_STREAM, PF_UNSPEC);

    if (xServerFd < 0)
    {
      #ifdef PANIC
      *logofs << "ServerProxy: PANIC! Call to socket failed. "
              << "Error is " << EGET() << " '" << ESTR()
              << "'.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Call to socket failed. "
           << "Error is " << EGET() << " '" << ESTR()
           << "'.\n";

      return -1;
    }

    #if defined(INFO) || defined (TEST)
    *logofs << "ServerProxy: Trying to connect to X server '"
            << xServerDisplay_ << "'.\n" << logofs_flush;
    #endif

    if (connect(xServerFd, xServerAddr_, xServerAddrLength_) < 0)
    {
      #ifdef WARNING
      *logofs << "ServerProxy: WARNING! Connection to '"
              << xServerDisplay_ << "' failed with error '"
              << ESTR() << "'. Retrying.\n" << logofs_flush;
      #endif

      close(xServerFd);

      if (--retryConnect == 0)
      {
        #ifdef PANIC
        *logofs << "ServerProxy: PANIC! Connection to '"
                << xServerDisplay_ << "' for channel ID#"
                << channelId << " failed. Error is "
                << EGET() << " '" << ESTR() << "'.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Connection to '"
             << xServerDisplay_ << "' failed. Error is "
             << EGET() << " '" << ESTR() << "'.\n";

        close(xServerFd);

        return -1;
      }

      if (activeChannels_ == 0)
      {
        sleep(2);
      }
      else
      {
        sleep(1);
      }
    }
    else
    {
      break;
    }
  }

  fdMap_[channelId] = xServerFd;
  channelMap_[xServerFd] = channelId;

  #ifdef TEST
  *logofs << "ServerProxy: X server descriptor FD#" << xServerFd 
          << " mapped to channel ID#" << channelId << ".\n"
          << logofs_flush;
  #endif

  //
  // Turn queuing off for path proxy-to-X-server.
  //

  if (control -> OptionServerNoDelay == 1)
  {
    SetNoDelay(xServerFd, control -> OptionServerNoDelay);
  }

  //
  // Set size of send and receive buffers at TCP/IP layer
  // to increase interactivity or throughput. Default is
  // usually 64KB and is achieved by leaving control
  // parameters to -1.
  //

  if (control -> OptionServerSendBuffer != -1)
  {
    SetSendBuffer(xServerFd, control -> OptionServerSendBuffer);
  }

  if (control -> OptionServerReceiveBuffer != -1)
  {
    SetReceiveBuffer(xServerFd, control -> OptionServerReceiveBuffer);
  }

  //
  // Load the persistent cache if this is
  // the first X channel.
  //

  if (handleCheckLoad(LOAD_IF_FIRST) < 0)
  {
    return -1;
  }

  transports_[channelId] = new Transport(xServerFd);

  if (transports_[channelId] == NULL)
  {
    return -1;
  }

  //
  // Starting from protocol level 3 client and server
  // caches are created in proxy and shared between all
  // channels. If remote proxy has older protocol level
  // pointers are NULL and channels must create their
  // own instances.
  //

  channels_[channelId] = new ServerChannel(transports_[channelId], compressor_,
                                               decompressor_);

  if (channels_[channelId] == NULL)
  {
    delete transports_[channelId];
    transports_[channelId] = NULL;

    return -1;
  }

  increaseActiveChannels(channelId);

  //
  // Propagate channel stores and caches to the new
  // channel.
  //

  channels_[channelId] -> setOpcodes(opcodeStore_);

  channels_[channelId] -> setStores(clientStore_, serverStore_);

  if (control -> isProtoStep3() == 1)
  {
    channels_[channelId] -> setCaches(clientCache_, serverCache_);
  }

  //
  // Let channel configure itself according
  // to control parameters.
  //

  channels_[channelId] -> handleConfiguration();

  return 1;
}

int ServerProxy::handleNewSyncConnection(int syncFd)
{
  #ifdef PANIC
  *logofs << "ServerProxy: PANIC! Function handleNewSyncConnection() "
          << "shouldn't be called at server side.\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Function handleNewSyncConnection() "
       << "shouldn't be called at server side.\n";

  HandleCleanup();

  return -1;
}

int ServerProxy::handleNewSyncConnectionFromProxy(int channelId)
{
  return handleNewGenericConnectionFromProxy(channelId, CHANNEL_SYNC, "localhost",
                                                 syncServerPort_, "sync");
}

int ServerProxy::handleNewKeybdConnection(int mediaFd)
{
  #ifdef PANIC
  *logofs << "ServerProxy: PANIC! Function handleNewKeybdConnection() "
          << "shouldn't be called at server side.\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Function handleNewKeybdConnection() "
       << "shouldn't be called at server side.\n";

  HandleCleanup();

  return -1;
}

int ServerProxy::handleNewKeybdConnectionFromProxy(int channelId)
{
  //
  // This is a special case. We connect keyboard to the
  // same X server where session is running. We should
  // write another wrapper that doesn't go through the
  // hostname or be able to connect to Unix sockets.
  //

  T_channel_type type = CHANNEL_KEYBD;
  char *hostname      = xServerDisplay_;
  int  port           = keybdServerPort_;
  char *label         = "embedded keyboard";

  if (checkNewGenericConnectionFromProxy(channelId, type, hostname, port, label) < 0)
  {
    return -1;
  }

  int keybdFd = socket(xServerAddrFamily_, SOCK_STREAM, PF_UNSPEC);

  if (keybdFd < 0)
  {
    #ifdef PANIC
    *logofs << "ServerProxy: PANIC! Call to socket failed. "
            << "Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to socket failed. "
         << "Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    return -1;
  }
  else if (connect(keybdFd, xServerAddr_, xServerAddrLength_) < 0)
  {
    #ifdef WARNING
    *logofs << "ServerProxy: WARNING! Connection to "
            << label << " server '" << xServerDisplay_
            << "' failed with error '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Warning" << ": Connection to "
         << label << " server '" << xServerDisplay_
         << "' failed with error '" << ESTR()
         << "'.\n";

    close(keybdFd);

    return -1;
  }

  //
  // Turn queuing off for path proxy-to-X-server.
  //

  if (control -> OptionServerNoDelay == 1)
  {
    SetNoDelay(keybdFd, control -> OptionServerNoDelay);
  }

  fdMap_[channelId] = keybdFd;
  channelMap_[keybdFd] = channelId;

  #ifdef TEST
  *logofs << "ServerProxy: Descriptor FD#" << keybdFd 
          << " mapped to channel ID#" << channelId << ".\n"
          << logofs_flush;
  #endif

  transports_[channelId] = new Transport(keybdFd);

  if (transports_[channelId] == NULL)
  {
    return -1;
  }

  channels_[channelId] = new KeybdChannel(transports_[channelId],
                                              compressor_, decompressor_);

  if (channels_[channelId] == NULL)
  {
    return -1;
  }

  increaseActiveChannels(channelId);

  channels_[channelId] -> handleConfiguration();

  return 1;
}

int ServerProxy::handleNewSambaConnection(int sambaFd)
{
  #ifdef PANIC
  *logofs << "ServerProxy: PANIC! Function handleNewSambaConnection() "
          << "shouldn't be called at server side.\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Function handleNewSambaConnection() "
       << "shouldn't be called at server side.\n";

  HandleCleanup();

  return -1;
}

int ServerProxy::handleNewSambaConnectionFromProxy(int channelId)
{
  //
  // Strangely enough, under some Windows OSes SMB
  // service doesn't bind to 'localhost'. Fall back
  // to 'localhost' if can't find computer name in
  // the environment. In future we should try to
  // bind to local device first and then try the
  // other IPs.
  //

  char *hostname = NULL;

  #ifdef __CYGWIN32__

  hostname = getenv("COMPUTERNAME");

  #endif

  if (hostname == NULL)
  {
    hostname = "localhost";
  }

  return handleNewGenericConnectionFromProxy(channelId, CHANNEL_SAMBA, hostname,
                                                 sambaServerPort_, "samba");
}

int ServerProxy::handleNewMediaConnection(int mediaFd)
{
  #ifdef PANIC
  *logofs << "ServerProxy: PANIC! Function handleNewMediaConnection() "
          << "shouldn't be called at server side.\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Function handleNewMediaConnection() "
       << "shouldn't be called at server side.\n";

  HandleCleanup();

  return -1;
}


int ServerProxy::handleNewMediaConnectionFromProxy(int channelId)
{
  return handleNewGenericConnectionFromProxy(channelId, CHANNEL_MEDIA, "localhost",
                                                 mediaServerPort_, "media");
}

int ServerProxy::checkNewGenericConnectionFromProxy(int channelId, T_channel_type type,
                                                        char *hostname, int port, char *label)
{
  if (channelId >= CONNECTIONS_LIMIT)
  {
    #ifdef PANIC
    *logofs << "ServerProxy: PANIC! Maximum mumber of available "
            << "channels exceeded.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Maximum mumber of available "
         << "channels exceeded.\n";

    return -1;
  }
  else if (channels_[channelId] != NULL)
  {
    #ifdef PANIC
    *logofs << "ServerProxy: PANIC! Trying to open a new connection "
            << "over an existing channel ID#" << channelId
            << " with FD#" << getFd(channelId) << ".\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Trying to open a new connection "
         << "over an existing channel ID#" << channelId
         << " with FD#" << getFd(channelId) << ".\n";

    return -1;
  }
  else if (port <= 0)
  {
    //
    // This happens when user has disabled
    // forwarding of the specific service.
    //

    #ifdef WARNING
    *logofs << "ServerProxy: WARNING! Refusing attempted connection "
            << "to " << label << " server.\n" << logofs_flush;
    #endif

    cerr << "Warning" << ": Refusing attempted connection "
         << "to " << label << " server.\n";

    return -1;
  }

  return 1;
}

int ServerProxy::handleNewGenericConnectionFromProxy(int channelId, T_channel_type type,
                                                         char *hostname, int port, char *label)
{
  if (checkNewGenericConnectionFromProxy(channelId, type, hostname, port, label) < 0)
  {
    return -1;
  }

  int serverPort = port;
  char *serverHost = hostname;
  int serverAddrFamily = AF_INET;
  sockaddr *serverAddr = NULL;
  unsigned int serverAddrLength = 0;

  #if defined(INFO) || defined (TEST)
  *logofs << "ServerProxy: Connecting to " << label
          << " server '" << serverHost << "' on TCP port '"
          << serverPort << "'.\n" << logofs_flush;
  #endif

  int serverIPAddr = GetHostAddress(serverHost);

  if (serverIPAddr == 0)
  {
    #ifdef PANIC
    *logofs << "ServerProxy: PANIC! Unknown " << label
            << " server host '" << serverHost << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Unknown " << label
         << " server host '" << serverHost
         << "'.\n";

    return -1;
  }

  sockaddr_in *serverAddrTCP = new sockaddr_in;

  serverAddrTCP -> sin_family = AF_INET;
  serverAddrTCP -> sin_port = htons(serverPort);
  serverAddrTCP -> sin_addr.s_addr = serverIPAddr;

  serverAddr = (sockaddr *) serverAddrTCP;
  serverAddrLength = sizeof(sockaddr_in);

  //
  // Connect to the requested server.
  //

  int serverFD = socket(serverAddrFamily, SOCK_STREAM, PF_UNSPEC);

  if (serverFD < 0)
  {
    #ifdef PANIC
    *logofs << "ServerProxy: PANIC! Call to socket failed. "
            << "Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to socket failed. "
         << "Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    delete serverAddrTCP;

    return -1;
  }
  else if (connect(serverFD, serverAddr, serverAddrLength) < 0)
  {
    #ifdef WARNING
    *logofs << "ServerProxy: WARNING! Connection to " << label
            << " server '" << serverHost << ":" << serverPort
            << "' failed with error '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Warning" << ": Connection to " << label
         << " server '" << serverHost << ":" << serverPort
         << "' failed with error '" << ESTR() << "'.\n";

    close(serverFD);

    delete serverAddrTCP;

    return -1;
  }

  delete serverAddrTCP;

  //
  // Turn queuing off for path proxy-to-server.
  //

  SetNoDelay(serverFD, 1);

  fdMap_[channelId] = serverFD;
  channelMap_[serverFD] = channelId;

  #ifdef TEST
  *logofs << "ServerProxy: Descriptor FD#" << serverFD 
          << " mapped to channel ID#" << channelId << ".\n"
          << logofs_flush;
  #endif

  transports_[channelId] = new Transport(serverFD);

  if (transports_[channelId] == NULL)
  {
    return -1;
  }

  switch (type)
  {
    case CHANNEL_SYNC:
    {
      channels_[channelId] = new SyncChannel(transports_[channelId],
                                                 compressor_, decompressor_);
      break;
    }
    case CHANNEL_KEYBD:
    {
      channels_[channelId] = new KeybdChannel(transports_[channelId],
                                                  compressor_, decompressor_);
      break;
    }
    case CHANNEL_SAMBA:
    {
      channels_[channelId] = new SambaChannel(transports_[channelId],
                                                  compressor_, decompressor_);
      break;
    }
    default:
    {
      channels_[channelId] = new MediaChannel(transports_[channelId],
                                                  compressor_, decompressor_);
      break;
    }
  }

  if (channels_[channelId] == NULL)
  {
    return -1;
  }

  #if defined(INFO) || defined (TEST)
  *logofs << "ServerProxy: Forwarded new connection to "
          << label << " server on port '" << serverPort
          << "'.\n" << logofs_flush;
  #endif

  cerr << "Info" << ": Forwarded new connection to "
       << label << " server on port '" << serverPort
       << "'.\n";

  increaseActiveChannels(channelId);

  channels_[channelId] -> handleConfiguration();

  return 1;
}

int ServerProxy::handleCheckLoad(T_proxy_load type)
{
  //
  // Issue a warning if we selected a cache
  // file but client side didn't tell us to
  // load it. This is probably because load
  // operation at remote side failed.
  //

  int channelCount = 0;

  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] != NULL &&
            channels_[channelId] -> getType() == CHANNEL_X)
    {
      channelCount++;
    }
  }

  if (channelCount == 0 && type == LOAD_IF_FIRST &&
          control -> PersistentCacheEnableLoad == 1 &&
              control -> PersistentCachePath != NULL &&
                  control -> PersistentCacheName != NULL &&
                      isTimestamp(lastLoadTs_) == 0)
  {
    #ifdef WARNING
    *logofs << "ServerProxy: WARNING! Cache file '" << control -> PersistentCachePath
            << "/" << control -> PersistentCacheName << "' not loaded.\n"
            << logofs_flush;
    #endif

    cerr << "Warning" << ": Cache file '" << control -> PersistentCachePath
         << "/" << control -> PersistentCacheName << "' not loaded.\n";

    //
    // Remove the cache file. In theory we can't be
    // sure of the reason why it was not loaded, but
    // in most cases it is because cache was not com-
    // patible or corrupted. Note also that even if
    // we don't remove the file now, cache would be
    // replaced at the time a new instance is created
    // at link shutdown.
    //

    #ifdef WARNING
    *logofs << "ServerProxy: WARNING! Removing supposedly "
            << "corrupted cache '" << control -> PersistentCachePath
            << "/" << control -> PersistentCacheName
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Warning" << ": Removing supposedly "
         << "corrupted cache '" << control -> PersistentCachePath
         << "/" << control -> PersistentCacheName << "'.\n";

    handleResetPersistentCache();
  }

  return 0;
}

int ServerProxy::handleCheckSave()
{
  //
  // Nothing to do on server proxy. Save must
  // be ordered by client side when the last
  // X channel is closed.
  //

  return 0;
}

int ServerProxy::handleLoadFromProxy()
{
  //
  // Check that either no X channel is
  // remaining or we are inside a reset.
  //

  int channelCount = 0;

  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] != NULL &&
            channels_[channelId] -> getType() == CHANNEL_X)
    {
      channelCount++;

      break;
    }
  }

  if (channelCount > 0 && reset_ != 1)
  {
    #ifdef PANIC
    *logofs << "ServerProxy: PANIC! Protocol violation "
            << "in command load with " << channelCount
            << " channels and reset " << reset_
            << ".\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Protocol violation "
         << "in command load from proxy.\n";

    return -1;
  } 
  else if (handleLoad() <= 0)
  {
    #ifdef PANIC
    *logofs << "ServerProxy: PANIC! Failed to load "
            << "stores from persistent cache.\n"
            << logofs_flush;
    #endif

    return -1;
  }

  return 1;
}

int ServerProxy::handleSaveFromProxy()
{
  int channelCount = 0;

  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] != NULL &&
            channels_[channelId] -> getType() == CHANNEL_X)
    {
      channelCount++;

      break;
    }
  }

  if (channelCount > 0 && reset_ != 1)
  {
    #ifdef PANIC
    *logofs << "ServerProxy: PANIC! Protocol violation "
            << "in command save with " << channelCount
            << " channels and reset " << reset_
            << ".\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Protocol violation "
         << "in command save from proxy.\n";

    return -1;
  }
  else if (handleSave() <= 0)
  {
    #ifdef PANIC
    *logofs << "ServerProxy: PANIC! Failed to save stores to persistent cache.\n"
            << logofs_flush;
    #endif

    return -1;
  }

  return 1;
}

int ServerProxy::handleSaveStores(ostream *cachefs, md5_state_t *md5StateStream,
                                      md5_state_t *md5StateClient) const
{
  if (clientStore_ -> saveRequestStores(cachefs, md5StateStream, md5StateClient,
                                            discard_checksum, use_data) < 0)
  {
    return -1;
  }
  else if (serverStore_ -> saveReplyStores(cachefs, md5StateStream, md5StateClient,
                                               use_checksum, discard_data) < 0)
  {
    return -1;
  }
  else if (serverStore_ -> saveEventStores(cachefs, md5StateStream, md5StateClient,
                                               use_checksum, discard_data) < 0)
  {
    return -1;
  }

  return 1;
}

int ServerProxy::handleLoadStores(istream *cachefs, md5_state_t *md5StateStream) const
{
  if (clientStore_ -> loadRequestStores(cachefs, md5StateStream,
                                            discard_checksum, use_data) < 0)
  {
    return -1;
  }
  else if (serverStore_ -> loadReplyStores(cachefs, md5StateStream,
                                               use_checksum, discard_data) < 0)
  {
    return -1;
  }
  else if (serverStore_ -> loadEventStores(cachefs, md5StateStream,
                                               use_checksum, discard_data) < 0)
  {
    return -1;
  }

  return 1;
}
