/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2003 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include "ReadBuffer.h"

#include "Transport.h"

//
// Set the verbosity level.
//

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG

ReadBuffer::ReadBuffer(Transport *transport, unsigned int initialReadSize,
                           unsigned int maximumReadSize)

  : transport_(transport), initialReadSize_(initialReadSize),
        maximumReadSize_(maximumReadSize)
{
  size_   = initialReadSize_;
  buffer_ = new unsigned char[size_];

  #ifdef VALGRIND

  memset(buffer_, 0, size_);

  #endif

  length_ = 0;
  start_  = 0;
}

ReadBuffer::~ReadBuffer()
{
  delete [] buffer_;
}

void ReadBuffer::partialReset()
{
  if (length_ == 0 && size_ > initialReadSize_)
  {
    //
    // Reset the buffer to the initial size
    // if for N milliseconds we didn't need
    // a bigger size.
    //

    #ifdef TEST
    *logofs << "ReadBuffer: Resetting buffer for FD#"
            << transport_ -> fd() << " from " << size_
            << " to " << initialReadSize_ << " bytes.\n"
            << logofs_flush;
    #endif

    delete [] buffer_;

    int newSize = initialReadSize_;

    unsigned char *newBuffer = new unsigned char[newSize];

    if (buffer_ == NULL)
    {
      #ifdef PANIC
      *logofs << "ReadBuffer: PANIC! Can't allocate "
              << newSize << " bytes of memory for buffer "
              << "in context [A].\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Can't allocate memory for "
           << "read buffer in context [A].\n";

      HandleCleanup();
    }

    buffer_ = newBuffer;
    size_   = newSize;

    #ifdef VALGRIND

    memset(buffer_, '\0', size_);

    #endif
  }
}

void ReadBuffer::fullReset()
{
  size_   = initialReadSize_;
  buffer_ = new unsigned char[size_];

  if (buffer_ == NULL)
  {
    #ifdef PANIC
    *logofs << "ReadBuffer: PANIC! Can't allocate "
            << size_ << " bytes of memory for buffer "
            << "in context [B].\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Can't allocate memory for "
         << "read buffer in context [B].\n";

    HandleCleanup();
  }

  #ifdef VALGRIND

  memset(buffer_, 0, size_);

  #endif

  length_ = 0;
  start_  = 0;
}

int ReadBuffer::readMessage()
{
  do
  {
    #ifdef DEBUG
    *logofs << "ReadBuffer: Buffer size is " << size_ 
            << " length is " << length_ << " and start is " 
            << start_ << ".\n" << logofs_flush;
    #endif

    if (length_ == size_)
    {
      //
      // The buffer is full. Increase its
      // size so we can read some more.
      //

      unsigned int newSize = (size_ < maximumReadSize_ ?
                                  size_ << 2 : size_ << 1);

      #ifdef TEST
      *logofs << "ReadBuffer: Resizing buffer for FD#"
              << transport_ -> fd() << " from " << size_
              << " to " << newSize << " bytes.\n"
              << logofs_flush;
      #endif

      unsigned char *newBuffer = new unsigned char[newSize];

      if (newBuffer == NULL)
      {
        #ifdef PANIC
        *logofs << "ReadBuffer: PANIC! Can't allocate "
                << newSize << " bytes of memory for buffer "
                << "in context [C].\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Can't allocate memory for "
             << "read buffer in context [C].\n";

        HandleCleanup();
      }

      #ifdef VALGRIND

      memset(newBuffer, '\0', newSize);

      #endif

      memcpy(newBuffer, buffer_ + start_, length_);

      delete [] buffer_;

      buffer_ = newBuffer;
      size_   = newSize;
    }
    else if (start_ != 0 && length_ != 0)
    {
      //
      // If any bytes are left over from last time 
      // (due to a partial message), shift them to 
      // the beginning of the buffer.
      //

      #ifdef TEST
      *logofs << "ReadBuffer: Moving " << length_ << " bytes of data "
              << "at beginning of buffer for FD#" << transport_ -> fd() 
              << ".\n" << logofs_flush;
      #endif

      memmove(buffer_, buffer_ + start_, length_);
    }

    start_ = 0;

    #ifdef DEBUG
    *logofs << "ReadBuffer: Buffer size is now " << size_ 
            << " length is " << length_ << " and start is " 
            << start_ << ".\n" << logofs_flush;
    #endif

    //
    // In theory we could get as much data as would fit in
    // the buffer, actually we impose a limit on amount of
    // data read. Infact we maybe need only a few bytes to
    // complete the next message and reading everything we:
    //
    // - Have to move more bytes at the next call.
    //
    // - Cause channel or proxy to consume too many
    //   ticks handling messages leaving the buffer
    //   in pending state.
    //
    // - Keep the buffer bigger than strictly needed.
    //

    int readLength = size_ - length_;

    if (length_ == 0)
    {
      if (readLength > (int) initialReadSize_)
      {
        readLength = initialReadSize_;
      }
    }
    else
    {
      if (readLength > (int) maximumReadSize_)
      {
        readLength = maximumReadSize_;
      }
    }

    unsigned char *readData = buffer_ + length_;

    #ifdef DEBUG
    *logofs << "ReadBuffer: Going to read " << readLength 
            << " bytes from FD#" << transport_ -> fd() << ".\n"
            << logofs_flush;
    #endif

    int bytesRead = transport_ -> read(readData, readLength);

    if (bytesRead > 0)
    {
      #ifdef TEST
      *logofs << "ReadBuffer: Read " << bytesRead << " bytes from FD#" 
              << transport_ -> fd() << ".\n" << logofs_flush;
      #endif

      length_ += bytesRead;
    }
    else if (bytesRead < 0)
    {
      #ifdef TEST
      *logofs << "ReadBuffer: Error detected reading from FD#"
              << transport_ -> fd() << ".\n" << logofs_flush;
      #endif

      return -1;
    }
    #ifdef TEST
    else
    {
      *logofs << "ReadBuffer: WARNING! No data read from FD#"
              << transport_ -> fd() << ".\n" << logofs_flush;
    }
    #endif
  }
  while (transport_ -> pending() > 0);

  return length_;
}

const unsigned char *ReadBuffer::getMessage(unsigned int &controlLength,
                                                unsigned int &dataLength)
{
  unsigned int trailerLength;

  #ifdef DEBUG
  *logofs << "ReadBuffer: Going to locate message with start at "
          << start_ << " and length " << length_ << ".\n"
          << logofs_flush;
  #endif

  int located = locateMessage(buffer_ + start_, buffer_ + start_ + length_,
                                  controlLength, dataLength, trailerLength);

  if (located <= 0)
  {
    //
    // No message or error was detected 
    // uncompressing the message.
    //

    #ifdef DEBUG
    *logofs << "ReadBuffer: No message was located for FD#"
            << transport_ -> fd() << ".\n" << logofs_flush;
    #endif

    return NULL;
  }
  else
  {
    const unsigned char *result = buffer_ + start_;

    if (dataLength > 0)
    {
      //
      // Message contains data, so go to the
      // first byte of payload.
      //

      result += trailerLength;

      start_  += (dataLength + trailerLength);
      length_ -= (dataLength + trailerLength);
    }
    else
    {
      //
      // It is a control message.
      //

      start_  += (controlLength + trailerLength);
      length_ -= (controlLength + trailerLength);
    }

    #ifdef DEBUG
    *logofs << "ReadBuffer: Located message for FD#" << transport_ -> fd()
            << " with control length " << controlLength << " and data length "
            << dataLength << ".\n" << logofs_flush;
    #endif

    return result;
  }
}
