/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2003 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#ifndef Proxy_H
#define Proxy_H

#include <sys/types.h>

#ifdef _AIX
#include <sys/select.h>
#endif

#include "Misc.h"
#include "Timestamp.h"

#include "Channel.h"
#include "Transport.h"
#include "EncodeBuffer.h"
#include "ProxyReadBuffer.h"

//
// Set the verbosity level.
//

#define WARNING
#define PANIC
#undef  TEST
#undef  DEBUG

class Proxy
{
  public:

  //
  // Maximum number of X connections supported.
  //

  static const int CONNECTIONS_LIMIT = 256;

  //
  // This makes possible 3 control
  // messages in a single frame.
  //

  static const int CONTROL_CODES_LENGTH = 9;

  Proxy(int fd);

  virtual ~Proxy();

  void setReadFDs(fd_set *fdSet, int &fdMax);

  void setWriteFDs(fd_set *fdSet, int &fdMax);

  void setTimeout(T_timestamp &tsMax);

  void setSchedule();

  int handleRead(fd_set &fdSet);

  int handleFlush(fd_set &fdSet);

  //
  // Perform operations on proxy link
  // or its own channels.
  //

  int handleRead();

  int handleRead(int fd);

  int handleSplit();

  int handleAbortSplit(int fd);

  int handleMotion();

  int handleWakeup();

  int handleCongestion();

  int handleFlush(T_flush type);

  int handleFlush(T_flush type, int fd);

  int handlePing();

  int handleReset();

  int handleFinish();

  int handleShutdown();

  int handleStatistics(int type, ostream *statofs);

  int handleAlert(int alert);

  virtual int handleNotify(T_notification_type type) = 0;

  int handleSocketConfiguration();

  int handleLinkConfiguration();

  int handleCacheConfiguration();

  virtual int handleNewXConnection(int clientFD) = 0;

  virtual int handleNewXConnectionFromProxy(int channelId) = 0;

  virtual int handleNewSyncConnection(int clientFD) = 0;

  virtual int handleNewSyncConnectionFromProxy(int channelId) = 0;

  virtual int handleNewKeybdConnection(int clientFD) = 0;

  virtual int handleNewKeybdConnectionFromProxy(int channelId) = 0;

  virtual int handleNewSambaConnection(int clientFD) = 0;

  virtual int handleNewSambaConnectionFromProxy(int channelId) = 0;

  virtual int handleNewMediaConnection(int clientFD) = 0;

  virtual int handleNewMediaConnectionFromProxy(int channelId) = 0;

  protected:

  //
  // Codes used for control messages in
  // proxy-to-proxy protocol.
  //
  // The following codes are currently
  // unused.
  //
  // CODE_ALERT_REPLY,
  // CODE_RESET_REPLY,
  // CODE_LOAD_REPLY,
  // CODE_SAVE_REPLY,
  // CODE_SHUTDOWN_REPLY,
  // CODE_CONFIGURATION_REQUEST,
  // CODE_CONFIGURATION_REPLY.
  //

  typedef enum
  {
    CODE_NEW_X_CONNECTION,
    CODE_NEW_SYNC_CONNECTION,
    CODE_NEW_KEYBD_CONNECTION,
    CODE_NEW_SAMBA_CONNECTION,
    CODE_NEW_MEDIA_CONNECTION,
    CODE_SWITCH_CONNECTION,
    CODE_DROP_CONNECTION,
    CODE_FINISH_CONNECTION,
    CODE_BEGIN_CONGESTION,
    CODE_END_CONGESTION,
    CODE_ALERT_REQUEST,
    CODE_ALERT_REPLY,
    CODE_RESET_REQUEST,
    CODE_RESET_REPLY,
    CODE_LOAD_REQUEST,
    CODE_LOAD_REPLY,
    CODE_SAVE_REQUEST,
    CODE_SAVE_REPLY,
    CODE_SHUTDOWN_REQUEST,
    CODE_SHUTDOWN_REPLY,
    CODE_PING_REQUEST,
    CODE_PING_REPLY,
    CODE_CONFIGURATION_REQUEST,
    CODE_CONFIGURATION_REPLY,
    CODE_STATISTICS_REQUEST,
    CODE_STATISTICS_REPLY,
    CODE_LAST_TAG
  } T_proxy_code;

  typedef enum
  {
    OPERATION_IN_MESSAGES,
    OPERATION_IN_CONFIGURATION,
    OPERATION_IN_STATISTICS,
    OPERATION_LAST_TAG
  } T_proxy_operation;

  int handleWrite();

  int handleFinish(int channelId);

  int handleFinishFromProxy(int channelId);

  int handleStatisticsFromProxy(int type);

  int handleStatisticsFromProxy(const unsigned char *message, unsigned int length);

  int handlePingFromProxy();

  int handleLoad();

  int handleSave();

  int handleSwitch(int channelId);

  int handleControl(T_proxy_code controlCode, int controlData = 0);

  public:

  //
  // Check if proxy or any channel has bytes
  // left in read buffer. Returns the file
  // descriptor corresponding to the first
  // matching channel.
  //

  int needPending() const;

  //
  // Check if any channels has prioritized
  // data to be sent.
  //

  int needPriority() const;

  //
  // Check if there are images currently
  // being streamed.
  //

  virtual int needSplit() const = 0;

  //
  // Check if any (server) channel needs
  // a timeout because of motion events
  // to flush.
  //

  virtual int needMotion() const = 0;

  //
  // Check if any (client) channel needs
  // to wakeup agent's clients.
  //

  virtual int needWakeup() const = 0;

  //
  // Check if any channel has grown its
  // buffer beyond the limit.
  //

  int needLimit() const;

  //
  // Check if any channel needs to check
  // its own congestion state.
  //

  int needCongestion() const;

  //
  // Check if proxy need a timeout to
  // write scheduled data.
  //

  int needFlush() const
  {
    if (transport_ -> blocked() == 0 &&
            transport_ -> length() > 0)
    {
      return fd_;
    }

    return -1;
  }

  //
  // Returns the number of active channels
  // that currently managed by this proxy.
  //

  int getChannels(T_channel_type type);

  //
  // If descriptor corresponds to a valid
  // channel, returns the type of traffic
  // carried by it.
  //

  T_channel_type getType(int fd);

  //
  // True if proxy is presently recovering
  // from a reset,
  //

  int getReset() const
  {
    return reset_;
  }

  //
  // Did remote peer signaled a shutdown?
  //

  int getShutdown() const
  {
    return shutdown_;
  }

  //
  // Remap these calls to the IO buffers.
  //

  int getLength(int fd) const
  {
    if (fd == fd_)
    {
      return transport_ -> length();
    }

    int channelId = getChannel(fd);

    if (channelId < 0 || channels_[channelId] == NULL)
    {
      return 0;
    }

    return transports_[channelId] -> length();
  }

  int getQueued(int fd) const
  {
    if (fd == fd_)
    {
      return transport_ -> queued();
    }

    int channelId = getChannel(fd);

    if (channelId < 0 || channels_[channelId] == NULL)
    {
      return 0;
    }

    return transports_[channelId] -> queued();
  }

  int getPending(int fd) const
  {
    if (fd == fd_)
    {
      return pending_;
    }

    int channelId = getChannel(fd);

    if (channelId < 0 || channels_[channelId] == NULL)
    {
      return 0;
    }

    return channels_[channelId] -> getPending();
  }

  //
  // Check if proxy or given channel has data
  // in buffers because of a blocking write.
  //

  int getBlocked(int fd) const
  {
    if (fd == fd_)
    {
      return transport_ -> blocked();
    }

    int channelId = getChannel(fd);

    if (channelId < 0 || channels_[channelId] == NULL)
    {
      return 0;
    }

    return transports_[channelId] -> blocked();
  }

  //
  // Check if an immediate flush of proxy link
  // is requested by channels. This is the case
  // when an X connection is waiting a reply.
  //

  int getPriority(int fd) const
  {
    if (fd == fd_)
    {
      return priority_;
    }

    int channelId = getChannel(fd);

    if (channelId < 0 || channels_[channelId] == NULL)
    {
      return 0;
    }

    return channels_[channelId] -> getPriority();
  }

  int getCongestion(int fd) const
  {
    if (fd == fd_)
    {
      return congestion_;
    }

    int channelId = getChannel(fd);

    if (channelId < 0 || channels_[channelId] == NULL)
    {
      return 0;
    }

    return channels_[channelId] -> getCongestion();
  }

  //
  // Let statistics class to get info from
  // message stores.
  //

  const ClientStore * const getClientStore() const
  {
    return clientStore_;
  }

  const ServerStore * const getServerStore() const
  {
    return serverStore_;
  }

  //
  // Implement persistence of cache on disk.
  //

  protected:

  typedef enum
  {
    LOAD_IF_ANY,
    LOAD_IF_FIRST
  } T_proxy_load;

  virtual int handleCheckLoad(T_proxy_load type) = 0;

  virtual int handleCheckSave() = 0;

  virtual int handleLoadFromProxy() = 0;

  virtual int handleSaveFromProxy() = 0;

  char *handleSaveStores(const char *savePath) const;

  virtual int handleSaveStores(ostream *cachefs, md5_state_t *md5StateStream,
                                   md5_state_t *md5StateClient) const = 0;

  int handleSaveVersion(unsigned char *buffer, int &major, int &minor, int &patch) const;

  void handleFailOnSave(const char *fullName, const char *failContext) const;

  const char *handleLoadStores(const char *loadPath, const char *loadName) const;

  virtual int handleLoadStores(istream *cachefs, md5_state_t *md5StateStream) const = 0;

  int handleLoadVersion(const unsigned char *buffer, int &major, int &minor, int &patch) const;

  void handleFailOnLoad(const char *fullName, const char *failContext) const;

  int handleResetPersistentCache();

  int handleResetOpcodes();

  int handleResetStores();

  int handleResetCaches();

  //
  // Choose a new timeout according to table.
  //

  void handleIncreaseRetry()
  {
    if (lastRetryIndex_ < 4)
    {
      lastRetryIndex_++;

      lastRetryTs_.tv_usec = (long) (control -> FlushTimeout * 1000 *
                                         lastRetryTable_[lastRetryIndex_]);
    }
    else
    {    
      lastRetryTs_.tv_usec = (long) (lastRetryTs_.tv_usec *
                                         lastRetryTable_[lastRetryIndex_]);
    }

    if (lastRetryTs_.tv_usec / 1000 >= control -> PingTimeout)
    {
      setTimestamp(lastRetryTs_, control -> PingTimeout);
    }
  }

  void handleResetRetry()
  {
    lastRetryIndex_ = 0;

    lastRetryTs_.tv_sec  = 0;
    lastRetryTs_.tv_usec = (long) (control -> FlushTimeout * 1000 *
                                       lastRetryTable_[lastRetryIndex_]);

    if (lastRetryTs_.tv_usec < control -> RetryTimeout * 1000)
    {
      lastRetryTs_.tv_usec = control -> RetryTimeout * 1000;
    }
  }

  //
  // Reset counters used to determine
  // the best time to flush the proxy
  // link.
  //

  void handleResetFlush();

  //
  // Update the timestamps to postpone
  // any wakeup timeout.
  //

  virtual void handleUpdateWakeup() const = 0;

  protected:

  virtual int getFd(int channelId) const = 0;
  virtual int getChannel(int fd) const = 0;

  virtual void cleanupChannelMap(int channelId) = 0;

  void increaseActiveChannels(int channelId);
  void decreaseActiveChannels(int channelId);

  //
  // Proxy create its own transport.
  //

  ProxyTransport *transport_;

  //
  // Compressor and decompressor are shared
  // between channels and message stores.
  //

  Compressor   *compressor_;
  Decompressor *decompressor_;

  //
  // Map NX specific opcodes.
  //

  OpcodeStore *opcodeStore_;

  //
  // Stores are shared between channels.
  //

  ClientStore *clientStore_;
  ServerStore *serverStore_;

  //
  // Starting from protocol level 3 even
  // client and server caches are shared
  // between channels.
  //

  ClientCache *clientCache_;
  ServerCache *serverCache_;

  //
  // Used for IO operations on proxy link.
  //

  int fd_;

  int inputChannel_;
  int outputChannel_;

  //
  // Used to read data sent from peer proxy.
  //

  ProxyReadBuffer readBuffer_;

  //
  // Used to send data to peer proxy.
  //

  EncodeBuffer encodeBuffer_;

  //
  // Control messages' array.
  //

  int controlLength_;

  unsigned char controlCodes_[CONTROL_CODES_LENGTH];

  //
  // Table of channel classes taking
  // care of open X connections.
  //

  Channel *channels_[CONNECTIONS_LIMIT];

  //
  // Table of open sockets.
  //

  Transport *transports_[CONNECTIONS_LIMIT];

  //
  // Lower and upper limit of active
  // and next channel to be handled
  // first.
  //

  int lowerChannel_;
  int upperChannel_;
  int firstChannel_;

  //
  // The number of active connections.
  //

  int activeChannels_;

  //
  // Proxy can be decoding messages,
  // handling a link reconfiguration,
  // or decoding statistics.
  //

  int operation_;

  //
  // True if there are pending messages
  // in proxy's read buffer.
  //

  int pending_;

  //
  // Force flush because of prioritized
  // control messages to send.
  //

  int priority_;

  //
  // Remote peer requested shutdown.
  //

  int shutdown_;

  //
  // We are in the middle of a reset.
  //

  int reset_;

  //
  // We are in the middle of a network
  // congestion in the path to remote
  // proxy.
  //

  int congestion_;

  //
  // Is there any channel at remote end
  // that's not consuming enough data?
  //

  int congestions_[CONNECTIONS_LIMIT];

  //
  // Timestamp of last packet received
  // or sent to remote proxy. Used to
  // detect lost connection.
  //

  T_timestamp lastBytesInTs_;
  T_timestamp lastBytesOutTs_;

  //
  // Proxy needs to monitor the TCP output
  // buffer to know when is time to send a
  // new packet. At each iteration choose
  // a new timeout according to the table.
  //

  T_timestamp lastRetryTs_;
  int         lastRetryIndex_;
  double      lastRetryTable_[5];

  //
  // Timestamp of last ping request sent
  // to remote proxy.
  //

  T_timestamp lastPingTs_;

  //
  // Timestamp of last 'no data received'
  // alert dialog showed to user.
  //

  T_timestamp lastAlertTs_;

  //
  // Timestamp of last buffer limit exceeded
  // that caused to stop reading from remote
  // proxy.
  //

  T_timestamp lastLimitTs_;

  //
  // Were message stores populated from
  // data on disk.
  //

  T_timestamp lastLoadTs_;

  //
  // Pointer to stream descriptor where
  // proxy is producing statistics.
  //

  ostream *lastStatisticsStream_;

  //
  // Cache values of own internal state
  // for performance reasons.
  //

  int stateReadProxy_;
  int stateReadChannels_;

  int stateUnblockProxy_;
  int stateUnblockChannels_;

  int stateLimit_;
  int stateRetry_;
  int statePending_;
  int statePriority_;
  int stateMotion_;
  int stateWakeup_;
  int stateCongestion_;
  int stateSplit_;
  int stateFlush_;

  int stateBlocked_;
  int stateQueued_;
  int stateLength_;
  int stateFlushable_;

  int stateNotify_;
};

#endif /* Proxy_H */
