/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2003 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#ifndef ClientChannel_H
#define ClientChannel_H

#include "Channel.h"

#include "SequenceQueue.h"

#include "ClientReadBuffer.h"

//
// Set the verbosity level.
//

#define PANIC
#undef  TEST
#undef  DEBUG

//
// Define this to log a line when a channel
// is created or destroyed.
//

#undef  REFERENCES

//
// This class implements the client
// side compression of X protocol.
//

class ClientChannel : public Channel
{
  public:

  ClientChannel(Transport *transport, Compressor *compressor,
                    Decompressor *decompressor);

  virtual ~ClientChannel();

  virtual int handleRead(EncodeBuffer &encodeBuffer);

  virtual int handleWrite(const unsigned char *message, unsigned int length);

  virtual int handleSplit(EncodeBuffer &encodeBuffer, MessageStore *store, 
                              const unsigned char *buffer, const unsigned int size);

  virtual int handleSplit(EncodeBuffer &encodeBuffer, int packetLimit);

  virtual int handleUnsplit(DecodeBuffer &decodeBuffer, MessageStore *store,
                                unsigned char *buffer, const unsigned int size);

  virtual int handleUnsplit(DecodeBuffer &decodeBuffer);

  virtual int handleAbortSplit(EncodeBuffer &encodeBuffer);

  virtual int handleAbortSplit(DecodeBuffer &decodeBuffer);

  virtual int handleMotion(EncodeBuffer &encodeBuffer, int forceMotion = 0)
  {
    return 0;
  }

  //
  // Wakeup agent's clients waiting
  // for sync or karma events.
  //

  virtual int handleWakeup(EncodeBuffer &encodeBuffer);

  virtual int handleConfiguration();

  virtual int handleFinish();

  virtual int handleNotify(T_notification_type type);
  
  virtual int handleReset();

  virtual int needSplit() const
  {
    return (finish_ == 0 && clientStore_ ->
                getSplitStore() -> getSize() > 0);
  }

  virtual int needMotion() const
  {
    return 0;
  }

  virtual int needWakeup() const
  {
    return (finish_ == 0 && (syncCounter_ + karmaCounter_ > 0));
  }

  virtual int updateWakeup()
  {
    if (karmaCounter_ > 0)
    {
      T_timestamp tsNow = getTimestamp();

      for (int client = lowerClient_;
               client <= upperClient_; client++)
      {
        if (isTimestamp(karmaClientTs_[client]) == 1)
        {
          karmaClientTs_[client] = tsNow;
        }
      }
    }

    return ((karmaCounter_ + syncCounter_) > 0);
  }

  virtual int needLimit()
  {
    return (transport_ -> length() >=
                control -> TransportXBufferLimit);
  }

  virtual int getLimit()
  {
    return (control -> TransportXBufferLimit);
  }

  virtual T_channel_type getType()
  {
    return CHANNEL_X;
  }

  void setBigEndian(int flag);

  protected:

  int handleFastReadRequest(EncodeBuffer &encodeBuffer, const unsigned char &opcode,
                                const unsigned char *&buffer, const unsigned int &size);

  int handleFastWriteReply(DecodeBuffer &decodeBuffer, unsigned char &opcode,
                               unsigned char *&buffer, unsigned int &size);

  int handleFastWriteEvent(DecodeBuffer &decodeBuffer, unsigned char &opcode,
                               unsigned char *&buffer, unsigned int &size);

  //
  // Short circuit simple replies at client
  // side and taint the original request as
  // a no-operation.
  //

  int handleTaintRequest(unsigned char &opcode);

  int handleTaintError(unsigned char request);

  //
  // How to handle sequence counter
  // in notification event.
  //

  enum T_notification
  {
    immediate,
    deferred
  };

  //
  // Send notification event to agent.
  //

  int handleWakeup(unsigned char type, enum T_notification how,
                       int client, int request, int position,
                           int completion);

  //
  // Commit target requests once they
  // are recomposed.
  //

  int handleCommit(EncodeBuffer &encodeBuffer, const unsigned char *buffer,
                       const unsigned int size);

  //
  // Model notification of congestion events.
  //

  virtual int isCongested()
  {
    return (transport_ -> blocked() == 1);
  }

  virtual int isReliable()
  {
    return 1;
  }

  void handleDecodeCharInfo(DecodeBuffer &, unsigned char *);

  //
  // Handle the MIT-SHM initialization
  // messages exchanged with the remote
  // proxy.
  //

  int handleShmemInit(EncodeBuffer &encodeBuffer, const unsigned char opcode,
                          unsigned int &stage, const unsigned char *buffer,
                              const unsigned int size);

  int handleShmemInit(DecodeBuffer &decodeBuffer, unsigned char &opcode,
                          unsigned char *&buffer, unsigned int &size);

  //
  // Own read buffer. It is able to identify
  // full messages read from X descriptor.
  //

  ClientReadBuffer readBuffer_;

  //
  // Opcode of last request coming from X
  // client.
  //

  unsigned char clientOpcode_;
  unsigned char serverOpcode_;

  //
  // Sequence number of last request coming
  // from X client or X server.
  //

  unsigned int clientSequence_;
  unsigned int serverSequence_;

  //
  // Last sequence number known by client. It can
  // be the real sequence generated by server or
  // the one of the last auto-generated event.
  //

  unsigned int lastSequence_;

  //
  // Used to identify replies based on sequence
  // number of original request.
  //

  SequenceQueue sequenceQueue_;

  //
  // Current client id selected as target
  // of an image split.
  //

  int splitClient_;

  //
  // Sleeping clients.
  //

  T_timestamp syncClientTs_[CONNECTIONS_LIMIT];
  T_timestamp karmaClientTs_[CONNECTIONS_LIMIT];

  //
  // Handle timeouts on sleeping
  // clients.
  //

  int syncCounter_;
  int karmaCounter_;

  //
  // How many sync requests we
  // have tainted so far.
  //

  int taintCounter_;

  private:

  //
  // Keep track of object 
  // creation and deletion.
  //

  #ifdef REFERENCES

  static int references_;

  #endif
};

#endif /* ClientChannel_H */
