/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2003 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include <unistd.h>
#include <sys/types.h>
#include <signal.h>
#include <fcntl.h>

#include "NX.h"

#include "Misc.h"

#include "Types.h"
#include "Timestamp.h"

#include "Control.h"
#include "Statistics.h"
#include "Proxy.h"
#include "Context.h"

#include "Keeper.h"

//
// Set the verbosity level.
//

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG
#undef  DUMP

#define PATH_LENGTH_LIMIT     512

//
// Need these from main loop to
// dispose the unused resources.
//

extern Control    *control;
extern Statistics *statistics;
extern Proxy      *proxy;
extern Context    *context;

//
// Need these to prevent cleanup
// of resources maintained by the
// main process.
//

extern ostream *statofs;

extern int useDaemonMode;
extern int useUnixSocket;

extern int lastDialog;
extern int lastWatchdog;
extern int lastKeeper;

//
// Release the most expensive resources.
//

static void PartialCleanup();

//
// Start a nxclient process in dialog mode.
//

int NXDialog(const char *caption, const char *message,
                 const char *type, int local, const char* display)
{
  //
  // Be sure log file is valid.
  //

  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  int pid;

  #if defined(INFO) || defined(TEST)
  *logofs << "NXDialog: Going to fork with NX pid '"
          << getpid() << "'.\n" << logofs_flush;
  #endif

  pid = fork();

  if (pid != 0)
  {
    if (pid < 0)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "NXDialog: PANIC! Function fork failed with result '"
              << pid << "'. Error is " << EGET() << " '" << ESTR()
              << "'.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Function fork failed with result '"
           << pid << "'. Error is " << EGET() << " '" << ESTR()
           << "'.\n";

      HandleCleanup();
    }
    #if defined(INFO) || defined(TEST)
    else
    {
      *logofs << "NXDialog: Created NX dialog process with pid '"
              << pid << "'.\n" << logofs_flush;
    }
    #endif

    return pid;
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "NXDialog: Executing child with pid '" << getpid()
          << "' and parent '" << getppid() << "'.\n"
          << logofs_flush;
  #endif

  //
  // Close all the inherited sockets as soon
  // as the new process is created. Use brute
  // force, just to be sure.
  //

  for (int i = 3; i < 256; i++)
  {
    if (fcntl(i, F_SETFD, 1) != 0 && EGET() != EBADF)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "NXDialog: PANIC! Cannot set close-on-exec "
              << "on FD#" << i << ". Error is " << EGET()
              << " '" << ESTR() << "'.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Cannot set close-on-exec on FD#"
           << i << ". Error is " << EGET() << " '" << ESTR()
           << "'.\n";

      HandleCleanup();
    }
  }

  #if defined(INFO) || defined(TEST)
  *logofs << "NXDialog: Running external NX dialog with caption '"
          << caption << "' message '" << message << "' type '"
          << type << "' local '" << local << "' display '"
          << display << "'.\n"
          << logofs_flush;
  #endif

  #ifdef __APPLE__

  char *client = "/usr/NX/bin/nxclient.app/Contents/MacOS/nxclient";

  #else

  char *client = "nxclient";

  #endif

  for (int i = 0; i < 2; i++)
  {
    if (local != 0)
    {
      execlp(client, client, "-dialog", type, "-caption", caption,
                 "-message", message, "-local", "-display", display, NULL);
    }
    else
    {
      execlp(client, client, "-dialog", type, "-caption", caption,
                 "-message", message, "-display", display, NULL);
    }

    #ifdef WARNING
    *logofs << "NXDialog: WARNING! Couldn't invoke 'nxclient'. Error is "
            << EGET() << " '" << ESTR() << "'.\n" << logofs_flush;
    #endif

    cerr << "Warning" << ": Couldn't invoke 'nxclient'. Error is "
         << EGET() << " '" << ESTR() << "'.\n";

    char newPath[PATH_LENGTH_LIMIT];

    strcpy(newPath, "/usr/NX/bin:/opt/NX/bin:/usr/local/NX/bin:");

    #ifdef __APPLE__

    strcat(newPath, "/usr/NX/bin/nxclient.app/Contents/MacOS:");

    #endif

    #ifdef __CYGWIN32__

    strcat(newPath, ".:");

    #endif

    int newLength = strlen(newPath);

    char *oldPath = getenv("PATH");

    strncpy(newPath + newLength, oldPath, PATH_LENGTH_LIMIT - newLength - 1);

    newPath[PATH_LENGTH_LIMIT - 1] = '\0';

    #ifdef WARNING
    *logofs << "NXDialog: WARNING! Trying with path '" << newPath
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Warning" << ": Trying with path '" << newPath
         << "'.\n";

    //
    // Solaris doesn't seem to have
    // function setenv().
    //

    #ifdef __sun

    char newEnv[PATH_LENGTH_LIMIT];

    sprintf(newEnv,"PATH=%s", newPath);

    putenv(newEnv);

    #else

    setenv("PATH", newPath, 1);

    #endif
  }

  //
  // Hopefully useless.
  //

  exit(0);
}

//
// Wait until timeout is elapsed
// and kill the parent process.
//

int NXWatchdog(int timeout)
{
  //
  // Be sure log file is valid.
  //

  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  int pid;

  #if defined(INFO) || defined(TEST)
  *logofs << "NXWatchdog: Going to fork with NX pid '"
          << getpid() << "'.\n" << logofs_flush;
  #endif

  pid = fork();

  if (pid != 0)
  {
    if (pid < 0)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "NXWatchdog: PANIC! Function fork failed with result '"
              << pid << "'. Error is " << EGET() << " '" << ESTR()
              << "'.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Function fork failed with result '"
           << pid << "'. Error is " << EGET() << " '" << ESTR()
           << "'.\n";

      HandleCleanup();
    }
    #if defined(INFO) || defined(TEST)
    else
    {
      *logofs << "NXWatchdog: Created NX watchdog process with pid '"
              << pid << "'.\n" << logofs_flush;
    }
    #endif

    return pid;
  }

  int parent = getppid();

  #if defined(INFO) || defined(TEST)
  *logofs << "NXWatchdog: Executing child with pid '" << getpid()
          << "' and parent '" << parent << "'.\n"
          << logofs_flush;
  #endif

  //
  // Get rid of unused resources.
  //

  PartialCleanup();

  EnableSignals();

  //
  // To cancel the watchdog just kill
  // this process.
  //

  T_timestamp startTs = getTimestamp();

  for (;;)
  {
    if (diffTimestamp(startTs, getTimestamp()) >= timeout * 1000)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "NXWatchdog: Timeout of " << timeout
              << " seconds raised in watchdog.\n"
              << logofs_flush;
      #endif

      //
      // Don't complain if parent is dead
      // and never kill the init process.
      //

      if (parent == getppid() && parent != 1)
      {
        if (kill(parent, SIGTERM) < 0 && EGET() != ESRCH)
        {
          #ifdef PANIC
          *logofs << "NXWatchdog: PANIC! Watchdog couldn't kill parent "
                  << "process with pid '" << parent << "'.\n"
                  << logofs_flush;
          #endif

          cerr << "Error" << ": Watchdog couldn't kill parent "
               << "process with pid '" << parent << "'.\n";
        }
      }
      #if defined(INFO) || defined(TEST)
      else
      {
        *logofs << "NXWatchdog: Parent process appears to be dead.\n"
                << logofs_flush;
      }
      #endif

      HandleCleanup();
    }

    sleep(1);
  }

  //
  // Hopefully useless.
  //

  exit(0);
}

int NXKeeper(int caches, int images, const char *root)
{
  //
  // Be sure log file is valid.
  //

  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  if (caches == 0 && images == 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "NXKeeper: No NX cache house-keeping needed.\n"
            << logofs_flush;
    #endif

    return 0;
  }

  int pid;

  #if defined(INFO) || defined(TEST)
  *logofs << "NXKeeper: Going to fork with NX pid '"
          << getpid() << "'.\n" << logofs_flush;
  #endif

  pid = fork();

  if (pid != 0)
  {
    if (pid < 0)
    {
      #if defined(INFO) || defined(TEST)
      *logofs << "NXKeeper: PANIC! Function fork failed with result '"
              << pid << "'. Error is " << EGET() << " '" << ESTR()
              << "'.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Function fork failed with result '"
           << pid << "'. Error is " << EGET() << " '" << ESTR()
           << "'.\n";

      HandleCleanup();
    }
    #if defined(INFO) || defined(TEST)
    else
    {
      *logofs << "NXKeeper: Created NX keeper process with pid '"
              << pid << "'.\n" << logofs_flush;
    }
    #endif

    return pid;
  }

  int parent = getppid();

  #if defined(INFO) || defined(TEST)
  *logofs << "NXKeeper: Executing child with pid '" << getpid()
          << "' and parent '" << parent << "'.\n"
          << logofs_flush;
  #endif

  #if defined(INFO) || defined(TEST)
  *logofs << "NXKeeper: Going to run with caches " << caches
          << " images " << images << " and root " << root
          << " at " << strTimestamp() << ".\n" << logofs_flush;
  #endif

  //
  // Create the house-keeper class.
  //

  int timeout = control -> KeeperTimeout;

  Keeper keeper(caches, images, root, 100);

  //
  // Get rid of unused resources. Root path
  // must be copied in keeper's constructor
  // before control is deleted.
  //
  
  PartialCleanup();

  EnableSignals();

  //
  // Decrease the priority of this process.
  //
  // The following applies to Cygwin: "Cygwin processes can be set to
  // IDLE_PRIORITY_CLASS, NORMAL_PRIORITY_CLASS, HIGH_PRIORITY_CLASS,
  // or REALTIME_PRIORITY_CLASS with the nice call. If you pass a
  // positive number to nice(), then the priority level will decrease
  // by one (within the above list of priorities). A negative number
  // would make it increase by one. It is not possible to change it
  // by more than one at a time without making repeated calls".
  //

  if (nice(5) < 0 && errno != 0)
  {
    #ifdef WARNING
    *logofs << "NXKeeper: WARNING! Failed to renice process to +5. "
            << "Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
     #endif

     cerr << "Warning" << ": Failed to renice process to +5. "
          << "Error is " << EGET() << " '" << ESTR()
          << "'.\n";
  }

  //
  // Delay the first run to give priority
  // to session startup.
  //

  usleep(timeout * 1000);

  //
  // House keeping of persistent caches
  // is performed only once.
  //

  if (caches != 0)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "NXKeeper: Going to cleanup NX cache directories "
            << "at " << strTimestamp() << ".\n" << logofs_flush;
    #endif

    keeper.keepCaches();
  }

  //
  // House keeping of images is only
  // performed at X server side.
  //

  if (images == 0)
  {
    HandleCleanup();
  }

  //
  // Run forever until we are killed
  // by parent.
  //

  for (;;)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "NXKeeper: Going to cleanup NX images directories "
            << "at " << strTimestamp() << ".\n" << logofs_flush;
    #endif

    if (keeper.keepImages() < 0)
    {
      HandleCleanup();
    }

    //
    // Exit if parent is dead.
    //

    if (parent != getppid() || parent == 1)
    {
      #ifdef WARNING
      *logofs << "NXKeeper: WARNING! Parent process appears to be dead. "
              << "Exiting keeper.\n" << logofs_flush;
      #endif

      cerr << "Warning" << ": Parent process appears to be dead. "
           << "Exiting keeper.\n";

      HandleCleanup();
    }

    #if defined(INFO) || defined(TEST)
    *logofs << "NXKeeper: Completed cleanup of NX images directories "
            << "at " << strTimestamp() << ".\n" << logofs_flush;
    #endif

    usleep(timeout * 1000);
  }

  //
  // Hopefully useless.
  //

  exit(0);
}

void PartialCleanup()
{
  DisableSignals();

  //
  // TODO: Under Cygwin we cannot reopen the statistics
  // stream if the inherited descriptor is left open by
  // the child process. By the way, I noted that deleting
  // statofs later can cause segmentation fault in cleanup.
  // The same happens trying to write to lockFileName or
  // unixSocketName, while, strangely enough, a strcpy()
  // having them for target seems to be OK. Reasons are
  // to be better investigated.
  //

  if (statofs != NULL)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Children: Freeing up statistics stream.\n"
            << logofs_flush;
    #endif

    delete statofs;

    statofs = NULL;
  }

  if (proxy != NULL)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Children: Freeing up proxy.\n"
            << logofs_flush;
    #endif

    delete proxy;

    proxy = NULL;
  }

  if (statistics != NULL)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Children: Freeing up statistics.\n"
            << logofs_flush;
    #endif

    delete statistics;

    statistics = NULL;
  }

  if (context != NULL)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Children: Freeing up context.\n"
            << logofs_flush;
    #endif

    delete context;

    context = NULL;
  }

  if (control != NULL)
  {
    #if defined(INFO) || defined(TEST)
    *logofs << "Children: Freeing up control.\n"
            << logofs_flush;
    #endif

    delete control;

    control = NULL;
  }

  //
  // Prevent deletion of unix socket
  // and lock file.
  //

  useDaemonMode = 0;
  useUnixSocket = 0;

  //
  // Don't let cleanup kill other
  // children.
  //

  lastDialog   = 0;
  lastWatchdog = 0;
  lastKeeper   = 0;

  EnableSignals();
}
