/*
 *  Copyright (c) by Shuu Yamaguchi <shuu@dotaster.com>
 *
 *  $Id: loader.c,v 3.15 2003/04/20 21:47:51 shuu Exp shuu $
 *
 *  Can be freely distributed and used under the terms of the GNU GPL.
 */
#include	<unistd.h>
#include	<sys/types.h>
#include	<sys/stat.h>
#include	<fcntl.h>
#include	<limits.h>
#include	<string.h>
#include	<stdio.h>
#include	<sys/mman.h>
#include	<sys/utsname.h>

#include	"murasaki.h"

#define	MODULE_BASEDIR	"/lib/modules/"
#define	CHAR_COMMENT	'#'
#define ENABLE_NONE_MODULE

static char *command_line[LIST_MAX+2];	/* 2 for modprobe + options */

/*
 * called: load_module
 */
static int
get_module(struct mu_op *opp,char **modules,char *fname)
{
	struct stat st;
	int fd;
	char *mp;
	int count;

	if ((fd = open(fname,O_RDONLY)) == -1) {
		syslog(LOG_LEVEL,"open error %s", fname);
		return 0;
	}
	if (fstat(fd,&st) == -1) {
		syslog(LOG_LEVEL,"fstat error %s", fname);
		return -1;
	}
	if ((mp = mmap(0,st.st_size,PROT_READ,MAP_SHARED,fd,0)) == MAP_FAILED) {
		syslog(LOG_LEVEL,"mmap error %s", fname);
		return -1;
	}
	close(fd);

	if ((count = opp->match_map(modules,mp,mp+st.st_size,opp->config)) == 0) {
		if (flag_debug > DEBUG_TRACE)
			syslog(LOG_LEVEL,"Not found in %s", fname);
	}
	munmap(mp,st.st_size);

	return count;
}


/*
 * called: load_module
 */
static char *
make_defaultpath(char *fname)
{
	struct utsname uts;
	static char fullpath[PATH_MAX];

	if (uname(&uts) == -1)
		return NULL;
	if (sizeof(MODULE_BASEDIR)-1 + 1 + strlen(fname) + strlen(uts.release) > PATH_MAX)
		return NULL;
	strcpy(fullpath,MODULE_BASEDIR);
	strcpy(fullpath+sizeof(MODULE_BASEDIR)-1,uts.release);
	strcat(fullpath+sizeof(MODULE_BASEDIR),"/");
	strcat(fullpath+sizeof(MODULE_BASEDIR),fname);

	return fullpath;
}

/*
 * called: load_module
 *
 * If module name begins with "alias-", it should be expand name and
 * be not register module_list
 */
static void
delete_alias(void)
{
	int i;
	for (i = 0;module_list[i] != NULL;i++) {
		if (strncmp(module_list[i],ALIAS_STR,sizeof(ALIAS_STR)-1) == 0) {
			remove_from_list(module_list,i);
			i--;		/* look the new module_list[i] */
		}
	}
}

/*
 * called: load_module
 *
 * When action is "remove", module arguments(AAA=xxx) need not only to
 * be given to 'modprobe' but also are worthless.
 * So, they should be removed.
 */
static void
delete_option(void)
{
	int i;

	for (i = 0; module_list[i] != NULL;i++) {
		if (strchr(module_list[i],'=') != NULL) {
			remove_from_list(module_list,i);
			i--;	/* look the new module_list[i] */
		}
	}
}

/*
 * called: main
 */
int
create_modulelist(struct mu_op *opp)
{
	char *default_path;
	int count,sum,i;

	if (opp->match_map == NULL)
		return INVALID;
	inform_config(opp->config);	/* print device configuration */

	/*
	 * 1st: search murasaki.XXXmap (XXX:usb,pci)
	 * 2nd: search modules.XXXmap
	 */
	if ((count = get_module(opp,module_list,opp->mappath)) == -1)
		return INVALID;
	sum = count;
	if ((default_path = make_defaultpath(opp->default_mapname)) == NULL)
		return INVALID;
	if ((count = get_module(opp,module_list,default_path)) == -1)
		return INVALID;
	sum += count;
	DPRINTF(LOG_LEVEL,"module sum = %d",sum);
	if (sum == 0) {	/* not found */
		beep(INVALID);
		syslog(LOG_LEVEL,"The device match nothing in mapfile");
		syslog(LOG_LEVEL,"Please change MODULE in following line to the appropriate module name, add it to %s",opp->mappath);
		print_config(opp->config);

		return GOOD;	/* it may be static link */
	}
	beep(GOOD);
	/*
	 * adding depended modules & expanding alias modules 
	 */
	if (get_depend(module_list) == INVALID) {
		DPRINTF(LOG_LEVEL,"get_depend return INVALID");
		return INVALID;
	}

	/*
	 *  clean up module_list[]
	 */
	delete_alias();

	if (opp->action == ACTION_REMOVE) {
		delete_stickymodule(module_list);
		delete_option();
	}

	DPRINT_LIST(module_list);
	for (i = 0;module_list[i] != NULL;i++) {
		if (strchr(module_list[i],'=') != NULL)
			continue;
		syslog(LOG_LEVEL,"%s %s",opp->action == ACTION_ADD ? "Loading" : "Unloading" , module_list[i]);
	}

	return GOOD;
}

/*
 * The number of modules is not 0, because create_modulelist() counts
 *  the sum of modules.
 */
int
load_module(struct mu_op *opp)
{
	int sum,i;
	char *loader;

	sum = count_of_list(module_list);
#ifdef	ENABLE_NONE_MODULE
	/*
	 * if module is "none", do nothing.
	 */
	if (sum == 1 && strcmp(module_list[0],"none") == 0) {
		DPRINTF(LOG_LEVEL,"\"none\" -> do nothing");
		return GOOD;
	}
#endif	/* ENABLE_NONE_MODULE */
	/* 
	 * module_list[0]:"modprobe"
	 * module_list[1]:"-as" or "-asr"
	 * module_list[2-]: module names
	 * module_list[END]: NULL names
	 */
	if ((loader = get_loader()) == NULL) {
		syslog(LOG_LEVEL,"fork error");
		return INVALID;
	}
	command_line[0] = loader;
	if (opp->action == ACTION_ADD)
		command_line[1] = "-as";
	else 	/* ACTION_REMOVE */
		command_line[1] = "-asr";
	for (i = 0;module_list[i] != NULL;i++) {
		command_line[i+2] = module_list[i];
	}
	command_line[i+2] = NULL;	/* verbose */

	executing(command_line[0],command_line,FORK_ON);

	return GOOD;
}
