/*

  Copyright (C) 2000, The MITRE Corporation

  Use of this software is subject to the terms of the GNU General
  Public License version 2.

  Please read the file LICENSE for the exact terms.

*/

/*
 * Classes for the Mobile Mesh Routing Protocol messages.
 *
 * Author: Kevin H. Grace, kgrace@mitre.org
 *         The MITRE Corporation
 *         202 Burlington Rd
 *         Bedford, MA  01730
 *         
 *
 * $Id: MmMsg.h,v 1.1.1.1 2003/10/24 10:34:57 br1 Exp $
 *   
 */
#ifndef __MmMsg_h
#define __MmMsg_h

#include <list.h>

#include <UtInterface.h>
#include <UtString.h>
#include <UtUtil.h>


class MsgTypes {
public:
  enum { Version = 1 };
  enum { tHello=1, tLsp=2 };
};

class Neighbor {
  unsigned int cAddr;
  unsigned int cMetric;
public:
  Neighbor(unsigned int addr, unsigned short metric) :
    cAddr(addr), cMetric(metric) {};
  Neighbor() :
    cAddr(0), cMetric(0) {};
  unsigned int  Addr() const { return cAddr; };
  unsigned int  Metric() const { return cMetric; };
  unsigned int& Addr() { return cAddr; };
  unsigned int& Metric() { return cMetric; };

  String Dump() const {
    String s;
    // s += String("<Neighbor> ");
    s += String("addr=") + DumpAddr(cAddr) + " ";
    s += String("metric=") + String((int)cMetric).Int() + " ";
    // s += "\n";
    return(s);
  }

  int operator ==(const Neighbor& rhs) const {
    return(cAddr == rhs.cAddr);
  }

  int operator <(const Neighbor& rhs) const {
    return(cAddr < rhs.cAddr);
  }
};

class ExtRoute {
  unsigned int cAddr;
  unsigned int cNetmask;
  unsigned int cMetric;
public:
  ExtRoute(unsigned int addr, unsigned int netmask, unsigned int metric) :
    cAddr(addr), cNetmask(netmask), cMetric(metric) {};
  ExtRoute() :
    cAddr(0), cNetmask(0), cMetric(0) {};
  unsigned int& Addr()    { return cAddr;    }
  unsigned int& Netmask() { return cNetmask; }
  unsigned int& Metric()  { return cMetric;  }
  unsigned int  Addr()    const   { return cAddr;    }
  unsigned int  Netmask() const   { return cNetmask; }
  unsigned int  Metric()  const   { return cMetric;  }
  String Dump() const {
    String s;
    s += String("<ExtRoute> ");
    s += DumpAddr(cAddr) + " / ";
    s += DumpAddr(cNetmask) + " ";
    s += String("m=") + String(cMetric);
    return(s);
  }
  int operator ==(const ExtRoute& rhs) const {
    return((cAddr == rhs.cAddr) && (cNetmask == rhs.cNetmask));
  }

  int operator <(const ExtRoute& rhs) const {
    if(cAddr < rhs.cAddr) return(1);
    if(cAddr != rhs.cAddr) return(0);
    return(cNetmask < rhs.cNetmask);
  }
};

/*
  Format of Hello Packets that are sent over the air:

  Hello     := <Version><Msg Type><# Neighbors><NeighborList>
  Version   := 1
  Msg Type  := 1
  Neighbor List            := <Neighbor Interface Addr> | <Neigbor List>


    0                   1                   2                   3
    0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   | Version = 1   | Msg Type = 1  |        # Neighbors            |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |                   Neighbor Interface Address                  |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
                                  ...                              
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |                   Neighbor Interface Address                  |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 
*/
class Hello {
  list<unsigned int> cNeighborList;
public:
  Hello(const list<unsigned int>& neighbors) :
    cNeighborList(neighbors)  {}
  Hello() {}
  list<unsigned int>& NeighborList() { return cNeighborList; }
  list<unsigned int> NeighborList() const { return cNeighborList; }
  
  // Extracts a Hello message from the given string
  bool Decode(String& s) {
    unsigned int offset = 0;
    bool ok = true;
    *this = Hello(); // Clear out this object
    unsigned short numNeighbors;
    list<unsigned int> neighbors;

    // Control structure...never loops
    do {
      Decode1(s,offset,ok);  if(!ok) break; // Skip Version
      Decode1(s,offset,ok);  if(!ok) break; // Skip Msg Type
      numNeighbors = Decode2(s,offset,ok);  if(!ok) break; 
      for(int i=0; ok && (i<numNeighbors); i++) {
	unsigned int neighborAddr = Decode4(s,offset,ok);  if(!ok) break;
	neighbors.push_back(neighborAddr);
      }
      *this = Hello(neighbors);
    } while(0);

    return(ok);
  }
  
  String Encode() const {
    String s;
    s += Encode1(MsgTypes::Version);
    s += Encode1(MsgTypes::tHello);
    unsigned short numNeighbors  = cNeighborList.size();
    s += Encode2(numNeighbors);
    list<unsigned int>::const_iterator it;
    for(it = cNeighborList.begin(); it != cNeighborList.end(); it++) {
      s += Encode4((*it));
    }
    return(s);
  }

  String Dump() const {
    String s;
    s += "[Hello] ";
    list<unsigned int>::const_iterator it;
    int i;
    for(i=0, it = cNeighborList.begin(); it != cNeighborList.end(); i++, it++) {
      s += String("\n   [") + String(i) + String("] ") + DumpAddr((*it));
    }
    return(s);
  }
    
};

  
/*
  Format of Link State Packets that are sent over the air:

  Lsp                      := <Lsp Header><Interface List><External Route List>
  Lsp Header               := <Version><Msg Type><Router Id><Sequence Number><Age>
                                <Hop Count><# Interfaces><# External Routes>
  Version                  := 1			       
  Msg Type                 := 2
  Interface List           := <Interface> | <Interface List>
  Interface                := <Local Interface Addr><Neighbor List>
  Neighbor List            := <Neighbor Interface Addr><Neighbor Metric> | <Neigbor List>


    0                   1                   2                   3
    0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |  Version = 1  |  Msg Type = 2 |                               |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |                           Router Id                           |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |                         Sequence Number                       |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |             Age               |           Hop Count           |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |         # Interfaces          |       # External Routes       |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |                    Local Interface Address                    |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |           # Neighbors         |                               |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |                   Neighbor Interface Address                  |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |                   Neighbor Interface Metric                   |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
                                  ...                              
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |                   Neighbor Interface Address                  |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |                   Neighbor Interface Metric                   |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+

                                  ...
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |                    External Route Address                     |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |                            Netmask                            |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |                            Metric                             |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
                                  ...
*/
class Lsp {
  unsigned int  cRouterId;
  unsigned int  cSeq;
  unsigned short cHops;
  unsigned short cAge;
public:
  class Face {
  public:
  private:
    unsigned int   cAddr;
    list<Neighbor> cNeighborList;
  public:
    Face(unsigned int addr, const list<Neighbor>& neighbors) :
      cAddr(addr), cNeighborList(neighbors) {};
    Face() {};
    unsigned int&    Addr()         { return cAddr;         }
    list<Neighbor>&  NeighborList() { return cNeighborList; }
    unsigned int           Addr()         const { return cAddr;         }
    const list<Neighbor>&  NeighborList() const { return cNeighborList; }
    String Dump() const {
      String s;
      // s += String("<Face> ");
      s += String("addr=") + DumpAddr(cAddr);
      int i;
      list<Neighbor>::const_iterator it;
      for(it = cNeighborList.begin(), i=0; it != cNeighborList.end(); it++, i++) {
	s += String("\n  neighbor[") + String(i) + String("] ") + (*it).Dump();
      }
      return(s);
    }
  };

private:
  list<Face> cFaceList;
  list<ExtRoute> cExtRouteList;
public:
  Lsp(unsigned int routerId, unsigned int seq, unsigned short hops, unsigned short age, 
      const list<Face>* faces = 0, const list<ExtRoute>* extRoutes = 0) :
    cRouterId(routerId), cSeq(seq), cHops(hops), cAge(age) {
    if(faces)     cFaceList = *(faces);
    if(extRoutes) cExtRouteList = *(extRoutes);
  }

  Lsp() : cRouterId(0), cSeq(0), cHops(0), cAge(0) {}
  
  unsigned int&    RouterId()     { return cRouterId;     }
  unsigned int&    Seq()          { return cSeq;          }
  unsigned short&  Hops()         { return cHops;         }
  unsigned short&  Age()          { return cAge;          }
  list<Face>&      FaceList()     { return cFaceList;     }
  list<ExtRoute>&  ExtRouteList() { return cExtRouteList; }

  unsigned int     RouterId()     const { return cRouterId;     }
  unsigned int     Seq()          const { return cSeq;          }
  unsigned short   Hops()         const { return cHops;         }
  unsigned short   Age()          const { return cAge;          }
  const list<Face>&       FaceList()     const { return cFaceList;     }
  const list<ExtRoute>&   ExtRouteList() const { return cExtRouteList; }

  String Dump() const {
    String s;
    s += String("[Lsp] ");
    s += String("router_id=") + DumpAddr(cRouterId) + " ";
    s += String("seq=") + String(cSeq) + " ";
    s += String("hops=") + String((int)cHops) + " ";
    s += String("age=") + String((int)cAge);
    int i;
    list<Face>::const_iterator it;
    for(it = cFaceList.begin(), i=0; it != cFaceList.end(); it++, i++) {
      s += String("\n face[") + String(i) + String("] ") + (*it).Dump();
    }

    list<ExtRoute>::const_iterator k;
    for(k = cExtRouteList.begin(), i=0; k != cExtRouteList.end(); k++, i++) {
      s += String("\n extRoute[") + String(i) + String("] ") + (*k).Dump();
    }
    return(s);
  }


  String Encode() const {
    String s;
    
    // First encode the LSP header
    s += Encode1(MsgTypes::Version);
    s += Encode1(MsgTypes::tLsp);
    s += Encode1(0); // Pad
    s += Encode1(0); // Pad
    s += Encode4(cRouterId);
    s += Encode4(cSeq);
    s += Encode2(cAge);
    s += Encode2(cHops);
    unsigned short numIf  = cFaceList.size();
    s += Encode2(numIf);
    unsigned short numExt = cExtRouteList.size();
    s += Encode2(numExt);
    
    // Now encode the face list
    list<Face>::const_iterator it;
    for(it = cFaceList.begin(); it != cFaceList.end(); it++) {
      unsigned int addr = (*it).Addr();
      s += Encode4(addr);
      const list<Neighbor>& neighborList = (*it).NeighborList();
      unsigned short numNeighbors = neighborList.size();
      s += Encode2(numNeighbors);
      unsigned short pad = 0;
      s += Encode2(pad);
      
      // Encode this local interface's neighbor list
      list<Neighbor>::const_iterator j;
      for(j = neighborList.begin(); j != neighborList.end(); j++) {
	const Neighbor &neighbor = (*j);
	s += Encode4(neighbor.Addr());
	s += Encode4(neighbor.Metric());
      }
    }

    // Now encode the external route list
    list<ExtRoute>::const_iterator k;
    for(k = cExtRouteList.begin(); k != cExtRouteList.end(); k++) {
      const ExtRoute &r = (*k);
      s += Encode4(r.Addr());
      s += Encode4(r.Netmask());
      s += Encode4(r.Metric());
    }

    return(s);
  }
  
  bool Decode(String& s) {
    bool ok = true;
    unsigned int offset = 0;
    *this = Lsp();  // Make sure the lsp is empty
    
    // Control structure...never loops
    do {
      // First decode the LSP header
      Decode1(s,offset,ok);  if(!ok) break; // Skip Version
      Decode1(s,offset,ok);  if(!ok) break; // Skip Msg Type
      Decode1(s,offset,ok);  if(!ok) break; // Skip Pad
      Decode1(s,offset,ok);  if(!ok) break; // Skip Pad
      cRouterId = Decode4(s,offset,ok);  if(!ok) break;
      cSeq      = Decode4(s,offset,ok);  if(!ok) break;
      cAge      = Decode2(s,offset,ok);  if(!ok) break;
      cHops     = Decode2(s,offset,ok);  if(!ok) break;
      unsigned short numIf  = Decode2(s,offset,ok);  if(!ok) break;
      unsigned short numExt = Decode2(s,offset,ok);  if(!ok) break;
      
      // Now decode the face list
      int i;
      for(i = 0; i < numIf; i++) {
	unsigned int addr = Decode4(s,offset,ok);  if(!ok) break;
	list<Neighbor> neighborList;
	unsigned short numNeighbors = Decode2(s,offset,ok);  if(!ok) break;
	Decode2(s,offset,ok);  if(!ok) break;  // Pad
	
	// Decode this local interface's neighbor list
	int j;
	for(j = 0; j < numNeighbors; j++) {
	  Neighbor n;
	  n.Addr()   = Decode4(s,offset,ok);  if(!ok) break;
	  n.Metric() = Decode4(s,offset,ok);  if(!ok) break;
	  neighborList.push_back(n);
	}
	cFaceList.push_back(Face(addr,neighborList));
      }
      
      // Now decode the external route list
      int k;
      for(k = 0; k < numExt; k++) {
	ExtRoute r;
	r.Addr()    = Decode4(s,offset,ok);  if(!ok) break;
	r.Netmask() = Decode4(s,offset,ok);  if(!ok) break;
	r.Metric()  = Decode4(s,offset,ok);  if(!ok) break;
	cExtRouteList.push_back(r);
      }
      
      // Here if we successfully decoded everything

    } while(0);
    return(ok);
  }
  
};

#endif




