/*
 * linux/drivers/mq1100fb.c
 *
 * Copyright © 2003 Keith Packard
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file COPYING in the main directory of this archive for
 * more details.
 *
 *	    MediaQ 1100/1132 LCD Controller Frame Buffer Driver
 *
 * Please direct your questions and comments on this driver to the following
 * email address:
 *
 *	keithp@keithp.com
 *
 * ChangeLog
 *
 * 2003-05-18: <keithp@keithp.com>
 *	- Ported from PCI development board to ARM H5400
 */

#include "mq1100init.c"

struct mq1100fb_par {
    struct fb_var_screeninfo	var;
    int				bpp;
    MqInitData			init;
};

struct mq1100fb_info {
    struct fb_info_gen  gen;
    unsigned long    	io;
    unsigned long    	fbmem;
    unsigned long	memsize;
    MqMap		map;
    struct mq1100fb_par	currentmode;
#ifdef CONFIG_PM
    struct pm_dev	*pm;
#endif
    struct MqFuncs	*funcs;
};

struct mq1100fb_palette {
    unsigned char	red,green,blue,transp;
};

static struct mq1100fb_info	fb_info;
static MqFormat			fb_format;
static int			mq1100fb_running;
static int			mq1100fb_active;
static struct display		disp;
static struct mq1100fb_palette	palette[256];
/*
 * Most fields here are overwritten as soon as MqDataToFormat is called.
 * So things like xres/yres and such are mostly for decoration here...
 */
static struct fb_var_screeninfo default_var = {
    320,	    /* xres */
    240,	    /* yres */
    320,	    /* xres_virtual */
    240,	    /* yres_virtual */
    0,		    /* xoffset */
    0,		    /* yoffset */
    16,		    /* bpp */
    0,		    /* grayscale */
    
    { 11, 5, 0 },   /* red */
    { 5, 6, 0 },    /* green */
    { 0, 5, 0 },    /* blue */
    { 0, 0, 0 },    /* trans */

    0,		    /* nonstd */
    0,		    /* activate */
    57,		    /* height */
    76,		    /* width */
    0,		    /* accel_flags */
};

/* This can be externally changed */
char *mq1100fb_name = "MQ1100";
static u16  pseudo_pal[16];

/* ------------------- chipset specific functions -------------------------- */

static int mq1100_encode_fix(struct fb_fix_screeninfo *fix,
			     const void *par,
			     struct fb_info_gen *info)
{
    struct mq1100fb_info * i = (struct mq1100fb_info *)info;
    debug ("enter\n");
    /*
     *  This function should fill in the 'fix' structure based on the values
     *  in the `par' structure.
     */
    memset(fix, 0, sizeof(struct fb_fix_screeninfo));
    strcpy(fix->id, mq1100fb_name);

    fix->smem_start = i->fbmem;
    fix->smem_len = i->memsize;

    fix->type = FB_TYPE_PACKED_PIXELS;
    fix->type_aux = 0;

    fix->visual = FB_VISUAL_TRUECOLOR;

    fix->xpanstep = fix->ywrapstep = 0;
    fix->ypanstep = 0;
    fix->line_length = fb_format.stride;
    fix->mmio_start = i->io;
    fix->mmio_len = MQ1100_REG_SIZE;

    fix->accel = FB_ACCEL_NONE;
    debug ("exit\n");

    return 0;
}

static int mq1100_decode_var(const struct fb_var_screeninfo *var,
			     void *par,
			     struct fb_info_gen *info)
{
    struct mq1100fb_par * p = (struct mq1100fb_par *)par;
    struct mq1100fb_info * i = (struct mq1100fb_info *)info;
    /*
     *  Get the video params out of 'var'. If a value doesn't fit, round it up,
     *  if it's too big, return -EINVAL.
     *
     *  Suggestion: Round up in the following order: bits_per_pixel, xres,
     *  yres, xres_virtual, yres_virtual, xoffset, yoffset, grayscale,
     *  bitfields, horizontal timing, vertical timing.
     */

    if (var->xres > 240 || var->yres > 320 || var->bits_per_pixel > 16) {
	    printk("Requested mode %dx%dx%d is out of range\n",
		   var->xres, var->yres, var->bits_per_pixel);
	    return -EINVAL;
    }

    if (var->xres * var->yres * var->bits_per_pixel / 8 > MQ1100_FB_SIZE) {
	    printk("Not enough VRAM for requested mode %dx%dx%d\n",
		   var->xres, var->yres, var->bits_per_pixel);
	    return -EINVAL;
    }

    debug ("enter i %X fb_info %X\n", (unsigned) i, (unsigned) &fb_info);
    p->var = *var;
    p->bpp = var->bits_per_pixel;
    switch (p->bpp) {
    case 8:
	p->var.red.offset = 0;
	p->var.green.offset = 0;
	p->var.blue.offset = 0;
	p->var.red.length = 6;
	p->var.green.length = 6;
	p->var.blue.length = 6;
	break;
    case 16:
	p->var.red.offset = 11;
	p->var.green.offset = 5;
	p->var.blue.offset = 0;
	p->var.red.length = 5;
	p->var.green.length = 6;
	p->var.blue.length = 5;
	break;
    default:
	return -EINVAL;
    }
    debug ("exit\n");

    return 0;
}

static int mq1100_encode_var(struct fb_var_screeninfo *var,
			     const void *par,
			     struct fb_info_gen *info)
{
    struct mq1100fb_par * p = (struct mq1100fb_par *)par;

    debug ("enter\n");
    /*
     *  Fill the 'var' structure based on the values in 'par' and maybe other
     *  values read out of the hardware.
     */

    *var = p->var;
    var->bits_per_pixel = p->bpp;
    debug ("exit\n");
    return 0;
}

static void mq1100_get_par(void *par, struct fb_info_gen *info)
{
    struct mq1100fb_par * p = (struct mq1100fb_par *)par;
    struct mq1100fb_info * i = (struct mq1100fb_info *)info;
    /*
     *  Fill the hardware's 'par' structure.
     */

    debug ("enter\n");
    *p = i->currentmode;
    debug ("exit\n");
}

static void mq1100_set_par(const void *par, struct fb_info_gen *info)
{
/*    struct mq1100fb_par * p = (struct mq1100fb_par *)par; */
    struct mq1100fb_info * i = (struct mq1100fb_info *)info;
    /*
     *  Set the hardware according to 'par'.
     */

    debug ("enter\n");
    if (mq1100fb_active)
    {
	debug ("already active\n");
    }
    else
    {
	/* i->currentmode = *p; */
	mq1100_lcd_enable (fb_info, MqFalse);
	mq1100_power (fb_info, MqFalse);
	mdelay (10);	/* give the mediaq a chance to go to sleep */
	mq1100_power (fb_info, MqTrue);
	mdelay (10);	/* give the mediaq a chance to wake up */
	MqInit (&i->map, &i->currentmode.init);
	debug ("MqPowerGC\n");
	MqPowerGC (&i->map, MqTrue);
	mq1100_lcd_enable (fb_info, MqTrue);
	mq1100fb_active = 1;
    }
    debug ("exit\n");
}

static int mq1100_getcolreg(unsigned regno, unsigned *red, unsigned *green,
			    unsigned *blue, unsigned *transp,
			    struct fb_info *info)
{
    struct mq1100fb_info * i = (struct mq1100fb_info *)info;
    int m;

    m = i->currentmode.bpp==8?256:16;
    if (regno >= m)
	return 1;

    *red = palette[regno].red;
    *green = palette[regno].green;
    *blue = palette[regno].blue;
    *transp = palette[regno].transp;

    return 0;
}

static int mq1100_setcolreg(unsigned regno, unsigned red, unsigned green,
			    unsigned blue, unsigned transp,
			    struct fb_info *info)
{
    struct mq1100fb_info * i = (struct mq1100fb_info *)info;
    int bpp;
    int m = 0;

    if (regno >= 256)
	return 1;

    palette[regno].red = red;
    palette[regno].green = green;
    palette[regno].blue = blue;
    palette[regno].transp = transp;

    bpp = i->currentmode.bpp; 
    m = bpp==8?256:16;
    if (regno >= m)
    {
	debug ("regno %d out of range (max %d)\n", regno, m);
	return 1;
    }
#if 1
    if (bpp == 8)
	;
    else
    /* RGB 565 */
    if (bpp == 16)
	pseudo_pal[regno] = ((red & 0xF800) |
			     ((green & 0xFC00) >> 5) |
			     ((blue & 0xF800) >> 11));
#endif
    return 0;
}

static int mq1100_pan_display(const struct fb_var_screeninfo *var,
			      struct fb_info_gen *info)
{
    /*
     *  Pan (or wrap, depending on the `vmode' field) the display using the
     *  `xoffset' and `yoffset' fields of the `var' structure.
     *  If the values don't fit, return -EINVAL.
     */

    /* ... */
    return 0;
}

static int mq1100_blank(int blank_mode, struct fb_info_gen *info)
{
    /*
     *  Blank the screen if blank_mode != 0, else unblank. If blank == NULL
     *  then the caller blanks by setting the CLUT (Color Look Up Table) to all
     *  black. Return 0 if blanking succeeded, != 0 if un-/blanking failed due
     *  to e.g. a video mode which doesn't support it. Implements VESA suspend
     *  and powerdown modes on hardware that supports disabling hsync/vsync:
     *    blank_mode == 2: suspend vsync
     *    blank_mode == 3: suspend hsync
     *    blank_mode == 4: powerdown
     */

    /* ... */
    return 0;
}

static void mq1100_set_disp(const void *par, struct display *disp,
			    struct fb_info_gen *info)
{
    struct mq1100fb_info * i = (struct mq1100fb_info *)info;
    struct fb_info * ii = &i->gen.info;
    struct mq1100fb_par * p = (struct mq1100fb_par *)par;
    /*
     *  Fill in a pointer with the virtual address of the mapped frame buffer.
     *  Fill in a pointer to appropriate low level text console operations (and
     *  optionally a pointer to help data) for the video mode `par' of your
     *  video hardware. These can be generic software routines, or hardware
     *  accelerated routines specifically tailored for your hardware.
     *  If you don't have any appropriate operations, you must fill in a
     *  pointer to dummy operations, and there will be no text output.
     */
    debug ("enter\n");
    debug ("bpp: %d\n", p->bpp);
    disp->screen_base = (void*) i->map.sync;
#ifdef FBCON_HAS_CFB8
    if (p->bpp == 8) {
	disp->dispsw = &fbcon_cfb8;
    } else
#endif
#ifdef FBCON_HAS_CFB16
    if (p->bpp == 16) {
	disp->dispsw = &fbcon_cfb16;
	disp->dispsw_data = ii->pseudo_palette;	/* console palette */
    } else
#endif
	disp->dispsw = &fbcon_dummy;
    debug ("fontwidthmask %X\n", disp->dispsw->fontwidthmask);

    debug ("exit\n");
}


void
mq1100_backlight_level(unsigned char level)
{
	fb_info.map.reg->FP.s.pwm_clock_selector_c8 = 0x0010 | ((0xff-level) << 8);
}


/* ------------ Interfaces to hardware functions ------------ */


#ifdef CONFIG_PM
/*
 * Power management hook.
 */
static int
mq1100fb_pm_callback(struct pm_dev *pm_dev, pm_request_t req, void *data)
{
    static int saved_backlight;

    if (req == PM_SUSPEND || req == PM_RESUME) {
	int state = (int)data;
	
	if (state == 0) {
	    /* Enter D0. */
	    mq1100_set_par (NULL, &fb_info);
	    fb_info.map.reg->FP.s.pwm_clock_selector_c8 = saved_backlight;
	} else {
	    /* Enter D1-D3.  Disable the LCD controller.  */
	    saved_backlight = fb_info.map.reg->FP.s.pwm_clock_selector_c8;
            MqPowerGC (&fb_info.map, MqFalse);
            mq1100_lcd_enable (fb_info, MqFalse);
            mq1100_power (fb_info, MqFalse);
 	    mq1100fb_active = 0;
	}
    }

    return 0;
}
#endif

struct fbgen_hwswitch mq1100_hwswitch = {
    NULL,
    mq1100_encode_fix,
    mq1100_decode_var,
    mq1100_encode_var,
    mq1100_get_par,
    mq1100_set_par,
    mq1100_getcolreg,
    mq1100_setcolreg,
    mq1100_pan_display,
    mq1100_blank,
    mq1100_set_disp
};

static struct fb_ops mq1100fb_ops = {
	fb_get_fix:fbgen_get_fix,
	fb_get_var:fbgen_get_var,
	fb_set_var:fbgen_set_var,
	fb_get_cmap:fbgen_get_cmap,
	fb_set_cmap:fbgen_set_cmap,
	fb_pan_display:fbgen_pan_display,
};

int __devinit
mq1100_probe (unsigned long io, unsigned long fbmem, struct MqFuncs *funcs)
{
    MqInitData *init = 0;
    fb_info.io = io;
    fb_info.fbmem = fbmem;
    debug ("io: %X fbmem: %X\n", 
	   (unsigned) fb_info.io, (unsigned) fb_info.fbmem);

    request_mem_region(fb_info.io, MQ1100_REG_SIZE, "mq1100fb");

    fb_info.map.reg = (volatile MqReg *) ioremap_nocache(fb_info.io, MQ1100_REG_SIZE);

    if (!fb_info.map.reg) {
	release_region(fb_info.io, MQ1100_REG_SIZE);
	debug("ioremap failed\n");
	return -1;
    }

    fb_info.memsize = MQ1100_FB_SIZE;
    request_mem_region(fb_info.fbmem, fb_info.memsize, "mq1100fb");

    fb_info.map.sync = (volatile unsigned char *)ioremap_nocache(fb_info.fbmem, fb_info.memsize);

    if (!fb_info.map.sync) {
	release_mem_region(fb_info.fbmem, fb_info.memsize);
	debug("ioremap failed\n");
	return -1;
    }
    fb_info.map.async = fb_info.map.sync + MQ1100_FB_SIZE;

    debug("MQ1100 board found : mem = %X,io = %X, mem_v = %X, io_v = %X\n",
	  (unsigned) fb_info.fbmem, 
	  (unsigned) fb_info.io, 
	  (unsigned) fb_info.map.sync, 
	  (unsigned) fb_info.map.reg);

    fb_info.gen.parsize = sizeof (struct mq1100fb_par);
    fb_info.gen.fbhw = &mq1100_hwswitch;

    strcpy(fb_info.gen.info.modename, mq1100fb_name);

    fb_info.gen.info.changevar = NULL;
    fb_info.gen.info.node = NODEV;
    fb_info.gen.info.fbops = &mq1100fb_ops;
    fb_info.gen.info.disp = &disp;

    fb_info.gen.info.switch_con = &fbgen_switch;
    fb_info.gen.info.updatevar = &fbgen_update_var;
    fb_info.gen.info.blank = &fbgen_blank;

    fb_info.gen.info.flags = FBINFO_FLAG_DEFAULT;
    fb_info.gen.info.fontname[0] = '\0';
    fb_info.gen.info.pseudo_palette = pseudo_pal;

    /* This should give a reasonable default video mode */

    init = funcs->mq1100_detect_lcd (&fb_info.map);
    
    MqDataToFormat (init, &fb_format, /*3780*/4210);
    default_var.xres = fb_format.width;
    default_var.yres = fb_format.height;
    default_var.xres_virtual = fb_format.width;
    default_var.yres_virtual = fb_format.height;
    default_var.bits_per_pixel = fb_format.bpp;
    default_var.height = fb_format.height_mm;
    default_var.width = fb_format.width_mm;
    default_var.activate |= FB_ACTIVATE_NOW;

    fb_info.funcs = funcs;
    
    fb_info.currentmode.var = default_var;
    fb_info.currentmode.bpp = default_var.bits_per_pixel;
    fb_info.currentmode.init = *init;

    fbgen_get_var(&disp.var, -1, &fb_info.gen.info);
    debug ("fbgen_set_disp %X\n", (unsigned) &fb_info.gen);
    fbgen_set_disp(-1, &fb_info.gen);

    debug ("register_framebuffer %X\n", (unsigned) &fb_info.gen);
    if (register_framebuffer(&fb_info.gen.info) < 0) {
	printk("Could not register MQ1100 framebuffer\n");
	return -EINVAL;
    }

    mq1100fb_running = 1;
    output("fb%d: %s frame buffer device %dx%d-%dbpp\n",
	   GET_FB_IDX(fb_info.gen.info.node), fb_info.gen.info.modename,
	   default_var.xres, default_var.yres, default_var.bits_per_pixel);

#ifdef CONFIG_PM
    /*
     * Note that the console registers this as well, but we want to
     * power down the display prior to sleeping.
     */
    fb_info.pm = pm_register(PM_SYS_DEV, PM_SYS_VGA, mq1100fb_pm_callback);
    if (fb_info.pm)
	fb_info.pm->data = &fb_info;
#endif

    return 0;
}

void __devexit mq1100_cleanup (void)
{
    debug ("enter\n");
    if (mq1100fb_running)
    {
	if (mq1100fb_active)
	{
	    MqPowerGC (&fb_info.map, MqFalse);
	    mq1100_lcd_enable (fb_info, MqFalse);
	    mq1100_power (fb_info, MqFalse);
	}
	unregister_framebuffer(&fb_info.gen.info);
	iounmap((void *)fb_info.map.reg);
	iounmap((void *)fb_info.map.sync);
	release_mem_region(fb_info.fbmem, fb_info.memsize);
	release_region(fb_info.io, MQ1100_REG_SIZE);
	mq1100fb_running = 0;
    }
    debug ("exit\n");
}

EXPORT_SYMBOL(mq1100_backlight_level);

MODULE_AUTHOR("Keith Packard <keithp@keithp.com>");
MODULE_DESCRIPTION("Framebuffer driver for MediaQ 1100 chips");
MODULE_LICENSE("GPL");
