// -*- linux-c -*-
/* 
 * arch/arm/kernel/dma-backpaq.c
 *
 * 06/30/03 Fae Ghodrat - This is a copy of dma-sa1100.c
 *                        It's used by backpaq-uda1341 which does not have dma.
 *                        It emulated a dma.
 *
 */

#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/init.h>
#include <linux/spinlock.h>
#include <linux/errno.h>
#include <linux/sched.h>

#include <asm/system.h>
#include <asm/irq.h>
#include <asm/hardware.h>
// #include <asm/dma.h>
#include <asm/arch/dma-backpaq.h>
#include <asm/system.h> /* smp_mb() */


#undef DEBUG
#ifdef DEBUG
#define DPRINTK( s, arg... )  printk( "dma<%p>: " s, regs , ##arg )
#else
#define DPRINTK( x... )
#endif


#define BACKPAQ_PLAY_CHANNEL 1
#define BACKPAQ_REC_CHANNEL 0

typedef struct {
	const char *device_id;	/* device name */
	u_long device;		/* this channel device, 0  if unused*/
	dma_callback_t callback;	/* to call when DMA completes */
	void *data;			/* ... with private data ptr */
} backpaq_dma_t;


// one channel for read one for write
static backpaq_dma_t dma_chan[BACKPAQ_DMA_CHANNELS];
static dma_regs_t dma_chan_regs[BACKPAQ_DMA_CHANNELS];

static spinlock_t dma_list_lock;

static void audio_dma_play_task_handler(void *x)
{
	// msleep(10);

	// printk("%s: calling call back function\n", __FUNCTION__);
	if (dma_chan[BACKPAQ_PLAY_CHANNEL].callback) {
		dma_chan[BACKPAQ_PLAY_CHANNEL].callback(dma_chan[BACKPAQ_PLAY_CHANNEL].data);
	}
}

static void audio_dma_rec_task_handler(void *x)
{
	// msleep(10);

	// printk("%s: calling call back function\n", __FUNCTION__);
	if (dma_chan[BACKPAQ_REC_CHANNEL].callback) {
		dma_chan[BACKPAQ_REC_CHANNEL].callback(dma_chan[BACKPAQ_REC_CHANNEL].data);
	}
}

static struct tq_struct audio_dma_play_task = {
	routine: audio_dma_play_task_handler
};

static struct tq_struct audio_dma_rec_task = {
	routine: audio_dma_rec_task_handler
};

// static void dma_irq_handler(int irq, void *dev_id, struct pt_regs *regs)
static void dma_irq_handler(int irq, int channel)
{
	/*
	dma_regs_t *dma_regs = &dma_chan_regs[channel];
	backpaq_dma_t *dma = &dma_chan[channel];
	int status = dma_regs->DCSR;
	// int status = dma_regs->RdDCSR;
	*/
	// printk("%s: interrupt received, dma = %x, dma_regs = %x, channel = %x\n", __FUNCTION__, dma, dma_regs, channel);
	/*
	if (status & DCSR_ERROR) {
	        // printk(KERN_CRIT "DMA on \"%s\" caused an error\n", dma->device_id);
		dma_regs->DCSR &= ~DCSR_ERROR;
		// dma_regs->ClrDCSR = DCSR_ERROR;
	}
	*/
	// dma_regs->ClrDCSR = status & DCSR_DONEA;
	// if (dma->callback) {
	        // if (status & DCSR_DONEA) {
	        // printk("%s: calling callback function\n", __FUNCTION__);
	        // dma_regs->DCSR &= ~(status & DCSR_DONEA);
		
		// dma->callback(dma->data);
		if (channel == BACKPAQ_PLAY_CHANNEL) {
			schedule_task(&audio_dma_play_task);
		} else if (channel == BACKPAQ_REC_CHANNEL) {
			schedule_task(&audio_dma_rec_task);
		} else {
			printk(KERN_ERR, "%s: %d is a bad dma channel number\n", __FUNCTION__, channel);
		}
		// }
	// }

}


/**
 *	sa1100_request_dma - allocate one of the SA11x0's DMA chanels
 *	@device: The SA11x0 peripheral targeted by this request
 *	@device_id: An ascii name for the claiming device
 *	@callback: Function to be called when the DMA completes
 *	@data: A cookie passed back to the callback function
 *	@dma_regs: Pointer to the location of the allocated channel's identifier
 *
 * 	This function will search for a free DMA channel and returns the
 * 	address of the hardware registers for that channel as the channel
 * 	identifier. This identifier is written to the location pointed by
 * 	@dma_regs. The list of possible values for @device are listed into
 * 	linux/include/asm-arm/arch-sa1100/dma.h as a dma_device_t enum.
 *
 * 	Note that reading from a port and writing to the same port are
 * 	actually considered as two different streams requiring separate
 * 	DMA registrations.
 *
 * 	The @callback function is called from interrupt context when one
 * 	of the two possible DMA buffers in flight has terminated. That
 * 	function has to be small and efficient while posponing more complex
 * 	processing to a lower priority execution context.
 *
 * 	If no channels are available, or if the desired @device is already in
 * 	use by another DMA channel, then an error code is returned.  This
 * 	function must be called before any other DMA calls.
 **/

int backpaq_request_dma (dma_device_t device, const char *device_id,
			 dma_callback_t callback, void *data,
			 dma_regs_t **dma_regs)
{
	backpaq_dma_t *dma = NULL;
	dma_regs_t *regs = NULL;
	int channel = 0;
	int err;
	long dmaIRQ = 0;

	*dma_regs = NULL;
	
	err = 0;
	if (device == DMA_RD) {
		channel = BACKPAQ_REC_CHANNEL;
	        dmaIRQ = IRQ_IPAQ_SLEEVE_5;
	} else {
		channel = BACKPAQ_PLAY_CHANNEL;
	        dmaIRQ = IRQ_IPAQ_SLEEVE_4;
	}
	
	spin_lock(&dma_list_lock);
	if (dma_chan[channel].device == device) {
		err = -EBUSY;
	} else if (!dma_chan[channel].device && !dma) {
		dma = &dma_chan[channel];
	}
	if (!err) {
		if (dma)
			dma->device = device;
		else
			err = -ENOSR;
	}
	spin_unlock(&dma_list_lock);
	if (err)
		return err;
	
	// regs = (dma_regs_t *)&DDAR;
	regs = &dma_chan_regs[channel];
	regs->DDAR = device;
	regs->DCSR = 0;
	regs->DBSA = 0;
	regs->DBTA = 0;

	// disable_irq(dmaIRQ); /* so we don't immediately get a FIFO empty interrupt. */ 
	err = request_irq(dmaIRQ, dma_irq_handler, SA_INTERRUPT,
			  device_id, channel);
	if (err) {
	        printk(KERN_ERR "%s: unable to request IRQ %ld for %s err=%d\n",
		       __FUNCTION__, dmaIRQ, device_id, err);
		dma->device = 0;
		return err;
	}
	

	dma->device_id = device_id;
	dma->callback = callback;
	dma->data = data;

	// regs->ClrDCSR = (DCSR_DONEA | DCSR_STRTA | DCSR_IE | DCSR_ERROR | DCSR_RUN);
	regs->DCSR &= ~(DCSR_DONEA | DCSR_STRTA | DCSR_IE | DCSR_ERROR | DCSR_RUN);
	regs->DDAR = device;
	

	printk(KERN_INFO "%s: Granting request for DMA channel %d "
	       "with interrupt %d\n", __FUNCTION__, channel, dmaIRQ);
	*dma_regs = regs;
	
	return 0;
}


/**
 * 	sa1100_free_dma - free a SA11x0 DMA channel
 * 	@regs: identifier for the channel to free
 *
 * 	This clears all activities on a given DMA channel and releases it
 * 	for future requests.  The @regs identifier is provided by a
 * 	successful call to sa1100_request_dma().
 **/

void backpaq_free_dma(dma_regs_t *regs)
{
	int i;
	long dmaIRQ;

	if( regs == NULL ) {
		printk(KERN_ERR "%s: NULL DMA register pointer\n", __FUNCTION__);
		return;
	}	  

	for(i=0; i<BACKPAQ_DMA_CHANNELS; i++) {
		// if (regs == (dma_regs_t *)&DDAR) {
	        if (dma_chan[i].device == regs->DDAR) {
		        printk(KERN_INFO "%s: Removing the DMA channel %d\n", __FUNCTION__, i);
			break;
		}
	}

	if (i >= BACKPAQ_DMA_CHANNELS) {
		printk(KERN_ERR "%s: bad DMA identifier\n", __FUNCTION__);
		return;
	}

	if (!dma_chan[i].device) {
		printk(KERN_ERR "%s: Trying to free a free DMA\n", __FUNCTION__);
		return;
	}
	
	// flush_scheduled_tasks(); // is this needed ???

	// regs->ClrDCSR = (DCSR_DONEA | DCSR_STRTA | DCSR_IE | DCSR_ERROR | DCSR_RUN);
	regs->DCSR &= ~(DCSR_DONEA | DCSR_STRTA | DCSR_IE | DCSR_ERROR | DCSR_RUN);

	if (dma_chan[i].device == DMA_RD) {
	        dmaIRQ = IRQ_IPAQ_SLEEVE_5;
	} else {
	        dmaIRQ = IRQ_IPAQ_SLEEVE_4;
	}

	disable_irq(dmaIRQ);
	free_irq(dmaIRQ, i);
	printk(KERN_INFO "%s: Released IRQ %d\n", __FUNCTION__, dmaIRQ);

	dma_chan[i].device = 0;
}


/**
 * 	sa1100_start_dma - submit a data buffer for DMA
 * 	@regs: identifier for the channel to use
 * 	@dma_ptr: buffer physical (or bus) start address
 * 	@size: buffer size
 *
 * 	This function hands the given data buffer to the hardware for DMA
 * 	access. If another buffer is already in flight then this buffer
 * 	will be queued so the DMA engine will switch to it automatically
 * 	when the previous one is done.  The DMA engine is actually toggling
 * 	between two buffers so at most 2 successful calls can be made before
 * 	one of them terminates and the callback function is called.
 *
 * 	The @regs identifier is provided by a successful call to
 * 	sa1100_request_dma().
 *
 * 	The @size must not be larger than %MAX_DMA_SIZE.  If a given buffer
 * 	is larger than that then it's the caller's responsibility to split
 * 	it into smaller chunks and submit them separately. If this is the
 * 	case then a @size of %CUT_DMA_SIZE is recommended to avoid ending
 * 	up with too small chunks. The callback function can be used to chain
 * 	submissions of buffer chunks.
 *
 * 	Error return values:
 * 	%-EOVERFLOW:	Given buffer size is too big.
 * 	%-EBUSY:	Both DMA buffers are already in use.
 * 	%-EAGAIN:	Both buffers were busy but one of them just completed
 * 			but the interrupt handler has to execute first.
 *
 * 	This function returs 0 on success.
 **/

int backpaq_start_dma(dma_regs_t *regs, dma_addr_t dma_ptr, u_int size)
{
	unsigned long flags;
	u_long status;
	int ret;
	
	if (dma_ptr & 3)
		printk(KERN_WARNING "DMA: unaligned start address (0x%08lx)\n",
		       (unsigned long)dma_ptr);
	
	if (size > MAX_DMA_SIZE)
		return -EOVERFLOW;
	
	// local_irq_save(flags);
	//status = regs->DCSR;
	// status = regs->RdDCSR;
	
	/* If DMA buffer is started, there's nothing else we can do. */
	/*
	if ((status & DCSR_STRTA) == DCSR_STRTA) {
		DPRINTK("start: st %#x busy\n", status);
		ret = -EBUSY;
		goto out;
	}
	*/
	/*
	if (!(status & DCSR_BIU) && !(status & DCSR_STRTA)) {
		if (status & DCSR_DONEA) {
		  // give a chance for the interrupt to be processed
			ret = -EAGAIN;
			goto out;
		}
	*/
		regs->DBSA = dma_ptr;
		regs->DBTA = size;
		regs->DCSR = (DCSR_STRTA | DCSR_IE | DCSR_RUN);
		// regs->SetDCSR = DCSR_STRTA | DCSR_IE | DCSR_RUN;
		DPRINTK("start a=%#x s=%d on A\n", dma_ptr, size);
		// } 
	ret = 0;
	
 out:
	// local_irq_restore(flags);
	return ret;
}


/**
 * 	sa1100_get_dma_pos - return current DMA position
 * 	@regs: identifier for the channel to use
 *
 * 	This function returns the current physical (or bus) address for the
 * 	given DMA channel.  If the channel is running i.e. not in a stopped
 * 	state then the caller must disable interrupts prior calling this
 * 	function and process the returned value before re-enabling them to
 * 	prevent races with the completion interrupt handler and the callback
 * 	function. The validation of the returned value is the caller's
 * 	responsibility as well -- the hardware seems to return out of range
 * 	values when the DMA engine completes a buffer.
 *
 * 	The @regs identifier is provided by a successful call to
 * 	sa1100_request_dma().
 *
 * We must determine whether buffer A or B is active.
 * Two possibilities: either we are in the middle of
 * a buffer, or the DMA controller just switched to the
 * next toggle but the interrupt hasn't been serviced yet.
 * The former case is straight forward.  In the later case,
 * we'll do like if DMA is just at the end of the previous
 * toggle since all registers haven't been reset yet.
 * This goes around the edge case and since we're always
 * a little behind anyways it shouldn't make a big difference.
 * If DMA has been stopped prior calling this then the
 * position is exact.
 **/

dma_addr_t backpaq_get_dma_pos(dma_regs_t *regs)
{
	int status;

	status = regs->DCSR;
	// status = regs->RdDCSR;
	// printk("%s: status = 0x%x, DCSR-STRTA = %d, DCSR-BIU = %d, DBTA = 0x%x\n", 
	// __FUNCTION__, status, status & DCSR_STRTA, status & DCSR_BIU, regs->DBTA);
	if (!(status & DCSR_BIU) &&  (status & DCSR_STRTA))
	        return (regs->DBTA); 
	        // return regs->DBSA; 
	else
		return -1;
}


/**
 * 	sa1100_reset_dma - reset a DMA channel
 * 	@regs: identifier for the channel to use
 *
 * 	This function resets and reconfigure the given DMA channel. This is
 * 	particularly useful after a sleep/wakeup event.
 *
 * 	The @regs identifier is provided by a successful call to
 * 	sa1100_request_dma().
 **/

void backpaq_reset_dma(dma_regs_t *regs)
{
	int i;

	if( regs == NULL ) {
		printk(KERN_ERR "%s: NULL DMA register pointer\n", __FUNCTION__);
		return;
	}	  

	for(i=0; i<BACKPAQ_DMA_CHANNELS; i++) {
	        if (dma_chan[i].device == regs->DDAR) {
		        // printk(KERN_INFO "%s: Found the DMA channel\n", __FUNCTION__);
			break;
		}
	}

	if (i >= BACKPAQ_DMA_CHANNELS) {
		printk(KERN_ERR "%s: bad DMA identifier\n", __FUNCTION__);
		return;
	}

	// if (regs == (dma_regs_t *)&DDAR) {
	// regs->ClrDCSR = (DCSR_DONEA | DCSR_STRTA | DCSR_IE | DCSR_ERROR | DCSR_RUN);
	regs->DCSR &= ~(DCSR_DONEA | DCSR_STRTA | DCSR_IE | DCSR_ERROR | DCSR_RUN);
	regs->DDAR = dma_chan[i].device;
	// }
}


EXPORT_SYMBOL(backpaq_request_dma);
EXPORT_SYMBOL(backpaq_free_dma);
EXPORT_SYMBOL(backpaq_start_dma);
EXPORT_SYMBOL(backpaq_get_dma_pos);
EXPORT_SYMBOL(backpaq_reset_dma);








