/*
 * linux/arch/arm/mach-omap/mux.c
 *
 * Utility to set the Omap MUX and PULL_DWN registers from a table in mux.h
 *
 * Copyright (C) 2003 Nokia Corporation
 *
 * Written by Tony Lindgren <tony.lindgren@nokia.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * $Id: mux.c
 * $Revision: 1.0
 * $Date: 2003/03/25
 *
 */
#include <linux/module.h>
#include <asm/system.h>
#include <asm/io.h>

#include <asm/arch/mux.h>

#define DEBUG_MUX

/*
 * Table of various FUNC_MUX and PULL_DWN combinations for each device.
 * See also reg_cfg_t above for the lookup table.
 */
static reg_cfg_set reg_cfg_table[] = {

	/* UART1 */
	{
		name:		"uart1_tx",
		busy:		0,
		debug:		0,
		mux_reg_name:	"FUNC_MUX_CTRL_9",
		mux_reg:	FUNC_MUX_CTRL_9,
		mask_offset:	21,
		mask:		BITS001,
		pull_dwn_name:	"PULL_DWN_CTRL_2",
		pull_dwn_reg:	PULL_DWN_CTRL_2,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	3
	},
	{
		name:		"uart1_rts",
		busy:		0,
		debug:		0,
		mux_reg_name:	"FUNC_MUX_CTRL_9",
		mux_reg:	FUNC_MUX_CTRL_9,
		mask_offset:	12,
		mask:		BITS001,
		pull_dwn_name:	"PULL_DWN_CTRL_2",
		pull_dwn_reg:	PULL_DWN_CTRL_2,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	0
	},

	/* UART2 (COM_UART_GATING), conflicts with USB2 */
	{
		name:		"uart2_tx",
		busy:		0,
		debug:		0,
		mux_reg_name:	"FUNC_MUX_CTRL_C",
		mux_reg:	FUNC_MUX_CTRL_C,
		mask_offset:	27,
		mask:		BITS001,
		pull_dwn_name:	"PULL_DWN_CTRL_3",
		pull_dwn_reg:	PULL_DWN_CTRL_3,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	3
	},
	{
		name:		"uart2_rx",
		busy:		0,
		debug:		0,
		mux_reg_name:	"FUNC_MUX_CTRL_C",
		mux_reg:	FUNC_MUX_CTRL_C,
		mask_offset:	18,
		mask:		BITS000,
		pull_dwn_name:	"PULL_DWN_CTRL_3",
		pull_dwn_reg:	PULL_DWN_CTRL_3,
		pull_dwn_val:	USE_PULL_DWN,
		pull_dwn_bit:	1
	},
	{
		name:		"uart2_cts",
		busy:		0,
		debug:		0,
		mux_reg_name:	"FUNC_MUX_CTRL_C",
		mux_reg:	FUNC_MUX_CTRL_C,
		mask_offset:	21,
		mask:		BITS000,
		pull_dwn_name:	"PULL_DWN_CTRL_3",
		pull_dwn_reg:	PULL_DWN_CTRL_3,
		pull_dwn_val:	USE_PULL_DWN,
		pull_dwn_bit:	1
	},
	{
		name:		"uart2_rts",
		busy:		0,
		debug:		0,
		mux_reg_name:	"FUNC_MUX_CTRL_C",
		mux_reg:	FUNC_MUX_CTRL_C,
		mask_offset:	24,
		mask:		BITS001,
		pull_dwn_name:	"PULL_DWN_CTRL_3",
		pull_dwn_reg:	PULL_DWN_CTRL_3,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	2
	},

	/* UART3 (GIGA_UART_GATING) */
	{
		name:		"uart3_tx",
		busy:		0,
		debug:		0,
		mux_reg_name:	"FUNC_MUX_CTRL_6",
		mux_reg:	FUNC_MUX_CTRL_6,
		mask_offset:	0,
		mask:		BITS001,
		pull_dwn_name:	"PULL_DWN_CTRL_0",
		pull_dwn_reg:	PULL_DWN_CTRL_0,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	30
	},
	{
		name:		"uart3_rx",
		busy:		0,
		debug:		0,
		mux_reg_name:	"FUNC_MUX_CTRL_6",
		mux_reg:	FUNC_MUX_CTRL_6,
		mask_offset:	3,
		mask:		BITS000,
		pull_dwn_name:	"PULL_DWN_CTRL_0",
		pull_dwn_reg:	PULL_DWN_CTRL_0,
		pull_dwn_val:	USE_PULL_DWN,
		pull_dwn_bit:	31
	},
	{
		name:		"uart3_cts",
		busy:		0,
		debug:		0,
		mux_reg_name:	"FUNC_MUX_CTRL_5",
		mux_reg:	FUNC_MUX_CTRL_5,
		mask_offset:	12,
		mask:		BITS010,
		pull_dwn_name:	"PULL_DWN_CTRL_0",
		pull_dwn_reg:	PULL_DWN_CTRL_0,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	24
	},
	{
		name:		"uart3_rts",
		busy:		0,
		debug:		0,
		mux_reg_name:	"FUNC_MUX_CTRL_5",
		mux_reg:	FUNC_MUX_CTRL_5,
		mask_offset:	15,
		mask:		BITS010,
		pull_dwn_name:	"PULL_DWN_CTRL_0",
		pull_dwn_reg:	PULL_DWN_CTRL_0,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	25
	},
	{
		name:		"uart3_clkreq",
		busy:		0,
		debug:		0,
		mux_reg_name:	"FUNC_MUX_CTRL_9",
		mux_reg:	FUNC_MUX_CTRL_9,
		mask_offset:	27,
		mask:		BITS000,
		pull_dwn_name:	"PULL_DWN_CTRL_2",
		pull_dwn_reg:	PULL_DWN_CTRL_2,
		pull_dwn_val:	USE_PULL_DWN,
		pull_dwn_bit:	5
	},
	/* USB master generic */
	{
		name:		"usb_vbus",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_7",
		mux_reg:	FUNC_MUX_CTRL_7,
		mask_offset:	9,
		mask:		BITS010,
		pull_dwn_name:	"PULL_DWN_CTRL_1",
		pull_dwn_reg:	PULL_DWN_CTRL_1,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	11
	},
	{
		name:		"gpio0",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_7",
		mux_reg:	FUNC_MUX_CTRL_7,
		mask_offset:	9,
		mask:		BITS000,
		pull_dwn_name:	"PULL_DWN_CTRL_1",
		pull_dwn_reg:	PULL_DWN_CTRL_1,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	11
	},
	{
		name:		"usb_puen",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_D",
		mux_reg:	FUNC_MUX_CTRL_D,
		mask_offset:	3,
		mask:		BITS000,
		pull_dwn_name:	"PULL_DWN_CTRL_3",
		pull_dwn_reg:	PULL_DWN_CTRL_3,
		pull_dwn_val:	USE_PULL_DWN,
		pull_dwn_bit:	5
	},
	{
		name:		"usb_clko",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_D",
		mux_reg:	FUNC_MUX_CTRL_D,
		mask_offset:	3,
		mask:		BITS001,
		pull_dwn_name:	"PULL_DWN_CTRL_3",
		pull_dwn_reg:	PULL_DWN_CTRL_3,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	5
	},

	/* USB1 master */
	{
		name:		"usb1_susp",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_8",
		mux_reg:	FUNC_MUX_CTRL_8,
		mask_offset:	27,
		mask:		BITS010,
		pull_dwn_name:	"PULL_DWN_CTRL_1",
		pull_dwn_reg:	PULL_DWN_CTRL_1,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	27
	},
	{
		name:		"usb1_seo",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_9",
		mux_reg:	FUNC_MUX_CTRL_9,
		mask_offset:	0,
		mask:		BITS010,
		pull_dwn_name:	"PULL_DWN_CTRL_1",
		pull_dwn_reg:	PULL_DWN_CTRL_1,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	28
	},
	{
		name:		"usb1_txen",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_9",
		mux_reg:	FUNC_MUX_CTRL_9,
		mask_offset:	3,
		mask:		BITS010,
		pull_dwn_name:	"PULL_DWN_CTRL_1",
		pull_dwn_reg:	PULL_DWN_CTRL_1,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	29
	},
	{
		name:		"usb1_txd",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_9",
		mux_reg:	FUNC_MUX_CTRL_9,
		mask_offset:	24,
		mask:		BITS001,
		pull_dwn_name:	"PULL_DWN_CTRL_2",
		pull_dwn_reg:	PULL_DWN_CTRL_2,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	4
	},
	{
		name:		"usb1_vp",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_A",
		mux_reg:	FUNC_MUX_CTRL_A,
		mask_offset:	3,
		mask:		BITS001,
		pull_dwn_name:	"PULL_DWN_CTRL_2",
		pull_dwn_reg:	PULL_DWN_CTRL_2,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	7
	},
	{
		name:		"usb1_vm",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_A",
		mux_reg:	FUNC_MUX_CTRL_A,
		mask_offset:	6,
		mask:		BITS001,
		pull_dwn_name:	"PULL_DWN_CTRL_2",
		pull_dwn_reg:	PULL_DWN_CTRL_2,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	8
	},
	{
		name:		"usb1_rcv",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_A",
		mux_reg:	FUNC_MUX_CTRL_A,
		mask_offset:	9,
		mask:		BITS001,
		pull_dwn_name:	"PULL_DWN_CTRL_2",
		pull_dwn_reg:	PULL_DWN_CTRL_2,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	9
	},
	{
		name:		"usb1_speed",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_A",
		mux_reg:	FUNC_MUX_CTRL_A,
		mask_offset:	12,
		mask:		BITS010,
		pull_dwn_name:	"PULL_DWN_CTRL_2",
		pull_dwn_reg:	PULL_DWN_CTRL_2,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	10
	},

	/* USB2 master */
	{
		name:		"usb2_susp",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_B",
		mux_reg:	FUNC_MUX_CTRL_B,
		mask_offset:	3,
		mask:		BITS001,
		pull_dwn_name:	"PULL_DWN_CTRL_2",
		pull_dwn_reg:	PULL_DWN_CTRL_2,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	17
	},
	{
		name:		"usb2_vp",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_B",
		mux_reg:	FUNC_MUX_CTRL_B,
		mask_offset:	6,
		mask:		BITS001,
		pull_dwn_name:	"PULL_DWN_CTRL_2",
		pull_dwn_reg:	PULL_DWN_CTRL_2,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	18
	},
	{
		name:		"usb2_txen",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_B",
		mux_reg:	FUNC_MUX_CTRL_B,
		mask_offset:	9,
		mask:		BITS001,
		pull_dwn_name:	"PULL_DWN_CTRL_2",
		pull_dwn_reg:	PULL_DWN_CTRL_2,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	19
	},
	{
		name:		"usb2_vm",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_C",
		mux_reg:	FUNC_MUX_CTRL_C,
		mask_offset:	18,
		mask:		BITS001,
		pull_dwn_name:	"PULL_DWN_CTRL_3",
		pull_dwn_reg:	PULL_DWN_CTRL_3,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	0
	},
	{
		name:		"usb2_rcv",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_C",
		mux_reg:	FUNC_MUX_CTRL_C,
		mask_offset:	21,
		mask:		BITS001,
		pull_dwn_name:	"PULL_DWN_CTRL_3",
		pull_dwn_reg:	PULL_DWN_CTRL_3,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	1
	},
	{
		name:		"usb2_seo",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_C",
		mux_reg:	FUNC_MUX_CTRL_C,
		mask_offset:	24,
		mask:		BITS010,
		pull_dwn_name:	"PULL_DWN_CTRL_3",
		pull_dwn_reg:	PULL_DWN_CTRL_3,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	2
	},
	{
		name:		"usb2_txd",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_C",
		mux_reg:	FUNC_MUX_CTRL_C,
		mask_offset:	27,
		mask:		BITS010,
		pull_dwn_name:	"PULL_DWN_CTRL_3",
		pull_dwn_reg:	PULL_DWN_CTRL_3,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	3
	},
	{
		name:		"gpio1",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_7",
		mux_reg:	FUNC_MUX_CTRL_7,
		mask_offset:	6,
		mask:		BITS000,
		pull_dwn_name:	"PULL_DWN_CTRL_1",
		pull_dwn_reg:	PULL_DWN_CTRL_1,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	10
	},
	{
		name:		"gpio2",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_7",
		mux_reg:	FUNC_MUX_CTRL_7,
		mask_offset:	3,
		mask:		BITS000,
		pull_dwn_name:	"PULL_DWN_CTRL_1",
		pull_dwn_reg:	PULL_DWN_CTRL_1,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	9
	},
	{
		name:		"gpio3",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_7",
		mux_reg:	FUNC_MUX_CTRL_7,
		mask_offset:	0,
		mask:		BITS000,
		pull_dwn_name:	"PULL_DWN_CTRL_1",
		pull_dwn_reg:	PULL_DWN_CTRL_1,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	8
	},
	{
		name:		"gpio4",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_6",
		mux_reg:	FUNC_MUX_CTRL_6,
		mask_offset:	27,
		mask:		BITS000,
		pull_dwn_name:	"PULL_DWN_CTRL_1",
		pull_dwn_reg:	PULL_DWN_CTRL_1,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	7
	},
	{
		name:		"gpio6",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_6",
		mux_reg:	FUNC_MUX_CTRL_6,
		mask_offset:	24,
		mask:		BITS000,
		pull_dwn_name:	"PULL_DWN_CTRL_1",
		pull_dwn_reg:	PULL_DWN_CTRL_1,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	6
	},
	{
		name:		"gpio7",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_6",
		mux_reg:	FUNC_MUX_CTRL_6,
		mask_offset:	21,
		mask:		BITS000,
		pull_dwn_name:	"PULL_DWN_CTRL_1",
		pull_dwn_reg:	PULL_DWN_CTRL_1,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	5
	},
	{
		name:		"gpio11",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_6",
		mux_reg:	FUNC_MUX_CTRL_6,
		mask_offset:	18,
		mask:		BITS000,
		pull_dwn_name:	"PULL_DWN_CTRL_1",
		pull_dwn_reg:	PULL_DWN_CTRL_1,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	4
	},
	{
		name:		"gpio12",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_6",
		mux_reg:	FUNC_MUX_CTRL_6,
		mask_offset:	15,
		mask:		BITS000,
		pull_dwn_name:	"PULL_DWN_CTRL_1",
		pull_dwn_reg:	PULL_DWN_CTRL_1,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	3
	},
	{
		name:		"gpio13",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_6",
		mux_reg:	FUNC_MUX_CTRL_6,
		mask_offset:	12,
		mask:		BITS000,
		pull_dwn_name:	"PULL_DWN_CTRL_1",
		pull_dwn_reg:	PULL_DWN_CTRL_1,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	2
	},
	{
		name:		"gpio14",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_6",
		mux_reg:	FUNC_MUX_CTRL_6,
		mask_offset:	9,
		mask:		BITS000,
		pull_dwn_name:	"PULL_DWN_CTRL_1",
		pull_dwn_reg:	PULL_DWN_CTRL_1,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	1
	},
	{
		name:		"gpio15",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_6",
		mux_reg:	FUNC_MUX_CTRL_6,
		mask_offset:	6,
		mask:		BITS000,
		pull_dwn_name:	"PULL_DWN_CTRL_1",
		pull_dwn_reg:	PULL_DWN_CTRL_1,
		pull_dwn_val:	NO_PULL_DWN,
		pull_dwn_bit:	0
	},
	{
		name:		"w5_armio5",
		busy:		0,
		debug:		1,
		mux_reg_name:	"FUNC_MUX_CTRL_C",
		mux_reg:	FUNC_MUX_CTRL_C,
		mask_offset:	24,
		mask:		BITS011,
		pull_dwn_name:	"",
		pull_dwn_reg:	0,
		pull_dwn_val:	0,
		pull_dwn_bit:	0
	},
};


/*
 * Sets the Omap MUX and PULL_DWN registers based on the table
 */
int omap_cfg_reg(reg_cfg_t reg_cfg)
{
	reg_cfg_set cfg;
	u_int reg_orig, reg, pull_dwn_orig, pull_dwn;

	cfg = reg_cfg_table[reg_cfg];

	/* Check the mux register in question */
	if (cfg.mux_reg) {
		reg_orig = readl(cfg.mux_reg);

		/* The mux registers always seem to be 3 bits long */
		reg = reg_orig & ~(0x7 << cfg.mask_offset);

		reg |= (cfg.mask << cfg.mask_offset);

		writel(reg, cfg.mux_reg);
	}

	/* Check for an associated pull down register */
	if (cfg.pull_dwn_reg) {
		pull_dwn_orig = readl(cfg.pull_dwn_reg);

		if (cfg.pull_dwn_val) {
			pull_dwn = pull_dwn_orig & ~(cfg.pull_dwn_bit);
		} else {
			pull_dwn = pull_dwn_orig | cfg.pull_dwn_bit;
		}

		writel(pull_dwn, cfg.pull_dwn_reg);
	}

#ifdef DEBUG_MUX
	if (cfg.debug) {
		printk("Omap: Setting register %s\n"
		       "      %s (0x%08x) = 0x%08x -> 0x%08x\n"
		       "      %s (0x%08x) = 0x%08x -> 0x%08x\n",
		       cfg.name, cfg.mux_reg_name, cfg.mux_reg, reg_orig, reg,
		       cfg.pull_dwn_name, cfg.pull_dwn_reg, pull_dwn_orig,
		       pull_dwn);
	}
#endif

	return 0;
}

EXPORT_SYMBOL(omap_cfg_reg);
