/* Start of file */

/* {{{ [fold] Comments */
/*
 * qce-ga, linux V4L driver for the Quickcam Express and Dexxa Quickcam
 *
 * memory.c - contains all needed memory management functions
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
/* }}} */

#include <linux/config.h>
#include <linux/version.h>
#ifdef CONFIG_SMP
#define __SMP__
#endif

#include <linux/wrapper.h>
#include <asm/io.h>
#include <linux/mm.h>		/* Required on Alpha, from Bob McElrath <mcelrath@draal.physics.wisc.edu> */
#include <asm/pgtable.h>	/* Required on Alpha */
#include <linux/vmalloc.h>	/* Required on Alpha */
#include <linux/pagemap.h>	/* pmd_offset requires this on SuSE supplied kernels */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
#define virt_to_page(page)	(MAP_NR(page))
#define page_address(x)		(x | PAGE_OFFSET)
#endif

#ifndef pte_offset
#define pte_offset(pmd,adr) pte_offset_map(pmd,adr)	/* Emulation for a kernel using the new rmap-vm */
#endif							/* Fix by Michele Balistreri <brain87@gmx.net> */

/* {{{ [fold] uvirt_to_kva(pgd_t *pgd, unsigned long adr) */
/*
 * Given PGD from the address space's page table, return the kernel
 * virtual mapping of the physical memory mapped at ADR.
 */
inline unsigned long uvirt_to_kva(pgd_t *pgd, unsigned long adr)
{
	unsigned long ret = 0UL;
	pmd_t *pmd;
	pte_t *ptep, pte;

	if (!pgd_none(*pgd)) {
		pmd = pmd_offset(pgd, adr);
		if (!pmd_none(*pmd)) {
			ptep = pte_offset(pmd, adr);
			pte = *ptep;
			if (pte_present(pte)) {
				ret = (unsigned long) page_address(pte_page(pte));
				ret |= (adr & (PAGE_SIZE - 1));
			}
		}
	}
	
	return ret;
}
/* }}} */
/* {{{ [fold] uvirt_to_bus(unsigned long adr) */
inline unsigned long uvirt_to_bus(unsigned long adr)
{
	unsigned long kva, ret;

	kva = uvirt_to_kva(pgd_offset(current->mm, adr), adr);
	ret = virt_to_bus((void *) kva);

	return ret;
}
/* }}} */
/* {{{ [fold] kvirt_to_bus(unsigned long adr) */
inline unsigned long kvirt_to_bus(unsigned long adr)
{
	unsigned long va, kva, ret;

	va = VMALLOC_VMADDR(adr);
	kva = uvirt_to_kva(pgd_offset_k(va), va);
	ret = virt_to_bus((void *) kva);

	return ret;
}
/* }}} */
/* {{{ [fold] kvirt_to_pa(unsigned long adr) */
/*
 * Here we want the physical address of the memory.
 * This is used when initializing the contents of the
 * area and marking the pages as reserved.
 */
inline unsigned long kvirt_to_pa(unsigned long adr)
{
	unsigned long va, kva, ret;

	va = VMALLOC_VMADDR(adr);
	kva = uvirt_to_kva(pgd_offset_k(va), va);
	ret = __pa(kva);

	return ret;
}
/* }}} */
/* {{{ [fold] rvmalloc(unsigned long size) */
void *rvmalloc(unsigned long size)
{
	void *mem;
	unsigned long adr, page;

	/* Round it off to PAGE_SIZE */
	size += (PAGE_SIZE - 1);
	size &= ~(PAGE_SIZE - 1);

	mem = vmalloc(size);
	if (!mem) return NULL;

	memset(mem, 0, size); /* Clear the ram out, no junk to the user */
	adr = (unsigned long) mem;
	while (size > 0) {
		page = kvirt_to_pa(adr);
		mem_map_reserve(virt_to_page(__va(page)));
		adr += PAGE_SIZE;
		if (size > PAGE_SIZE)
			size -= PAGE_SIZE;
		else
			size = 0;
	}
	return mem;
}
/* }}} */
/* {{{ [fold] rvfree(void *mem, unsigned long size) */
void rvfree(void *mem, unsigned long size)
{
	unsigned long adr, page;

	if (!mem) return;

	size += (PAGE_SIZE - 1);
	size &= ~(PAGE_SIZE - 1);
	adr = (unsigned long) mem;
	
	while (size > 0) {
		page = kvirt_to_pa(adr);
		mem_map_unreserve(virt_to_page(__va(page)));
		
		adr += PAGE_SIZE;
		
		if (size > PAGE_SIZE)
			size -= PAGE_SIZE;
		else
			size = 0;
	}
	vfree(mem);
}
/* }}} */

/* End of file */
