/*
* Driver interface to the ASIC Complasion chip on the iPAQ H3800
*
* Copyright 2001 Compaq Computer Corporation.
*
* Use consistent with the GNU GPL is permitted,
* provided that this copyright notice is
* preserved in its entirety in all copies and derived works.
*
* COMPAQ COMPUTER CORPORATION MAKES NO WARRANTIES, EXPRESSED OR IMPLIED,
* AS TO THE USEFULNESS OR CORRECTNESS OF THIS CODE OR ITS
* FITNESS FOR ANY PARTICULAR PURPOSE.
*
* Author:  Andrew Christian
*          <Andrew.Christian@compaq.com>
*          October 2001
*/

#include <linux/module.h>
#include <linux/version.h>
#include <linux/config.h>

#include <linux/init.h>

#include <asm/arch/hardware.h>
#include <asm/arch-sa1100/h3600_hal.h>
#include <asm/hardware/ipaq-asic2.h>
#include <asm/arch/hardware.h>
#include <asm/arch-sa1100/h3600_hal.h>

#include "asic2_shared.h"

// Define this to see all suspend/resume init/cleanup messages
#define DEBUG_INIT()  \
        if (1) printk(KERN_NOTICE "%s\n", __FUNCTION__)

#define PDEBUG(format,arg...) printk(KERN_DEBUG __FILE__ ":%s - " format "\n", __FUNCTION__ , ## arg )
#define PALERT(format,arg...) printk(KERN_ALERT __FILE__ ":%s - " format "\n", __FUNCTION__ , ## arg )
#define PERROR(format,arg...) printk(KERN_ERR __FILE__ ":%s - " format "\n", __FUNCTION__ , ## arg )

/***********************************************************************************
 *      Shared resources                                                           
 *                                                                                 
 *      EX1 on Clock      24.576 MHz crystal (OWM,ADC,PCM,SPI,PWM,UART,SD,Audio)       
 *      EX2               33.8688 MHz crystal (SD Controller and Audio)
 ***********************************************************************************/

struct asic_shared {
	spinlock_t        lock;
	int               clock_ex1;    /* 24.765 MHz crystal */
	int               clock_ex2;    /* 33.8688 MHz crystal */
} g_shared;

void h3600_asic_shared_add( unsigned long *s, enum ASIC_SHARED v )
{
	unsigned long flags;
	if (0) PDEBUG("%lx %d",*s,v);

	if ( (*s & v) == v )    /* Already set */
		return;

	spin_lock_irqsave( &g_shared.lock, flags );

	if ( v & ASIC_SHARED_CLOCK_EX1 && !(*s & ASIC_SHARED_CLOCK_EX1)) {
		if ( !g_shared.clock_ex1++ ) {
			if (0) PDEBUG(": enabling EX1");
			H3800_ASIC2_CLOCK_Enable |= ASIC2_CLOCK_EX1;
		}
	}

	if ( v & ASIC_SHARED_CLOCK_EX2 && !(*s & ASIC_SHARED_CLOCK_EX2)) {
		if ( !g_shared.clock_ex2++ ) {
			if (0) PDEBUG(": enabling EX2");
			H3800_ASIC2_CLOCK_Enable |= ASIC2_CLOCK_EX2;
		}
	}

	spin_unlock_irqrestore( &g_shared.lock, flags );
	*s |= v;
}

void h3600_asic_shared_release( unsigned long *s, enum ASIC_SHARED v )
{
	unsigned long flags;
	if (0) PDEBUG("%lx %d",*s,v);

	spin_lock_irqsave( &g_shared.lock, flags );

	if ( v & ASIC_SHARED_CLOCK_EX1 && !(~*s & ASIC_SHARED_CLOCK_EX1)) {
		if ( !--g_shared.clock_ex1 ) {
			if (0) PDEBUG(": disabling EX1");
			H3800_ASIC2_CLOCK_Enable &= ~ASIC2_CLOCK_EX1;
		}
	}

	if ( v & ASIC_SHARED_CLOCK_EX2 && !(~*s & ASIC_SHARED_CLOCK_EX2)) {
		if ( !--g_shared.clock_ex2 ) {
			if (0) PDEBUG(": disabling EX2");
			H3800_ASIC2_CLOCK_Enable &= ~ASIC2_CLOCK_EX2;
		}
	}

	spin_unlock_irqrestore( &g_shared.lock, flags );
	*s &= ~v;
}

void h3600_asic_shared_debug (int *ex1, int *ex2)
{
	*ex1 = g_shared.clock_ex1;
	*ex2 = g_shared.clock_ex2;
}

int __init h3600_asic_shared_init( void )
{
	DEBUG_INIT();
	spin_lock_init(&g_shared.lock);
	H3800_ASIC2_CLOCK_Enable &= ~(ASIC2_CLOCK_EX1 | ASIC2_CLOCK_EX2);
	return 0;
}

void h3600_asic_shared_cleanup( void )
{
	DEBUG_INIT();

	if ( H3800_ASIC2_CLOCK_Enable & (ASIC2_CLOCK_EX1 | ASIC2_CLOCK_EX2) ) {
		PDEBUG(": Error - EX still enabled");
		H3800_ASIC2_CLOCK_Enable &= ~(ASIC2_CLOCK_EX1 | ASIC2_CLOCK_EX2);
	}
}

void h3600_asic_clock_enable (unsigned long bit, int on)
{
	unsigned long flags;

	local_irq_save (flags);

	if (on)
		H3800_ASIC2_CLOCK_Enable |= bit;
	else
		H3800_ASIC2_CLOCK_Enable &= ~bit;

	local_irq_restore (flags);
}
