/* 
 * arch/sh/mm/sq.c
 * 
 * SH4 Store Queue support.
 *
 * Copyright (c) 2001 M. R. Brown <mrbrown@0xd6.org>
 * Copyright (c) 2001 Paul Mundt  <lethal@chaoticdreams.org>
 *
 * Released under the terms of the GNU GPL v2.0
 *
 * Some concepts/code "gracefully" borrowed from arch/sh/mm/ioremap.c.
 */
#include <linux/init.h>
#include <linux/list.h>
#include <linux/slab.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/mmzone.h>

#include <asm/page.h>
#include <asm/pgalloc.h>
#include <asm/sq.h>
#include <asm/io.h>

/*
 * Due to the overlapping nature of the store queues, we can't guarantee
 * data consistency on a prefetch with more than one driver accessing the
 * store queues at the same time.
 */
static DECLARE_MUTEX(sq_sem);

/* SQ Zone */
static zone_t *sq_zone = 0;

/**
 * sq_init - Allocate zone
 *
 * Allocates SQ address space as a zone.
 */
int __init sq_init(void)
{
	if (sq_zone)
		BUG();
	if (SQ_ADDRBASE & ~PAGE_MASK)
		BUG();

	sq_zone = kmalloc(sizeof(struct zone_struct), GFP_KERNEL);
	if (!sq_zone)
		return -ENOMEM;
	memset(sq_zone, 0, sizeof(struct zone_struct));

	/*
	 * Setup the SQ area as a zone
	 */
	sq_zone->zone_start_paddr = SQ_ADDRBASE;
	sq_zone->size             = SQ_ADDRMAX - SQ_ADDRBASE;
	sq_zone->free_pages       = sq_zone->size / PAGE_SIZE;
	sq_zone->lock             = SPIN_LOCK_UNLOCKED;

	INIT_LIST_HEAD(&sq_zone->free_area->free_list);
	return 0;
}

/**
 * sq_exit - Free the zone
 *
 * Validates that there no regions left allocated, and releases the
 * previously allocated zone.
 */
void __exit sq_exit(void)
{
	if (!list_empty(&sq_zone->free_area->free_list))
		BUG();

	kfree(sq_zone);
}

/**
 * sq_alloc_pages - Allocate pages
 *
 * @size: size of region to allocate
 *
 * Page aligns @size, allocates pages from available location, inserts
 * region into region list in the zone, and returns a pointer to the
 * allocated region.
 */
static void *sq_alloc_pages(unsigned long size)
{
	unsigned long start;
	unsigned int nr_pages = size / PAGE_SIZE;
	void *ret;
	unsigned long i;

	printk("sq_alloc_pages(): start, nr_pages is %d\n", nr_pages);
	printk("sq_alloc_pages: consistency check 1\n");
	if (!sq_zone->free_pages) {
		printk("sq_alloc_pages(): returning NULL\n");
		return NULL;
	}
	printk("sq_alloc_pages(): free pages check passed\n");

	down(&sq_sem);

	if (nr_pages > sq_zone->free_pages) {
		printk(KERN_WARNING "sq.c: Can't allocate %d pages, "
				    "allocating %d instead.\n",
				    nr_pages, (int)sq_zone->free_pages);
		nr_pages = sq_zone->free_pages;
		printk("sq_alloc_pages(): nr_pages is now %d\n", nr_pages);
	}

	printk("sq_alloc_pages(): zone_start_paddr + size = %08x\n", sq_zone->zone_start_paddr + sq_zone->size);
	printk("sq_alloc_pages(): free_pages * PAGE_SIZE = %08x\n", sq_zone->free_pages * PAGE_SIZE);

	/* Align the start address to 32 bytes. */
	start = SQ_ALIGN((sq_zone->zone_start_paddr + sq_zone->size) -
		(sq_zone->free_pages * PAGE_SIZE));
	printk("sq_alloc_pages(): start = %08x\n", start);

	sq_zone->free_pages -= nr_pages;
	printk("sq_alloc_pages(): free_pages decr., now = %d\n", sq_zone->free_pages);

	ret = (void *)start;
	printk("sq_alloc_pages(): ret = %08x\n", ret);

	/* 
	 * This isn't really what the free_list was intended for.. but it's as
	 * good a spot as any to store information on allocated regions.
	 */
	{
		struct sq_area *area;

		area = kmalloc(sizeof(struct sq_area), GFP_KERNEL);
		if (!area) {
			up(&sq_sem);
			printk("sq_alloc_pages(): up called\n");
			return NULL;
		}
		area->addr = ret;
		printk("sq_alloc_pages(): area->addr = %08x\n", area->addr);
		area->size = nr_pages * PAGE_SIZE;
		printk("sq_alloc_pages(): area->size = %08x\n", area->size);
		
		list_add((struct list_head *)area, &sq_zone->free_area->free_list);
		printk("sq_alloc_pages(): list_add done\n");
	}

	up(&sq_sem);
	printk("sq_alloc_pages(): returning ret\n");
	return ret;
}

/**
 * sq_free_pages - Free allocated pages
 *
 * @region: pointer to physical base address of region to free
 * 
 * Releases the allocated region @region and removes it from the
 * list of allocated regions in the zone.
 */
static void sq_free_pages(void *region)
{
	struct list_head *lh;
	struct sq_area *area = 0;
	int found = 0;

	down(&sq_sem);

	/*
	 * Spin the list looking for the region ..
	 */
	list_for_each(lh, &sq_zone->free_area->free_list) {
		area = (struct sq_area *)lh;
		
		if (region == area->addr) {
			found++;
			break;
		}
	}

	/*
	 * Don't do anything if we got an invalid region
	 */
	if (!found) {
		up(&sq_sem);
		return;
	}
	
	sq_zone->free_pages += area->size / PAGE_SIZE;

	list_del(lh);
	kfree(area);

	up(&sq_sem);
}


/**
 * sq_ioremap - Map a physical address to a store queue
 * @phys_addr: the physical address to map
 * @size: the size of address range
 *
 * Maps the physical address range to an address range suitable for store
 * queue operations.  Returns the address of the allocated store queue.
 */
void *sq_ioremap(unsigned long phys_addr, unsigned long size)
{
	unsigned long offset, last_addr;
	unsigned long flags = 0;
	void *addr;

	/* Don't allow wraparound or zero size */
	last_addr = phys_addr + size - 1;
	if (!size || last_addr < phys_addr)
		return NULL;

	/* Page-align the mapping */
	offset = phys_addr & ~PAGE_MASK;
	phys_addr &= PAGE_MASK;
	size = PAGE_ALIGN(last_addr) - phys_addr;
	printk("sq_ioremap(): phys_addr = %08x, size = %08x, offset = %08x\n", phys_addr, size, offset);

	/* Allocate the store queue area and map it */
	printk("sq_ioremap(): calling sq_alloc_pages()\n");
	addr = sq_alloc_pages(size);
	if (!addr)
		return NULL;
	printk("sq_ioremap(): done with sq_alloc_pages()\n");

	return (void *)(offset + (char *)addr);
}


/**
 * sq_iounmap - Unmap a store queue
 * @addr: the store queue address to unmap
 *
 * Unmaps a previously mapped store queue address.
 */
void sq_iounmap(void *addr)
{
	sq_free_pages(addr);
}


/**
 * sq_flush - Flush (prefetch) the store queue cache
 * @addr: the store queue address to flush
 *
 * Executes a prefetch instruction on the specified store queue cache,
 * so that the cached data is written to physical memory.
 */
__inline__ void sq_flush(void *addr)
{
	(unsigned long *)addr = SQ_ALIGN((unsigned long)addr);
	printk("sq_flush: flushing %08x\n", addr);
	__asm__ __volatile__ ("pref @%0": "=r" (addr) : : "memory");
}

/**
 * sq_write - Write data to memory using the store queue cache
 * @addr: the store queue address
 * @buffer: pointer to the write buffer
 * @size: size of the buffer in bytes
 *
 * Copies the contents of @buffer to @addr, flushing the store queue cache
 * every 32 bytes.  Note that because store queues are 32-byte aligned only
 * @size MOD 32 bytes are sent, so your buffer size should already be a
 * multiple of 32 bytes.
 */
void sq_write(void *addr, void *buffer, unsigned long size)
{
	BUG();
}
