/*
 * linux/drivers/video/pl110fb.h
 *    -- ARM PrimeCell PL110 LCD controller frame buffer device
 *
 * Copyright (C) 2002 Lineo, Inc.
 *  
 * Portions Copyright (C) 2001 Sharp Microelectronics of the Americas, Inc.
 * 		       CAMAS, WA
 *
 * based in part on sa1100fb.h, which is Copyright (C) Eric A. Thomas
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file COPYING in the main directory of this archive
 * for more details.
 */

/*
 * Color LCD Controller registers
 */
typedef struct {
    volatile u32	timing0;		/* Horizontal axis panel control	*/
    volatile u32	timing1;		/* Vertical axis panel control		*/
    volatile u32	timing2;		/* clock and signal polarity control	*/
    volatile u32	timing3;		/* line end control			*/
    volatile dma_addr_t	upbase;			/* upper panel frame base address	*/
    volatile dma_addr_t	lpbase;			/* lower panel frame base address	*/
    volatile u32	intrEnable;		/* interrupt enable mask		*/
    volatile u32	control;		/* LCD panel pixel parameters		*/
    volatile u32	rawIntrStatus;		/* raw interrupt status			*/
    volatile u32	maskedIntrStatus;	/* masked interrupt status		*/
    volatile dma_addr_t	upcurr;			/* upper panel current address		*/
    volatile dma_addr_t	lpcurr;			/* lower panel current address		*/
    volatile dma_addr_t	lpoverflow;		/* SDRAM fb base			*/
    volatile u32	reservedcc[115];	/* reserved				*/
    volatile u32	palette[128];		/* 256 x 16-bit color palette		*/
} lcdRegs_t;


/*
 * LCDTiming0 Register Bit Field constants
 *
 * NOTE: Ensure the argument to the following macros is greater
 * than zero.
 */ 
#define LCD_TIMING0_HBP(n)	_SBF(24,((n)-1))		/* Horiz Back Porch */ 
#define LCD_TIMING0_HFP(n)	_SBF(16,((n)-1))		/* Horiz Front Porch */ 
#define LCD_TIMING0_HSW(n)	_SBF(8,((n)-1))			/* Horiz sync Pulse Width */ 
#define LCD_TIMING0_PPL(n)	_SBF(2,((((n)/16)-1)&0x3F))	/* Pixels per line */ 

/*
 * LCDTiming1 Register Bit Field constants
 *
 * NOTE: Ensure the argument to the following macros is greater
 * than zero.
 */ 
#define LCD_TIMING1_VBP(n)	_SBF(24,(n))		/* Vertical Back Porch */ 
#define LCD_TIMING1_VFP(n)	_SBF(16,(n))		/* Vertical Front Porch */ 
#define LCD_TIMING1_VSW(n)	_SBF(10,(n))		/* Vertical Synchronization Pulse */ 
#define LCD_TIMING1_LPP(n)	_SBF(0,((n)-1))		/* Lines per Panel */ 

/*
 * LCDTiming2 Register Bit Field constants
 *
 * NOTE: Ensure the argument to the following macros is greater
 * than two.
 */ 
#define LCD_TIMING2_BCD 	_BIT(26)		/* Bypass Pixel Clock Divider */
#define LCD_TIMING2_CPL(n)	_SBF(16,((n)-1)&0x3FF)	/* Clocks Per Line */ 
#define LCD_TIMING2_IOE		_BIT(14)		/* Invert Output Enable */ 
#define LCD_TIMING2_IPC		_BIT(13)		/* Invert Panel Clock */ 
#define LCD_TIMING2_IHS 	_BIT(12)		/* Invert Horizontal Synchronization */ 
							/* set == HSYNC is active low	     */
#define LCD_TIMING2_IVS		_BIT(11)		/* Invert Vertical Synchronization */ 
							/* set == VSYNC is active low	     */
#define LCD_TIMING2_ACB(n)	_SBF(6,((n)-1))		/* AC Bias Pin Frequency */ 
#define LCD_TIMING2_CLKSEL	 _BIT(5)		/* Clock Selector */ 
#define LCD_TIMING2_PCD(n)	_SBF(0,((n)-2))		/* Panel Clock Divisor */ 


/*
 * LCDTiming3 Register Bit Field constants
 *
 * NOTE: Ensure the argument to the following macros is greater
 * than one.
 */ 
#define LCD_TIMING3_LEE		_BIT(16)		/* Line End Enable */ 
#define LCD_TIMING3_LED(n)	_SBF(0,((n)-1))		/* Line End Signal Delay */ 


/*
 * intrEnable, rawIntrStatus, maskedIntrStatus bit field positions
 */
#define LCD_STATUS_MBERROR	_BIT(4)			/* Master Bus Error */ 
#define LCD_STATUS_VCOMP 	_BIT(3)			/* Vertical Compare */ 
#define LCD_STATUS_LNBU 	_BIT(2)			/* LCD Next addr. Base Update*/ 
#define LCD_STATUS_FUF 		_BIT(1)			/* FIFO underflow */ 


/*
 * Control Register Bit Field constants
 */ 
#define LCD_CTRL_WATERMARK	_BIT(16)		/* LCD DMA FIFO Watermark Level */ 
#define LCD_CTRL_LDMAFIFOTME	_BIT(15)		/* LCD DMA FIFO Test Mode Enable */ 

#define LCD_CTRL_VCOMP(n)	_SBF(12,((n)&0x3))	/* Generate interrupt at:	*/
#define LCD_CTRL_VCOMP_SVS	_SBF(12,0)		/*   start of vertical sync	*/
#define LCD_CTRL_VCOMP_SBP	_SBF(12,1)		/*   start of back porch 	*/
#define LCD_CTRL_VCOMP_SAV	_SBF(12,2)		/*   start of active video	*/
#define LCD_CTRL_VCOMP_SFP	_SBF(12,3)		/*   start of front porch	*/

#define LCD_CTRL_PWR		_BIT(11)		/* LCD Power Enable		*/ 
#define LCD_CTRL_BEPO		_BIT(10)		/* Big Endian Pixel Order	*/ 
#define LCD_CTRL_BEBO		_BIT(9)			/* Big Endian Byte Order	*/ 
#define LCD_CTRL_BGR		_BIT(8)			/* Swap Red and Blue (RGB to BGR) */ 
#define LCD_CTRL_DUAL		_BIT(7)			/* Dual Panel STN		*/ 
#define LCD_CTRL_MON8		_BIT(6)			/* Monochrome LCD has 8-bit interface */ 
#define LCD_CTRL_TFT 		_BIT(5)			/* TFT LCD			*/ 

#define LCD_CTRL_BW_COLOR	_SBF(4,0)		/* STN LCD is Color		*/ 
#define LCD_CTRL_BW_MONO	_SBF(4,1)		/* STN LCD is Monochrome	*/

#define LCD_CTRL_BPP1		_SBF(1,0)		/* Bits per pixel		*/ 
#define LCD_CTRL_BPP2		_SBF(1,1)
#define LCD_CTRL_BPP4		_SBF(1,2)
#define LCD_CTRL_BPP8		_SBF(1,3)
#define LCD_CTRL_BPP16		_SBF(1,4)
#define LCD_CTRL_BPP24		_SBF(1,5)

#define LCD_CTRL_ENABLE		_BIT(0)			/* LCD Controller Enable	*/ 



typedef struct {
    volatile u32 setup;					/* Setup		*/ 
    volatile u32 control;				/* Control		*/ 
    volatile u32 timing1;				/* HR-TFT Timing 1	*/ 
    volatile u32 timing2;				/* HR-TFT Timing 2	*/ 
} lcdicpRegs_t;



/*
 * LCDICP Setup Register Bit Fields
 *
 * NOTE: Ensure the argument to the following macros is greater
 * than zero.
 */ 
#define	LCDICP_SETUP_MODE_BYPASS	_SBF(0,0)
#define	LCDICP_SETUP_MODE_HRTFT		_SBF(0,1)
#define	LCDICP_SETUP_MODE_DMTN		_SBF(0,2)
#define	LCDICP_SETUP_HORIZ_REVERSE	_SBF(2,0)
#define	LCDICP_SETUP_HORIZ_NORMAL	_SBF(2,1)
#define	LCDICP_SETUP_VERT_REVERSE	_SBF(3,0)
#define	LCDICP_SETUP_VERT_NORMAL	_SBF(3,1)
/* Calculates bit field value from actual pixels per line */ 
#define LCDICP_SETUP_PPL(n)		_SBF(4,((n)-1))
#define LCDICP_SETUP_POWER		_BIT(13)	/* lh7a400 only */


/*
 * LCDICP Control Register Bit Fields
 */ 
#define LCDICP_CONTROL_SPSEN		_BIT(0)
#define LCDICP_CONTROL_CLSEN		_BIT(1)
#define LCDICP_CONTROL_UBLEN		_BIT(2)
#define LCDICP_CONTROL_DISP		_BIT(3)
#define LCDICP_CONTROL_EN0		_BIT(4)
#define LCDICP_CONTROL_EN1		_BIT(5)
#define LCDICP_CONTROL_EN2		_BIT(6)
#define LCDICP_CONTROL_EN3		_BIT(7)


/*
 * LCDICP Timing 1 Register Bit Fields
 *
 * NOTE: Ensure the argument to the following macros is greater
 * than zero.
 */ 
#define LCDICP_TIMING1_LPDEL(n)		_SBF(0,((n)-1)&0xF)
#define LCDICP_TIMING1_REVDEL(n)	_SBF(4,((n)-1)&0xF)
#define LCDICP_TIMING1_PSDEL(n)		_SBF(8,((n)-1)&0xF)
#define LCDICP_TIMING1_CLSDEL(n)	_SBF(8,((n)-1)&0xF)


/*
 * LCDICP Timing 2 Register Bit Fields
 *
 * NOTE: Ensure the argument to the following macros is greater
 * than zero.
 */ 
#define LCDICP_TIMING2_PSDEL2(n)	_SBF(0,((n)-1)&0x1FF)
#define LCDICP_TIMING2_CLSDEL2(n)	_SBF(0,((n)-1)&0x1FF)
#define LCDICP_TIMING2_SPLVALUE(n)	_SBF(9,((n)-1)&0x7F)


/*
 * These are the bitfields for each
 * display depth that we support.
 */
struct pl110fb_rgb {
	struct fb_bitfield	red;
	struct fb_bitfield	green;
	struct fb_bitfield	blue;
	struct fb_bitfield	transp;
};

/*
 * This structure describes the machine which we are running on.
 */
struct pl110fb_mach_info {
	u_long		pixclock;

	u_short		xres;
	u_short		yres;

	u_char		bpp;
	u_char		hsync_len;		/* horiz sync pulse width	*/
	u_char		left_margin;		/* horiz back porch		*/
	u_char		right_margin;		/* horiz front porch		*/

	u_char		vsync_len;		/* vertical sync pulse width	*/
	u_char		upper_margin;		/* vertical back porch		*/
	u_char		lower_margin;		/* vertical front porch		*/
	u_char		sync;

	u_int		cmap_greyscale:1,
			cmap_inverse:1,
			cmap_static:1,
			unused:29;

	u_long		 LCDtiming2;
	u_long		 LCDtiming3;
	u_long		 LCDcontrol;
	u_long		 LCDICPsetup;
	u_long		 LCDICPcontrol;
	u_long		 LCDICPtiming1;
	u_long		 LCDICPtiming2;
};


/* Shadows for LCD/LCDICP controller registers */
struct pl110fb_lcd_reg {
	u_long		 LCDtiming0;
	u_long		 LCDtiming1;
	u_long		 LCDtiming2;
	u_long		 LCDtiming3;
	u_long		 LCDcontrol;
	u_long		 LCDICPsetup;
	u_long		 LCDICPcontrol;
	u_long		 LCDICPtiming1;
	u_long		 LCDICPtiming2;
};

#define RGB_8	(0)
#define RGB_16	(1)
#define NR_RGB	2

struct pl110fb_info {
	struct fb_info		fb;
	signed int		currcon;

	struct pl110fb_rgb	*rgb[NR_RGB];

	u_int			max_bpp;
	u_int			max_xres;
	u_int			max_yres;

	/*
	 * These are the addresses we mapped
	 * the framebuffer memory region to.
	 */
	dma_addr_t		map_dma;
	u_char *		map_cpu;
	u_int			map_size;

	u_char *		screen_cpu;
	dma_addr_t		screen_dma;
	u_int			palette_size;

	dma_addr_t		upbase;
	dma_addr_t		lpbase;

	u_long			LCDtiming2;
	u_long			LCDtiming3;
	u_long			LCDcontrol;
	u_long			LCDICPsetup;
	u_long			LCDICPcontrol;
	u_long			LCDICPtiming1;
	u_long			LCDICPtiming2;

	u_int			cmap_inverse:1,
				cmap_static:1,
				unused:30;

	u_long			reg_LCDtiming0;
	u_long			reg_LCDtiming1;
	u_long			reg_LCDtiming2;
	u_long			reg_LCDtiming3;
	u_long			reg_LCDcontrol;
	u_long			reg_LCDICPsetup;
	u_long			reg_LCDICPcontrol;
	u_long			reg_LCDICPtiming1;
	u_long			reg_LCDICPtiming2;

	volatile u_char		state;
	volatile u_char		task_state;
	struct semaphore	ctrlr_sem;
	wait_queue_head_t	ctrlr_wait;
	struct tq_struct	task;

#ifdef CONFIG_PM
	struct pm_dev		*pm;
#endif
#ifdef CONFIG_CPU_FREQ
	struct notifier_block	clockchg;
#endif
};

#define __type_entry(ptr,type,member) ((type *)((char *)(ptr)-offsetof(type,member)))

#define TO_INF(ptr,member)	__type_entry(ptr,struct pl110fb_info,member)

/*
 * These are the actions for set_ctrlr_state
 */
#define C_DISABLE		(0)
#define C_ENABLE		(1)
#define C_DISABLE_CLKCHANGE	(2)
#define C_ENABLE_CLKCHANGE	(3)
#define C_REENABLE		(4)
#define C_DISABLE_PM		(5)
#define C_ENABLE_PM		(6)

#define PL110_NAME	"PL110"

/*
 *  Debug macros 
 */
#if DEBUG
#  define DPRINTK(fmt, args...)	printk("%s: " fmt, __FUNCTION__ , ## args)
#else
#  define DPRINTK(fmt, args...)
#endif

/*
 * Minimum X and Y resolutions
 */
#define MIN_XRES	64
#define MIN_YRES	64

