/*****************************************************************************
 * Sharp L3 LH7X algorithm/adapter module.
 *
 *  By Russell King,
 *
 *    gratuitously ripped from sa1111-uda1341.c by John Dorsey.
 *
 *    gratuitously ripped from l3-sa1111.c by Jim Gleason who
 *    replaced the guts with code originally written by SuryanG and
 *    KovitzP as the file LH79520_evb_codec_l3_driver.c (See below),
 *    and modified as needed to integrate with Linux.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 *****************************************************************************
 *
 * Workfile: LH79520_evb_codec_l3_driver.c
 * Revision: 1.0
 * Author:   SuryanG
 * Date:     Jan 29 2002 19:39:06
 *
 * Project: LH79520 Evaluation Board
 *
 * Description:
 *    This file contains the low-level driver functions that are used
 *    by the I2S audio codec control interface.
 *
 * References:
 *    (1) LH79520 User's Guide
 *
 *    (2) UDA1341TS Economy audio CODEC for MiniDisc home stereo and
 *        portable applications, Product Specification, 
 *        Philips Semiconductors
 *
 * Revision History:
 * 
 *    Rev 1.0   Jan 29 2002 19:39:06   SuryanG
 * Initial revision.
 * 
 *    Rev 1.3   Jan 23 2002 15:46:16   SuryanG
 * Added function headers.
 * 
 *    Rev 1.2   Jan 04 2002 10:12:50   KovitzP
 * CPLD bit descriptions moved to LH79520_evb.h
 * 
 *    Rev 1.1   Dec 28 2001 13:29:38   KovitzP
 * Modified for updated Timer header files.
 * 
 *    Rev 1.0   Sep 05 2001 10:56:20   SuryanG
 * Initial revision.
 * 
 *	COPYRIGHT (C) 2001 SHARP MICROELECTRONICS OF THE AMERICAS, INC.
 *		CAMAS, WA
 *
 ****************************************************************************/

#include <linux/config.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/module.h>

#include <linux/delay.h>
#include <linux/errno.h>

#include <linux/l3/l3.h>
// #include <linux/l3/uda1341.h>

#include <asm/hardware.h>
#include <asm/semaphore.h>

#include <asm/arch/hardware.h>
#include <asm/arch/cpld.h>

#define CLK_LOW_DATA_LOW_OP     _BIT(2)
#define CLK_LOW_DATA_HI_OP      (_BIT(2) | _BIT(0))

#define CLK_HI_DATA_LOW_OP      (_BIT(2) | _BIT(1))
#define CLK_HI_DATA_HI_OP       (_BIT(2) | _BIT(1) | _BIT(0))

#define INP_MODE_CLK_LOW        ((uint16_t) (~(_BIT(1)) & ~_BIT(2))) 
#define INP_MODE_CLK_HI         ((uint16_t) (_BIT(1))) 

static unsigned char l3_addr = -1;

static cpldRegs_t *cpld = (cpldRegs_t *)CPLD_BASE;

/**********************************************************************
* 
* Function: l3_lh7x_min_delay
*
* Purpose:
*   Delay a minimum time necessary to meet all L3 timing requirements.
*   (Currently we are using 1 u-second).
*
**********************************************************************/
static inline void l3_lh7x_min_delay(void)
{
	udelay(1);		/* Delay 1 u-Second */
	return;
}

/**********************************************************************
*
* Function: l3_lh7x_send_stop
*
* Purpose:
*   Sends a stop on the L3 mode line
*
* Processing:
*   Set the CPLD L3 pin to go low, wait for 1 microsecond and set it
*   high.
*
* Parameters: 
*   None
*
* Outputs: None
*
* Returns: Nothing
*
* Notes: 
*
**********************************************************************/
static void l3_lh7x_send_stop(void)
{
	CPLD_L3_MODE_LOW;
	l3_lh7x_min_delay();	/* Delay to meet min L3 timing reqs */
	CPLD_L3_MODE_HI;

	return;
}

/**********************************************************************
*
* Function: l3_lh7x_set_addr
*
* Purpose:
*   Sets the L3 protocol state (Address)
*
* Processing:
*   Generate L3 clock and shift out data on the L3 data line from
*   the CPLD.
*
* Parameters: 
*   addr - DATA0, DATA1 or STATUS
*
* Outputs: None
*
* Returns: Nothing
*
* Notes: 
*
**********************************************************************/
static void l3_lh7x_set_addr(unsigned char addr)
{
	int32_t i;
	uint16_t send;
    
	CPLD_L3_MODE_LOW; /* address mode */
 
	for(i = 0; i < 8; i++)
	{
		/* D[2:0] either 101 or 100 */
		send = (uint8_t)(((addr >> i) & _BIT(0)) ?
		CLK_LOW_DATA_HI_OP : CLK_LOW_DATA_LOW_OP); 
		cpld->l3_reg = send;
		barrier();
		l3_lh7x_min_delay();	/* Delay to meet min L3 timing reqs */
		/* D[2:0] either 111 or 110 */
		send = (uint8_t)(((addr >> i) & _BIT(0)) ? 
		CLK_HI_DATA_HI_OP : CLK_HI_DATA_LOW_OP); 
		cpld->l3_reg = send;
		barrier();
		l3_lh7x_min_delay();	/* Delay to meet min L3 timing reqs */
	}

	CPLD_L3_MODE_HI;
	l3_addr = addr;   

	return;
}

/**********************************************************************
*
* Function: l3_lh7x_send_byte
* Function: l3_lh7x_send_msg
*
* Purpose:
*   Writes data out on the L3 lines, to the codec.
*
* Processing:
*   Change modes (addr) if necessary. While there are bytes to write,
*   generate L3 clock and shift out data on the L3 data line from
*   the CPLD.
*
* Parameters: 
*   addr - DATA0, DATA1 or STATUS
*   data - data byte to be written
*
* Outputs: None
*
* Returns: Nothing
*
* Notes: 
*
**********************************************************************/
static inline void l3_lh7x_send_byte(unsigned char addr, unsigned char data)
{
	int32_t i;
	uint16_t send;

	if(addr != l3_addr)
	{
		l3_lh7x_set_addr(addr);
	}

	l3_lh7x_send_stop();
	for(i = 0; i < 8; i++)
	{
		/* D[2:0] either 101 or 100 */            
		send = (uint8_t)(((data >> i) & _BIT(0)) ? 
		CLK_LOW_DATA_HI_OP : CLK_LOW_DATA_LOW_OP); 
		cpld->l3_reg = send;
		barrier();
		l3_lh7x_min_delay();	/* Delay to meet min L3 timing reqs */
		/* D[2:0] either 111 or 110 */
		send = (uint8_t)(((data >> i) & _BIT(0)) ? 
		CLK_HI_DATA_HI_OP : CLK_HI_DATA_LOW_OP);   
		cpld->l3_reg = send;
		barrier();
		l3_lh7x_min_delay();	/* Delay to meet min L3 timing reqs */
	}    

	l3_lh7x_send_stop();

	return;
}

static void l3_lh7x_send_msg(struct l3_msg *msg)
{
	int len = msg->len;
	char *p = msg->buf;

	if (len > 1) {
		while ((len--) > 1)
			l3_lh7x_send_byte(msg->addr, *p++);
	}
	l3_lh7x_send_byte(msg->addr, *p);

	return;
}

/**********************************************************************
*
* Function: char l3_lh7x_recv_byte
* Function: l3_lh7x_recv_msg
*
* Purpose:
*   Reads data on the L3 lines from the codec.
*
* Processing:
*   Change modes (addr) if necessary. Configure input mode and generate L3 
*   clock until the data is read. 
*
* Parameters: 
*
* Outputs: None
*
* Returns: data byte read
*
* Notes: 
*
**********************************************************************/
static inline unsigned char l3_lh7x_recv_byte(unsigned char addr)
{
	unsigned char data;
	uint32_t i;
	unsigned char temp;

	if(addr != l3_addr)
	{
		l3_lh7x_set_addr(addr);
	}

	temp = 0;

	l3_lh7x_send_stop();

	for (i = 0; i < 8; i++)
	{
		/* input mode, clock low */
		cpld->l3_reg = (uint16_t) (cpld->l3_reg & INP_MODE_CLK_LOW); 
		barrier();
		l3_lh7x_min_delay();	/* Delay to meet min L3 timing reqs */
		/*input mode, clock high */
		cpld->l3_reg = (uint16_t) (cpld->l3_reg | INP_MODE_CLK_HI); 
		barrier();
		l3_lh7x_min_delay();	/* Delay to meet min L3 timing reqs */
		temp = (unsigned char)((cpld->l3_reg & _BIT(0))? 
		(temp | _BIT(i)) : (temp & ~_BIT(i)));
	}

	l3_lh7x_send_stop(); 
	data = (unsigned char)(temp >> 1);

	return(data);
}

static void l3_lh7x_recv_msg(struct l3_msg *msg)
{
	int len = msg->len;
	char *p = msg->buf;

	if (len > 1) {
		while ((len--) > 1)
			*p++ = l3_lh7x_recv_byte(msg->addr);
	}
	*p = l3_lh7x_recv_byte(msg->addr);

	return;
}

static int l3_lh7x_xfer(struct l3_adapter *adap, struct l3_msg msgs[], int num)
{
	int i;

	for (i = 0; i < num; i++) {
		struct l3_msg *pmsg = &msgs[i];

		if (pmsg->flags & L3_M_RD)
			l3_lh7x_recv_msg(pmsg);
		else
			l3_lh7x_send_msg(pmsg);
	}

	return num;
}

static struct l3_algorithm l3_lh7x_algo = {
	name:		"L3 LH7x algorithm",
	xfer:		l3_lh7x_xfer,
};

static DECLARE_MUTEX(lh7x_lock);

static struct l3_adapter l3_lh7x_adapter = {
	owner:		THIS_MODULE,
	name:		"l3-lh7x",
	algo:		&l3_lh7x_algo,
	lock:		&lh7x_lock,
};

static int __init l3_lh7x_init(void)
{
	int ret = -ENODEV;

	ret = l3_add_adapter(&l3_lh7x_adapter);

	return ret;
}

static void __exit l3_lh7x_exit(void)
{
	l3_del_adapter(&l3_lh7x_adapter);

	return;
}

module_init(l3_lh7x_init);
module_exit(l3_lh7x_exit);

