/*****************************************************************

Copyright (c) 2000 Matthias Elter
              2004 Aaron J. Seigo <aseigo@kde.org>

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************/

#include <qframe.h>
#include <qvalidator.h>
#include <qlayout.h>
#include <qpainter.h>
#include <qstyle.h>

#include <kdebug.h>
#include <khelpmenu.h>
#include <klocale.h>
#include <kglobal.h>
#include <kpopupmenu.h>
#include <kiconloader.h>
#include <kconfig.h>
#include <kstdguiitem.h>

#include "addcontainer_mnu.h"
#include "container_applet.h"
#include "container_extension.h"
#include "containerarea.h"
#include "extensionmanager.h"
#include "kicker.h"
#include "removecontainer_mnu.h"

#include "panelextension.h"
#include "panelextension.moc"

// KDE4: make these say Panel_(somenumber)
PanelExtension::PanelExtension(const QString& configFile, QWidget *parent, const char *name)
    : DCOPObject(QCString("ChildPanel_") + QString::number( (ulong) this ).latin1() )
    , KPanelExtension(configFile, KPanelExtension::Stretch, 0, parent, name)
    , _opMnu( 0 )
    , _configFile(configFile)
    , _built( false )
{
    setAcceptDrops(!Kicker::the()->isImmutable());
    setCustomMenu(opMenu());

    QVBoxLayout* _layout = new QVBoxLayout(this);

    // container area
    _containerArea = new ContainerArea( config(), this, opMenu() );
    connect(_containerArea, SIGNAL(maintainFocus(bool)), this, SIGNAL(maintainFocus(bool)));
    _layout->addWidget(_containerArea);

    _containerArea->setFrameStyle(QFrame::NoFrame);
    _containerArea->viewport()->installEventFilter(this);
    _containerArea->configure();

    // Make sure the containerarea has the right orientation from the
    // beginning.
    positionChange(position());

    connect(Kicker::the(), SIGNAL(configurationChanged()), SLOT(configurationChanged()));

    // we wait to get back to the event loop to start up the container area so that
    // the main panel in ExtensionManager will be assigned and we can tell in a
    // relatively non-hackish way that we are (or aren't) the "main panel"
    QTimer::singleShot(0, this, SLOT(populateContainerArea()));
}

PanelExtension::~PanelExtension()
{
}

void PanelExtension::populateContainerArea()
{
    _containerArea->show();

    if (ExtensionManager::the()->isMainPanel(topLevelWidget()))
    {
        setObjId("Panel");
        _containerArea->initialize(true);
    }
    else
    {
        _containerArea->initialize(false);
    }
}

void PanelExtension::configurationChanged()
{
    _containerArea->configure();
}

QPopupMenu* PanelExtension::opMenu()
{
    if (_opMnu != 0)
    {
        return _opMnu;
    }

    _opMnu = new QPopupMenu(this);
    connect(_opMnu, SIGNAL(aboutToShow()), this, SLOT(slotBuildOpMenu()));

    return _opMnu;
}

void PanelExtension::positionChange(Position p)
{
    _containerArea->setPosition(p);
}

QSize PanelExtension::sizeHint(Position p, QSize maxSize) const
{
    QSize size;

    if( p == Left || p == Right )
    {
        size = QSize( sizeInPixels(),
                      _containerArea->heightForWidth(sizeInPixels()));
    }
    else
    {
        size = QSize( _containerArea->widthForHeight(sizeInPixels()),
                      sizeInPixels() );
    }

    return size.boundedTo( maxSize );
}

bool PanelExtension::eventFilter(QObject*, QEvent * e)
{
    if ( e->type() == QEvent::MouseButtonPress )
    {
        QMouseEvent* me = (QMouseEvent*) e;
        if ( me->button() == RightButton )
        {
            opMenu()->exec(me->globalPos());
            return true;
        }
    }

    if ( e->type() == QEvent::Resize )
    {
        emit updateLayout();
    }

    return false;
}

void PanelExtension::setPanelSize(int size)
{
    setSize(static_cast<Size>(size), customSize());

    // save the size setting here if it isn't a custom setting
    config()->setGroup("General");
    config()->writeEntry("Size", size);
    config()->sync();
}

void PanelExtension::addKMenuButton()
{
    _containerArea->addKMenuButton();
}

void PanelExtension::addDesktopButton()
{
    _containerArea->addDesktopButton();
}

void PanelExtension::addWindowListButton()
{
    _containerArea->addWindowListButton();
}

void PanelExtension::addURLButton(const QString &url)
{
    _containerArea->addURLButton(url);
}

void PanelExtension::addBrowserButton(const QString &startDir)
{
    _containerArea->addBrowserButton(startDir);
}

void PanelExtension::addServiceButton(const QString& desktopEntry)
{
    _containerArea->addServiceButton(desktopEntry);
}

void PanelExtension::addServiceMenuButton(const QString &, const QString& relPath)
{
    _containerArea->addServiceMenuButton(relPath);
}

void PanelExtension::addNonKDEAppButton(const QString &filePath, const QString &icon, const QString &cmdLine, bool inTerm)
{
    _containerArea->addNonKDEAppButton(filePath, filePath, icon, cmdLine, inTerm);
}

void PanelExtension::addNonKDEAppButton(const QString &title, const QString &filePath, const QString &icon, const QString &cmdLine, bool inTerm)
{
    _containerArea->addNonKDEAppButton(title, filePath, icon, cmdLine, inTerm);
}

void PanelExtension::addApplet(const QString &desktopFile)
{
    _containerArea->addApplet(desktopFile);
}

void PanelExtension::addAppletContainer(const QString &desktopFile)
{
    // KDE4: this appears in the DCOP interface.
    // but it's such a bad name, can this go away?
    addApplet(desktopFile);
}

void PanelExtension::restart()
{
    Kicker::the()->restart();
}

void PanelExtension::configure()
{
    Kicker::the()->configure();
}

void PanelExtension::slotBuildOpMenu()
{
    if (_built || !_opMnu)
        return;

    _built = true;

    if (!Kicker::the()->isImmutable())
    {
        // TODO: should this really be in ContainerArea???
        // setup addmenu and removemenu
        if (_containerArea->canAddContainers())
        {
            AddContainerMenu* addMnu = new AddContainerMenu(_containerArea,
                                                            true, this);
            _opMnu->insertItem(i18n("&Add to Panel"), addMnu);

            RemoveContainerMenu* removeMnu = new RemoveContainerMenu(_containerArea,
                                                                     true, this);
            _opMnu->insertItem(i18n("&Remove From Panel"), removeMnu);
        }

        _opMnu->insertItem(SmallIconSet("configure"),
                           i18n("&Configure Panel..."),
                           this, SLOT(showConfig()));

        if (_containerArea->canAddContainers())
        {
            _opMnu->insertSeparator();
        }
    }

    KHelpMenu* help = new KHelpMenu( this, KGlobal::instance()->aboutData(), false);
    _opMnu->insertItem(SmallIconSet("help"), KStdGuiItem::help().text(), help->menu());
    _opMnu->adjustSize();
}

void PanelExtension::showConfig()
{
    Kicker::the()->showConfig(_configFile);
}

MenubarExtension::MenubarExtension(const AppletInfo& info)
    : PanelExtension(info.configFile()),
      m_menubar(0)
{
}

MenubarExtension::~MenubarExtension()
{
    if (m_menubar)
    {
        m_menubar->setImmutable(false);
        _containerArea->slotSaveContainerConfig();
    }
}

void MenubarExtension::populateContainerArea()
{
    PanelExtension::populateContainerArea();
    ContainerList containers = _containerArea->containers("All");
    for (ContainerIterator it(containers); it.current() ; ++it)
    {
        if (it.current()->appletType() == "Applet")
        {
            AppletContainer* applet = dynamic_cast<AppletContainer*>(it.current());
            if (applet->info().desktopFile() == "menuapplet.desktop")
            {
                m_menubar = applet;
                break;
            }
        }
    }

    if (!m_menubar)
    {
        m_menubar = _containerArea->addApplet("menuapplet.desktop");
    }

    m_menubar->setImmutable(true);
}

