/*
 *  Copyright (C) 2002-2005  The DOSBox Team
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "dosbox.h"
#include "mem.h"
#include "inout.h"
#include "int10.h"


static Bit8u static_functionality[0x10]=
{
 /* 0 */ 0xff,  // All modes supported #1
 /* 1 */ 0xff,  // All modes supported #2
 /* 2 */ 0x0f,  // All modes supported #3
 /* 3 */ 0x00, 0x00, 0x00, 0x00,  // reserved
 /* 7 */ 0x07,  // 200, 350, 400 scan lines
 /* 8 */ 0x04,  // total number of character blocks available in text modes
 /* 9 */ 0x02,  // maximum number of active character blocks in text modes
 /* a */ 0xff,  // Misc Flags Everthing supported 
 /* b */ 0x0c,  // Support for Display combination and intensity/blinking
 /* c */ 0x00,  // reserved
 /* d */ 0x00,  // reserved
 /* e */ 0x00,  // Change to add new functions
 /* f */ 0x00   // reserved
};

static Bit16u map_offset[8]={
	0x0000,0x4000,0x8000,0xc000,
	0x2000,0x6000,0xa000,0xe000
};

void INT10_LoadFont(PhysPt font,bool reload,Bitu count,Bitu offset,Bitu map,Bitu height) {
	PhysPt where=PhysMake(0xa000,map_offset[map & 0x7]+offset*32);
	IO_Write(0x3c4,0x2);IO_Write(0x3c5,0x4);	//Enable plane 2
	IO_Write(0x3ce,0x6);Bitu old_6=IO_Read(0x3cf);
	IO_Write(0x3cf,0x0);	//Disable odd/even and a0000 adressing
	for (Bitu i=0;i<count;i++) {
		MEM_BlockCopy(where,font,height);
		where+=32;
		font+=height;
	}
	IO_Write(0x3c4,0x2);
	IO_Write(0x3c5,0xf);	//Enable all planes
	IO_Write(0x3ce,0x6);
	IO_Write(0x3cf,old_6);	//odd/even and b8000 adressing
	/* Reload tables and registers with new values based on this height */
	if (reload) {
		//Max scanline 
		Bit16u base=real_readw(BIOSMEM_SEG,BIOSMEM_CRTC_ADDRESS);
		IO_Write(base,0x9);
		IO_Write(base+1,(IO_Read(base+1) & 0xe0)|(height-1));
		//Vertical display end bios says, but should stay the same?
		//Rows setting in bios segment
		real_writeb(BIOSMEM_SEG,BIOSMEM_NB_ROWS,(CurMode->sheight/height)-1);
		real_writeb(BIOSMEM_SEG,BIOSMEM_CHAR_HEIGHT,height);
		//TODO Reprogram cursor size?
	}
}

void INT10_SetupRomMemory(void) {
/* This should fill up certain structures inside the Video Bios Rom Area */
	PhysPt rom_base=PhysMake(0xc000,0);
	Bitu i;
	int10.rom.used=3;
	if (machine==MCH_VGA) {
		// set up the start of the ROM
		phys_writew(rom_base+0,0xaa55);
		phys_writeb(rom_base+2,0x40);		// Size of ROM: 64 512-blocks = 32KB
		phys_writeb(rom_base+0x1e,0x49);	// IBM string
		phys_writeb(rom_base+0x1f,0x42);
		phys_writeb(rom_base+0x20,0x4d);
		phys_writeb(rom_base+0x21,0x00);
		int10.rom.used=0x100;
	}
	int10.rom.font_8_first=RealMake(0xC000,int10.rom.used);
	for (i=0;i<128*8;i++) {
		phys_writeb(rom_base+int10.rom.used++,int10_font_08[i]);
	}
	int10.rom.font_8_second=RealMake(0xC000,int10.rom.used);
	for (i=0;i<128*8;i++) {
		phys_writeb(rom_base+int10.rom.used++,int10_font_08[i+128*8]);
	}
	int10.rom.font_14=RealMake(0xC000,int10.rom.used);
	for (i=0;i<256*14;i++) {
		phys_writeb(rom_base+int10.rom.used++,int10_font_14[i]);
	}
	int10.rom.font_16=RealMake(0xC000,int10.rom.used);
	for (i=0;i<256*16;i++) {
		phys_writeb(rom_base+int10.rom.used++,int10_font_16[i]);
	}
	int10.rom.static_state=RealMake(0xC000,int10.rom.used);
	for (i=0;i<0x10;i++) {
		phys_writeb(rom_base+int10.rom.used++,static_functionality[i]);
	}
	for (i=0;i<128*8;i++) {
		phys_writeb(PhysMake(0xf000,0xfa6e)+i,int10_font_08[i]);
	}
	RealSetVec(0x1F,int10.rom.font_8_second);

	if (machine == MCH_VGA) { //EGA/VGA. Just to be safe
		/* Sum of all bytes in rom module 256 should be 0 */
		Bit8u sum = 0;
		for (i = 0;i < 32 * 1024;i++) //32 KB romsize
			sum += phys_readb(rom_base + i); //OVERFLOW IS OKAY
		sum = 256 - sum;
		phys_writeb(rom_base+int10.rom.used++,sum);
	}
};




Bit8u int10_font_08[256 * 8] = {
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x7e, 0x81, 0xa5, 0x81, 0xbd, 0x99, 0x81, 0x7e,
  0x7e, 0xff, 0xdb, 0xff, 0xc3, 0xe7, 0xff, 0x7e,
  0x6c, 0xfe, 0xfe, 0xfe, 0x7c, 0x38, 0x10, 0x00,
  0x10, 0x38, 0x7c, 0xfe, 0x7c, 0x38, 0x10, 0x00,
  0x38, 0x7c, 0x38, 0xfe, 0xfe, 0x7c, 0x38, 0x7c,
  0x10, 0x10, 0x38, 0x7c, 0xfe, 0x7c, 0x38, 0x7c,
  0x00, 0x00, 0x18, 0x3c, 0x3c, 0x18, 0x00, 0x00,
  0xff, 0xff, 0xe7, 0xc3, 0xc3, 0xe7, 0xff, 0xff,
  0x00, 0x3c, 0x66, 0x42, 0x42, 0x66, 0x3c, 0x00,
  0xff, 0xc3, 0x99, 0xbd, 0xbd, 0x99, 0xc3, 0xff,
  0x0f, 0x07, 0x0f, 0x7d, 0xcc, 0xcc, 0xcc, 0x78,
  0x3c, 0x66, 0x66, 0x66, 0x3c, 0x18, 0x7e, 0x18,
  0x3f, 0x33, 0x3f, 0x30, 0x30, 0x70, 0xf0, 0xe0,
  0x7f, 0x63, 0x7f, 0x63, 0x63, 0x67, 0xe6, 0xc0,
  0x99, 0x5a, 0x3c, 0xe7, 0xe7, 0x3c, 0x5a, 0x99,
  0x80, 0xe0, 0xf8, 0xfe, 0xf8, 0xe0, 0x80, 0x00,
  0x02, 0x0e, 0x3e, 0xfe, 0x3e, 0x0e, 0x02, 0x00,
  0x18, 0x3c, 0x7e, 0x18, 0x18, 0x7e, 0x3c, 0x18,
  0x66, 0x66, 0x66, 0x66, 0x66, 0x00, 0x66, 0x00,
  0x7f, 0xdb, 0xdb, 0x7b, 0x1b, 0x1b, 0x1b, 0x00,
  0x3e, 0x63, 0x38, 0x6c, 0x6c, 0x38, 0xcc, 0x78,
  0x00, 0x00, 0x00, 0x00, 0x7e, 0x7e, 0x7e, 0x00,
  0x18, 0x3c, 0x7e, 0x18, 0x7e, 0x3c, 0x18, 0xff,
  0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18, 0x18, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x7e, 0x3c, 0x18, 0x00,
  0x00, 0x18, 0x0c, 0xfe, 0x0c, 0x18, 0x00, 0x00,
  0x00, 0x30, 0x60, 0xfe, 0x60, 0x30, 0x00, 0x00,
  0x00, 0x00, 0xc0, 0xc0, 0xc0, 0xfe, 0x00, 0x00,
  0x00, 0x24, 0x66, 0xff, 0x66, 0x24, 0x00, 0x00,
  0x00, 0x18, 0x3c, 0x7e, 0xff, 0xff, 0x00, 0x00,
  0x00, 0xff, 0xff, 0x7e, 0x3c, 0x18, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x30, 0x78, 0x78, 0x30, 0x30, 0x00, 0x30, 0x00,
  0x6c, 0x6c, 0x6c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x6c, 0x6c, 0xfe, 0x6c, 0xfe, 0x6c, 0x6c, 0x00,
  0x30, 0x7c, 0xc0, 0x78, 0x0c, 0xf8, 0x30, 0x00,
  0x00, 0xc6, 0xcc, 0x18, 0x30, 0x66, 0xc6, 0x00,
  0x38, 0x6c, 0x38, 0x76, 0xdc, 0xcc, 0x76, 0x00,
  0x60, 0x60, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x30, 0x60, 0x60, 0x60, 0x30, 0x18, 0x00,
  0x60, 0x30, 0x18, 0x18, 0x18, 0x30, 0x60, 0x00,
  0x00, 0x66, 0x3c, 0xff, 0x3c, 0x66, 0x00, 0x00,
  0x00, 0x30, 0x30, 0xfc, 0x30, 0x30, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x60,
  0x00, 0x00, 0x00, 0xfc, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x00,
  0x06, 0x0c, 0x18, 0x30, 0x60, 0xc0, 0x80, 0x00,
  0x7c, 0xc6, 0xce, 0xde, 0xf6, 0xe6, 0x7c, 0x00,
  0x30, 0x70, 0x30, 0x30, 0x30, 0x30, 0xfc, 0x00,
  0x78, 0xcc, 0x0c, 0x38, 0x60, 0xcc, 0xfc, 0x00,
  0x78, 0xcc, 0x0c, 0x38, 0x0c, 0xcc, 0x78, 0x00,
  0x1c, 0x3c, 0x6c, 0xcc, 0xfe, 0x0c, 0x1e, 0x00,
  0xfc, 0xc0, 0xf8, 0x0c, 0x0c, 0xcc, 0x78, 0x00,
  0x38, 0x60, 0xc0, 0xf8, 0xcc, 0xcc, 0x78, 0x00,
  0xfc, 0xcc, 0x0c, 0x18, 0x30, 0x30, 0x30, 0x00,
  0x78, 0xcc, 0xcc, 0x78, 0xcc, 0xcc, 0x78, 0x00,
  0x78, 0xcc, 0xcc, 0x7c, 0x0c, 0x18, 0x70, 0x00,
  0x00, 0x30, 0x30, 0x00, 0x00, 0x30, 0x30, 0x00,
  0x00, 0x30, 0x30, 0x00, 0x00, 0x30, 0x30, 0x60,
  0x18, 0x30, 0x60, 0xc0, 0x60, 0x30, 0x18, 0x00,
  0x00, 0x00, 0xfc, 0x00, 0x00, 0xfc, 0x00, 0x00,
  0x60, 0x30, 0x18, 0x0c, 0x18, 0x30, 0x60, 0x00,
  0x78, 0xcc, 0x0c, 0x18, 0x30, 0x00, 0x30, 0x00,
  0x7c, 0xc6, 0xde, 0xde, 0xde, 0xc0, 0x78, 0x00,
  0x30, 0x78, 0xcc, 0xcc, 0xfc, 0xcc, 0xcc, 0x00,
  0xfc, 0x66, 0x66, 0x7c, 0x66, 0x66, 0xfc, 0x00,
  0x3c, 0x66, 0xc0, 0xc0, 0xc0, 0x66, 0x3c, 0x00,
  0xf8, 0x6c, 0x66, 0x66, 0x66, 0x6c, 0xf8, 0x00,
  0xfe, 0x62, 0x68, 0x78, 0x68, 0x62, 0xfe, 0x00,
  0xfe, 0x62, 0x68, 0x78, 0x68, 0x60, 0xf0, 0x00,
  0x3c, 0x66, 0xc0, 0xc0, 0xce, 0x66, 0x3e, 0x00,
  0xcc, 0xcc, 0xcc, 0xfc, 0xcc, 0xcc, 0xcc, 0x00,
  0x78, 0x30, 0x30, 0x30, 0x30, 0x30, 0x78, 0x00,
  0x1e, 0x0c, 0x0c, 0x0c, 0xcc, 0xcc, 0x78, 0x00,
  0xe6, 0x66, 0x6c, 0x78, 0x6c, 0x66, 0xe6, 0x00,
  0xf0, 0x60, 0x60, 0x60, 0x62, 0x66, 0xfe, 0x00,
  0xc6, 0xee, 0xfe, 0xfe, 0xd6, 0xc6, 0xc6, 0x00,
  0xc6, 0xe6, 0xf6, 0xde, 0xce, 0xc6, 0xc6, 0x00,
  0x38, 0x6c, 0xc6, 0xc6, 0xc6, 0x6c, 0x38, 0x00,
  0xfc, 0x66, 0x66, 0x7c, 0x60, 0x60, 0xf0, 0x00,
  0x78, 0xcc, 0xcc, 0xcc, 0xdc, 0x78, 0x1c, 0x00,
  0xfc, 0x66, 0x66, 0x7c, 0x6c, 0x66, 0xe6, 0x00,
  0x78, 0xcc, 0xe0, 0x70, 0x1c, 0xcc, 0x78, 0x00,
  0xfc, 0xb4, 0x30, 0x30, 0x30, 0x30, 0x78, 0x00,
  0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xfc, 0x00,
  0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x78, 0x30, 0x00,
  0xc6, 0xc6, 0xc6, 0xd6, 0xfe, 0xee, 0xc6, 0x00,
  0xc6, 0xc6, 0x6c, 0x38, 0x38, 0x6c, 0xc6, 0x00,
  0xcc, 0xcc, 0xcc, 0x78, 0x30, 0x30, 0x78, 0x00,
  0xfe, 0xc6, 0x8c, 0x18, 0x32, 0x66, 0xfe, 0x00,
  0x78, 0x60, 0x60, 0x60, 0x60, 0x60, 0x78, 0x00,
  0xc0, 0x60, 0x30, 0x18, 0x0c, 0x06, 0x02, 0x00,
  0x78, 0x18, 0x18, 0x18, 0x18, 0x18, 0x78, 0x00,
  0x10, 0x38, 0x6c, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0x30, 0x30, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0x76, 0x00,
  0xe0, 0x60, 0x60, 0x7c, 0x66, 0x66, 0xdc, 0x00,
  0x00, 0x00, 0x78, 0xcc, 0xc0, 0xcc, 0x78, 0x00,
  0x1c, 0x0c, 0x0c, 0x7c, 0xcc, 0xcc, 0x76, 0x00,
  0x00, 0x00, 0x78, 0xcc, 0xfc, 0xc0, 0x78, 0x00,
  0x38, 0x6c, 0x60, 0xf0, 0x60, 0x60, 0xf0, 0x00,
  0x00, 0x00, 0x76, 0xcc, 0xcc, 0x7c, 0x0c, 0xf8,
  0xe0, 0x60, 0x6c, 0x76, 0x66, 0x66, 0xe6, 0x00,
  0x30, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00,
  0x0c, 0x00, 0x0c, 0x0c, 0x0c, 0xcc, 0xcc, 0x78,
  0xe0, 0x60, 0x66, 0x6c, 0x78, 0x6c, 0xe6, 0x00,
  0x70, 0x30, 0x30, 0x30, 0x30, 0x30, 0x78, 0x00,
  0x00, 0x00, 0xcc, 0xfe, 0xfe, 0xd6, 0xc6, 0x00,
  0x00, 0x00, 0xf8, 0xcc, 0xcc, 0xcc, 0xcc, 0x00,
  0x00, 0x00, 0x78, 0xcc, 0xcc, 0xcc, 0x78, 0x00,
  0x00, 0x00, 0xdc, 0x66, 0x66, 0x7c, 0x60, 0xf0,
  0x00, 0x00, 0x76, 0xcc, 0xcc, 0x7c, 0x0c, 0x1e,
  0x00, 0x00, 0xdc, 0x76, 0x66, 0x60, 0xf0, 0x00,
  0x00, 0x00, 0x7c, 0xc0, 0x78, 0x0c, 0xf8, 0x00,
  0x10, 0x30, 0x7c, 0x30, 0x30, 0x34, 0x18, 0x00,
  0x00, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0x76, 0x00,
  0x00, 0x00, 0xcc, 0xcc, 0xcc, 0x78, 0x30, 0x00,
  0x00, 0x00, 0xc6, 0xd6, 0xfe, 0xfe, 0x6c, 0x00,
  0x00, 0x00, 0xc6, 0x6c, 0x38, 0x6c, 0xc6, 0x00,
  0x00, 0x00, 0xcc, 0xcc, 0xcc, 0x7c, 0x0c, 0xf8,
  0x00, 0x00, 0xfc, 0x98, 0x30, 0x64, 0xfc, 0x00,
  0x1c, 0x30, 0x30, 0xe0, 0x30, 0x30, 0x1c, 0x00,
  0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x00,
  0xe0, 0x30, 0x30, 0x1c, 0x30, 0x30, 0xe0, 0x00,
  0x76, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x10, 0x38, 0x6c, 0xc6, 0xc6, 0xfe, 0x00,
  0x78, 0xcc, 0xc0, 0xcc, 0x78, 0x18, 0x0c, 0x78,
  0x00, 0xcc, 0x00, 0xcc, 0xcc, 0xcc, 0x7e, 0x00,
  0x1c, 0x00, 0x78, 0xcc, 0xfc, 0xc0, 0x78, 0x00,
  0x7e, 0xc3, 0x3c, 0x06, 0x3e, 0x66, 0x3f, 0x00,
  0xcc, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0x7e, 0x00,
  0xe0, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0x7e, 0x00,
  0x30, 0x30, 0x78, 0x0c, 0x7c, 0xcc, 0x7e, 0x00,
  0x00, 0x00, 0x78, 0xc0, 0xc0, 0x78, 0x0c, 0x38,
  0x7e, 0xc3, 0x3c, 0x66, 0x7e, 0x60, 0x3c, 0x00,
  0xcc, 0x00, 0x78, 0xcc, 0xfc, 0xc0, 0x78, 0x00,
  0xe0, 0x00, 0x78, 0xcc, 0xfc, 0xc0, 0x78, 0x00,
  0xcc, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00,
  0x7c, 0xc6, 0x38, 0x18, 0x18, 0x18, 0x3c, 0x00,
  0xe0, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00,
  0xc6, 0x38, 0x6c, 0xc6, 0xfe, 0xc6, 0xc6, 0x00,
  0x30, 0x30, 0x00, 0x78, 0xcc, 0xfc, 0xcc, 0x00,
  0x1c, 0x00, 0xfc, 0x60, 0x78, 0x60, 0xfc, 0x00,
  0x00, 0x00, 0x7f, 0x0c, 0x7f, 0xcc, 0x7f, 0x00,
  0x3e, 0x6c, 0xcc, 0xfe, 0xcc, 0xcc, 0xce, 0x00,
  0x78, 0xcc, 0x00, 0x78, 0xcc, 0xcc, 0x78, 0x00,
  0x00, 0xcc, 0x00, 0x78, 0xcc, 0xcc, 0x78, 0x00,
  0x00, 0xe0, 0x00, 0x78, 0xcc, 0xcc, 0x78, 0x00,
  0x78, 0xcc, 0x00, 0xcc, 0xcc, 0xcc, 0x7e, 0x00,
  0x00, 0xe0, 0x00, 0xcc, 0xcc, 0xcc, 0x7e, 0x00,
  0x00, 0xcc, 0x00, 0xcc, 0xcc, 0x7c, 0x0c, 0xf8,
  0xc3, 0x18, 0x3c, 0x66, 0x66, 0x3c, 0x18, 0x00,
  0xcc, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0x78, 0x00,
  0x18, 0x18, 0x7e, 0xc0, 0xc0, 0x7e, 0x18, 0x18,
  0x38, 0x6c, 0x64, 0xf0, 0x60, 0xe6, 0xfc, 0x00,
  0xcc, 0xcc, 0x78, 0xfc, 0x30, 0xfc, 0x30, 0x30,
  0xf8, 0xcc, 0xcc, 0xfa, 0xc6, 0xcf, 0xc6, 0xc7,
  0x0e, 0x1b, 0x18, 0x3c, 0x18, 0x18, 0xd8, 0x70,
  0x1c, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0x7e, 0x00,
  0x38, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00,
  0x00, 0x1c, 0x00, 0x78, 0xcc, 0xcc, 0x78, 0x00,
  0x00, 0x1c, 0x00, 0xcc, 0xcc, 0xcc, 0x7e, 0x00,
  0x00, 0xf8, 0x00, 0xf8, 0xcc, 0xcc, 0xcc, 0x00,
  0xfc, 0x00, 0xcc, 0xec, 0xfc, 0xdc, 0xcc, 0x00,
  0x3c, 0x6c, 0x6c, 0x3e, 0x00, 0x7e, 0x00, 0x00,
  0x38, 0x6c, 0x6c, 0x38, 0x00, 0x7c, 0x00, 0x00,
  0x30, 0x00, 0x30, 0x60, 0xc0, 0xcc, 0x78, 0x00,
  0x00, 0x00, 0x00, 0xfc, 0xc0, 0xc0, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xfc, 0x0c, 0x0c, 0x00, 0x00,
  0xc3, 0xc6, 0xcc, 0xde, 0x33, 0x66, 0xcc, 0x0f,
  0xc3, 0xc6, 0xcc, 0xdb, 0x37, 0x6f, 0xcf, 0x03,
  0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00,
  0x00, 0x33, 0x66, 0xcc, 0x66, 0x33, 0x00, 0x00,
  0x00, 0xcc, 0x66, 0x33, 0x66, 0xcc, 0x00, 0x00,
  0x22, 0x88, 0x22, 0x88, 0x22, 0x88, 0x22, 0x88,
  0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa,
  0xdb, 0x77, 0xdb, 0xee, 0xdb, 0x77, 0xdb, 0xee,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 0x18, 0x18,
  0x18, 0x18, 0xf8, 0x18, 0xf8, 0x18, 0x18, 0x18,
  0x36, 0x36, 0x36, 0x36, 0xf6, 0x36, 0x36, 0x36,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0x36, 0x36, 0x36,
  0x00, 0x00, 0xf8, 0x18, 0xf8, 0x18, 0x18, 0x18,
  0x36, 0x36, 0xf6, 0x06, 0xf6, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x00, 0x00, 0xfe, 0x06, 0xf6, 0x36, 0x36, 0x36,
  0x36, 0x36, 0xf6, 0x06, 0xfe, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0xfe, 0x00, 0x00, 0x00,
  0x18, 0x18, 0xf8, 0x18, 0xf8, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xf8, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x1f, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0xff, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xff, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0xff, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x1f, 0x18, 0x1f, 0x18, 0x18, 0x18,
  0x36, 0x36, 0x36, 0x36, 0x37, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x37, 0x30, 0x3f, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3f, 0x30, 0x37, 0x36, 0x36, 0x36,
  0x36, 0x36, 0xf7, 0x00, 0xff, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xff, 0x00, 0xf7, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x37, 0x30, 0x37, 0x36, 0x36, 0x36,
  0x00, 0x00, 0xff, 0x00, 0xff, 0x00, 0x00, 0x00,
  0x36, 0x36, 0xf7, 0x00, 0xf7, 0x36, 0x36, 0x36,
  0x18, 0x18, 0xff, 0x00, 0xff, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0xff, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xff, 0x00, 0xff, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 0xff, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x3f, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x1f, 0x18, 0x1f, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1f, 0x18, 0x1f, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 0x3f, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0xff, 0x36, 0x36, 0x36,
  0x18, 0x18, 0xff, 0x18, 0xff, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0xf8, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x1f, 0x18, 0x18, 0x18,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
  0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
  0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
  0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x76, 0xdc, 0xc8, 0xdc, 0x76, 0x00,
  0x00, 0x78, 0xcc, 0xf8, 0xcc, 0xf8, 0xc0, 0xc0,
  0x00, 0xfc, 0xcc, 0xc0, 0xc0, 0xc0, 0xc0, 0x00,
  0x00, 0xfe, 0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x00,
  0xfc, 0xcc, 0x60, 0x30, 0x60, 0xcc, 0xfc, 0x00,
  0x00, 0x00, 0x7e, 0xd8, 0xd8, 0xd8, 0x70, 0x00,
  0x00, 0x66, 0x66, 0x66, 0x66, 0x7c, 0x60, 0xc0,
  0x00, 0x76, 0xdc, 0x18, 0x18, 0x18, 0x18, 0x00,
  0xfc, 0x30, 0x78, 0xcc, 0xcc, 0x78, 0x30, 0xfc,
  0x38, 0x6c, 0xc6, 0xfe, 0xc6, 0x6c, 0x38, 0x00,
  0x38, 0x6c, 0xc6, 0xc6, 0x6c, 0x6c, 0xee, 0x00,
  0x1c, 0x30, 0x18, 0x7c, 0xcc, 0xcc, 0x78, 0x00,
  0x00, 0x00, 0x7e, 0xdb, 0xdb, 0x7e, 0x00, 0x00,
  0x06, 0x0c, 0x7e, 0xdb, 0xdb, 0x7e, 0x60, 0xc0,
  0x38, 0x60, 0xc0, 0xf8, 0xc0, 0x60, 0x38, 0x00,
  0x78, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x00,
  0x00, 0xfc, 0x00, 0xfc, 0x00, 0xfc, 0x00, 0x00,
  0x30, 0x30, 0xfc, 0x30, 0x30, 0x00, 0xfc, 0x00,
  0x60, 0x30, 0x18, 0x30, 0x60, 0x00, 0xfc, 0x00,
  0x18, 0x30, 0x60, 0x30, 0x18, 0x00, 0xfc, 0x00,
  0x0e, 0x1b, 0x1b, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xd8, 0xd8, 0x70,
  0x30, 0x30, 0x00, 0xfc, 0x00, 0x30, 0x30, 0x00,
  0x00, 0x76, 0xdc, 0x00, 0x76, 0xdc, 0x00, 0x00,
  0x38, 0x6c, 0x6c, 0x38, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00,
  0x0f, 0x0c, 0x0c, 0x0c, 0xec, 0x6c, 0x3c, 0x1c,
  0x78, 0x6c, 0x6c, 0x6c, 0x6c, 0x00, 0x00, 0x00,
  0x70, 0x18, 0x30, 0x60, 0x78, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x3c, 0x3c, 0x3c, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

Bit8u int10_font_14[256 * 14] = {
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x7e, 0x81, 0xa5, 0x81, 0x81, 0xbd, 0x99, 0x81,
  0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0xff,
  0xdb, 0xff, 0xff, 0xc3, 0xe7, 0xff, 0x7e, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x6c, 0xfe, 0xfe,
  0xfe, 0xfe, 0x7c, 0x38, 0x10, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x10, 0x38, 0x7c, 0xfe, 0x7c,
  0x38, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x3c, 0x3c, 0xe7, 0xe7, 0xe7, 0x18, 0x18,
  0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x3c,
  0x7e, 0xff, 0xff, 0x7e, 0x18, 0x18, 0x3c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18,
  0x3c, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xe7, 0xc3, 0xc3,
  0xe7, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0x42, 0x42, 0x66, 0x3c,
  0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
  0xc3, 0x99, 0xbd, 0xbd, 0x99, 0xc3, 0xff, 0xff,
  0xff, 0xff, 0x00, 0x00, 0x1e, 0x0e, 0x1a, 0x32,
  0x78, 0xcc, 0xcc, 0xcc, 0x78, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0x66, 0x66, 0x3c, 0x18,
  0x7e, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x3f, 0x33, 0x3f, 0x30, 0x30, 0x30, 0x70, 0xf0,
  0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0x63,
  0x7f, 0x63, 0x63, 0x63, 0x67, 0xe7, 0xe6, 0xc0,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0xdb, 0x3c,
  0xe7, 0x3c, 0xdb, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x80, 0xc0, 0xe0, 0xf8, 0xfe, 0xf8,
  0xe0, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x02, 0x06, 0x0e, 0x3e, 0xfe, 0x3e, 0x0e, 0x06,
  0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x3c,
  0x7e, 0x18, 0x18, 0x18, 0x7e, 0x3c, 0x18, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x66, 0x66,
  0x66, 0x66, 0x00, 0x66, 0x66, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7f, 0xdb, 0xdb, 0xdb, 0x7b, 0x1b,
  0x1b, 0x1b, 0x1b, 0x00, 0x00, 0x00, 0x00, 0x7c,
  0xc6, 0x60, 0x38, 0x6c, 0xc6, 0xc6, 0x6c, 0x38,
  0x0c, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0xfe, 0xfe, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18,
  0x18, 0x18, 0x7e, 0x3c, 0x18, 0x7e, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x7e, 0x3c,
  0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x0c, 0xfe, 0x0c, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x60,
  0xfe, 0x60, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0xc0, 0xc0,
  0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x28, 0x6c, 0xfe, 0x6c, 0x28, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10,
  0x38, 0x38, 0x7c, 0x7c, 0xfe, 0xfe, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xfe, 0x7c,
  0x7c, 0x38, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x3c, 0x3c, 0x3c, 0x18, 0x18, 0x00, 0x18,
  0x18, 0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x66,
  0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x6c, 0x6c, 0xfe, 0x6c,
  0x6c, 0x6c, 0xfe, 0x6c, 0x6c, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x7c, 0xc6, 0xc2, 0xc0, 0x7c, 0x06,
  0x86, 0xc6, 0x7c, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc2, 0xc6, 0x0c, 0x18, 0x30, 0x66,
  0xc6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x6c,
  0x6c, 0x38, 0x76, 0xdc, 0xcc, 0xcc, 0x76, 0x00,
  0x00, 0x00, 0x00, 0x30, 0x30, 0x30, 0x60, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0c, 0x18, 0x30, 0x30, 0x30, 0x30,
  0x30, 0x18, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x30, 0x18, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x18,
  0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x66, 0x3c, 0xff, 0x3c, 0x66, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18,
  0x7e, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x30, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x02, 0x06, 0x0c, 0x18,
  0x30, 0x60, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xce, 0xde, 0xf6, 0xe6,
  0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x38, 0x78, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6,
  0x06, 0x0c, 0x18, 0x30, 0x60, 0xc6, 0xfe, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0x06, 0x06,
  0x3c, 0x06, 0x06, 0xc6, 0x7c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0c, 0x1c, 0x3c, 0x6c, 0xcc, 0xfe,
  0x0c, 0x0c, 0x1e, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xfe, 0xc0, 0xc0, 0xc0, 0xfc, 0x06, 0x06, 0xc6,
  0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x60,
  0xc0, 0xc0, 0xfc, 0xc6, 0xc6, 0xc6, 0x7c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0xc6, 0x06, 0x0c,
  0x18, 0x30, 0x30, 0x30, 0x30, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0x7c, 0xc6,
  0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x7c, 0xc6, 0xc6, 0xc6, 0x7e, 0x06, 0x06, 0x0c,
  0x78, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18,
  0x18, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x30, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x06, 0x0c, 0x18, 0x30, 0x60, 0x30,
  0x18, 0x0c, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x7e, 0x00, 0x00, 0x7e, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x30,
  0x18, 0x0c, 0x06, 0x0c, 0x18, 0x30, 0x60, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0x0c,
  0x18, 0x18, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xde, 0xde, 0xde,
  0xdc, 0xc0, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x10, 0x38, 0x6c, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6,
  0xc6, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfc, 0x66,
  0x66, 0x66, 0x7c, 0x66, 0x66, 0x66, 0xfc, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0,
  0xc0, 0xc0, 0xc2, 0x66, 0x3c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xf8, 0x6c, 0x66, 0x66, 0x66, 0x66,
  0x66, 0x6c, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xfe, 0x66, 0x62, 0x68, 0x78, 0x68, 0x62, 0x66,
  0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x66,
  0x62, 0x68, 0x78, 0x68, 0x60, 0x60, 0xf0, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0,
  0xc0, 0xde, 0xc6, 0x66, 0x3a, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xfe, 0xc6,
  0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x3c, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x0c,
  0x0c, 0x0c, 0x0c, 0x0c, 0xcc, 0xcc, 0x78, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xe6, 0x66, 0x6c, 0x6c,
  0x78, 0x6c, 0x6c, 0x66, 0xe6, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xf0, 0x60, 0x60, 0x60, 0x60, 0x60,
  0x62, 0x66, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xc6, 0xee, 0xfe, 0xfe, 0xd6, 0xc6, 0xc6, 0xc6,
  0xc6, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xe6,
  0xf6, 0xfe, 0xde, 0xce, 0xc6, 0xc6, 0xc6, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x38, 0x6c, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x6c, 0x38, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfc, 0x66, 0x66, 0x66, 0x7c, 0x60,
  0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xd6, 0xde, 0x7c,
  0x0c, 0x0e, 0x00, 0x00, 0x00, 0x00, 0xfc, 0x66,
  0x66, 0x66, 0x7c, 0x6c, 0x66, 0x66, 0xe6, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0x60,
  0x38, 0x0c, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7e, 0x7e, 0x5a, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
  0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0xc6, 0x6c, 0x38, 0x10, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6,
  0xd6, 0xd6, 0xfe, 0x7c, 0x6c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0x6c, 0x38, 0x38, 0x38,
  0x6c, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x66, 0x66, 0x66, 0x66, 0x3c, 0x18, 0x18, 0x18,
  0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xc6,
  0x8c, 0x18, 0x30, 0x60, 0xc2, 0xc6, 0xfe, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x3c, 0x30, 0x30, 0x30,
  0x30, 0x30, 0x30, 0x30, 0x3c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x80, 0xc0, 0xe0, 0x70, 0x38, 0x1c,
  0x0e, 0x06, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x3c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
  0x3c, 0x00, 0x00, 0x00, 0x10, 0x38, 0x6c, 0xc6,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00,
  0x30, 0x30, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc,
  0x76, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe0, 0x60,
  0x60, 0x78, 0x6c, 0x66, 0x66, 0x66, 0x7c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c,
  0xc6, 0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1c, 0x0c, 0x0c, 0x3c, 0x6c, 0xcc,
  0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc6,
  0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x6c,
  0x64, 0x60, 0xf0, 0x60, 0x60, 0x60, 0xf0, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x76,
  0xcc, 0xcc, 0xcc, 0x7c, 0x0c, 0xcc, 0x78, 0x00,
  0x00, 0x00, 0xe0, 0x60, 0x60, 0x6c, 0x76, 0x66,
  0x66, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18,
  0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x06,
  0x00, 0x0e, 0x06, 0x06, 0x06, 0x06, 0x66, 0x66,
  0x3c, 0x00, 0x00, 0x00, 0xe0, 0x60, 0x60, 0x66,
  0x6c, 0x78, 0x6c, 0x66, 0xe6, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xec, 0xfe, 0xd6, 0xd6, 0xd6,
  0xc6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xdc, 0x66, 0x66, 0x66, 0x66, 0x66, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c,
  0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xdc, 0x66, 0x66,
  0x66, 0x7c, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x76, 0xcc, 0xcc, 0xcc, 0x7c,
  0x0c, 0x0c, 0x1e, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xdc, 0x76, 0x66, 0x60, 0x60, 0xf0, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c,
  0xc6, 0x70, 0x1c, 0xc6, 0x7c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x10, 0x30, 0x30, 0xfc, 0x30, 0x30,
  0x30, 0x36, 0x1c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc,
  0x76, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x66, 0x66, 0x66, 0x66, 0x3c, 0x18, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc6,
  0xc6, 0xd6, 0xd6, 0xfe, 0x6c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0x6c, 0x38,
  0x38, 0x6c, 0xc6, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0x7e,
  0x06, 0x0c, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xfe, 0xcc, 0x18, 0x30, 0x66, 0xfe, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x0e, 0x18, 0x18, 0x18,
  0x70, 0x18, 0x18, 0x18, 0x0e, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00, 0x18,
  0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x70, 0x18, 0x18, 0x18, 0x0e, 0x18, 0x18, 0x18,
  0x70, 0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x38,
  0x6c, 0xc6, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0, 0xc0, 0xc2,
  0x66, 0x3c, 0x0c, 0x06, 0x7c, 0x00, 0x00, 0x00,
  0xcc, 0xcc, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc,
  0x76, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x18, 0x30,
  0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc6, 0x7c, 0x00,
  0x00, 0x00, 0x00, 0x10, 0x38, 0x6c, 0x00, 0x78,
  0x0c, 0x7c, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xcc, 0xcc, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 0x60,
  0x30, 0x18, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc,
  0x76, 0x00, 0x00, 0x00, 0x00, 0x38, 0x6c, 0x38,
  0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc, 0x76, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x66,
  0x60, 0x66, 0x3c, 0x0c, 0x06, 0x3c, 0x00, 0x00,
  0x00, 0x10, 0x38, 0x6c, 0x00, 0x7c, 0xc6, 0xfe,
  0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xcc, 0xcc, 0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc6,
  0x7c, 0x00, 0x00, 0x00, 0x00, 0x60, 0x30, 0x18,
  0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc6, 0x7c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x00, 0x38,
  0x18, 0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x3c, 0x66, 0x00, 0x38, 0x18, 0x18,
  0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x60,
  0x30, 0x18, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18,
  0x3c, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6, 0x10,
  0x38, 0x6c, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0x00,
  0x00, 0x00, 0x38, 0x6c, 0x38, 0x00, 0x38, 0x6c,
  0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0x00, 0x00, 0x00,
  0x18, 0x30, 0x60, 0x00, 0xfe, 0x66, 0x60, 0x7c,
  0x60, 0x66, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xcc, 0x76, 0x36, 0x7e, 0xd8, 0xd8,
  0x6e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x6c,
  0xcc, 0xcc, 0xfe, 0xcc, 0xcc, 0xcc, 0xce, 0x00,
  0x00, 0x00, 0x00, 0x10, 0x38, 0x6c, 0x00, 0x7c,
  0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0x00, 0x7c, 0xc6, 0xc6,
  0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x60,
  0x30, 0x18, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6,
  0x7c, 0x00, 0x00, 0x00, 0x00, 0x30, 0x78, 0xcc,
  0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x76, 0x00,
  0x00, 0x00, 0x00, 0x60, 0x30, 0x18, 0x00, 0xcc,
  0xcc, 0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0x00, 0xc6, 0xc6, 0xc6,
  0xc6, 0x7e, 0x06, 0x0c, 0x78, 0x00, 0x00, 0xc6,
  0xc6, 0x38, 0x6c, 0xc6, 0xc6, 0xc6, 0xc6, 0x6c,
  0x38, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6, 0x00,
  0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x18, 0x3c, 0x66, 0x60,
  0x60, 0x66, 0x3c, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x38, 0x6c, 0x64, 0x60, 0xf0, 0x60, 0x60,
  0x60, 0xe6, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x66, 0x66, 0x3c, 0x18, 0x7e, 0x18, 0x7e, 0x18,
  0x18, 0x00, 0x00, 0x00, 0x00, 0xf8, 0xcc, 0xcc,
  0xf8, 0xc4, 0xcc, 0xde, 0xcc, 0xcc, 0xc6, 0x00,
  0x00, 0x00, 0x00, 0x0e, 0x1b, 0x18, 0x18, 0x18,
  0x7e, 0x18, 0x18, 0x18, 0x18, 0xd8, 0x70, 0x00,
  0x00, 0x18, 0x30, 0x60, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 0x0c,
  0x18, 0x30, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18,
  0x3c, 0x00, 0x00, 0x00, 0x00, 0x18, 0x30, 0x60,
  0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x30, 0x60, 0x00, 0xcc,
  0xcc, 0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x76, 0xdc, 0x00, 0xdc, 0x66, 0x66,
  0x66, 0x66, 0x66, 0x00, 0x00, 0x00, 0x76, 0xdc,
  0x00, 0xc6, 0xe6, 0xf6, 0xfe, 0xde, 0xce, 0xc6,
  0xc6, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x6c, 0x6c,
  0x3e, 0x00, 0x7e, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x38, 0x6c, 0x6c, 0x38, 0x00,
  0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x30, 0x30, 0x00, 0x30, 0x30, 0x60,
  0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0xc0, 0xc0, 0xc0,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0x06, 0x06, 0x06, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xc0, 0xc0, 0xc6, 0xcc, 0xd8,
  0x30, 0x60, 0xdc, 0x86, 0x0c, 0x18, 0x3e, 0x00,
  0x00, 0xc0, 0xc0, 0xc6, 0xcc, 0xd8, 0x30, 0x66,
  0xce, 0x9e, 0x3e, 0x06, 0x06, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x00, 0x18, 0x18, 0x3c, 0x3c, 0x3c,
  0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x6c, 0xd8, 0x6c, 0x36, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd8, 0x6c,
  0x36, 0x6c, 0xd8, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44,
  0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x55, 0xaa,
  0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa,
  0x55, 0xaa, 0x55, 0xaa, 0xdd, 0x77, 0xdd, 0x77,
  0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77,
  0xdd, 0x77, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xf8,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0xf8, 0x18, 0xf8, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0xf6, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xfe, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x18, 0xf8,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x36, 0x36,
  0x36, 0x36, 0x36, 0xf6, 0x06, 0xf6, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe,
  0x06, 0xf6, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0xf6, 0x06, 0xfe,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0xfe, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x18,
  0x18, 0xf8, 0x18, 0xf8, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xf8, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x1f,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xff, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x1f, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x1f, 0x18, 0x1f, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x37, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 0x30, 0x3f,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x3f, 0x30, 0x37, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0xf7, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0x00, 0xf7, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 0x30, 0x37,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xff, 0x00, 0xff, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x36, 0x36, 0x36, 0x36,
  0x36, 0xf7, 0x00, 0xf7, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x18, 0x18, 0x18, 0x18, 0x18, 0xff,
  0x00, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xff, 0x00, 0xff, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xff, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x3f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x1f,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x1f, 0x18, 0x1f, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x3f, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0xff, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x18, 0xff,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x1f, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0,
  0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
  0xf0, 0xf0, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0x0f,
  0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
  0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc, 0xd8,
  0xd8, 0xdc, 0x76, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xfc, 0xc6, 0xc6, 0xfc,
  0xc0, 0xc0, 0x40, 0x00, 0x00, 0x00, 0xfe, 0xc6,
  0xc6, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x6c,
  0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0xc6, 0x60, 0x30, 0x18, 0x30,
  0x60, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x7e, 0xd8, 0xd8, 0xd8, 0xd8,
  0x70, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x66, 0x66, 0x66, 0x66, 0x7c, 0x60, 0x60, 0xc0,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7e, 0x18, 0x3c, 0x66, 0x66, 0x66,
  0x3c, 0x18, 0x7e, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x38, 0x6c, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0x6c,
  0x38, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x6c,
  0xc6, 0xc6, 0xc6, 0x6c, 0x6c, 0x6c, 0xee, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x1e, 0x30, 0x18, 0x0c,
  0x3e, 0x66, 0x66, 0x66, 0x3c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0xdb, 0xdb,
  0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x03, 0x06, 0x7e, 0xdb, 0xdb, 0xf3, 0x7e, 0x60,
  0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1c, 0x30,
  0x60, 0x60, 0x7c, 0x60, 0x60, 0x30, 0x1c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xfe, 0x00, 0x00, 0xfe, 0x00,
  0x00, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x18, 0x7e, 0x18, 0x18, 0x00, 0x00,
  0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x18,
  0x0c, 0x06, 0x0c, 0x18, 0x30, 0x00, 0x7e, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x0c, 0x18, 0x30, 0x60,
  0x30, 0x18, 0x0c, 0x00, 0x7e, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0e, 0x1b, 0x1b, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xd8, 0xd8,
  0x70, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18,
  0x18, 0x00, 0x7e, 0x00, 0x18, 0x18, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc,
  0x00, 0x76, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x38, 0x6c, 0x6c, 0x38, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x0f, 0x0c, 0x0c, 0x0c, 0x0c,
  0x0c, 0xec, 0x6c, 0x3c, 0x1c, 0x00, 0x00, 0x00,
  0x00, 0xd8, 0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x70,
  0xd8, 0x30, 0x60, 0xc8, 0xf8, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x7c, 0x7c, 0x7c, 0x7c, 0x7c, 0x7c, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

Bit8u int10_font_16[256 * 16] = {
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7e, 0x81, 0xa5, 0x81, 0x81, 0xbd,
  0x99, 0x81, 0x81, 0x7e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7e, 0xff, 0xdb, 0xff, 0xff, 0xc3,
  0xe7, 0xff, 0xff, 0x7e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x6c, 0xfe, 0xfe, 0xfe,
  0xfe, 0x7c, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x7c, 0xfe,
  0x7c, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x3c, 0x3c, 0xe7, 0xe7,
  0xe7, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x3c, 0x7e, 0xff, 0xff,
  0x7e, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x3c,
  0x3c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe7, 0xc3,
  0xc3, 0xe7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 0x42,
  0x42, 0x66, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xc3, 0x99, 0xbd,
  0xbd, 0x99, 0xc3, 0xff, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x1e, 0x0e, 0x1a, 0x32, 0x78, 0xcc,
  0xcc, 0xcc, 0xcc, 0x78, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0x66, 0x66, 0x66, 0x3c,
  0x18, 0x7e, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3f, 0x33, 0x3f, 0x30, 0x30, 0x30,
  0x30, 0x70, 0xf0, 0xe0, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7f, 0x63, 0x7f, 0x63, 0x63, 0x63,
  0x63, 0x67, 0xe7, 0xe6, 0xc0, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x18, 0xdb, 0x3c, 0xe7,
  0x3c, 0xdb, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x80, 0xc0, 0xe0, 0xf0, 0xf8, 0xfe, 0xf8,
  0xf0, 0xe0, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x02, 0x06, 0x0e, 0x1e, 0x3e, 0xfe, 0x3e,
  0x1e, 0x0e, 0x06, 0x02, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18,
  0x7e, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66,
  0x66, 0x00, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7f, 0xdb, 0xdb, 0xdb, 0x7b, 0x1b,
  0x1b, 0x1b, 0x1b, 0x1b, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x7c, 0xc6, 0x60, 0x38, 0x6c, 0xc6, 0xc6,
  0x6c, 0x38, 0x0c, 0xc6, 0x7c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xfe, 0xfe, 0xfe, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18,
  0x7e, 0x3c, 0x18, 0x7e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x7e, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x0c, 0xfe,
  0x0c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x60, 0xfe,
  0x60, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0xc0,
  0xc0, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x24, 0x66, 0xff,
  0x66, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x38, 0x7c,
  0x7c, 0xfe, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0xfe, 0x7c, 0x7c,
  0x38, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x3c, 0x3c, 0x3c, 0x18, 0x18,
  0x18, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x66, 0x66, 0x66, 0x24, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x6c, 0x6c, 0xfe, 0x6c, 0x6c,
  0x6c, 0xfe, 0x6c, 0x6c, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x7c, 0xc6, 0xc2, 0xc0, 0x7c, 0x06,
  0x06, 0x86, 0xc6, 0x7c, 0x18, 0x18, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xc2, 0xc6, 0x0c, 0x18,
  0x30, 0x60, 0xc6, 0x86, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x6c, 0x6c, 0x38, 0x76, 0xdc,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x30, 0x30, 0x30, 0x60, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0c, 0x18, 0x30, 0x30, 0x30, 0x30,
  0x30, 0x30, 0x18, 0x0c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x30, 0x18, 0x0c, 0x0c, 0x0c, 0x0c,
  0x0c, 0x0c, 0x18, 0x30, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x3c, 0xff,
  0x3c, 0x66, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x7e,
  0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x18, 0x18, 0x30, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x02, 0x06, 0x0c, 0x18,
  0x30, 0x60, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0xc3, 0xc3, 0xdb, 0xdb,
  0xc3, 0xc3, 0x66, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x38, 0x78, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x7e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0x06, 0x0c, 0x18, 0x30,
  0x60, 0xc0, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0x06, 0x06, 0x3c, 0x06,
  0x06, 0x06, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0c, 0x1c, 0x3c, 0x6c, 0xcc, 0xfe,
  0x0c, 0x0c, 0x0c, 0x1e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0xc0, 0xc0, 0xc0, 0xfc, 0x06,
  0x06, 0x06, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x60, 0xc0, 0xc0, 0xfc, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0xc6, 0x06, 0x06, 0x0c, 0x18,
  0x30, 0x30, 0x30, 0x30, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0x7c, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0x7e, 0x06,
  0x06, 0x06, 0x0c, 0x78, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00,
  0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00,
  0x00, 0x18, 0x18, 0x30, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x06, 0x0c, 0x18, 0x30, 0x60,
  0x30, 0x18, 0x0c, 0x06, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0x00, 0x00,
  0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x60, 0x30, 0x18, 0x0c, 0x06,
  0x0c, 0x18, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0x0c, 0x18, 0x18,
  0x18, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xde, 0xde,
  0xde, 0xdc, 0xc0, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x10, 0x38, 0x6c, 0xc6, 0xc6, 0xfe,
  0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfc, 0x66, 0x66, 0x66, 0x7c, 0x66,
  0x66, 0x66, 0x66, 0xfc, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0, 0xc0, 0xc0,
  0xc0, 0xc2, 0x66, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xf8, 0x6c, 0x66, 0x66, 0x66, 0x66,
  0x66, 0x66, 0x6c, 0xf8, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0x66, 0x62, 0x68, 0x78, 0x68,
  0x60, 0x62, 0x66, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0x66, 0x62, 0x68, 0x78, 0x68,
  0x60, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0, 0xc0, 0xde,
  0xc6, 0xc6, 0x66, 0x3a, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xfe, 0xc6,
  0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1e, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
  0xcc, 0xcc, 0xcc, 0x78, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xe6, 0x66, 0x66, 0x6c, 0x78, 0x78,
  0x6c, 0x66, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xf0, 0x60, 0x60, 0x60, 0x60, 0x60,
  0x60, 0x62, 0x66, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc3, 0xe7, 0xff, 0xff, 0xdb, 0xc3,
  0xc3, 0xc3, 0xc3, 0xc3, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xe6, 0xf6, 0xfe, 0xde, 0xce,
  0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfc, 0x66, 0x66, 0x66, 0x7c, 0x60,
  0x60, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
  0xc6, 0xd6, 0xde, 0x7c, 0x0c, 0x0e, 0x00, 0x00,
  0x00, 0x00, 0xfc, 0x66, 0x66, 0x66, 0x7c, 0x6c,
  0x66, 0x66, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0x60, 0x38, 0x0c,
  0x06, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xff, 0xdb, 0x99, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3,
  0xc3, 0x66, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0xdb,
  0xdb, 0xff, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc3, 0xc3, 0x66, 0x3c, 0x18, 0x18,
  0x3c, 0x66, 0xc3, 0xc3, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc3, 0xc3, 0xc3, 0x66, 0x3c, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xff, 0xc3, 0x86, 0x0c, 0x18, 0x30,
  0x60, 0xc1, 0xc3, 0xff, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x30, 0x30, 0x30, 0x30, 0x30,
  0x30, 0x30, 0x30, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x80, 0xc0, 0xe0, 0x70, 0x38,
  0x1c, 0x0e, 0x06, 0x02, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
  0x0c, 0x0c, 0x0c, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x10, 0x38, 0x6c, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00,
  0x30, 0x30, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xe0, 0x60, 0x60, 0x78, 0x6c, 0x66,
  0x66, 0x66, 0x66, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc0,
  0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1c, 0x0c, 0x0c, 0x3c, 0x6c, 0xcc,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xfe,
  0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x6c, 0x64, 0x60, 0xf0, 0x60,
  0x60, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xcc, 0xcc,
  0xcc, 0xcc, 0xcc, 0x7c, 0x0c, 0xcc, 0x78, 0x00,
  0x00, 0x00, 0xe0, 0x60, 0x60, 0x6c, 0x76, 0x66,
  0x66, 0x66, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x00, 0x38, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x06, 0x06, 0x00, 0x0e, 0x06, 0x06,
  0x06, 0x06, 0x06, 0x06, 0x66, 0x66, 0x3c, 0x00,
  0x00, 0x00, 0xe0, 0x60, 0x60, 0x66, 0x6c, 0x78,
  0x78, 0x6c, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xe6, 0xff, 0xdb,
  0xdb, 0xdb, 0xdb, 0xdb, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xdc, 0x66, 0x66,
  0x66, 0x66, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xdc, 0x66, 0x66,
  0x66, 0x66, 0x66, 0x7c, 0x60, 0x60, 0xf0, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xcc, 0xcc,
  0xcc, 0xcc, 0xcc, 0x7c, 0x0c, 0x0c, 0x1e, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xdc, 0x76, 0x66,
  0x60, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0x60,
  0x38, 0x0c, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x10, 0x30, 0x30, 0xfc, 0x30, 0x30,
  0x30, 0x30, 0x36, 0x1c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xcc, 0xcc, 0xcc,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0xc3, 0xc3,
  0xc3, 0x66, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0xc3, 0xc3,
  0xdb, 0xdb, 0xff, 0x66, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0x66, 0x3c,
  0x18, 0x3c, 0x66, 0xc3, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7e, 0x06, 0x0c, 0xf8, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xcc, 0x18,
  0x30, 0x60, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0e, 0x18, 0x18, 0x18, 0x70, 0x18,
  0x18, 0x18, 0x18, 0x0e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x70, 0x18, 0x18, 0x18, 0x0e, 0x18,
  0x18, 0x18, 0x18, 0x70, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x76, 0xdc, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x6c, 0xc6,
  0xc6, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0, 0xc0, 0xc0,
  0xc2, 0x66, 0x3c, 0x0c, 0x06, 0x7c, 0x00, 0x00,
  0x00, 0x00, 0xcc, 0x00, 0x00, 0xcc, 0xcc, 0xcc,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x0c, 0x18, 0x30, 0x00, 0x7c, 0xc6, 0xfe,
  0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x10, 0x38, 0x6c, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xcc, 0x00, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x60, 0x30, 0x18, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x38, 0x6c, 0x38, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 0x60, 0x60,
  0x66, 0x3c, 0x0c, 0x06, 0x3c, 0x00, 0x00, 0x00,
  0x00, 0x10, 0x38, 0x6c, 0x00, 0x7c, 0xc6, 0xfe,
  0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0x00, 0x00, 0x7c, 0xc6, 0xfe,
  0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x60, 0x30, 0x18, 0x00, 0x7c, 0xc6, 0xfe,
  0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x66, 0x00, 0x00, 0x38, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x3c, 0x66, 0x00, 0x38, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x60, 0x30, 0x18, 0x00, 0x38, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xc6, 0x00, 0x10, 0x38, 0x6c, 0xc6, 0xc6,
  0xfe, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x38, 0x6c, 0x38, 0x00, 0x38, 0x6c, 0xc6, 0xc6,
  0xfe, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x30, 0x60, 0x00, 0xfe, 0x66, 0x60, 0x7c,
  0x60, 0x60, 0x66, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x6e, 0x3b, 0x1b,
  0x7e, 0xd8, 0xdc, 0x77, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3e, 0x6c, 0xcc, 0xcc, 0xfe, 0xcc,
  0xcc, 0xcc, 0xcc, 0xce, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x10, 0x38, 0x6c, 0x00, 0x7c, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0x00, 0x00, 0x7c, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x60, 0x30, 0x18, 0x00, 0x7c, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x30, 0x78, 0xcc, 0x00, 0xcc, 0xcc, 0xcc,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x60, 0x30, 0x18, 0x00, 0xcc, 0xcc, 0xcc,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0x00, 0x00, 0xc6, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7e, 0x06, 0x0c, 0x78, 0x00,
  0x00, 0xc6, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xc6, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x18, 0x7e, 0xc3, 0xc0, 0xc0, 0xc0,
  0xc3, 0x7e, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x38, 0x6c, 0x64, 0x60, 0xf0, 0x60, 0x60,
  0x60, 0x60, 0xe6, 0xfc, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc3, 0x66, 0x3c, 0x18, 0xff, 0x18,
  0xff, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xfc, 0x66, 0x66, 0x7c, 0x62, 0x66, 0x6f,
  0x66, 0x66, 0x66, 0xf3, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x0e, 0x1b, 0x18, 0x18, 0x18, 0x7e, 0x18,
  0x18, 0x18, 0x18, 0x18, 0xd8, 0x70, 0x00, 0x00,
  0x00, 0x18, 0x30, 0x60, 0x00, 0x78, 0x0c, 0x7c,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x0c, 0x18, 0x30, 0x00, 0x38, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x30, 0x60, 0x00, 0x7c, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x30, 0x60, 0x00, 0xcc, 0xcc, 0xcc,
  0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x76, 0xdc, 0x00, 0xdc, 0x66, 0x66,
  0x66, 0x66, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00,
  0x76, 0xdc, 0x00, 0xc6, 0xe6, 0xf6, 0xfe, 0xde,
  0xce, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x3c, 0x6c, 0x6c, 0x3e, 0x00, 0x7e, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x38, 0x6c, 0x6c, 0x38, 0x00, 0x7c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x30, 0x30, 0x00, 0x30, 0x30, 0x60,
  0xc0, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xc0,
  0xc0, 0xc0, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x06,
  0x06, 0x06, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xc0, 0xc0, 0xc2, 0xc6, 0xcc, 0x18, 0x30,
  0x60, 0xce, 0x9b, 0x06, 0x0c, 0x1f, 0x00, 0x00,
  0x00, 0xc0, 0xc0, 0xc2, 0xc6, 0xcc, 0x18, 0x30,
  0x66, 0xce, 0x96, 0x3e, 0x06, 0x06, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x00, 0x18, 0x18, 0x18,
  0x3c, 0x3c, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x36, 0x6c, 0xd8,
  0x6c, 0x36, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xd8, 0x6c, 0x36,
  0x6c, 0xd8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44,
  0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44,
  0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa,
  0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa,
  0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77,
  0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xf8,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 0xf8,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xf6,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x18, 0xf8,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x36, 0x36, 0x36, 0x36, 0x36, 0xf6, 0x06, 0xf6,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x06, 0xf6,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0xf6, 0x06, 0xfe,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xfe,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 0xf8,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x1f,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x1f,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xff,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x1f,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x37,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 0x30, 0x3f,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0x30, 0x37,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0xf7, 0x00, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0xf7,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 0x30, 0x37,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0x36, 0xf7, 0x00, 0xf7,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x00, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0xff,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x3f,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x1f,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x18, 0x1f,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xff,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x18, 0xff,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xf8,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
  0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
  0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
  0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc, 0xd8,
  0xd8, 0xd8, 0xdc, 0x76, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x78, 0xcc, 0xcc, 0xcc, 0xd8, 0xcc,
  0xc6, 0xc6, 0xc6, 0xcc, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0xc6, 0xc6, 0xc0, 0xc0, 0xc0,
  0xc0, 0xc0, 0xc0, 0xc0, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0x6c, 0x6c, 0x6c,
  0x6c, 0x6c, 0x6c, 0x6c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xfe, 0xc6, 0x60, 0x30, 0x18,
  0x30, 0x60, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0xd8, 0xd8,
  0xd8, 0xd8, 0xd8, 0x70, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x66, 0x66,
  0x66, 0x7c, 0x60, 0x60, 0xc0, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x76, 0xdc, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x7e, 0x18, 0x3c, 0x66, 0x66,
  0x66, 0x3c, 0x18, 0x7e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x38, 0x6c, 0xc6, 0xc6, 0xfe,
  0xc6, 0xc6, 0x6c, 0x38, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x6c, 0xc6, 0xc6, 0xc6, 0x6c,
  0x6c, 0x6c, 0x6c, 0xee, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1e, 0x30, 0x18, 0x0c, 0x3e, 0x66,
  0x66, 0x66, 0x66, 0x3c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0xdb, 0xdb,
  0xdb, 0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x03, 0x06, 0x7e, 0xdb, 0xdb,
  0xf3, 0x7e, 0x60, 0xc0, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1c, 0x30, 0x60, 0x60, 0x7c, 0x60,
  0x60, 0x60, 0x30, 0x1c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6,
  0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0x00, 0x00, 0xfe,
  0x00, 0x00, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x7e, 0x18,
  0x18, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x30, 0x18, 0x0c, 0x06, 0x0c,
  0x18, 0x30, 0x00, 0x7e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x0c, 0x18, 0x30, 0x60, 0x30,
  0x18, 0x0c, 0x00, 0x7e, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0e, 0x1b, 0x1b, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
  0xd8, 0xd8, 0xd8, 0x70, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x7e,
  0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc, 0x00,
  0x76, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x38, 0x6c, 0x6c, 0x38, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18,
  0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x0f, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0xec,
  0x6c, 0x6c, 0x3c, 0x1c, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xd8, 0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x70, 0xd8, 0x30, 0x60, 0xc8, 0xf8, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x7c, 0x7c, 0x7c, 0x7c,
  0x7c, 0x7c, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};



