/*  XMMS - Cross-platform multimedia player
 *  Copyright (C) 1998-2003  Peter Alm, Mikael Alm, Olle Hallnas,
 *                           Thomas Nilsson and 4Front Technologies
 *  Copyright (C) 1999-2003  Haavard Kvaalen
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public Licensse as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#if defined(HAVE_CONFIG_H)
#include "config.h"
#endif

#include <glib.h>
#include <glib/gprintf.h>
#include <stdlib.h>
#include <string.h>
#include <gdk/gdk.h>
#include <getopt.h>
#include <ctype.h>

#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <gdk/gdkx.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <sm.h>

#ifdef HAVE_SCHED_H
# include <sched.h>
#elif defined HAVE_SYS_SCHED_H
# include <sys/sched.h>
#endif

#include "main.h"

#include "controlsocket.h"
#include "dnd.h"
#include "dock.h"
#include "effect.h"
#include "equalizer.h"
#include "general.h"
#include "hints.h"
#include "input.h"
#include "output.h"
#include "mainwin.h"
#include "playlist.h"
#include "playlistwin.h"
#include "pluginenum.h"
#include "prefswin.h"
#include "skin.h"
#include "skinwin.h"
#include "util.h"
#include "visualization.h"
#include "libbeep/configfile.h"
#include "libbeep/beepctrl.h"
#include "libbeep/util.h"

#include "pixmaps.h"


Config cfg;

GList *disabled_iplugins = NULL;

GList *dock_window_list = NULL;

gboolean fb_visible = FALSE;

/* For x11r5 session management */
static char **restart_argv;
static int restart_argc;

gboolean pposition_broken = FALSE;

extern gchar *plugin_dir_list[];

const GtkTargetEntry _xmms_drop_types[] = {
    {"text/plain", 0, XMMS_DROP_PLAINTEXT},
    {"text/uri-list", 0, XMMS_DROP_URLENCODED},
    {"STRING", 0, XMMS_DROP_STRING},
    {"interface/x-winamp-skin", 0, XMMS_DROP_SKIN},
    {"application/x-font-ttf", 0, XMMS_DROP_FONT},
};

static void make_beep_dir(void)
{
    gchar *filename;

    filename = g_strconcat(g_get_home_dir(), "/", BMP_RCPATH, NULL);
    mkdir(filename, S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH);
    g_free(filename);

    filename =
	g_strconcat(g_get_home_dir(), "/", BMP_RCPATH, "/Skins", NULL);
    mkdir(filename, S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH);
    g_free(filename);

    filename =
	g_strconcat(g_get_home_dir(), "/", BMP_RCPATH, "/Plugins", NULL);
    mkdir(filename, S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH);
    g_free(filename);
}

static void read_config(void)
{
    ConfigFile *cfgfile;
    gchar *filename;
    gint i, length;

    memset(&cfg, 0, sizeof(Config));
    cfg.autoscroll = TRUE;
    cfg.player_x = 20;
    cfg.player_y = 20;
    cfg.always_on_top = FALSE;
    cfg.sticky = FALSE;
    cfg.always_show_cb = TRUE;
    cfg.convert_underscore = TRUE;
    cfg.convert_twenty = TRUE;
    cfg.show_numbers_in_pl = TRUE;
    cfg.snap_windows = TRUE;
    cfg.save_window_position = TRUE;
    cfg.dim_titlebar = TRUE;
    cfg.get_info_on_load = TRUE;
    cfg.get_info_on_demand = TRUE;
    cfg.eq_doublesize_linked = TRUE;
    cfg.player_visible = TRUE;
    cfg.no_playlist_advance = FALSE;
    cfg.smooth_title_scroll = TRUE;
    cfg.random_skin_on_play = FALSE;
    cfg.mainwin_use_xfont = FALSE;
    cfg.use_pl_metadata = TRUE;
    cfg.warn_about_unplayables = TRUE;

    cfg.playlist_x = 295;
    cfg.playlist_y = 20;
    cfg.playlist_width = 300;
    cfg.playlist_height = 232;
    cfg.playlist_transparent = FALSE;

    cfg.filesel_path = NULL;
    cfg.playlist_path = NULL;

    cfg.equalizer_x = 20;
    cfg.equalizer_y = 136;

    cfg.snap_distance = 10;
    cfg.pause_between_songs_time = 2;

    cfg.vis_type = VIS_ANALYZER;
    cfg.analyzer_mode = ANALYZER_NORMAL;
    cfg.analyzer_type = ANALYZER_BARS;
    cfg.analyzer_peaks = TRUE;
    cfg.scope_mode = SCOPE_DOT;
    cfg.vu_mode = VU_SMOOTH;
    cfg.vis_refresh = REFRESH_FULL;
    cfg.analyzer_falloff = FALLOFF_FAST;
    cfg.peaks_falloff = FALLOFF_SLOW;
    cfg.disabled_iplugins = NULL;
    cfg.enabled_gplugins = NULL;
    cfg.mouse_change = 8;

    cfg.xmms_compat_mode = TRUE;

    cfg.gentitle_format = NULL;

    filename =
	g_build_filename(g_get_home_dir(), BMP_RCPATH, "config", NULL);

    cfgfile = xmms_cfg_open_file(filename);
    if (cfgfile) {
	xmms_cfg_read_boolean(cfgfile, "xmms", "allow_multiple_instances",
			      &cfg.allow_multiple_instances);
	xmms_cfg_read_boolean(cfgfile, "xmms", "use_realtime",
			      &cfg.use_realtime);
	xmms_cfg_read_boolean(cfgfile, "xmms", "always_show_cb",
			      &cfg.always_show_cb);
	xmms_cfg_read_boolean(cfgfile, "xmms", "convert_underscore",
			      &cfg.convert_underscore);
	xmms_cfg_read_boolean(cfgfile, "xmms", "convert_%20",
			      &cfg.convert_twenty);
	xmms_cfg_read_boolean(cfgfile, "xmms", "show_numbers_in_pl",
			      &cfg.show_numbers_in_pl);
	xmms_cfg_read_boolean(cfgfile, "xmms", "snap_windows",
			      &cfg.snap_windows);
	xmms_cfg_read_boolean(cfgfile, "xmms", "save_window_positions",
			      &cfg.save_window_position);
	xmms_cfg_read_boolean(cfgfile, "xmms", "dim_titlebar",
			      &cfg.dim_titlebar);
	xmms_cfg_read_boolean(cfgfile, "xmms", "get_info_on_load",
			      &cfg.get_info_on_load);
	xmms_cfg_read_boolean(cfgfile, "xmms", "get_info_on_demand",
			      &cfg.get_info_on_demand);
	xmms_cfg_read_boolean(cfgfile, "xmms", "eq_doublesize_linked",
			      &cfg.eq_doublesize_linked);
	xmms_cfg_read_boolean(cfgfile, "xmms", "no_playlist_advance",
			      &cfg.no_playlist_advance);
	xmms_cfg_read_boolean(cfgfile, "xmms", "sort_jump_to_file",
			      &cfg.sort_jump_to_file);
	xmms_cfg_read_boolean(cfgfile, "xmms", "use_pl_metadata",
			      &cfg.use_pl_metadata);
	xmms_cfg_read_boolean(cfgfile, "xmms", "warn_about_unplayables",
			      &cfg.warn_about_unplayables);
	xmms_cfg_read_boolean(cfgfile, "xmms", "smooth_title_scroll",
			      &cfg.smooth_title_scroll);
	xmms_cfg_read_boolean(cfgfile, "xmms",
			      "use_backslash_as_dir_delimiter",
			      &cfg.use_backslash_as_dir_delimiter);
	xmms_cfg_read_int(cfgfile, "xmms", "player_x", &cfg.player_x);
	xmms_cfg_read_int(cfgfile, "xmms", "player_y", &cfg.player_y);
	xmms_cfg_read_boolean(cfgfile, "xmms", "player_shaded",
			      &cfg.player_shaded);
	xmms_cfg_read_boolean(cfgfile, "xmms", "player_visible",
			      &cfg.player_visible);
	xmms_cfg_read_boolean(cfgfile, "xmms", "shuffle", &cfg.shuffle);
	xmms_cfg_read_boolean(cfgfile, "xmms", "repeat", &cfg.repeat);
	xmms_cfg_read_boolean(cfgfile, "xmms", "doublesize",
			      &cfg.doublesize);
	xmms_cfg_read_boolean(cfgfile, "xmms", "autoscroll_songname",
			      &cfg.autoscroll);
	xmms_cfg_read_int(cfgfile, "xmms", "timer_mode", &cfg.timer_mode);
	xmms_cfg_read_int(cfgfile, "xmms", "vis_type", &cfg.vis_type);
	xmms_cfg_read_int(cfgfile, "xmms", "analyzer_mode",
			  &cfg.analyzer_mode);
	xmms_cfg_read_int(cfgfile, "xmms", "analyzer_type",
			  &cfg.analyzer_type);
	xmms_cfg_read_boolean(cfgfile, "xmms", "analyzer_peaks",
			      &cfg.analyzer_peaks);
	xmms_cfg_read_int(cfgfile, "xmms", "scope_mode", &cfg.scope_mode);
	xmms_cfg_read_int(cfgfile, "xmms", "vu_mode", &cfg.vu_mode);
	xmms_cfg_read_int(cfgfile, "xmms", "vis_refresh_rate",
			  &cfg.vis_refresh);
	xmms_cfg_read_int(cfgfile, "xmms", "analyzer_falloff",
			  &cfg.analyzer_falloff);
	xmms_cfg_read_int(cfgfile, "xmms", "peaks_falloff",
			  &cfg.peaks_falloff);
	xmms_cfg_read_int(cfgfile, "xmms", "playlist_x", &cfg.playlist_x);
	xmms_cfg_read_int(cfgfile, "xmms", "playlist_y", &cfg.playlist_y);
	xmms_cfg_read_int(cfgfile, "xmms", "playlist_width",
			  &cfg.playlist_width);
	xmms_cfg_read_int(cfgfile, "xmms", "playlist_height",
			  &cfg.playlist_height);
	xmms_cfg_read_boolean(cfgfile, "xmms", "playlist_shaded",
			      &cfg.playlist_shaded);
	xmms_cfg_read_boolean(cfgfile, "xmms", "playlist_visible",
			      &cfg.playlist_visible);
	xmms_cfg_read_boolean(cfgfile, "xmms", "playlist_transparent",
			      &cfg.playlist_transparent);
	xmms_cfg_read_string(cfgfile, "xmms", "playlist_font",
			     &cfg.playlist_font);
	xmms_cfg_read_boolean(cfgfile, "xmms", "use_fontsets",
			      &cfg.use_fontsets);
	xmms_cfg_read_boolean(cfgfile, "xmms", "mainwin_use_xfont",
			      &cfg.mainwin_use_xfont);
	xmms_cfg_read_string(cfgfile, "xmms", "mainwin_font",
			     &cfg.mainwin_font);
	xmms_cfg_read_int(cfgfile, "xmms", "playlist_position",
			  &cfg.playlist_position);
	xmms_cfg_read_int(cfgfile, "xmms", "equalizer_x",
			  &cfg.equalizer_x);
	xmms_cfg_read_int(cfgfile, "xmms", "equalizer_y",
			  &cfg.equalizer_y);
	xmms_cfg_read_int(cfgfile, "xmms", "snap_distance",
			  &cfg.snap_distance);
	xmms_cfg_read_boolean(cfgfile, "xmms", "equalizer_visible",
			      &cfg.equalizer_visible);
	xmms_cfg_read_boolean(cfgfile, "xmms", "equalizer_active",
			      &cfg.equalizer_active);
	xmms_cfg_read_boolean(cfgfile, "xmms", "equalizer_shaded",
			      &cfg.equalizer_shaded);
	xmms_cfg_read_boolean(cfgfile, "xmms", "equalizer_autoload",
			      &cfg.equalizer_autoload);
	xmms_cfg_read_boolean(cfgfile, "xmms", "easy_move",
			      &cfg.easy_move);
	xmms_cfg_read_float(cfgfile, "xmms", "equalizer_preamp",
			    &cfg.equalizer_preamp);
	for (i = 0; i < 10; i++) {
	    gchar eqtext[18];

	    g_sprintf(eqtext, "equalizer_band%d", i);
	    xmms_cfg_read_float(cfgfile, "xmms", eqtext,
				&cfg.equalizer_bands[i]);
	}
	xmms_cfg_read_string(cfgfile, "xmms", "eqpreset_default_file",
			     &cfg.eqpreset_default_file);
	xmms_cfg_read_string(cfgfile, "xmms", "eqpreset_extension",
			     &cfg.eqpreset_extension);
	xmms_cfg_read_string(cfgfile, "xmms", "skin", &cfg.skin);
	xmms_cfg_read_string(cfgfile, "xmms", "output_plugin",
			     &cfg.outputplugin);
	xmms_cfg_read_string(cfgfile, "xmms", "enabled_gplugins",
			     &cfg.enabled_gplugins);
	xmms_cfg_read_string(cfgfile, "xmms", "enabled_vplugins",
			     &cfg.enabled_vplugins);
	xmms_cfg_read_string(cfgfile, "xmms", "enabled_eplugins",
			     &cfg.enabled_eplugins);
	xmms_cfg_read_string(cfgfile, "xmms", "filesel_path",
			     &cfg.filesel_path);
	xmms_cfg_read_string(cfgfile, "xmms", "playlist_path",
			     &cfg.playlist_path);
	xmms_cfg_read_string(cfgfile, "xmms", "disabled_iplugins",
			     &cfg.disabled_iplugins);
	xmms_cfg_read_boolean(cfgfile, "xmms", "use_eplugins",
			      &cfg.use_eplugins);
	xmms_cfg_read_boolean(cfgfile, "xmms", "always_on_top",
			      &cfg.always_on_top);
	xmms_cfg_read_boolean(cfgfile, "xmms", "sticky", &cfg.sticky);
	xmms_cfg_read_boolean(cfgfile, "xmms", "random_skin_on_play",
			      &cfg.random_skin_on_play);
	xmms_cfg_read_boolean(cfgfile, "xmms", "pause_between_songs",
			      &cfg.pause_between_songs);
	xmms_cfg_read_int(cfgfile, "xmms", "pause_between_songs_time",
			  &cfg.pause_between_songs_time);
	xmms_cfg_read_int(cfgfile, "xmms", "mouse_wheel_change",
			  &cfg.mouse_change);
	xmms_cfg_read_boolean(cfgfile, "xmms", "show_wm_decorations",
			      &cfg.show_wm_decorations);
	xmms_cfg_read_boolean(cfgfile, "xmms", "xmms_compat_mode",
			      &cfg.xmms_compat_mode);
	if (xmms_cfg_read_int
	    (cfgfile, "xmms", "url_history_length", &length)) {
	    for (i = 1; i <= length; i++) {
		gchar str[19], *temp;

		g_sprintf(str, "url_history%d", i);
		if (xmms_cfg_read_string(cfgfile, "xmms", str, &temp))
		    cfg.url_history = g_list_append(cfg.url_history, temp);
	    }
	}
	xmms_cfg_read_string(cfgfile, "xmms", "generic_title_format",
			     &cfg.gentitle_format);

	xmms_cfg_free(cfgfile);
    }


    if (cfg.playlist_font && strlen(cfg.playlist_font) == 0) {
	g_free(cfg.playlist_font);
	cfg.playlist_font = NULL;
    }

    if (cfg.mainwin_font && strlen(cfg.mainwin_font) == 0) {
	g_free(cfg.mainwin_font);
	cfg.mainwin_font = NULL;
    }

    if (!cfg.playlist_font)
	cfg.playlist_font = g_strdup("Sans Bold 12");
    if (!cfg.mainwin_font)
	cfg.mainwin_font = g_strdup("Sans Bold 9");
    if (!cfg.gentitle_format)
	cfg.gentitle_format = g_strdup("%p - %t");
    if (!cfg.outputplugin) {
#ifdef HAVE_OSS
	cfg.outputplugin =
	    g_strdup_printf("%s/%s/libOSS.so", PLUGIN_DIR,
			    plugin_dir_list[0]);
#elif defined(sun)
	cfg.outputplugin =
	    g_strdup_printf("%s/%s/libSolaris.so", PLUGIN_DIR,
			    plugin_dir_list[0]);
#else
	/*
	 * FIXME: This implisitly means the output plugin that is first
	 * in the alphabet will be used (usually the disk writer plugin)
	 */
	cfg.outputplugin = g_strdup("");
#endif
    }
    if (cfg.eqpreset_default_file == NULL)
	cfg.eqpreset_default_file = g_strdup("dir_default.preset");
    if (cfg.eqpreset_extension == NULL)
	cfg.eqpreset_extension = g_strdup("preset");

    g_free(filename);
}

void save_config(void)
{
    GList *d_iplist, *node;
    gchar *temp;
    gchar *filename, *str;
    gint i;
    ConfigFile *cfgfile;

    if (cfg.disabled_iplugins)
	g_free(cfg.disabled_iplugins);
    if (disabled_iplugins && (g_list_length(disabled_iplugins) > 0)) {
	d_iplist = disabled_iplugins;
	cfg.disabled_iplugins =
	    g_strdup(g_basename
		     (((InputPlugin *) d_iplist->data)->filename));
	d_iplist = d_iplist->next;
	while (d_iplist != NULL) {
	    temp = cfg.disabled_iplugins;
	    cfg.disabled_iplugins =
		g_strconcat(temp, ",",
			    g_basename(((InputPlugin *) d_iplist->data)->
				       filename), NULL);
	    g_free(temp);
	    d_iplist = d_iplist->next;
	}
    } else
	cfg.disabled_iplugins = g_strdup("");

    filename =
	g_build_filename(g_get_home_dir(), BMP_RCPATH, "config", NULL);
    cfgfile = xmms_cfg_open_file(filename);
    if (!cfgfile)
	cfgfile = xmms_cfg_new();
    xmms_cfg_write_boolean(cfgfile, "xmms", "allow_multiple_instances",
			   cfg.allow_multiple_instances);
    xmms_cfg_write_boolean(cfgfile, "xmms", "use_realtime",
			   cfg.use_realtime);
    xmms_cfg_write_boolean(cfgfile, "xmms", "always_show_cb",
			   cfg.always_show_cb);
    xmms_cfg_write_boolean(cfgfile, "xmms", "convert_underscore",
			   cfg.convert_underscore);
    xmms_cfg_write_boolean(cfgfile, "xmms", "convert_%20",
			   cfg.convert_twenty);
    xmms_cfg_write_boolean(cfgfile, "xmms", "show_numbers_in_pl",
			   cfg.show_numbers_in_pl);
    xmms_cfg_write_boolean(cfgfile, "xmms", "snap_windows",
			   cfg.snap_windows);
    xmms_cfg_write_boolean(cfgfile, "xmms", "save_window_positions",
			   cfg.save_window_position);
    xmms_cfg_write_boolean(cfgfile, "xmms", "dim_titlebar",
			   cfg.dim_titlebar);
    xmms_cfg_write_boolean(cfgfile, "xmms", "use_pl_metadata",
			   cfg.use_pl_metadata);
    xmms_cfg_write_boolean(cfgfile, "xmms", "warn_about_unplayables",
			   cfg.warn_about_unplayables);
    xmms_cfg_write_boolean(cfgfile, "xmms", "get_info_on_load",
			   cfg.get_info_on_load);
    xmms_cfg_write_boolean(cfgfile, "xmms", "get_info_on_demand",
			   cfg.get_info_on_demand);
    xmms_cfg_write_boolean(cfgfile, "xmms", "eq_doublesize_linked",
			   cfg.eq_doublesize_linked);
    xmms_cfg_write_boolean(cfgfile, "xmms", "no_playlist_advance",
			   cfg.no_playlist_advance);
    xmms_cfg_write_boolean(cfgfile, "xmms", "sort_jump_to_file",
			   cfg.sort_jump_to_file);
    xmms_cfg_write_boolean(cfgfile, "xmms", "smooth_title_scroll",
			   cfg.smooth_title_scroll);
    xmms_cfg_write_boolean(cfgfile, "xmms",
			   "use_backslash_as_dir_delimiter",
			   cfg.use_backslash_as_dir_delimiter);
    /*  dock_get_widget_pos(mainwin, &cfg.player_x, &cfg.player_y); */
    xmms_cfg_write_int(cfgfile, "xmms", "player_x", cfg.player_x);
    xmms_cfg_write_int(cfgfile, "xmms", "player_y", cfg.player_y);
    xmms_cfg_write_boolean(cfgfile, "xmms", "player_shaded",
			   cfg.player_shaded);
    xmms_cfg_write_boolean(cfgfile, "xmms", "player_visible",
			   cfg.player_visible);
    xmms_cfg_write_boolean(cfgfile, "xmms", "shuffle",
			   mainwin_shuffle->tb_selected);
    xmms_cfg_write_boolean(cfgfile, "xmms", "repeat",
			   mainwin_repeat->tb_selected);
    xmms_cfg_write_boolean(cfgfile, "xmms", "doublesize", cfg.doublesize);
    xmms_cfg_write_boolean(cfgfile, "xmms", "autoscroll_songname",
			   cfg.autoscroll);
    xmms_cfg_write_int(cfgfile, "xmms", "timer_mode", cfg.timer_mode);
    xmms_cfg_write_int(cfgfile, "xmms", "vis_type", cfg.vis_type);
    xmms_cfg_write_int(cfgfile, "xmms", "analyzer_mode",
		       cfg.analyzer_mode);
    xmms_cfg_write_int(cfgfile, "xmms", "analyzer_type",
		       cfg.analyzer_type);
    xmms_cfg_write_boolean(cfgfile, "xmms", "analyzer_peaks",
			   cfg.analyzer_peaks);
    xmms_cfg_write_int(cfgfile, "xmms", "scope_mode", cfg.scope_mode);
    xmms_cfg_write_int(cfgfile, "xmms", "vu_mode", cfg.vu_mode);
    xmms_cfg_write_int(cfgfile, "xmms", "vis_refresh_rate",
		       cfg.vis_refresh);
    xmms_cfg_write_int(cfgfile, "xmms", "analyzer_falloff",
		       cfg.analyzer_falloff);
    xmms_cfg_write_int(cfgfile, "xmms", "peaks_falloff",
		       cfg.peaks_falloff);
    /*  dock_get_widget_pos(playlistwin, &cfg.playlist_x, &cfg.playlist_y); */
    xmms_cfg_write_int(cfgfile, "xmms", "playlist_x", cfg.playlist_x);
    xmms_cfg_write_int(cfgfile, "xmms", "playlist_y", cfg.playlist_y);
    xmms_cfg_write_int(cfgfile, "xmms", "playlist_width",
		       cfg.playlist_width);
    xmms_cfg_write_int(cfgfile, "xmms", "playlist_height",
		       cfg.playlist_height);
    xmms_cfg_write_boolean(cfgfile, "xmms", "playlist_shaded",
			   cfg.playlist_shaded);
    xmms_cfg_write_boolean(cfgfile, "xmms", "playlist_visible",
			   cfg.playlist_visible);
    xmms_cfg_write_boolean(cfgfile, "xmms", "playlist_transparent",
			   cfg.playlist_transparent);
    xmms_cfg_write_string(cfgfile, "xmms", "playlist_font",
			  cfg.playlist_font);
    xmms_cfg_write_boolean(cfgfile, "xmms", "use_fontsets",
			   cfg.use_fontsets);
    xmms_cfg_write_boolean(cfgfile, "xmms", "mainwin_use_xfont",
			   cfg.mainwin_use_xfont);
    xmms_cfg_write_string(cfgfile, "xmms", "mainwin_font",
			  cfg.mainwin_font);
    xmms_cfg_write_int(cfgfile, "xmms", "playlist_position",
		       get_playlist_position());
    /*  dock_get_widget_pos(equalizerwin, &cfg.equalizer_x, &cfg.equalizer_y); */
    xmms_cfg_write_int(cfgfile, "xmms", "equalizer_x", cfg.equalizer_x);
    xmms_cfg_write_int(cfgfile, "xmms", "equalizer_y", cfg.equalizer_y);
    xmms_cfg_write_int(cfgfile, "xmms", "snap_distance",
		       cfg.snap_distance);
    xmms_cfg_write_boolean(cfgfile, "xmms", "equalizer_visible",
			   cfg.equalizer_visible);
    xmms_cfg_write_boolean(cfgfile, "xmms", "equalizer_shaded",
			   cfg.equalizer_shaded);
    xmms_cfg_write_boolean(cfgfile, "xmms", "equalizer_active",
			   cfg.equalizer_active);
    xmms_cfg_write_boolean(cfgfile, "xmms", "equalizer_autoload",
			   cfg.equalizer_autoload);
    xmms_cfg_write_boolean(cfgfile, "xmms", "easy_move", cfg.easy_move);
    xmms_cfg_write_boolean(cfgfile, "xmms", "use_eplugins",
			   cfg.use_eplugins);
    xmms_cfg_write_boolean(cfgfile, "xmms", "always_on_top",
			   cfg.always_on_top);
    xmms_cfg_write_boolean(cfgfile, "xmms", "sticky", cfg.sticky);
    xmms_cfg_write_float(cfgfile, "xmms", "equalizer_preamp",
			 cfg.equalizer_preamp);
    xmms_cfg_write_boolean(cfgfile, "xmms", "random_skin_on_play",
			   cfg.random_skin_on_play);
    xmms_cfg_write_boolean(cfgfile, "xmms", "pause_between_songs",
			   cfg.pause_between_songs);
    xmms_cfg_write_int(cfgfile, "xmms", "pause_between_songs_time",
		       cfg.pause_between_songs_time);
    xmms_cfg_write_int(cfgfile, "xmms", "mouse_wheel_change",
		       cfg.mouse_change);
    xmms_cfg_write_boolean(cfgfile, "xmms", "show_wm_decorations",
			   cfg.show_wm_decorations);
    xmms_cfg_write_boolean(cfgfile, "xmms", "xmms_compat_mode",
			   cfg.xmms_compat_mode);
    xmms_cfg_write_string(cfgfile, "xmms", "eqpreset_default_file",
			  cfg.eqpreset_default_file);
    xmms_cfg_write_string(cfgfile, "xmms", "eqpreset_extension",
			  cfg.eqpreset_extension);
    for (i = 0; i < 10; i++) {
	str = g_strdup_printf("equalizer_band%d", i);
	xmms_cfg_write_float(cfgfile, "xmms", str, cfg.equalizer_bands[i]);
	g_free(str);
    }
    if (skin->path)
	xmms_cfg_write_string(cfgfile, "xmms", "skin", skin->path);
    else
	xmms_cfg_remove_key(cfgfile, "xmms", "skin");
    if (get_current_output_plugin())
	xmms_cfg_write_string(cfgfile, "xmms", "output_plugin",
			      get_current_output_plugin()->filename);
    else
	xmms_cfg_remove_key(cfgfile, "xmms", "output_plugin");

    str = general_stringify_enabled_list();
    if (str) {
	xmms_cfg_write_string(cfgfile, "xmms", "enabled_gplugins", str);
	g_free(str);
    } else
	xmms_cfg_remove_key(cfgfile, "xmms", "enabled_gplugins");

    str = vis_stringify_enabled_list();
    if (str) {
	xmms_cfg_write_string(cfgfile, "xmms", "enabled_vplugins", str);
	g_free(str);
    } else
	xmms_cfg_remove_key(cfgfile, "xmms", "enabled_vplugins");

    str = effect_stringify_enabled_list();
    if (str) {
	xmms_cfg_write_string(cfgfile, "xmms", "enabled_eplugins", str);
	g_free(str);
    } else
	xmms_cfg_remove_key(cfgfile, "xmms", "enabled_eplugins");

    xmms_cfg_write_string(cfgfile, "xmms", "disabled_iplugins",
			  cfg.disabled_iplugins);
    if (cfg.filesel_path)
	xmms_cfg_write_string(cfgfile, "xmms", "filesel_path",
			      cfg.filesel_path);
    if (cfg.playlist_path)
	xmms_cfg_write_string(cfgfile, "xmms", "playlist_path",
			      cfg.playlist_path);
    xmms_cfg_write_int(cfgfile, "xmms", "url_history_length",
		       g_list_length(cfg.url_history));
    for (node = cfg.url_history, i = 1; node;
	 node = g_list_next(node), i++) {
	str = g_strdup_printf("url_history%d", i);
	xmms_cfg_write_string(cfgfile, "xmms", str, node->data);
	g_free(str);
    }
    xmms_cfg_write_string(cfgfile, "xmms", "generic_title_format",
			  cfg.gentitle_format);

    xmms_cfg_write_file(cfgfile, filename);
    xmms_cfg_free(cfgfile);

    g_free(filename);

    playlist_save(g_build_filename
		  (g_get_home_dir(), BMP_RCPATH, "bmp.m3u", NULL), FALSE);
}

gchar *xmms_get_gentitle_format(void)
{
    return cfg.gentitle_format;
}

static struct option long_options[] = {
    {"help", 0, NULL, 'h'},
    {"session", 1, NULL, 'n'},
    {"rew", 0, NULL, 'r'},
    {"play", 0, NULL, 'p'},
    {"pause", 0, NULL, 'u'},
    {"play-pause", 0, NULL, 't'},
    {"stop", 0, NULL, 's'},
    {"fwd", 0, NULL, 'f'},
    {"enqueue", 0, NULL, 'e'},
    {"show-main-window", 0, NULL, 'm'},
    {"version", 0, NULL, 'v'},
    {"sm-client-id", 1, NULL, 'i'},
    {"xmms", 0, NULL, 'x'},
    {0, 0, 0, 0}
};

void display_usage(void)
{
    g_fprintf(stderr, _("Usage: beep [options] [files] ...\n\n"
			"Options:\n" "--------\n"));
    g_fprintf(stderr, "\n-h, --help		");
    /* -h, --help switch */
    g_fprintf(stderr, _("Display this text and exit."));
    g_fprintf(stderr, "\n-n, --session		");
    /* -n, --session switch */
    g_fprintf(stderr, _("Select XMMS session (Default: 0)"));
    g_fprintf(stderr, "\n-r, --rew		");
    /* -r, --rew switch */
    g_fprintf(stderr, _("Skip backwards in playlist"));
    g_fprintf(stderr, "\n-p, --play		");
    /* -p, --play switch */
    g_fprintf(stderr, _("Start playing current playlist"));
    g_fprintf(stderr, "\n-u, --pause		");
    /* -u, --pause switch */
    g_fprintf(stderr, _("Pause current song"));
    g_fprintf(stderr, "\n-s, --stop		");
    /* -s, --stop switch */
    g_fprintf(stderr, _("Stop current song"));
    g_fprintf(stderr, "\n-t, --play-pause	");
    /* -t, --play-pause switch */
    g_fprintf(stderr, _("Pause if playing, play otherwise"));
    g_fprintf(stderr, "\n-f, --fwd		");
    /* -f, --fwd switch */
    g_fprintf(stderr, _("Skip forward in playlist"));
    g_fprintf(stderr, "\n-e, --enqueue		");
    /* -e, --enqueue switch */
    g_fprintf(stderr, _("Don't clear the playlist"));
    g_fprintf(stderr, "\n-m, --show-main-window	");
    /* -m, --show-main-window switch */
    g_fprintf(stderr, _("Show the main window."));
    g_fprintf(stderr, "\n-i, --sm-client-id		");
    /* -i, --sm-client-id switch */
    g_fprintf(stderr, _("Previous session ID"));
    g_fprintf(stderr, "\n-v, --version		");
    /* -v, --version switch */
    g_fprintf(stderr, _("Print version number and exit."));
/* 	g_fprintf(stderr, "\n-x, --xmms                   "); */
/* 	/\* -x, --xmms switch *\/ */
/* 	g_fprintf(stderr, _("XMMS compatibility mode")); */
    g_fprintf(stderr, "\n\n");

    exit(0);
}

struct cmdlineopt {
    GList *filenames;
    int session;
    gboolean play, stop, pause, fwd, rew, play_pause, playcd;
    gboolean enqueue, mainwin, remote;
    gboolean load_skins;
    char *previous_session_id;

    gboolean xmms_compat_mode;
};

void parse_cmd_line(int argc, char **argv, struct cmdlineopt *opt)
{
    int c, i;
    char *filename;

    memset(opt, 0, sizeof(struct cmdlineopt));
    opt->session = -1;
    while ((c =
	    getopt_long(argc, argv, "chn:rpusfemvtLS", long_options,
			NULL)) != -1) {
	switch (c) {
	case 'h':
	    display_usage();
	    break;
	case 'n':
	    opt->session = atoi(optarg);
	    break;
	case 'r':
	    opt->rew = TRUE;
	    break;
	case 'p':
	    opt->play = TRUE;
	    break;
	case 'u':
	    opt->pause = TRUE;
	    break;
	case 's':
	    opt->stop = TRUE;
	    break;
	case 'f':
	    opt->fwd = TRUE;
	    break;
	case 't':
	    opt->play_pause = TRUE;
	    break;
	case 'm':
	    opt->mainwin = TRUE;
	    break;
	case 'e':
	    opt->enqueue = TRUE;
	    break;
	case 'v':
	    g_printf("%s %s\n", PACKAGE, VERSION);
	    exit(0);
	    break;
	case 'i':
	    opt->previous_session_id = g_strdup(optarg);
	    break;
	case 'c':
	    opt->playcd = TRUE;
	    break;
/* 		        case 'x': */
/* 				opt->xmms_compat_mode = TRUE; */
/* 				break; */
	case 'S':
	    opt->load_skins = TRUE;
	    break;
	}
    }
    for (i = optind; i < argc; i++) {
	if (argv[i][0] == '/' || strstr(argv[i], "://"))
	    filename = g_strdup(argv[i]);
	else {
	    char *tmp = g_get_current_dir();
	    filename = g_strconcat(tmp, "/", argv[i], NULL);
	    g_free(tmp);
	}
	opt->filenames = g_list_prepend(opt->filenames, filename);
    }
    opt->filenames = g_list_reverse(opt->filenames);
}

void handle_cmd_line_options(struct cmdlineopt *opt, gboolean remote)
{
    if (opt->session == -1) {
	if (!remote)
	    opt->session = ctrlsocket_get_session_id();
	else
	    opt->session = 0;
    }

    if (opt->filenames != NULL) {
	GList *node;
	int pos = 0;

	if (opt->load_skins) {
	    node = opt->filenames;
	    xmms_remote_set_skin(opt->session, node->data);
	    skin_install_skin(node->data);
	    while (node) {
		g_free(node->data);
		node = g_list_next(node);
	    }
	    g_list_free(opt->filenames);
	} else {
	    if (opt->enqueue && opt->play)
		pos = xmms_remote_get_playlist_length(opt->session);
	    if (!opt->enqueue)
		xmms_remote_playlist_clear(opt->session);
	    xmms_remote_playlist_add(opt->session, opt->filenames);
	    node = opt->filenames;
	    while (node) {
		g_free(node->data);
		node = g_list_next(node);
	    }
	    g_list_free(opt->filenames);
	    if (opt->enqueue && opt->play &&
		xmms_remote_get_playlist_length(opt->session) > pos)
		xmms_remote_set_playlist_pos(opt->session, pos);
	    if (!opt->enqueue)
		xmms_remote_play(opt->session);
	}
    }
    if (opt->rew)
	xmms_remote_playlist_prev(opt->session);
    if (opt->play)
	xmms_remote_play(opt->session);
    if (opt->pause)
	xmms_remote_pause(opt->session);
    if (opt->stop)
	xmms_remote_stop(opt->session);
    if (opt->fwd)
	xmms_remote_playlist_next(opt->session);
    if (opt->play_pause)
	xmms_remote_play_pause(opt->session);
    if (opt->mainwin)
	xmms_remote_main_win_toggle(opt->session, TRUE);
    if (opt->playcd)
	play_medium();
}


void segfault_handler(int sig)
{
    g_printf(_("\nReceived SIGSEGV\n\n"
	       "This could be a bug in BMP. If you don't know why this happened,\n"
	       "please send a mail to bmp-bugs@linux-media.net\n\n"));
    exit(1);
}

static gboolean
pposition_configure(GtkWidget * w, GdkEventConfigure * event,
		    gpointer data)
{
    gint x, y;
    gdk_window_get_deskrelative_origin(w->window, &x, &y);
    if (x != 0 || y != 0)
	pposition_broken = TRUE;
    gtk_widget_destroy(w);

    return FALSE;
}

void check_pposition(void)
{
    GtkWidget *window;
    GdkBitmap *mask;
    GdkGC *gc;
    GdkColor pattern;

    window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    g_signal_connect(G_OBJECT(window), "configure_event",
		     G_CALLBACK(pposition_configure), NULL);
    gtk_widget_set_uposition(window, 0, 0);
    gtk_widget_realize(window);

    gtk_widget_set_usize(window, 1, 1);
    gdk_window_set_decorations(window->window, 0);

    mask = gdk_pixmap_new(window->window, 1, 1, 1);
    gc = gdk_gc_new(mask);
    pattern.pixel = 0;
    gdk_gc_set_foreground(gc, &pattern);
    gdk_draw_rectangle(mask, gc, TRUE, 0, 0, 1, 1);
    gdk_gc_destroy(gc);
    gtk_widget_shape_combine_mask(window, mask, 0, 0);

    gtk_widget_show(window);

    while (g_main_iteration(FALSE));
}

#if 0

static GdkFilterReturn
save_yourself_filter(GdkXEvent * xevent, GdkEvent * event, gpointer data)
{
    Atom save_yourself, protocols;

    save_yourself = gdk_atom_intern("WM_SAVE_YOURSELF", FALSE);
    protocols = gdk_atom_intern("WM_PROTOCOLS", FALSE);

    if (((XEvent *) xevent)->type == ClientMessage) {
	XClientMessageEvent *cme = (XClientMessageEvent *) xevent;
	if (cme->message_type == protocols &&
	    (Atom) cme->data.l[0] == save_yourself) {
	    save_config();
	    XSetCommand(GDK_DISPLAY(),
			GDK_WINDOW_XWINDOW(mainwin->window),
			restart_argv, restart_argc);
	    return GDK_FILTER_REMOVE;
	}
    }

    return GDK_FILTER_CONTINUE;
}
#endif

static void enable_x11r5_session_management(int argc, char **argv)
{
    /*
     * X11R5 Session Management 
     *
     * Most of xmms' options does not make sense when we are
     * restarted, so we drop them all.
     */

    /* GdkAtom save_yourself; */
    /* Atom *temp, *temp2; */
    /* int i, n; */

    restart_argc = 1;
    restart_argv = g_malloc(sizeof(char *));
    restart_argv[0] = g_strdup(argv[0]);

    XSetCommand(GDK_DISPLAY(), GDK_WINDOW_XWINDOW(mainwin->window),
		restart_argv, restart_argc);
#if 0
    save_yourself = gdk_atom_intern("WM_SAVE_YOURSELF", FALSE);

    /*
     * It would be easier if we could call gdk_add_client_message_filter()
     * here but GDK only allows one filter per message and want's to
     * filter "WM_PROTOCOLS" itself.
     */
    gdk_window_add_filter(mainwin->window, save_yourself_filter, NULL);
    if (XGetWMProtocols
	(GDK_DISPLAY(), GDK_WINDOW_XWINDOW(mainwin->window), &temp, &n)) {
	for (i = 0; i < n; i++)
	    if (temp[i] == save_yourself)
		return;
	temp2 = g_new(Atom, n + 1);
	for (i = 0; i < n; i++)
	    temp2[i] = temp[i];
	temp2[i] = save_yourself;
	XSetWMProtocols(GDK_DISPLAY(), GDK_WINDOW_XWINDOW(mainwin->window),
			temp2, n + 1);
	if (n > 0)
	    XFree(temp);
	g_free(temp2);
    } else
	XSetWMProtocols(GDK_DISPLAY(), GDK_WINDOW_XWINDOW(mainwin->window),
			&save_yourself, 1);
#endif
}

/*void controls_playpause(GtkWidget * w, gpointer data) {

	if (get_input_paused()) {
		gtk_button_set_label(data,"Pause");	
	}

	else {
		gtk_button_set_label(data,"Play");
	}


	input_pause();
}
*/



int main(int argc, char **argv)
{
    struct cmdlineopt options;
#if defined(HAVE_SCHED_SETSCHEDULER) && defined(HAVE_SCHED_GET_PRIORITY_MAX)
    struct sched_param sparam;
#endif

    if (!gtk_init_check(&argc, &argv)) {
	if (argc < 2) {
	    g_error("Unable to open display, exiting.");
	    exit(1);
	}
	handle_cmd_line_options(&options, TRUE);
	exit(0);
    }

#ifdef ENABLE_NLS
    bindtextdomain(PACKAGE, LOCALEDIR);
    bind_textdomain_codeset(PACKAGE, "UTF-8");
    textdomain(PACKAGE);
#endif

    signal(SIGPIPE, SIG_IGN);	/* for controlsocket.c */
    signal(SIGSEGV, segfault_handler);
    g_thread_init(NULL);
    if (!g_thread_supported()) {
	g_printf(_("Sorry, threads isn't supported on your platform.\n\n"
		   "If you're on a libc5 based linux system and installed GLIB & GTK+ before you\n"
		   "installed LinuxThreads you need to recompile GLIB & GTK+\n"));
	exit(1);
    }
    parse_cmd_line(argc, argv, &options);

#if defined(HAVE_SRANDOMDEV)
    srandomdev();
#else
    srandom(time(NULL));
#endif

    read_config();

    if (geteuid() == 0) {
#if defined(HAVE_SCHED_SETSCHEDULER) && defined(HAVE_SCHED_GET_PRIORITY_MAX)
	if (cfg.use_realtime) {
	    sparam.sched_priority = sched_get_priority_max(SCHED_RR);
	    sched_setscheduler(0, SCHED_RR, &sparam);
	}
#endif
	setuid(getuid());
    }

    make_beep_dir();
    if (options.session != -1 || !setup_ctrlsocket()) {
	handle_cmd_line_options(&options, TRUE);
	exit(0);
    }
    gdk_rgb_init();
    gtk_widget_set_default_colormap(gdk_rgb_get_cmap());
    gtk_widget_set_default_visual(gdk_rgb_get_visual());

    if ((gtk_major_version == 2) && (gtk_minor_version < 2)) {
	g_printf(_
		 ("Sorry, your GTK+ version (%d.%d.%d) doesn't work with BMP.\n"
		  "Please use GTK+ %s or newer.\n"), gtk_major_version,
		 gtk_minor_version, gtk_micro_version, "2.2.0");
	exit(0);
    }

    check_wm_hints();
    check_pposition();
    mainwin_create();

    /* Plugins might start threads that can call gtk */

    if (!XInitThreads()) {
	g_message("BMP: XInitThreads() failed\n");
	return 1;
    }

    GDK_THREADS_ENTER();

    init_plugins();

    playlistwin_create();

    equalizerwin_create();
    init_skins(cfg.skin);

    create_popups();

    make_skin_window();

    create_prefs_window();
    util_read_menu_rc();
    read_volume(VOLSET_STARTUP);

    playlist_load(g_build_filename
		  (g_get_home_dir(), BMP_RCPATH, "bmp.m3u", NULL));

    playlist_set_position(cfg.playlist_position);
    GDK_THREADS_LEAVE();
    start_ctrlsocket();
    handle_cmd_line_options(&options, FALSE);
    GDK_THREADS_ENTER();
    mainwin_set_info_text();

    gtk_widget_show(mainwin);
    if (pposition_broken && cfg.player_x != -1 && cfg.save_window_position)
	dock_set_uposition(mainwin, cfg.player_x, cfg.player_y);


    if (!cfg.player_visible
	&& (cfg.playlist_visible || cfg.equalizer_visible))
	mainwin_real_hide();
    else
	mainwin_show(TRUE);
    if (cfg.playlist_visible)
	playlistwin_show(TRUE);
    if (cfg.equalizer_visible)
	equalizerwin_show(TRUE);

    draw_main_window(TRUE);

    mainwin_timeout_tag = gtk_timeout_add(10, idle_func, NULL);
    playlist_start_get_info_thread();

    enable_x11r5_session_management(argc, argv);
    sm_init(argc, argv, options.previous_session_id);

/*	GtkWidget * controls = gtk_window_new(GTK_WINDOW_POPUP);
	GtkWidget * button = gtk_button_new_with_label(_("Pause"));
	GtkWidget * box = gtk_vbox_new (FALSE, 0);
	
	gtk_container_add(controls,box);

	GtkRequisition * controls_size;
	gint x,y,width,height,depth;
	
	controls_size->width = 64;
	controls_size->height = 200;
	
	gtk_widget_size_request(controls,controls_size);
	gdk_window_get_geometry(playlistwin->window,&x,&y,&width,&height,&depth);
	
	gtk_window_move(controls,x+width+1,y);
	gtk_box_pack_start(box,button,FALSE,FALSE,0);

	gtk_window_set_decorated(controls,FALSE);	
	gtk_widget_show_all(controls);

	g_signal_connect(button,"clicked",controls_playpause,button);

*/

    gtk_main();
    GDK_THREADS_LEAVE();

    return 0;
}


#if 0
/* xmms.desktop comment */
char *comment = _("X Multimedia System");
#endif
