/* AbiWord
 * Copyright (C) 1998 AbiSource, Inc.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA.
 */

#include "ap_Features.h"

#include "ut_assert.h"
#include "ut_vector.h"
#include "ut_locale.h"
#include "ap_UnixToolbar_StyleCombo.h"
#include "ap_Toolbar_Id.h"
#include "xap_Frame.h"
#include "xap_UnixFontManager.h"
#include "xap_UnixFont.h"
#include "pd_Style.h"
#include "xad_Document.h"
#include "xap_App.h"
#include "ev_UnixToolbar.h"
#include "ut_debugmsg.h"


/*****************************************************************/

int sort_cb(gconstpointer a, gconstpointer b)
{
	return UT_XML_strcmp((const XML_Char*)a, (const XML_Char*)b);
}

EV_Toolbar_Control * AP_UnixToolbar_StyleCombo::static_constructor(EV_Toolbar * pToolbar,
														  XAP_Toolbar_Id id)
{
	AP_UnixToolbar_StyleCombo * p = new AP_UnixToolbar_StyleCombo(pToolbar,id);
	return p;
}

AP_UnixToolbar_StyleCombo::AP_UnixToolbar_StyleCombo(EV_Toolbar * pToolbar,
													 XAP_Toolbar_Id id)
	: EV_Toolbar_Control(pToolbar/*,id*/), 
	  m_pDefaultDesc(NULL)
{
	UT_ASSERT(id==AP_TOOLBAR_ID_FMT_STYLE);

	m_nPixels = 120;		// TODO: do a better calculation
	m_nLimit = 15;         // 15 styles before the scroll bar??.
	m_pFrame = static_cast<AP_UnixFrame *>(static_cast<EV_UnixToolbar *>(pToolbar)->getFrame());
}

AP_UnixToolbar_StyleCombo::~AP_UnixToolbar_StyleCombo(void)
{
	freeStyles();
	pango_font_description_free(m_pDefaultDesc);
}

/*****************************************************************/

bool AP_UnixToolbar_StyleCombo::populate(void)
{
	// clear anything that's already there
	m_vecContents.clear();

	// populate the vector

#if 1
	// HACK: for now, just hardwire it
	// NB if you change the case of the labels, it will stop working
	// unless you also change all the places where the style appears!
	m_vecContents.addItem("Normal");
	m_vecContents.addItem("Heading 1");
	m_vecContents.addItem("Heading 2");
	m_vecContents.addItem("Heading 3");
	m_vecContents.addItem("Plain Text");
	m_vecContents.addItem("Block Text");
#else

	AD_Document * pAD_Doc = m_pFrame->getCurrentDoc();
	if(!pAD_Doc)
	{
		return false;
	}
	PD_Document *pDocument = static_cast<PD_Document *>(pAD_Doc);

	// TODO: need a view/doc pointer to get this right
	// ALSO: will need to repopulate as new styles added
	// HYP:  only call this method from shared code? 
	const char * szName;
	const PD_Style * pStyle;

	for (UT_uint32 k=0; (pDocument->enumStyles(k,&szName,&pStyle)); k++)
	{
		m_vecContents.addItem(szName);
	}
#endif 

	return true;
}


bool AP_UnixToolbar_StyleCombo::repopulate(void)
{
	// repopulate the vector from the current document
    // If ithere is one present

	AD_Document * pAD_Doc = m_pFrame->getCurrentDoc();
	if(!pAD_Doc)
	{
		return false;
	}
	PD_Document *pDocument = static_cast<PD_Document *>(pAD_Doc);


	// clear anything that's already there
	m_vecContents.clear();
	freeStyles();

	// defaults for style combo
	if (m_pDefaultDesc == NULL) {
		XAP_UnixFont *pDefaultFont = XAP_UnixFontManager::pFontManager->getDefaultFont ();
		m_pDefaultDesc = pango_font_description_new ();
		pango_font_description_set_family (m_pDefaultDesc, pDefaultFont->getName ());
		// TODO hardcoded default size
		pango_font_description_set_size (m_pDefaultDesc, 12 * PANGO_SCALE);
		pango_font_description_set_style (m_pDefaultDesc, pDefaultFont->getPangoStyle ());
		pango_font_description_set_weight (m_pDefaultDesc, pDefaultFont->getPangoWeight ());
	}

	const char * szName;
	PD_Style * pStyle;
	GSList *list = NULL;

	for (UT_uint32 k=0; (pDocument->enumStyles(k,&szName,const_cast<const PD_Style**>(&pStyle))); k++)
	{
		if (!pStyle) {
			UT_DEBUGMSG(("no style instance for '%s'\n", szName));
		}

		/* TODO Numbered Headings are also lists, we need to show them tough
		if (pStyle->isList())
			continue;
		*/

		list = g_slist_prepend (list, (char *)szName);

		/* wysiwyg styles are disabled for now 
		PangoFontDescription *desc = pango_font_description_copy (m_pDefaultDesc);
		getPangoAttrs(pStyle, desc);
		m_mapStyles.insert(szName, desc);
		*/
	}

	// Ok, it's a bit hackish to put them in a list for sorting first
	// but somehow the vector's qsort totally failed for me
	if (list) 
	{
		list = g_slist_sort(list, (GCompareFunc)sort_cb);		
		do 
		{
			m_vecContents.addItem((const char *)list->data);

		} while (NULL != (list = g_slist_next(list)));
	}		

	return true;
}

const PangoFontDescription* 
AP_UnixToolbar_StyleCombo::getStyle (const gchar *name) {

	const PangoFontDescription *desc = m_mapStyles.pick(name);
	if (desc == NULL) {
		repopulate();
		desc = m_mapStyles.pick(name);
	}
	return desc;
}

/*!
* \todo ROB parse more attributes like font-color, background-color
*/
void 		 
AP_UnixToolbar_StyleCombo::getPangoAttrs (PD_Style *pStyle, 
										  PangoFontDescription *desc) {

	UT_return_if_fail (pStyle);
	UT_LocaleTransactor t (LC_NUMERIC, "C");

	const XML_Char *value = NULL;

	if (pStyle->getPropertyExpand ("font-family", value)) {
		pango_font_description_set_family (desc, value);
	}

	if (pStyle->getPropertyExpand ("font-size", value)) {
		pango_font_description_set_size (desc, UT_convertToDimension (value, DIM_PT) * PANGO_SCALE);
	}

	if (pStyle->getPropertyExpand ("font-style", value)) {
		PangoStyle style = PANGO_STYLE_NORMAL;
		if (!UT_strcmp (value, "italic"))
			style = PANGO_STYLE_ITALIC;
		pango_font_description_set_style (desc, style);
	}

	if (pStyle->getPropertyExpand ("font-weight", value)) {
		PangoWeight weight = PANGO_WEIGHT_NORMAL;
		if (!UT_strcmp (value, "bold"))
			weight = PANGO_WEIGHT_BOLD;
		pango_font_description_set_weight (desc, weight);
	}
}

void 
AP_UnixToolbar_StyleCombo::freeStyles() {

	UT_GenericVector<PangoFontDescription*> *pVec = m_mapStyles.enumerate();
	for (UT_uint32 i = 0; i < pVec->size(); i++) {
		pango_font_description_free(pVec->getNthItem(i));
	}
	delete pVec;
}
