/*****************************************************************************
 * x11params_ui.c: X11 Parameters Xaw GUI
 *****************************************************************************
 * $Id: x11params_ui.c,v 1.4 2004/09/29 20:28:46 pingus77 Exp $
 *****************************************************************************
 * Copyright (C) 2004 Pingus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <math.h>

#include "config.h"

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/XawInit.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Paned.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Scrollbar.h>

#ifdef HAVE_XPM
# include <X11/xpm.h>
#endif

#include "strtab.h"
#include "xdtv.h"
#include "grab.h"
#include "x11.h"
#include "toolbox.h"
#include "popup_ui.h"
#include "blackborder.h"
#include "divx.h"
#include "channel.h"
#include "vop.h"

#ifdef HAVE_XPM
# include "xdtv.xpm"
#ifdef HAVE_PIXMAPS
# include "../pixmaps/x11_x11options.xpm"
# include "../pixmaps/x11_xvcolorkeyinfo.xpm"
# include "../pixmaps/x11_wmoffby.xpm"
# include "../pixmaps/x11_pixsize.xpm"
# include "../pixmaps/x11_openmainoptions.xpm"
# include "../pixmaps/x11_chooseblackborder.xpm"
# include "../pixmaps/x11_capwidthheight.xpm"
# include "../pixmaps/x11_applysave.xpm"
#endif
#endif

/*-------------------------------------------------------------------------*/
extern void save_cb (Widget widget, XtPointer clientdata, XtPointer call_data);
extern void OptionsAction (Widget, XEvent *, String *, Cardinal *);
extern void action_cb (Widget widget, XtPointer clientdata, XtPointer call_data);
extern Widget app_shell;
extern Widget c_cap;
extern Display *dpy;
extern Atom wm_protocols[2];
extern int debug;
extern int xawpopup;
extern char xawxvmessage [1024];
extern int blackborder;
extern int width_capture, height_capture;
extern int width_capture_max, height_capture_max;
extern int cur_capture;
extern int oldswitch;

extern int fs_width, fs_height;
extern int fs_xoff, fs_yoff;
extern int pix_width, pix_height;
extern char title[256];

static Widget x11params_shell, x11params_paned;
static Widget c_fullscreen_mode;
static Widget c_fullscreen_resolution;
static Widget c_blackborder_mode;
static Widget c_xv_info;
static Widget t_fullscreen_wmoffby_str;
static Widget t_pixmaps_size_str;
static Widget t_xv_colorkey_str;
static Widget l_blackborder=NULL, s_blackborder=NULL;
static Widget l_capt_width, s_capt_height;
static Widget l_capt_height, s_capt_width;
static Widget c_x11;
static Widget c_wmoffby;
static Widget c_pixmaps_size;
static Widget c_gframes;
static Widget c_blackborders;
static Widget c_openmain;
static Widget c_x11paramssave;
static Widget c_autograb;

int x11_fullscreen_resolution;
int x11_blackborder_mode;
char x11_fullscreen_resolution_str[256];
char x11_fullscreen_wmoffby_str[256];
char x11_pixmaps_size_str[256];
char x11_xv_colorkey_str[256];
char * colorkey_str;
char * wmoffby_str;
char * pixmaps_size_str;


#ifdef HAVE_XPM
  Pixmap icon_pixmap;
  Pixmap icon_shapemask;
#endif

/*------------------------------------------------------------------------*/

#define CAPT_WIDTH_MIN        64
// +15 to see the cursor in xaw GUI with xaw lib (no problem with others Xaw3d, XawM, neXtaw etc...
#define CAPT_WIDTH_MAX       783
#define CAPT_HEIGHT_MIN       64
// +15 to see the cursor in xaw GUI with xaw lib (no problem with others Xaw3d, XawM, neXtaw etc...
#define CAPT_HEIGHT_MAX      591 

/*------------------------------------------------------------------------*/

struct CALL_ACTION call_options_4 = { 0, "Options", {NULL} };

/*--------------------------------------------------------------------------*/

enum
{
  X11_FULLSCREEN_MODE,
  X11_FULLSCREEN_RESOLUTION,
  X11_BLACKBORDER_MODE,
  X11_AUTOGRAB
};

static struct STRTAB x11_method_list[] = {
  {0, "New Method"},
  {1, "Old Method"},
  {-1, NULL}
};

static struct STRTAB x11_autograb_list[] = {
  {0, "Off"},
  {1, "On"},
  {-1, NULL}
};

static struct STRTAB blackborder_mode_list[] = {
  {0, "1.33:1 (4:3)"},
  {1, "1.66:1"},  
  {2, "1.78:1 (16:9)"},
  {3, "1.85:1"},
  {4, "2.20:1 (Super Panavision 70)"},  
  {5, "2.35:1 (20:9)"},
  {6, "2.40:1 (Panavision)"},
  {7, "2.55:1 (Cinemascope)"},
  {8, "2.65:1 (Cinerama)"},
  {9, "2.76:1 (Ultra Panavision 70)"},    
  {-1, NULL}
};

static struct STRTAB x11_resolution_list[] = {
  {0, "320 x 200"},
  {1, "320 x 240"},
  {2, "400 x 300"},
  {3, "512 x 384"},
  {4, "640 x 480"},
  {5, "768 x 576"},
  {6, "800 x 600"},
  {7, "1024 x 768"},
  {8, "1152 x 864"},
  {9, "1280 x 1024"},
  {10, "1400 x 1050"},
  {11, "1600 x 1200"},
  {12, "1800 x 1440"},
  {13, "1900 x 1440"},
  {-1, NULL}
};

/*-------------------------------------------------------------------------*/

static void
x11params_menu_cb (Widget widget, XtPointer clientdata, XtPointer call_data)
{
  int cd = (int) clientdata;
  int j;
  char label[64];

  switch (cd)
    {
    case X11_FULLSCREEN_MODE:
      if (-1 != (j = popup_menu (widget, "Fullscreen Mode", x11_method_list)))
        {
          oldswitch = j;
          sprintf (label, "%-" LABEL_WIDTH4 "s: %s", "Fullscreen Mode",
                   int_to_str (oldswitch, x11_method_list));
          XtVaSetValues (c_fullscreen_mode, XtNlabel, label, NULL);
	  if (debug)
	    fprintf(stderr,"Fullscreen mode: %d\n",oldswitch);
        }
      break;
        
    case X11_FULLSCREEN_RESOLUTION:
      if (-1 != (j = popup_menu (widget, "Fullscreen Resolution", x11_resolution_list)))
        {

          x11_fullscreen_resolution = j;
          sprintf (label, "%-" LABEL_WIDTH4 "s: %s", "Fullscreen Resolution",
                   int_to_str (x11_fullscreen_resolution, x11_resolution_list));
          XtVaSetValues (c_fullscreen_resolution, XtNlabel, label, NULL);
          if (debug)
	    fprintf(stderr,"Fullscreen Resolution: %d\n",x11_fullscreen_resolution);
          sscanf (int_to_str (x11_fullscreen_resolution, x11_resolution_list), "%d x %d", &fs_width, &fs_height);
        }
      break;

    case X11_BLACKBORDER_MODE:
      if (-1 != (j = popup_menu (widget, "Blackborder Mode", blackborder_mode_list)))
        {

          x11_blackborder_mode = j;
          sprintf (label, "%-" LABEL_WIDTH "s: %s", "Blackborder Mode",
                   int_to_str (x11_blackborder_mode, blackborder_mode_list));
          XtVaSetValues (c_blackborder_mode, XtNlabel, label, NULL);
          if (debug)
	    fprintf(stderr,"Blackborder Mode: %d\n",x11_blackborder_mode);
	  switch (j)
	  {
	    case 0: // 1.33:1 (4:3)
	    set_ybar_ratio(0);
            if (s_blackborder)
              set_float (s_blackborder, XtNtopOfThumb, 0);
	      return;
	    case 1: // 1.66:1
	    set_ybar_ratio((int)(0.20*YBAR_DEN/2));
            if (s_blackborder)
              set_float (s_blackborder, XtNtopOfThumb, (int)(0.20*YBAR_DEN/2));
	      return;	      
	    case 2: // 1.78:1 (16:9)
	    set_ybar_ratio((int)(0.25*YBAR_DEN/2));
            if (s_blackborder)
              set_float (s_blackborder, XtNtopOfThumb, (int)(0.25*YBAR_DEN/2));
	      return;
	    case 3: // 1.85:1
	    set_ybar_ratio((int)(0.28*YBAR_DEN/2));
            if (s_blackborder)
              set_float (s_blackborder, XtNtopOfThumb, (int)(0.28*YBAR_DEN/2));
	      return;
	    case 4: // 2.20:1 (Super Panavision 70)
	    set_ybar_ratio((int)(0.40*YBAR_DEN/2));
            if (s_blackborder)
              set_float (s_blackborder, XtNtopOfThumb, (int)(0.40*YBAR_DEN/2));
	      return;	      
	    case 5: // 2.35:1 (20:9)
	    set_ybar_ratio((int)(0.43*YBAR_DEN/2));
            if (s_blackborder)
              set_float (s_blackborder, XtNtopOfThumb, (int)(0.43*YBAR_DEN/2));
	      return;     
	    case 6: // 2.40:1 (Panavision)
	    set_ybar_ratio((int)(0.45*YBAR_DEN/2));
            if (s_blackborder)
              set_float (s_blackborder, XtNtopOfThumb, (int)(0.45*YBAR_DEN/2));
	      return;
	    case 7: // 2.55:1 (Cinemascope)
	    set_ybar_ratio((int)(0.48*YBAR_DEN/2));
            if (s_blackborder)
              set_float (s_blackborder, XtNtopOfThumb, (int)(0.48*YBAR_DEN/2));
	      return;
	    case 8: // 2.65:1 (Cinerama)
	    set_ybar_ratio((int)(0.50*YBAR_DEN/2));
            if (s_blackborder)
              set_float (s_blackborder, XtNtopOfThumb, (int)(0.50*YBAR_DEN/2));
	      return;
	    case 9: // 2.76:1 (Ultra Panavision 70)
	    set_ybar_ratio((int)(0.52*YBAR_DEN/2));
            if (s_blackborder)
              set_float (s_blackborder, XtNtopOfThumb, (int)(0.52*YBAR_DEN/2));
	      return;           
	  }
	}
      break;
    case X11_AUTOGRAB:
      if (-1 != (j = popup_menu (widget, "Auto Grabdisplay", x11_autograb_list)))
        {
          vop_autograb = j;
          sprintf (label, "%-" LABEL_WIDTH4 "s: %s", "Auto Grabdisplay",
                   int_to_str (vop_autograb, x11_autograb_list));
          XtVaSetValues (c_autograb, XtNlabel, label, NULL);
	  if (debug)
	    fprintf(stderr,"Auto Grabdisplay: %d\n",vop_autograb);
	  if (!vop_autograb)
	    XtVaSetValues(c_cap, XtNsensitive, True, NULL);
	  else
	    XtVaSetValues(c_cap, XtNsensitive, False, NULL);	    
        }
      break;
      
    default:
      break;
    }
}

static void
x11_jump_scb (Widget widget, XtPointer clientdata, XtPointer call_data)
{
  char *name;
  float value;
  char label[64];

  value = *(float *) call_data;
  name = XtName (XtParent (widget));

  if (0 == strcmp(name, "x11blackborder"))
    {
      set_ybar_ratio(rint(value * (YBAR_DEN/2)));
    }

   if (0 == strcmp ("capt_width", name))
    {
      int width_capture_max_prec = width_capture_max;

      width_capture_max = (int)(value*(CAPT_WIDTH_MAX-CAPT_WIDTH_MIN)+CAPT_WIDTH_MIN + 16) & ~31;

      if(width_capture_max != width_capture_max_prec) {
	video_new_size ();

	if (s_capt_width)
	  set_float (s_capt_width, XtNtopOfThumb,
		     (float)(width_capture_max-CAPT_WIDTH_MIN)/
		     (CAPT_WIDTH_MAX-CAPT_WIDTH_MIN));
	  sprintf (label, "%-" LABEL_WIDTH "s: %4d", "Cap. Width", width_capture_max);
	if (l_capt_width)
	  XtVaSetValues (l_capt_width, XtNlabel, label, NULL);
      }
    }
   if (0 == strcmp ("capt_height", name))
    {
      int height_capture_max_prec = height_capture_max;

      height_capture_max = (int)(value*(CAPT_HEIGHT_MAX-CAPT_HEIGHT_MIN)+CAPT_HEIGHT_MIN + 16) & ~31;

      if(height_capture_max != height_capture_max_prec) {
	video_new_size ();
	if (s_capt_height)
	  set_float (s_capt_height, XtNtopOfThumb,
		     (float)(height_capture_max-CAPT_HEIGHT_MIN)/
		     (CAPT_HEIGHT_MAX-CAPT_HEIGHT_MIN));
	  sprintf (label, "%-" LABEL_WIDTH "s: %4d", "Cap. Height", height_capture_max);
	if (l_capt_height)
	  XtVaSetValues (l_capt_height, XtNlabel, label, NULL);
      }
    }
}



static void
x11_scroll_scb (Widget widget, XtPointer clientdata, XtPointer call_data)
{
  int move = (int) call_data;
  Dimension length;
  float shown, top1, top2;
  char *name;
  float step = 1.0;
  float range = 1.0;

  name = XtName (XtParent (widget));

  XtVaGetValues (widget,
                 XtNlength, &length,
                 XtNshown, &shown, XtNtopOfThumb, &top1, NULL);

  if (strcmp(name, "x11blackborder") == 0)
    {
      range = (float)(YBAR_DEN/2);
      step = 1.0;
    }
  else if (strcmp (name,"capt_width") == 0)
    {
      range = (float)(CAPT_WIDTH_MAX-CAPT_WIDTH_MIN);
      step = 32.0;
    }
else if (strcmp (name,"capt_height") == 0)
    {
      range = (float)(CAPT_HEIGHT_MAX-CAPT_HEIGHT_MIN);
      step = 32.0;

    }

  if (move > 0)
    top2 = top1 + step/range;
  else
    top2 = top1 - step/range;
  if (top2 < 0.0)
    top2 = 0.0;
  if (top2 > 1.0)
    top2 = 1.0;

  x11_jump_scb (widget, clientdata, &top2);

}

/*-------------------------------------------------------------------------*/

void
set_capture_size (int width, int height)
{
  char label[64];

  if (s_capt_width)
    set_float(s_capt_width, XtNtopOfThumb, (float) (width - CAPT_WIDTH_MIN)/
	    (CAPT_WIDTH_MAX - CAPT_WIDTH_MIN));
  sprintf (label, "%-" LABEL_WIDTH "s: %4d", "Cap. Width",
           width);
  if (l_capt_width)
    XtVaSetValues (l_capt_width, XtNlabel, label, NULL);

  if (s_capt_height)
    set_float(s_capt_height, XtNtopOfThumb, (float) (height - CAPT_HEIGHT_MIN)/
	    (CAPT_HEIGHT_MAX - CAPT_HEIGHT_MIN));
  sprintf (label, "%-" LABEL_WIDTH "s: %4d", "Cap. Height",
           height);
  if (l_capt_height)
    XtVaSetValues (l_capt_height, XtNlabel, label, NULL);
}

void
set_blackborder (int j)
{
  if (cur_capture != CAPTURE_GRABDISPLAY)
    return;

  if (is_divx_in_progress())
    return;

  switch (j)
    {
    case 0:
      set_ybar_ratio(get_ybar_ratio()-1);
      break;

    case 1:
      set_ybar_ratio(get_ybar_ratio()+1);
      break;
    }
    
  sprintf (title, "BlackBorder: %d",get_ybar_ratio());  
  set_timer_title ();    
}

void update_blackborder(void)
{
  char label[64];
  static int oldy= -1 ;
  int y = get_ybar_ratio();

  if(y==oldy)
    return;

  oldy = y;
  sprintf(label, "%-" LABEL_WIDTH "s: %d", "Blackborder", get_ybar_ratio());
  if (l_blackborder)
    XtVaSetValues (l_blackborder, XtNlabel, label, NULL);

  if (get_ybar_ratio() == 0)
    sprintf (label, "%-" LABEL_WIDTH "s: %s", "Blackborder Mode","1.33:1 (4:3)");
  else if (get_ybar_ratio() == (int)(0.20*YBAR_DEN/2))
    sprintf (label, "%-" LABEL_WIDTH "s: %s", "Blackborder Mode","1.66:1");    
  else if (get_ybar_ratio() == (int)(0.25*YBAR_DEN/2))
    sprintf (label, "%-" LABEL_WIDTH "s: %s", "Blackborder Mode","1.78:1 (16:9)");
  else if (get_ybar_ratio() == (int)(0.28*YBAR_DEN/2))
    sprintf (label, "%-" LABEL_WIDTH "s: %s", "Blackborder Mode","1.85:1");    
  else if (get_ybar_ratio() == (int)(0.40*YBAR_DEN/2))
    sprintf (label, "%-" LABEL_WIDTH "s: %s", "Blackborder Mode","2.20:1 (Super Panavision 70)");  
  else if (get_ybar_ratio() == (int)(0.43*YBAR_DEN/2))
    sprintf (label, "%-" LABEL_WIDTH "s: %s", "Blackborder Mode","2.35:1 (20:9)");
  else if (get_ybar_ratio() == (int)(0.45*YBAR_DEN/2))
    sprintf (label, "%-" LABEL_WIDTH "s: %s", "Blackborder Mode","2.40:1 (Panavision)");
  else if (get_ybar_ratio() == (int)(0.48*YBAR_DEN/2))
    sprintf (label, "%-" LABEL_WIDTH "s: %s", "Blackborder Mode","2.55:1 (Cinemascope)");
  else if (get_ybar_ratio() == (int)(0.50*YBAR_DEN/2))
    sprintf (label, "%-" LABEL_WIDTH "s: %s", "Blackborder Mode","2.65:1 (Cinerama)");            
  else if (get_ybar_ratio() == (int)(0.52*YBAR_DEN/2))
    sprintf (label, "%-" LABEL_WIDTH "s: %s", "Blackborder Mode","2.76:1 (Ultra Panavision 70)");                    
  else
    sprintf (label, "%-" LABEL_WIDTH "s: %s", "Blackborder Mode","-");

  if (c_blackborder_mode)
    XtVaSetValues (c_blackborder_mode, XtNlabel, label, NULL);

  if (s_blackborder)
    set_float (s_blackborder, XtNtopOfThumb, (float)y*2/YBAR_DEN);

  if (!l_blackborder || !s_blackborder) oldy=-1;  
}

/*--------------------------------------------------------------------------*/

static void
close_x11paramswin (Widget widget, XtPointer clientdata, XtPointer call_data)
{
  XtCallActionProc (x11params_shell, "x11Params", NULL, NULL, 0);
}

static void
xv_info (Widget widget, XtPointer clientdata, XtPointer call_data)
{
  PopupMessage(xawxvmessage);
}

void
x11ParamsAction (Widget widget, XEvent * event, String * params, Cardinal * num_params)
{
  static int mapped = 0, first = 1;

  if (event && event->type == ClientMessage)
    {
      if (event->xclient.data.l[0] == wm_protocols[1])
        {
          if (debug)
            fprintf (stderr, "Config: wm_save_yourself\n");
          XSetCommand (XtDisplay (x11params_shell), XtWindow (x11params_shell), NULL, 0);
          return;
        }
    }

  if (x11params_shell == NULL)
    {
      fprintf(stderr, "x11ParamsAction: ooops: x11params_shell is NULL\n");
            if (xawpopup)
              PopupMessage("x11ParamsAction:\nooops: x11params_shell is NULL\n");
      return;
    }

  if (mapped)
    {
      XtPopdown (x11params_shell);
      mapped = 0;
    }
  else
    {
      XtPopup (x11params_shell, XtGrabNone);
      mapped = 1;
      if (first)
        {
          XSetWMProtocols (XtDisplay (x11params_shell), XtWindow (x11params_shell), wm_protocols, 2);
          first = 0;
        }
    }
}

/*--------------------------------------------------------------------------*/

void
create_x11paramswin()
{
  Widget c,p;
  char label[64];
#ifdef HAVE_XPM  
  Pixmap icon_pixmap;
  Pixmap icon_shapemask;
#endif  

  x11params_shell = XtVaAppCreateShell ("x11Params", "XdTV",
                                   topLevelShellWidgetClass,
                                   dpy, XtNclientLeader, app_shell, XtNvisual,vinfo->visual,
				   XtNcolormap,colormap, XtNdepth,vinfo->depth,
				   NULL);

  x11params_paned =
    XtVaCreateManagedWidget ("paned", panedWidgetClass, x11params_shell, NULL);


  /* x11 Label */
  c_x11 = XtVaCreateManagedWidget ("lx11", labelWidgetClass, x11params_paned,
                           PANED_FIX, NULL);
  
  /* Fullscreen Mode Menu */
  c_fullscreen_mode = c = XtVaCreateManagedWidget ("fullscreen_mode", commandWidgetClass, x11params_paned,
                               PANED_FIX, NULL);
  XtAddCallback (c, XtNcallback, x11params_menu_cb, (XtPointer) X11_FULLSCREEN_MODE);
  sprintf (label, "%-" LABEL_WIDTH4 "s: %s", "Fullscreen Mode",
           int_to_str (oldswitch, x11_method_list));
  XtVaSetValues (c, XtNlabel, label, NULL);
  
  /* Fullscreen Resolution Menu */
  c_fullscreen_resolution = c = XtVaCreateManagedWidget ("fullscreen_resolution", commandWidgetClass, x11params_paned,
                               PANED_FIX, NULL);
  if (fs_width == 0 || fs_height == 0)
  {
    fs_width = 640;
    fs_height = 480;
  }
  sprintf (x11_fullscreen_resolution_str, "%d x %d", fs_width, fs_height);
  x11_fullscreen_resolution = str_to_int (x11_fullscreen_resolution_str, x11_resolution_list);
  XtAddCallback (c, XtNcallback, x11params_menu_cb, (XtPointer) X11_FULLSCREEN_RESOLUTION);
  sprintf (label, "%-" LABEL_WIDTH4 "s: %s", "Fullscreen Resolution",
           int_to_str (x11_fullscreen_resolution, x11_resolution_list));
  XtVaSetValues (c, XtNlabel, label, NULL);

  /* Auto Grabdisplay Menu */
  c_autograb = c = XtVaCreateManagedWidget ("Auto_Grabdisplay", commandWidgetClass, x11params_paned,
                               PANED_FIX, NULL);
  XtAddCallback (c, XtNcallback, x11params_menu_cb, (XtPointer) X11_AUTOGRAB);
  sprintf (label, "%-" LABEL_WIDTH4 "s: %s", "Auto Grabdisplay",
           int_to_str (vop_autograb, x11_autograb_list));
  XtVaSetValues (c, XtNlabel, label, NULL);  

  /* wm-off-by Label */
  c_wmoffby = XtVaCreateManagedWidget ("lwmoffby", labelWidgetClass, x11params_paned,
                           PANED_FIX, NULL);
  t_fullscreen_wmoffby_str = c = XtVaCreateManagedWidget ("wmoffby", asciiTextWidgetClass, x11params_paned,
                                                PANED_FIX, NULL);
  sprintf (x11_fullscreen_wmoffby_str, "%d x %d", fs_xoff, fs_yoff);
  XtVaSetValues (t_fullscreen_wmoffby_str, XtNstring, x11_fullscreen_wmoffby_str, NULL);
  if (x11_fullscreen_wmoffby_str == NULL) strcpy(x11_fullscreen_wmoffby_str,"0 x 0");


  /* Pixmaps Channel Size Label */
  c_pixmaps_size = XtVaCreateManagedWidget ("lpixmaps_size", labelWidgetClass, x11params_paned,
                           PANED_FIX, NULL);

  t_pixmaps_size_str = c = XtVaCreateManagedWidget ("pixmaps_size", asciiTextWidgetClass, x11params_paned,
                                                PANED_FIX, NULL);
  sprintf (x11_pixmaps_size_str, "%d x %d", pix_width, pix_height);
  XtVaSetValues (t_pixmaps_size_str, XtNstring, x11_pixmaps_size_str, NULL);
  if (x11_pixmaps_size_str == NULL) strcpy(x11_pixmaps_size_str,"128 x 96");


  /* XV Colorkey Label + Info */
  c_xv_info = c = XtVaCreateManagedWidget ("lxv_info", commandWidgetClass, x11params_paned, PANED_FIX, NULL);
  XtAddCallback (c, XtNcallback, xv_info, (XtPointer) NULL);

  t_xv_colorkey_str = c = XtVaCreateManagedWidget ("xv_colorkey", asciiTextWidgetClass, x11params_paned,
                                                PANED_FIX, NULL);
  sprintf (x11_xv_colorkey_str, "%d", get_colorkey(dpy));
  XtVaSetValues (t_xv_colorkey_str, XtNstring, x11_xv_colorkey_str, NULL);


  /* Capt Width / Height Label */
  c_gframes = XtVaCreateManagedWidget ("lgframes", labelWidgetClass, x11params_paned,
                           PANED_FIX, NULL);

  /* Capt_width Slider */
 p =
   XtVaCreateManagedWidget ("capt_width", panedWidgetClass, x11params_paned,
                               XtNorientation, XtEvertical, PANED_FIX, NULL);

  l_capt_width = XtVaCreateManagedWidget ("lcapt_width", labelWidgetClass, p,
                               XtNshowGrip, False, NULL);

  s_capt_width = XtVaCreateManagedWidget ("scapt_width", scrollbarWidgetClass, p,
					  PANED_FIX, NULL);

  set_float (s_capt_width, XtNtopOfThumb,
	     (float)(width_capture_max-CAPT_WIDTH_MIN)/
	     (CAPT_WIDTH_MAX-CAPT_WIDTH_MIN));

  sprintf (label, "%-" LABEL_WIDTH "s: %4d", "Cap. Width", width_capture_max);
  XtVaSetValues (l_capt_width, XtNlabel, label, NULL);

  if(width_capture>0) {
    XtVaSetValues(l_capt_width, XtNsensitive, False, NULL);
    XtVaSetValues(s_capt_width, XtNsensitive, False, NULL);
  }

  XtAddCallback (s_capt_width, XtNjumpProc, x11_jump_scb, NULL);
  XtAddCallback (s_capt_width, XtNscrollProc, x11_scroll_scb, NULL);


  /* Capt_height Slider */
  p =
    XtVaCreateManagedWidget ("capt_height", panedWidgetClass, x11params_paned,
                               XtNorientation, XtEvertical, PANED_FIX, NULL);

  l_capt_height = XtVaCreateManagedWidget ("lcapt_height", labelWidgetClass, p,
                               XtNshowGrip, False, NULL);

  s_capt_height = XtVaCreateManagedWidget ("scapt_height", scrollbarWidgetClass, p,
					  PANED_FIX, NULL);

  set_float (s_capt_height, XtNtopOfThumb,
	     (float)(height_capture_max-CAPT_HEIGHT_MIN)/
	     (CAPT_HEIGHT_MAX-CAPT_HEIGHT_MIN));

  sprintf (label, "%-" LABEL_WIDTH "s: %4d", "Cap. Height", height_capture_max);
  XtVaSetValues (l_capt_height, XtNlabel, label, NULL);

  if(height_capture>0) {
    XtVaSetValues(l_capt_height, XtNsensitive, False, NULL);
    XtVaSetValues(s_capt_height, XtNsensitive, False, NULL);
  }

  XtAddCallback (s_capt_height, XtNjumpProc, x11_jump_scb, NULL);
  XtAddCallback (s_capt_height, XtNscrollProc, x11_scroll_scb, NULL);


  /* Blackborder Label */
  c_blackborders = XtVaCreateManagedWidget ("l_blackborder", labelWidgetClass, x11params_paned,
                           PANED_FIX, NULL);

  /* Blackborder Mode Menu */
  c_blackborder_mode = c = XtVaCreateManagedWidget ("Blackborder_Mode", commandWidgetClass, x11params_paned,
                               PANED_FIX, NULL);
  XtAddCallback (c, XtNcallback, x11params_menu_cb, (XtPointer) X11_BLACKBORDER_MODE);
  if (get_ybar_ratio() == 0)
    sprintf (label, "%-" LABEL_WIDTH "s: %s", "Blackborder Mode","1.33:1 (4:3)");
  else if (get_ybar_ratio() == (int)(0.20*YBAR_DEN/2))
    sprintf (label, "%-" LABEL_WIDTH "s: %s", "Blackborder Mode","1.66:1");    
  else if (get_ybar_ratio() == (int)(0.25*YBAR_DEN/2))
    sprintf (label, "%-" LABEL_WIDTH "s: %s", "Blackborder Mode","1.78:1 (16:9)");
  else if (get_ybar_ratio() == (int)(0.28*YBAR_DEN/2))
    sprintf (label, "%-" LABEL_WIDTH "s: %s", "Blackborder Mode","1.85:1");    
  else if (get_ybar_ratio() == (int)(0.40*YBAR_DEN/2))
    sprintf (label, "%-" LABEL_WIDTH "s: %s", "Blackborder Mode","2.20:1 (Super Panavision 70)");  
  else if (get_ybar_ratio() == (int)(0.43*YBAR_DEN/2))
    sprintf (label, "%-" LABEL_WIDTH "s: %s", "Blackborder Mode","2.35:1 (20:9)");
  else if (get_ybar_ratio() == (int)(0.45*YBAR_DEN/2))
    sprintf (label, "%-" LABEL_WIDTH "s: %s", "Blackborder Mode","2.40:1 (Panavision)");
  else if (get_ybar_ratio() == (int)(0.48*YBAR_DEN/2))
    sprintf (label, "%-" LABEL_WIDTH "s: %s", "Blackborder Mode","2.55:1 (Cinemascope)");
  else if (get_ybar_ratio() == (int)(0.50*YBAR_DEN/2))
    sprintf (label, "%-" LABEL_WIDTH "s: %s", "Blackborder Mode","2.65:1 (Cinerama)");            
  else if (get_ybar_ratio() == (int)(0.52*YBAR_DEN/2))
    sprintf (label, "%-" LABEL_WIDTH "s: %s", "Blackborder Mode","2.76:1 (Ultra Panavision 70)");                    
  else
    sprintf (label, "%-" LABEL_WIDTH "s: %s", "Blackborder Mode","-");
  XtVaSetValues (c, XtNlabel, label, NULL);


  /* Blackborder Slider */
  p =
    XtVaCreateManagedWidget ("x11blackborder", panedWidgetClass, x11params_paned,
                             XtNorientation, XtEvertical, PANED_FIX, NULL);

  l_blackborder = XtVaCreateManagedWidget ("lblackborder", labelWidgetClass, p,
                               XtNshowGrip, False, NULL);

  s_blackborder = XtVaCreateManagedWidget ("sblackborder", scrollbarWidgetClass, p,
                                          PANED_FIX, NULL);
  update_blackborder();
  XtAddCallback (s_blackborder, XtNjumpProc, x11_jump_scb, NULL);
  XtAddCallback (s_blackborder, XtNscrollProc, x11_scroll_scb, NULL);

    
/* Open Main Options Menu */
  c_openmain = XtVaCreateManagedWidget ("lopenmain", commandWidgetClass, x11params_paned, PANED_FIX, NULL);
    XtAddCallback (c_openmain, XtNcallback, action_cb, (XtPointer) & call_options_4);  
  
    
/* Button to close x11 Parameters GUI */

  c_x11paramssave = c = XtVaCreateManagedWidget ("lx11paramssave", commandWidgetClass, x11params_paned, PANED_FIX, NULL);
  XtAddCallback (c, XtNcallback, save_cb, (XtPointer) NULL);
  XtAddCallback (c, XtNcallback, close_x11paramswin, (XtPointer) NULL);

#ifdef HAVE_XPM
  XpmCreatePixmapFromData(XtDisplay(x11params_shell),
                          RootWindowOfScreen(XtScreen(x11params_shell)),
                          icon_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(x11params_shell, XtNiconPixmap, icon_pixmap, NULL);
#ifdef HAVE_PIXMAPS  
  XpmCreatePixmapFromData(XtDisplay(x11params_paned),
                          RootWindowOfScreen(XtScreen(x11params_paned)),
                          x11_x11options_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_x11, XtNbitmap, icon_pixmap, NULL);
  XpmCreatePixmapFromData(XtDisplay(x11params_paned),
                          RootWindowOfScreen(XtScreen(x11params_paned)),
                          x11_wmoffby_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_wmoffby, XtNbitmap, icon_pixmap, NULL); 
  XpmCreatePixmapFromData(XtDisplay(x11params_paned),
                          RootWindowOfScreen(XtScreen(x11params_paned)),
                          x11_pixsize_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_pixmaps_size, XtNbitmap, icon_pixmap, NULL);   
  XpmCreatePixmapFromData(XtDisplay(x11params_paned),
                          RootWindowOfScreen(XtScreen(x11params_paned)),
                          x11_xvcolorkeyinfo_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_xv_info, XtNbitmap, icon_pixmap, NULL);  
  XpmCreatePixmapFromData(XtDisplay(x11params_paned),
                          RootWindowOfScreen(XtScreen(x11params_paned)),
                          x11_capwidthheight_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_gframes, XtNbitmap, icon_pixmap, NULL);    
  XpmCreatePixmapFromData(XtDisplay(x11params_paned),
                          RootWindowOfScreen(XtScreen(x11params_paned)),
                          x11_chooseblackborder_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_blackborders, XtNbitmap, icon_pixmap, NULL);    
  XpmCreatePixmapFromData(XtDisplay(x11params_paned),
                          RootWindowOfScreen(XtScreen(x11params_paned)),
                          x11_openmainoptions_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_openmain, XtNbitmap, icon_pixmap, NULL);        
  XpmCreatePixmapFromData(XtDisplay(x11params_paned),
                          RootWindowOfScreen(XtScreen(x11params_paned)),
                          x11_applysave_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_x11paramssave, XtNbitmap, icon_pixmap, NULL);    
#endif  
#endif

}

void update_colorkey(void)
{
  int i;
  if(!t_xv_colorkey_str) return;
  XtVaGetValues (t_xv_colorkey_str, XtNstring, &colorkey_str, NULL);
  sscanf (colorkey_str, "%d", &i);
  if (debug)
    fprintf(stderr,"colorkey = %s\n",colorkey_str);
  set_colorkey(dpy, i);
}

void update_wmoffby(void)
{
  if(!t_fullscreen_wmoffby_str) return;
  XtVaGetValues (t_fullscreen_wmoffby_str, XtNstring, &wmoffby_str, NULL);
  if (debug)
    fprintf(stderr,"wmoffby = %s\n",wmoffby_str);
  if (2 != sscanf (wmoffby_str, "%d x %d", &fs_xoff, &fs_yoff))
  fs_xoff = fs_yoff = 0;
  if (debug)
    fprintf(stderr,"fs_xoff & fs_yoff = %d & %d\n",fs_xoff, fs_yoff);
}

void update_pixmaps_size(void)
{
  if(!t_pixmaps_size_str) return;
  XtVaGetValues (t_pixmaps_size_str, XtNstring, &pixmaps_size_str, NULL);
  if (debug)
    fprintf(stderr,"pixmaps_size = %s\n",pixmaps_size_str);
  if (2 != sscanf (pixmaps_size_str, "%d x %d", &pix_width, &pix_height))
  {
    pix_width = 128;
    pix_height = 96;
  }
  if (debug)
    fprintf(stderr,"pix_width & pix_height = %d & %d\n",pix_width, pix_height);
}
