/*****************************************************************************
 * divx.c: DivX recording
 *****************************************************************************
 * $Id: divx.c,v 1.112 2004/12/05 18:41:35 pingus77 Exp $
 *****************************************************************************
 * Copyright (C) 2001 Keuleu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <pthread.h>
#include <math.h>
#include <sys/time.h>
#include "config.h"
#include "audio.h"
#include "writefile.h"
#include "aop.h"
#include "mixer.h"

#include "divx.h"
#include "avilib.h"
#include "popup_ui.h"

#include <X11/Intrinsic.h>
#include "strtab.h"
#include "channel.h"
#include "blackborder.h"
#include "memcpy.h"
#include "grab.h"
#include "divx_ui.h"
#include "http.h"

#ifdef HAVE_DIVX4LINUX
# include <encore2.h>
#endif

#ifdef HAVE_FFMPEG
#if LIBAVCODEC_BUILD < 4641
#  error we dont support libavcodec prior to build 4641, get the latest libavcodec CVS
#endif
#endif

#ifdef HAVE_XVID
# include <xvid.h>
#ifdef XVID_API
#define HAVE_XVID10 /* XVID 1.0 */
#else
#define HAVE_XVID09 /* XVID 0.9 */
#endif
#endif

extern struct GRABBER *grabbers[],grab_avi;
extern int grabber;
extern void avi_wait_write_audio(void);
extern int fs;
extern char channel_title[256];

#undef  NOAUDIO
#ifndef NOAUDIO
#define SYNCHRO
#endif

void video_get_size(int *w, int *h);
void video_set_size(int w, int h, int force);
void video_get_capture_size(int *w, int *h);
void video_set_capture_size(int w, int h);
void video_lock_size(void);
void video_unlock_size(void);
void set_capture_temp_grab(int grab, int mask);

divx_t divx = {
#if defined (HAVE_XVID)
 XVID,         // codec to use
#elif defined (HAVE_FFMPEG)
 FFMPEG_MPEG4, // codec to use
#elif defined (HAVE_DIVX4LINUX)
 DIVX4LINUX,   // codec to use
#else
 UNCOMPRESSED,
#endif
    384, // width
    288, // height
 800000, // bitrate (bits/second)
      1, // quality
      2, // min_quantizer
      8, // max_quantizer
  4*441, // audio_buffer_size
     48, // audio_fragments
   2048, // audio_sizefragment
      0, // stereo
#ifdef HAVE_LAME
      1, // compress_audio
#else
      0, // compress_audio
#endif
    128, // mp3_bitrate
      5, // mp3_quality
      0, // mp3_vbr
      8, // mp3_vbr_quality
  25000, // fps * 1000
   0.08, // maxgap between audio and video in seconds.
   0.00, // delay between audio and video
      1, // display_frames
      0, // sub
      0, // chg
   NULL, // filename
      0, // stream
  63427, // http_port
};
static int       slw; //status of the last AVI_write
static int displaynextframe = 1;

extern int debug;

char xawmessage [1024];
char xawmesstmp [256];
char lastfilename[256] = {'\0'};
char *pplayer = "mplayer -nofs -quiet -nosound ";
extern int xawpopup;

#ifndef NOAUDIO
/*****************************************************************************
 *                       RAW AUDIO DEFINITIONS
 ****************************************************************************/
#define DEC_FREQ    44100

static pthread_t audio_thread;
static int       achans;  // nb of audio channels
static int       aformat = 0; // Audio format
static unsigned long audio_frames, audio_buffers;
static int       fps_num, fps_den;

static void *    audio_record(void *);

/*****************************************************************************
 *                       MP3 AUDIO DEFINITIONS
 ****************************************************************************/
#ifdef HAVE_LAME

#include "lame/lame.h"

static lame_global_flags *gfp = NULL;
static char *    lame_buffer = NULL;
static int       lame_buffer_size = 0;

inline static int       lame_encode(short int *raw_buffer, int samples_per_channel);
static void      lame_stop(void);

#endif // HAVE_LAME

#ifdef SYNCHRO
static long long nbsamplesaudios;
static unsigned long dropped_audio_buffers, dropped_video_frames;
static long long dropped_audio_samples;
extern double videostampmin,videostampmax;
double audiostamp=0.0;
#endif // SYNCHRO
#endif // NOAUDIO


/*****************************************************************************
 *                       DIVX4 VIDEO DEFINITIONS
 ****************************************************************************/
static void *    handle = NULL;

#ifdef HAVE_DIVX4LINUX
# if ENCORE_VERSION >= 20021024
   static DivXBitmapInfoHeader *format =NULL;
   static SETTINGS *settings = NULL;
# else
   static ENC_PARAM param;
# endif
static ENC_FRAME frame;
#endif

#ifdef HAVE_FFMPEG
static AVCodec *codec = NULL;
static AVCodecContext *codec_context = NULL;
static AVFrame *picture_420p = NULL;
#endif
#ifdef HAVE_XVID
#ifdef HAVE_XVID09
static XVID_ENC_FRAME  xframe;
static XVID_ENC_PARAM  xparam;
static int const xmotion_presets[7] = {
        0,                                                        // Q 0
        PMV_EARLYSTOP16,                                          // Q 1
        PMV_EARLYSTOP16,                                          // Q 2
        PMV_EARLYSTOP16 | PMV_HALFPELREFINE16,                    // Q 3
        PMV_EARLYSTOP16 | PMV_HALFPELREFINE16,                    // Q 4
        PMV_EARLYSTOP16 | PMV_HALFPELREFINE16 | PMV_EARLYSTOP8 |  // Q 5
        PMV_HALFPELREFINE8,
        PMV_EARLYSTOP16 | PMV_HALFPELREFINE16 | PMV_EXTSEARCH16 | // Q 6
        PMV_USESQUARES16 | PMV_EARLYSTOP8 | PMV_HALFPELREFINE8
};
static int const xgeneral_presets[7] = {
        XVID_H263QUANT,                               // Q 0
        XVID_MPEGQUANT,                               // Q 1
        XVID_H263QUANT,                               // Q 2
        XVID_H263QUANT | XVID_HALFPEL,                // Q 3
        XVID_H263QUANT | XVID_HALFPEL | XVID_INTER4V, // Q 4
        XVID_H263QUANT | XVID_HALFPEL | XVID_INTER4V, // Q 5
        XVID_H263QUANT | XVID_HALFPEL | XVID_INTER4V  // Q 6
};
#else /* XVID 1.0 */
static xvid_enc_frame_t  xframe;
static xvid_enc_create_t xparam;
static xvid_enc_plugin_t xparam_plugin;
static xvid_enc_stats_t xstats;
static int const xmotion_presets[7] = {
  /* quality 0 */
  0,
  /* quality 1 */
  XVID_ME_ADVANCEDDIAMOND16,
  /* quality 2 */
  XVID_ME_ADVANCEDDIAMOND16 | XVID_ME_HALFPELREFINE16,
  /* quality 3 */
  XVID_ME_ADVANCEDDIAMOND16 | XVID_ME_HALFPELREFINE16 |
  XVID_ME_ADVANCEDDIAMOND8 | XVID_ME_HALFPELREFINE8,
  /* quality 4 */
  XVID_ME_ADVANCEDDIAMOND16 | XVID_ME_HALFPELREFINE16 |
  XVID_ME_ADVANCEDDIAMOND8 | XVID_ME_HALFPELREFINE8 |
  XVID_ME_CHROMA_PVOP | XVID_ME_CHROMA_BVOP,
  /* quality 5 */
  XVID_ME_ADVANCEDDIAMOND16 | XVID_ME_HALFPELREFINE16 |
  XVID_ME_ADVANCEDDIAMOND8 | XVID_ME_HALFPELREFINE8 |
  XVID_ME_CHROMA_PVOP | XVID_ME_CHROMA_BVOP,
  /* quality 6 */
  XVID_ME_ADVANCEDDIAMOND16 | XVID_ME_HALFPELREFINE16 | XVID_ME_EXTSEARCH16 |
  XVID_ME_ADVANCEDDIAMOND8 | XVID_ME_HALFPELREFINE8 | XVID_ME_EXTSEARCH8 |
  XVID_ME_CHROMA_PVOP | XVID_ME_CHROMA_BVOP,
};
static int const xvop_presets[7] = {
  /* quality 0 */
  0,
  /* quality 1 */
  0,
  /* quality 2 */
  XVID_VOP_HALFPEL,
  /* quality 3 */
  XVID_VOP_HALFPEL | XVID_VOP_INTER4V,
  /* quality 4 */
  XVID_VOP_HALFPEL | XVID_VOP_INTER4V,
  /* quality 5 */
  XVID_VOP_HALFPEL | XVID_VOP_INTER4V |
  XVID_VOP_TRELLISQUANT,
  /* quality 6 */
  XVID_VOP_HALFPEL | XVID_VOP_INTER4V |
  XVID_VOP_TRELLISQUANT | XVID_VOP_HQACPRED,
};
#endif
#endif /* HAVE_XVID*/

/************** conversion format tables *****************/
#ifdef HAVE_DIVX4LINUX
static video_fmt* fmts_conv_divx=NULL;
static video_fmt fmts_divx[]={
  VIDEO_YVU420,
  VIDEO_YUYV,
  VIDEO_RGB24B,
  VIDEO_RGB32B,
  0};
#if ENCORE_VERSION >= 20021024
#define FOURCC(A, B, C, D) ( ((uint8_t) (A)) | (((uint8_t) (B))<<8) | (((uint8_t) (C))<<16) | (((uint8_t) (D))<<24) )
static int fmts_xaw_to_biCompression[MAX_VIDEO_FMT] = {
  [VIDEO_YVU420] = FOURCC('Y','V','1','2'),
  [VIDEO_YUYV]   = FOURCC('Y','U','Y','2'),
  [VIDEO_RGB24B] = 0,
  [VIDEO_RGB32B] = 0,
};
static int fmts_xaw_to_biBitCount[MAX_VIDEO_FMT] = {
  [VIDEO_YVU420] = 0,
  [VIDEO_YUYV]   = 0,
  [VIDEO_RGB24B] = 24,
  [VIDEO_RGB32B] = 32,
};
#else
static int fmts_xaw_to_divxcolorspace[MAX_VIDEO_FMT]= {
  [VIDEO_YVU420] = ENC_CSP_YV12,
  [VIDEO_YUYV]   = ENC_CSP_YUY2,
  [VIDEO_RGB24B] = ENC_CSP_RGB24,
  [VIDEO_RGB32B] = ENC_CSP_RGB32,
};
#endif
#endif

#ifdef HAVE_XVID
static video_fmt fmts_xvid[]={
  VIDEO_RGB15,
  VIDEO_RGB16,
#ifdef HAVE_XVID09
  VIDEO_RGB32B,
  VIDEO_RGB24B,
#else
  VIDEO_RGB32,
  VIDEO_RGB24,
#endif
  VIDEO_YUYV,
  VIDEO_YVU420,
  0};
static video_fmt* fmts_conv_xvid=NULL;
int fmts_xaw_to_xvidcolorspace[MAX_VIDEO_FMT]= {
  [VIDEO_RGB15]  = XVID_CSP_RGB555,
  [VIDEO_RGB16]  = XVID_CSP_RGB565,
#ifdef HAVE_XVID09
  [VIDEO_RGB24B] = XVID_CSP_RGB24,
  [VIDEO_RGB32B] = XVID_CSP_RGB32,
  [VIDEO_RGB24]  = XVID_CSP_RGB24, /* for grab-avi */
  [VIDEO_RGB32]  = XVID_CSP_RGB32,
#else
  [VIDEO_RGB24]  = XVID_CSP_BGR,
  [VIDEO_RGB32]  = XVID_CSP_BGRA,
#endif
  [VIDEO_YUYV]   = XVID_CSP_YUY2,
  [VIDEO_UYVY]   = XVID_CSP_YUY2,
  [VIDEO_YVU420] = XVID_CSP_YV12,
};
#endif

static video_fmt *fmts_conv_uncompressed=NULL;
static video_fmt fmts_uncompressed[]= {
  VIDEO_RGB15,
  VIDEO_RGB16,
  VIDEO_RGB24,
  VIDEO_RGB32,
  VIDEO_YUYV,
  VIDEO_UYVY,
  VIDEO_YVU420,
  VIDEO_YUV420,
  0};
struct STRTAB fourccs_raw[] = {
  {VIDEO_RGB15_LE,"\017RGB"},
  {VIDEO_RGB16_LE,"\020RGB"},
  {VIDEO_RGB24_LE,"\030RGB"},
  {VIDEO_RGB32_LE,"\040RGB"},
  {VIDEO_RGB15_BE,"\017BGR"},
  {VIDEO_RGB16_BE,"\020BGR"},
  {VIDEO_RGB24_BE,"\030BGR"},
  {VIDEO_RGB32_BE,"\040BGR"},
  {VIDEO_RGB08,  "\010RGB"},
  {VIDEO_GRAY1,  "\001RGB"},
  {VIDEO_YUYV,   "YUY2"},
  {VIDEO_UYVY,   "UYVY"},
  {VIDEO_YVU420, "YV12"},
  {VIDEO_YUV420, "I420"},
  {VIDEO_YUV420, "IYUV"},
  {-1,NULL}
};
/************** end conversion format tables *****************/

  // Doc says max encoded frame size is around 6 bytes per pixel...
  // I like the "around" ....
#define ENC_FRAME_SIZE (6*768*576)
static unsigned char * enc_frame = NULL;

static avi_t *   avifile = NULL;
static FILE *    subfile = NULL;

static int       prev_w = 0;
static int       prev_h = 0;
static int       prev_w_capture = 0;
static int       prev_h_capture = 0;
static unsigned long video_frames;
static unsigned long lastsubend;
static struct timeval tv_start;
static int width_rec, height_rec;
   /* recorded for the case where divx.width/height change...*/
static video_fmt fmt_conv;

extern int       AVI_errno;
static int       divx_in_progress = 0, divx_audio=0;
static int restart=0;

/*****************************************************************************
 *                       DIVX VIDEO FUNCTIONS
 ****************************************************************************/

static char * get_filename(void) {
  char *filename=NULL;
  char begin_name[256];
  struct stat buf;
  char *divx_file;
  update_divxfilename();
  if (stat (divx.filename, &buf) == 0 &&  S_ISDIR(buf.st_mode)) {
    
    // Get TV Program Title if Nxtvepg send it to xdTV    
    if (channel_title[0] != 0)
      sprintf(begin_name, "movie-%s",channel_title);
    else
      strcpy(begin_name,"movie");
    
    /* case divx.filename is a directory */  
    divx_file = snap_filename
      (begin_name, cur_sender!=-1 ? channels[cur_sender]->name : "unknown", "avi");
    filename=malloc(strlen(divx.filename)+strlen(divx_file)+2);
    strcpy(filename,divx.filename);
    strcat(filename, "/");
    strcat(filename, divx_file);
  } else {
    int l;
    filename=strdup(divx.filename);
    l=strlen(filename);
    if(l>=4 && strcmp(filename+l-4,".avi")==0){
      static int nb;
      if(!restart) nb=1; else {
	/* when recording is restarted automatically.
	   (when the file reaches the 2Gb/4Gb limit
	   or when there's a new http client)
	   filename.avi becomes filename_2.avi...
	*/
	char *f2=filename;
	filename=malloc(l+10);
	*(f2+l-4)=0;
	sprintf(filename,"%s_%d.avi",f2,nb);
	free(f2);
	nb++;
      }
    }
  }
  if(debug){
    fprintf(stderr, "divx.filename = %s\n",divx.filename);
    fprintf(stderr, "filename      = %s\n",filename);
  }
  return filename;
}

static char* get_subfilename(char *avi_filename) {
  int l=strlen(avi_filename);
  char *f=NULL;
  if(l<4 || strcmp(avi_filename+l-4,".avi")!=0){
    f=malloc(l+5);
    strcpy(f,avi_filename);
    strcat(f, ".sub");
  } else {
    f=strdup(avi_filename);
    strcpy(f+l-3,"sub");
  }
  return f;
}

void divx_init(int width, int height)
{
#ifdef HAVE_DIVX4LINUX
# if defined(ENC_OPT_VERSION) && defined (ENCORE_VERSION)
   int version_binary;
# endif
#endif
#ifdef HAVE_XVID
#ifdef HAVE_XVID09
  XVID_INIT_PARAM xinit;
#else
  xvid_gbl_init_t xinit;
  xvid_plugin_single_t single;
#endif
#endif
#if defined (HAVE_DIVX4LINUX) || defined (HAVE_XVID)
  int ret;
#endif
  struct timezone tz;
  time_t t;
#ifndef SYNCHRO
  int fps = (double)cur_maxfpsnum * 1000 / cur_maxfpsden + 0.5;
#else
  int fps = divx.fps;
#endif
  char *fourcc = NULL;
  char *filename = NULL;
  char *codec_name = NULL;
  struct timeval tv;
  strcpy(xawmessage,"Record Informations:\n\n");

  if (handle != NULL)
    {
      fprintf(stderr, "divx_init: already started\n");
      if (xawpopup && !divx.stream && !fs)
        PopupMessage("divx_init: already started\n");
      return;
    }

  if (debug)
    {
      fprintf(stderr, "DivX init...\n");
      switch(divx.codec)
        {
#ifdef HAVE_DIVX4LINUX
        case DIVX4LINUX:
            codec_name = "DivX";
            break;
#endif

#ifdef HAVE_FFMPEG
          case FFMPEG_MPEG4:
            codec_name = "FFMPEG - MPEG4\n";
            break;
          case FFMPEG_MPEG1:
            codec_name = "FFMPEG - MPEG1VIDEO\n";
          break;
#if LIBAVCODEC_BUILD >= 4734
          case FFMPEG_FFVHUFF:
            codec_name = "FFMPEG - FFVHUFF\n";
          break;
#endif
          case FFMPEG_HUFFYUV:
            codec_name = "FFMPEG - HUFFYUV\n";
          break;
#endif

#ifdef HAVE_XVID
        case XVID:
            codec_name = "XviD\n";
            break;
#endif

	case UNCOMPRESSED:
	  codec_name = "UnCompressed\n";
	  break;

        }
      fprintf(stderr, "divx.codec             = %s\n", codec_name);
      fprintf(stderr, "divx.width             = %d\n", divx.width);
      fprintf(stderr, "divx.height            = %d\n", divx.height);
      fprintf(stderr, "divx.bitrate           = %ld\n", divx.bitrate);
      fprintf(stderr, "divx.quality           = %d\n", divx.quality);
      fprintf(stderr, "divx.min_quantizer     = %d\n", divx.min_quantizer);
      fprintf(stderr, "divx.max_quantizer     = %d\n", divx.max_quantizer);
      fprintf(stderr, "divx.audio_buffer_size = %d\n", divx.audio_buffer_size);
      fprintf(stderr, "divx.stereo            = %d\n", divx.stereo);
      fprintf(stderr, "divx.compress_audio    = %d\n", divx.compress_audio);
      fprintf(stderr, "divx.mp3_bitrate       = %d\n", divx.mp3_bitrate);

      fprintf(stderr, "divx.mp3_quality       = %d\n", divx.mp3_quality);
      fprintf(stderr, "divx.mp3_vbr           = %d\n", divx.mp3_vbr);
      fprintf(stderr, "divx.mp3_vbr_quality   = %d\n", divx.mp3_vbr_quality);
#ifdef SYNCHRO
      fprintf(stderr, "divx.fps               = %d\n", divx.fps);
      fprintf(stderr, "divx.maxgap            = %f\n", divx.maxgap);
#endif
      fprintf(stderr, "divx.display_frames    = %d\n", divx.display_frames);
    }

  prev_w = width;
  prev_h = height;
  width_rec = divx.width;
  height_rec = divx.height;

  if(width_rec>cur_maxwidth) width_rec = cur_maxwidth;
  if(height_rec>cur_maxheight) height_rec = cur_maxheight;
  width = width_rec; height = height_rec;
  if(fps / 1000.0 > (double)cur_maxfpsnum / cur_maxfpsden - 0.01) {
    fps_num = cur_maxfpsnum;
    fps_den = cur_maxfpsden;
  } else {
    fps_num = fps;
    fps_den = 1000;
  }

  if (debug)
    fprintf(stderr, "divx_init: requested video size: w = %d, h = %d\n", width, height);

  video_get_capture_size(&prev_w_capture, &prev_h_capture);
  video_set_capture_size(width, height);

  if (debug)
    fprintf(stderr, "divx_init: allocated video size: w = %d, h = %d\n", width, height);

  height_rec = height -= 2 * get_ybar(height);
  if (debug)
    fprintf(stderr, "divx_init: encoded video size:   w = %d, h = %d\n", width, height);

  switch(divx.codec)
    {
#ifdef HAVE_DIVX4LINUX
    case DIVX4LINUX:
      if(fmts_conv_divx==NULL) {
	fmts_conv_divx=malloc(MAX_VIDEO_FMT*sizeof(video_fmt));
	setpreferred_list(fmts_divx,fmts_conv_divx);
      }
      // Check for DivX codec version
#if defined(ENC_OPT_VERSION) && defined (ENCORE_VERSION)
      version_binary = encore(0, ENC_OPT_VERSION, 0, 0);
      sprintf(xawmesstmp, "divx_init: divx4linux DivX encoder version = %d\n", version_binary);
      fprintf(stderr, xawmesstmp);
      strcat(xawmessage,xawmesstmp);
      if (version_binary != ENCORE_VERSION)
        {
          //...interfaces are incompatible, encoding can't be performed
	  sprintf(xawmesstmp, "divx4linux DivX encoder claim its version is %d and I was expecting encoder version %d: I give up...\n",
                  version_binary, ENCORE_VERSION);
	  fprintf(stderr, xawmesstmp);
	  strcat(xawmessage,xawmesstmp);
	  divx_stop();
          return;
        }
#else
      sprintf(xawmesstmp, "divx_init: Warning: at compile time, divx4linux was too old to perform version checking\n");
      fprintf(stderr, xawmesstmp);
      strcat(xawmessage,xawmesstmp);
#endif

#if ENCORE_VERSION >= 20021024
    if ((settings = malloc(sizeof(SETTINGS)))==NULL) {
      sprintf(xawmesstmp, "out of memory");
      fprintf(stderr, xawmesstmp);
      strcat(xawmessage,xawmesstmp);
      divx_stop();
      return;
    }
    if ((format = malloc(sizeof(DivXBitmapInfoHeader)))==NULL) {
      sprintf(xawmesstmp, "out of memory");
      fprintf(stderr, xawmesstmp);
      strcat(xawmessage,xawmesstmp);
      divx_stop();
      return;
    }
    memset (settings, 0, sizeof(SETTINGS));
    memset (format, 0, sizeof(DivXBitmapInfoHeader));

    settings->vbr_mode = RCMODE_502_1PASS;
    settings->bitrate = divx.bitrate;
    settings->quantizer = (float)(divx.bitrate/1000);
    settings->use_bidirect = 0;
    settings->input_clock = fps_num;
    settings->input_frame_period = fps_den;
    settings->quality = divx.quality;
    settings->complexity_modulation = 0.5;
    settings->enable_crop = 0;
    settings->enable_resize = 0;

    format->biSize = sizeof(DivXBitmapInfoHeader);
    format->biWidth = width;
    format->biHeight = height;
#else
      memset(&param, 0, sizeof(param));
      param.x_dim = width;
      param.y_dim = height;
      param.framerate = (float)fps_num/fps_den;
      param.min_quantizer = divx.min_quantizer;
      param.max_quantizer = divx.max_quantizer;
      param.quality = divx.quality; // (1 - fastest, 5 - best)
      param.bitrate = divx.bitrate;

      ret = encore(0, ENC_OPT_INIT, &param, 0);
      if (ret != ENC_OK)
        {
          sprintf(xawmesstmp, "divx_init: divx4linux encore initialization failed\n");
	  fprintf(stderr, xawmesstmp);
	  strcat(xawmessage,xawmesstmp);
	  divx_stop();
          return;
        }

      handle=param.handle;
#endif
      enc_frame = (unsigned char *)malloc(ENC_FRAME_SIZE);
      if (enc_frame == NULL)
        {
          sprintf(xawmesstmp, "divx_init: couldn't allocate memory for encoded frame: releasing divx4linux DivX encoder...\n");
	  fprintf(stderr, xawmesstmp);
	  strcat(xawmessage,xawmesstmp);
          divx_stop();
          return;
        }
      memset(&frame, 0, sizeof(frame));
      frame.bitstream = enc_frame;

      fmt_conv = fmts_conv_divx[x11_pixmap_format];
#if ENCORE_VERSION >= 20021024
      format->biCompression = fmts_xaw_to_biCompression[fmt_conv];
      format->biBitCount = fmts_xaw_to_biBitCount[fmt_conv];
      ret = encore((void*)&handle, ENC_OPT_INIT, format, settings);
       if (ret != ENC_OK)
         {
           sprintf(xawmesstmp, "divx_init: divx4linux encore initialization failed\n");
	   fprintf(stderr, xawmesstmp);
	   strcat(xawmessage,xawmesstmp);
	   sprintf(xawmesstmp, "error value is :%d\n",ret);
	   fprintf(stderr, xawmesstmp);
	   strcat(xawmessage,xawmesstmp);
	   divx_stop();
           return;
         }
#else
       memset(&frame, 0, sizeof(frame));
       frame.bitstream = enc_frame;
       frame.colorspace = fmts_xaw_to_divxcolorspace[fmt_conv];
#endif
       break;
#endif // HAVE_DIVX4LINUX

#ifdef HAVE_FFMPEG
    case FFMPEG_MPEG4:
    case FFMPEG_MPEG1:
#if LIBAVCODEC_BUILD >= 4734
    case FFMPEG_FFVHUFF:    
#endif
    case FFMPEG_HUFFYUV:
      avcodec_init();
      sprintf(xawmesstmp, "divx_init: initializing ffmpeg libavcodec version %d (build %d)\n", avcodec_version(), avcodec_build());
      fprintf(stderr, xawmesstmp);
      strcat(xawmessage,xawmesstmp);
      if (divx.codec == FFMPEG_MPEG4)
        {
          register_avcodec(&mpeg4_encoder);
          codec = avcodec_find_encoder(CODEC_ID_MPEG4);
        }
      else if (divx.codec == FFMPEG_MPEG1)
        {
          register_avcodec(&mpeg1video_encoder);
          codec = avcodec_find_encoder(CODEC_ID_MPEG1VIDEO);
        }
#if LIBAVCODEC_BUILD >= 4734
      else if (divx.codec == FFMPEG_FFVHUFF)
        {
          register_avcodec(&ffvhuff_encoder);
          codec = avcodec_find_encoder(CODEC_ID_FFVHUFF);
        }
#endif
      else if (divx.codec == FFMPEG_HUFFYUV)
        {
          register_avcodec(&huffyuv_encoder);
          codec = avcodec_find_encoder(CODEC_ID_HUFFYUV);
        }
      if (codec == NULL)
         {
            sprintf(xawmesstmp, "divx_init: couldn't find %s codec\n", codec_name);
            fprintf(stderr, xawmesstmp);
            strcat(xawmessage,xawmesstmp);
	    divx_stop();
            return;
          }

      if (codec_context == NULL)
        codec_context = handle = avcodec_alloc_context();
      picture_420p = avcodec_alloc_frame();

      /* All default parameters are borrowed from MPlayer */
      if (divx.codec == FFMPEG_MPEG4)
        codec_context->codec_id            = CODEC_ID_MPEG4;
      else if (divx.codec == FFMPEG_MPEG1)
        codec_context->codec_id            = CODEC_ID_MPEG1VIDEO;
#if LIBAVCODEC_BUILD >= 4734
      else if (divx.codec == FFMPEG_FFVHUFF)
        codec_context->codec_id            = CODEC_ID_FFVHUFF;
#endif
      else if (divx.codec == FFMPEG_HUFFYUV)
        codec_context->codec_id            = CODEC_ID_HUFFYUV;
      codec_context->codec_type            = CODEC_TYPE_VIDEO;
      codec_context->bit_rate              = divx.bitrate;
      codec_context->bit_rate_tolerance    = divx.bitrate*10;
      codec_context->width                 = width;
      codec_context->height                = height;
#if LIBAVCODEC_BUILD >= 4662
      codec_context->frame_rate		   = fps_num;
      codec_context->frame_rate_base       = fps_den;
#else
      codec_context->frame_rate            = fps_num*FRAME_RATE_BASE/fps_den;
#endif
      codec_context->qmin                  = divx.min_quantizer;
      codec_context->qmax                  = divx.max_quantizer;
      codec_context->max_qdiff             = 3;
      codec_context->qcompress             = 0.5;
      codec_context->qblur                 = 0.5;
      codec_context->max_b_frames          = 0;
      codec_context->b_quant_factor        = 1.25;
      codec_context->rc_strategy           = 2;
      codec_context->b_frame_strategy      = 0;
#ifdef CODEC_FLAG_PART
      codec_context->b_quant_offset        = 1.25;
      codec_context->luma_elim_threshold   = 0;
      codec_context->chroma_elim_threshold = 0;
      codec_context->rtp_payload_size      = 0;
      /*
      if(param_packet_size )
        codec_context->rtp_mode            = 1;
      */
      codec_context->rtp_mode              = 0;
      codec_context->strict_std_compliance = 0;
#endif
      codec_context->i_quant_factor        = 0.8;
      codec_context->i_quant_offset        = 0.0;
      codec_context->rc_qsquish            = 1.0;
      codec_context->rc_qmod_amp           = 0;
      codec_context->rc_qmod_freq          = 0;
      codec_context->rc_eq                 = "tex^qComp";
      codec_context->rc_max_rate           = 0*1000;
      codec_context->rc_min_rate           = 0*1000;
      codec_context->rc_buffer_size        = 0*1000;
      codec_context->rc_buffer_aggressivity= 1.0;
      codec_context->rc_initial_cplx       = 0;

      fmt_conv = VIDEO_YUV420;
      /* HUFFYUV uses YUV422P colorspace by standard,
         but also accepts YUV420P which make files 20% smaller 
      */
      /* Note this choice probably makes files unreadable on Win platforms
         but saves time to write new conversion/buffer handling functions.
	 (the default format YUV422P is taken for recent version
	 of ffmpeg, because in this case the good choice is FFVHUFF !)
      */
#if LIBAVCODEC_BUILD >= 4734
      if (divx.codec == FFMPEG_FFVHUFF) {
	codec_context->pix_fmt=PIX_FMT_YUV420P;
	codec_context->strict_std_compliance=-1;
        if(divx.bitrate < 5000000)
          codec_context->context_model = 1;
      }
      if (divx.codec == FFMPEG_HUFFYUV) {
	fmt_conv = VIDEO_YUV422P;
	codec_context->pix_fmt=PIX_FMT_YUV422P;
      }
#else
      if (divx.codec == FFMPEG_HUFFYUV) {
	codec_context->pix_fmt=PIX_FMT_YUV420P;
	codec_context->strict_std_compliance=-1;
      }
#endif

      codec_context->mpeg_quant            = 0;
      codec_context->dct_algo              = 0;
      codec_context->idct_algo             = 0;
      codec_context->lumi_masking          = 0.0;
      codec_context->temporal_cplx_masking = 0.0;
      codec_context->spatial_cplx_masking  = 0.0;
      codec_context->p_masking             = 0.0;
      codec_context->dark_masking          = 0.0;
      codec_context->gop_size           = 250; // lavc_param_keyint
      codec_context->flags              = 0;   // MPlayer default

      /* 4mv is currently buggy with B frames */
      /*
      if (lavc_param_vmax_b_frames > 0 && lavc_param_v4mv) {
        printf("4MV with B-Frames not yet supported -> 4MV disabled\n");
        lavc_param_v4mv = 0;
      }
      codec_context->flags|= lavc_param_v4mv ? CODEC_FLAG_4MV : 0;
#ifdef CODEC_FLAG_PART
      codec_context->flags|= lavc_param_data_partitioning;
#endif
      if(lavc_param_gray) codec_context->flags|= CODEC_FLAG_GRAY;
      if(lavc_param_normalize_aqp) codec_context->flags|= CODEC_FLAG_NORMALIZE_AQP;
      if(lavc_param_interlaced_dct) codec_context->flags|= CODEC_FLAG_INTERLACED_DCT;
      */

      /*
       *         0    none (very low quality)
                 1    full (slow)
                 2    log (low quality)
                 3    phods (low quality)
                 4    EPZS (default)
                 5    X1 (experimental)
      */
      codec_context->me_method = ME_ZERO+ divx.quality;

      /* fixed qscale :p */
      /*
      if (lavc_param_vqscale)
        {
          //printf("Using constant qscale = %d (VBR)\n", lavc_param_vqscale);
          lavc_venc_context->flags        |= CODEC_FLAG_QSCALE;
          lavc_venc_context->quality       = 0;
        }
      */

      // We are using only a one pass compression
#if LIBAVCODEC_BUILD >= 4734
      if (divx.codec != FFMPEG_FFVHUFF && divx.codec != FFMPEG_HUFFYUV)
#else
      if (divx.codec != FFMPEG_HUFFYUV)
#endif
        codec_context->flags |= CODEC_FLAG_PASS1;

      picture_420p->linesize[0] = codec_context->width;
      picture_420p->linesize[1] = codec_context->width / 2;
      picture_420p->linesize[2] = codec_context->width / 2;

      if (avcodec_open(codec_context, codec) < 0)
        {
          sprintf(xawmesstmp, "divx_init: couldn't open %s codec\n", codec_name);
          fprintf(stderr, xawmesstmp);
          strcat(xawmessage,xawmesstmp);
	  divx_stop();
          return;
        }

      if (codec_context->codec->encode == NULL)
        {
          sprintf(xawmesstmp, "divx_init: %s codec initialization failed (ctx->codec->encode == NULL)!\n", codec_name);
          fprintf(stderr, xawmesstmp);
          strcat(xawmessage,xawmesstmp);
	  divx_stop();
          return;
        }

      enc_frame = (unsigned char *)malloc(ENC_FRAME_SIZE);
      if (enc_frame == NULL)
        {
          sprintf(xawmesstmp, "divx_init: couldn't allocate memory for encoded frame: releasing ffmpeg DivX encoder...\n");
          fprintf(stderr, xawmesstmp);
          strcat(xawmessage,xawmesstmp);
          divx_stop();
          return;
        }

      break;
#endif // HAVE_FFMPEG

#ifdef HAVE_XVID
    case XVID:
      if(fmts_conv_xvid==NULL) {
	fmts_conv_xvid=malloc(MAX_VIDEO_FMT*sizeof(video_fmt));
	setpreferred_list(fmts_xvid,fmts_conv_xvid);
      }
      memset(&xinit, 0, sizeof(xinit));
#ifdef HAVE_XVID09
      xvid_init(NULL, 0, &xinit, NULL);
      sprintf(xawmesstmp, "divx_init: initializing XviD core version %d\n", xinit.core_build);
      fprintf(stderr, xawmesstmp);
      strcat(xawmessage,xawmesstmp);
      if (debug)
        fprintf(stderr, "divx_init: XviD API version = %d.%d\n", xinit.api_version>>16, xinit.api_version&0xFF);
      if (xinit.api_version != API_VERSION)
        {
          //...interfaces are incompatible, encoding can't be performed
          sprintf(xawmesstmp, "XviD core claim its API version is %d.%d and I was expecting API version %d%.d: I give up...\n",
                  xinit.api_version>>16, xinit.api_version&0xFF,
                  API_VERSION>>16, API_VERSION&0xFF);
          fprintf(stderr, xawmesstmp);
          strcat(xawmessage,xawmesstmp);
	  divx_stop();
          return;
        }
#else /* XVID 1.0 */
      sprintf(xawmesstmp, "divx_init: XviD version=%d.%d.%d API=%d.%d\n",
		XVID_VERSION_MAJOR(XVID_VERSION),
		XVID_VERSION_MINOR(XVID_VERSION),
		XVID_VERSION_PATCH(XVID_VERSION),
		XVID_API_MAJOR(XVID_API),
		XVID_API_MINOR(XVID_API));
      fprintf(stderr, xawmesstmp);
      strcat(xawmessage,xawmesstmp);
      xinit.version = XVID_VERSION;
      if(xvid_global(NULL, XVID_GBL_INIT, &xinit, NULL)<0) {
	sprintf(xawmesstmp,"xvid initialization failed\n");
        fprintf(stderr, xawmesstmp);
        strcat(xawmessage,xawmesstmp);
	divx_stop();
      }
      if(xinit.version != XVID_VERSION) {
        sprintf(xawmesstmp, "divx_init: XviD executed version=%d.%d.%d\n",
		XVID_VERSION_MAJOR(xinit.version),
		XVID_VERSION_MINOR(xinit.version),
		XVID_VERSION_PATCH(xinit.version));
      fprintf(stderr, xawmesstmp);
      strcat(xawmessage,xawmesstmp);
      }
#endif
      memset(&xparam, 0, sizeof(xparam));
      xparam.width = width;
      xparam.height = height;
      xparam.fincr = fps_den;
      xparam.fbase = fps_num;
#ifdef HAVE_XVID09
      xparam.rc_bitrate = divx.bitrate;
      xparam.rc_reaction_delay_factor = -1; // forces default value
      xparam.rc_averaging_period = -1; // forces default value
      xparam.rc_buffer = -1; // forces default value
      xparam.max_quantizer = divx.max_quantizer;
      xparam.min_quantizer = divx.min_quantizer;
#else  /* XVID 1.0 */
      memset(&single, 0, sizeof(xvid_plugin_single_t));
      single.version = XVID_VERSION;
      single.bitrate = divx.bitrate;
      xparam_plugin.func = xvid_plugin_single;
      xparam_plugin.param = &single;
      xparam.num_plugins = 1;
      xparam.plugins = &xparam_plugin;
      xparam.version = XVID_VERSION;
#endif
      xparam.max_key_interval = 10*fps_num/fps_den; // default value is too big
      ret = xvid_encore(NULL, XVID_ENC_CREATE, &xparam, NULL);
      if (ret != 0)
        {
          sprintf(xawmesstmp, "divx_init: XviD encore initialization failed ret=%d\n",ret);
          fprintf(stderr, xawmesstmp);
          strcat(xawmessage,xawmesstmp);
	  divx_stop();
          return;
        }
      handle = xparam.handle;

      memset(&xframe, 0, sizeof(xframe));
      fmt_conv = fmts_conv_xvid[x11_pixmap_format];
#ifdef HAVE_XVID09
      xframe.general = xgeneral_presets[divx.quality];
      xframe.motion = xmotion_presets[divx.quality];
      xframe.quant = 0;
      xframe.intra = -1;
      xframe.colorspace = fmts_xaw_to_xvidcolorspace[fmt_conv];
#else
      xframe.version = XVID_VERSION;
      xframe.vop_flags = xvop_presets[divx.quality];
      xframe.motion = xmotion_presets[divx.quality];
      xframe.input.csp = fmts_xaw_to_xvidcolorspace[fmt_conv];
#endif      

      enc_frame = (unsigned char *)malloc(ENC_FRAME_SIZE);
      if (enc_frame == NULL)
        {
          sprintf(xawmesstmp, "divx_init: couldn't allocate memory for encoded frame: releasing XviD encoder...\n");
          fprintf(stderr, xawmesstmp);
          strcat(xawmessage,xawmesstmp);
          divx_stop();
          return;
        }
      xframe.bitstream = enc_frame;
      break;
#endif // HAVE_XVID
  case UNCOMPRESSED:
    if(fmts_conv_uncompressed==NULL) {
      fmts_conv_uncompressed=malloc(MAX_VIDEO_FMT*sizeof(video_fmt));
      setpreferred_list(fmts_uncompressed,fmts_conv_uncompressed);
    }
    sprintf(xawmesstmp, "divx_init: Uncompressed Codec used\n");
    fprintf(stderr, xawmesstmp);
    strcat(xawmessage,xawmesstmp);
    fmt_conv = fmts_conv_uncompressed[x11_pixmap_format];
    }

  // Open and setup AVI file
  filename = get_filename();
  sprintf(lastfilename,"%s '%s'&",pplayer, filename);
  
  if (debug)
    fprintf(stderr,"nxtvepg title: %s\n",channel_title);
  
  avifile = AVI_open_output_file(filename);
  if(divx.sub) {
    char *f=get_subfilename(filename);
    subfile=fopen(f,"w");
    free(f);
    if(subfile==NULL) perror("fopen sub");
    lastsubend=0;
  }
  if(subfile) {
    fprintf(subfile,"FORMAT=%f\n\n",(double)fps_num/fps_den);
  }
  free(filename);

  if (avifile == NULL)
    {
      AVI_print_error("divx_init: AVI_open_output_file:");
      divx_stop();
      return;
    }

  switch (divx.codec)
    {
#ifdef HAVE_DIVX4LINUX
    case DIVX4LINUX:
#if (ENCORE_MAJOR_VERSION >= 5010)
      fourcc = "DX50";
#else
      fourcc = "DIVX";
#endif
      break;
#endif

#ifdef HAVE_FFMPEG
    case FFMPEG_MPEG4:
      fourcc = "DIVX";
      break;
    case FFMPEG_MPEG1:
      fourcc = "MPEG";
      break;
#if LIBAVCODEC_BUILD >= 4734
    case FFMPEG_FFVHUFF:
      fourcc = "FFVH";
      AVI_set_extradata(avifile, codec_context->extradata,codec_context->extradata_size);
      break;
#endif
    case FFMPEG_HUFFYUV:
      fourcc = "HFYU";
      /* Internal codec configuration data needed in extradata */
      AVI_set_extradata(avifile, codec_context->extradata,codec_context->extradata_size);
      break;
#endif

#ifdef HAVE_XVID
    case XVID:
      fourcc = "XVID";
      break;
#endif
    case UNCOMPRESSED:
      fourcc = int_to_str(fmt_conv, fourccs_raw);
      if(fourcc==NULL) TRAP("should never happen\n");
      break;
    }
  AVI_set_video(avifile, width, height, fps_num, fps_den, fourcc);

#ifdef NOAUDIO
  AVI_set_audio(avifile, 0 /* channels*/,
                         0 /* rate */,
                         0 /* bits */,
                         0 /* formats */,
                         0 /* brate */,
                         0 /* framesize */,
                         0 /* vbr */);
#else

 /* the stereo mode could be automatically detected by the card !*/
 if(divx.stereo) achans = 2; else achans = 1;

#ifdef HAVE_LAME
 if (divx.compress_audio)
   {
     gfp = lame_init();

     lame_set_num_channels(gfp, achans);

     lame_set_in_samplerate(gfp, DEC_FREQ);
     lame_set_compression_ratio(gfp, 0.0);
     if(!divx.mp3_vbr)
       lame_set_brate(gfp, divx.mp3_bitrate);
     else {
       lame_set_VBR(gfp, vbr_mtrh);
       lame_set_VBR_q(gfp,divx.mp3_vbr_quality);
     }

     // mode = 0,1,2,3 = stereo, jstereo, dual channel (not supported), mono
     // default: lame picks based on compression ration and input channels
     if (achans == 1)
       lame_set_mode(gfp, 3);
     else
       lame_set_mode(gfp, 1);

     // quality=0..9.  0=best (very slow).  9=worst.
     // recommended:  2     near-best quality, not too slow
     //               5     good quality, fast
     //               7     ok quality, really fast
     lame_set_quality(gfp, divx.mp3_quality);

     if (lame_init_params(gfp) < 0)
       {
         sprintf(xawmesstmp, "divx_init: lame_init_params failed\n");
         fprintf(stderr, xawmesstmp);
         strcat(xawmessage,xawmesstmp);
         divx_stop();
         return;
       }

     lame_buffer_size = (1.25*divx.audio_buffer_size)+7200;
     lame_buffer = malloc(lame_buffer_size);
     if (lame_buffer == NULL)
       {
         sprintf(xawmesstmp, "divx_init: could not allocate lame buffer\n");
         fprintf(stderr, xawmesstmp);
         strcat(xawmessage,xawmesstmp);
         divx_stop();
         return;
       }

     aformat = WAVE_FORMAT_MPEGLAYER3;
     AVI_set_audio(avifile, achans, lame_get_out_samplerate(gfp), 16, aformat, (divx.mp3_bitrate * 1000)/8,lame_get_framesize(gfp),divx.mp3_vbr);
     if (debug)
       fprintf (stderr, "divx_init: audio: WAVE_FORMAT_MPEGLAYER3\n");
   }
 else
   {
     aformat = WAVE_FORMAT_PCM;
     AVI_set_audio(avifile, achans, DEC_FREQ, 16, aformat, 0, 0, 0);
     if (debug)
       fprintf (stderr, "divx_init: audio: WAVE_FORMAT_PCM\n");
   }
#else
 aformat = WAVE_FORMAT_PCM;
 AVI_set_audio(avifile, achans, DEC_FREQ, 16, aformat, 0, 0, 0);
 if (debug)
   fprintf (stderr, "divx_init: audio: WAVE_FORMAT_PCM\n");
#endif // HAVE_LAME

#endif // NOAUDIO

 /* http must be started when the header of the avifile is fixed
    but before the write of any frame to catch the header */

 if(divx.stream)
 {
   fprintf(stderr,"GUI Streaming activation: %d\n",divx.stream);
   avifile->send_data = http_send_data;
   if(!restart) http_init_server(avifile);
 }

 set_capture_temp_grab(1, 1);

 // Let's go
  video_frames = 0;
  slw = 0;
#ifndef NOAUDIO
  audio_frames = 0;
  audio_buffers = 0;
#ifdef SYNCHRO
  nbsamplesaudios = 0;
  dropped_video_frames = 0;
  dropped_audio_samples = 0;
  dropped_audio_buffers = 0;
#endif
  gettimeofday(&tv, NULL);
  if(audiostamp==0.0) audiostamp=tv.tv_usec/1000000.0+tv.tv_sec;
  if(videostampmin==0.0)
    videostampmin=videostampmax=tv.tv_usec/1000000.0+tv.tv_sec;
  if (!is_aop_running() && grabbers[grabber] != &grab_avi)
    divx_audio_start();
#endif

  time(&t);
  sprintf(xawmesstmp, "Recording started %s", ctime(&t));
  fprintf(stderr,xawmesstmp);
  strcat(xawmessage,xawmesstmp);
  gettimeofday(&tv_start, &tz);

  divx_in_progress = 1;
}

void divx_restart(void) {
  restart=1;
  divx_stop();
  divx_init(prev_w, prev_h);
  restart=0;
}

void
divx_encode(unsigned char *src_frame)
{
#ifdef HAVE_DIVX4LINUX
   ENC_RESULT res;
#endif
#if defined(HAVE_DIVX4LINUX) || defined(HAVE_XVID)
  int ret;
#endif
#ifdef HAVE_FFMPEG
  int enc_size;
#endif
  int flag;


  /* only the video thread can run divx_stop */
  if(slw)
    {
      sprintf(xawmesstmp, "last AVI_write_status = %d\n",slw);
      fprintf(stderr,xawmesstmp);
      strcat(xawmessage,xawmesstmp);
      AVI_print_error("reason");
      displaynextframe = 0;
      if(AVI_errno==AVI_ERR_SIZELIM)
	divx_restart();
      else
	divx_stop();
      return;
    }

#ifdef SYNCHRO
  if(((double)video_frames * fps_den / fps_num - videostampmax
      + divx.delay >
      (double)nbsamplesaudios / DEC_FREQ - audiostamp + divx.maxgap)
     || (videostampmin > audiostamp + 4.0) ) { //if audio is blocked
    dropped_video_frames++;
    return;
  }
#endif

  switch (divx.codec)
    {
#ifdef HAVE_DIVX4LINUX
    case DIVX4LINUX:
      frame.image=convert1
	(src_frame, x11_pixmap_format, fmt_conv, width_rec, height_rec);
      ret = encore(handle, ENC_OPT_ENCODE, &frame, &res);
      if (ret != ENC_OK)
        {
          sprintf(xawmesstmp, "divx_encode: error encoding frame (divx4linux)!\n");
          fprintf(stderr,xawmesstmp);
          strcat(xawmessage,xawmesstmp);
          slw = AVI_write_frame(avifile, frame.bitstream, 0, AVI_FLAG_NONE);
        }
      else
        {
#if ENCORE_VERSION >= 20021024
          flag = (res.cType == 'I') ? AVI_FLAG_KEYFRAME : AVI_FLAG_NONE;
#else
	  flag = res.is_key_frame ? AVI_FLAG_KEYFRAME : AVI_FLAG_NONE;
#endif
          slw = AVI_write_frame(avifile, frame.bitstream, frame.length, flag);
        }
      break;
#endif // HAVE_DIVX4LINUX

#ifdef HAVE_FFMPEG
    case FFMPEG_MPEG4:
    case FFMPEG_MPEG1:
#if LIBAVCODEC_BUILD >= 4734
    case FFMPEG_FFVHUFF:
#endif
    case FFMPEG_HUFFYUV:
      picture_420p->data[0] = convert1
	(src_frame, x11_pixmap_format, fmt_conv, width_rec, height_rec);
      picture_420p->data[1] =
	picture_420p->data[0] + width_rec*height_rec;
      picture_420p->data[2] =
	picture_420p->data[1] + width_rec*height_rec / 4;
      if(fmt_conv==VIDEO_YUV422P)
	picture_420p->data[2] += width_rec*height_rec / 4;
      picture_420p->key_frame = -1;
      enc_size = avcodec_encode_video(codec_context, enc_frame, ENC_FRAME_SIZE, picture_420p);
      if (enc_size > 0) {
	flag = codec_context->coded_frame->key_frame ? AVI_FLAG_KEYFRAME : AVI_FLAG_NONE;
	slw = AVI_write_frame(avifile, enc_frame, enc_size, flag);
      } else {
	sprintf(xawmesstmp, "divx_encode: error encoding frame (ffmpeg)!\n");
	fprintf(stderr,xawmesstmp);
	strcat(xawmessage,xawmesstmp);
	slw = AVI_write_frame(avifile, enc_frame, 0, AVI_FLAG_NONE);
      }
      break;
#endif // HAVE_FFMPEG

#ifdef HAVE_XVID
    case XVID:
#ifdef HAVE_XVID09
      xframe.image = convert1
	(src_frame, x11_pixmap_format, fmt_conv, width_rec, height_rec);
      xframe.intra = -1;
      ret = xvid_encore(handle, XVID_ENC_ENCODE, &xframe, NULL);
#else
      xframe.input.plane[0] = convert1
	    (src_frame, x11_pixmap_format, fmt_conv, width_rec, height_rec);
      if(fmt_conv==VIDEO_YVU420) {
	int s=width_rec*height_rec;
	xframe.input.plane[1] = xframe.input.plane[0]+s;
	xframe.input.plane[2] = xframe.input.plane[1]+s/4;
	xframe.input.stride[0] = width_rec;
	xframe.input.stride[1] = xframe.input.stride[2] = width_rec/4;
      } else
	xframe.input.stride[0] = size_img(fmt_conv,width_rec,1);
      xframe.type = XVID_TYPE_AUTO;
      xframe.length = ENC_FRAME_SIZE;
      memset(&xstats, 0, sizeof(xstats));
      xstats.version = XVID_VERSION;
      ret = xvid_encore(handle, XVID_ENC_ENCODE, &xframe, &xstats);
#endif
#ifdef HAVE_XVID09
      if(ret>0) ret=-1000-ret;
#endif
      if (ret < 0)
        {
          sprintf(xawmesstmp, "divx_encode: error encoding frame (XviD)! ret=%d\n",ret);
          fprintf(stderr,xawmesstmp);
          strcat(xawmessage,xawmesstmp);
          slw = AVI_write_frame(avifile, xframe.bitstream, 0, AVI_FLAG_NONE);
        }
      else
        {
#ifdef HAVE_XVID09
          flag = (xframe.intra > 0) ? AVI_FLAG_KEYFRAME : AVI_FLAG_NONE;
          slw = AVI_write_frame(avifile, xframe.bitstream, xframe.length, flag);
#else
          flag = (xframe.out_flags & XVID_KEYFRAME)
	    ? AVI_FLAG_KEYFRAME : AVI_FLAG_NONE;
          slw = AVI_write_frame(avifile, xframe.bitstream, ret, flag);
#endif
        }
      break;
#endif // HAVE_XVID
    case UNCOMPRESSED: {
      unsigned char *image;
      int size = size_img(fmt_conv,width_rec,height_rec);
      image=convert1
	(src_frame, x11_pixmap_format, fmt_conv, width_rec, height_rec);
      slw = AVI_write_frame(avifile, image, size, AVI_FLAG_KEYFRAME);
      break;
    }}
  video_frames++;
  if(divx.stream && !divx.display_frames)
    http_read_clients();
  //segfaults if executed when norm changes and frames are displayed
}

void
divx_stop(void)
{
#if defined(HAVE_DIVX4LINUX) || defined(HAVE_XVID)
  int ret;
#endif
  double elapsed_time;
  struct timeval tv_stop;
  struct timezone tz;
  time_t t;

  divx_in_progress = 0;

  gettimeofday(&tv_stop, &tz);
  time(&t);
  sprintf(xawmesstmp, "Recording stopped %s", ctime(&t));
  fprintf(stderr,xawmesstmp);
  strcat(xawmessage,xawmesstmp);

  if (debug)
    fprintf(stderr, "divx_stop...\n");

#ifndef NOAUDIO
  divx_audio_stop();
  if (debug)
    fprintf(stderr, "divx_stop: audio stopped\n");

  if(is_aop_running())
    aop_wait_write_audio();
  else if(grabbers[grabber] == &grab_avi) {
    avi_wait_write_audio();
  }

#ifdef HAVE_LAME
  if (divx.compress_audio)
    lame_stop();
#endif // HAVE_LAME
#endif

  if(avifile) avifile->send_data = NULL;
  if(!restart) http_close_server();

  // Set the real FPS
  elapsed_time = ((double)tv_stop.tv_sec + (double)tv_stop.tv_usec*1.0e-6)
    - ((double)tv_start.tv_sec + (double)tv_start.tv_usec*1.0e-6);

  set_capture_temp_grab(0, 1);

#ifndef SYNCHRO
  fps_den = 1000000;
  fps_num = (double)video_frames * fps_den / elapsed_time + 0.5;

  if (avifile) {
    avifile->fps_num = fps_num;
    avifile->fps_den = fps_den;
  }
#else
  if(avifile && (avifile->fps_num != fps_num || avifile->fps_den != fps_den))
  {
    sprintf(xawmesstmp, "ATTENTION fps has changed during recording\n");
    fprintf(stderr,xawmesstmp);
    strcat(xawmessage,xawmesstmp);
  }
#endif

#ifndef NOAUDIO
#ifndef SYNCHRO
  sprintf(xawmesstmp, "%ld frames recorded in %f seconds => FPS = %f\n%ld audio frames recorded (%ld audio buffers)\n", video_frames, elapsed_time, (double)fps_num/fps_den, audio_frames,audio_buffers);
  fprintf(stderr,xawmesstmp);
  strcat(xawmessage,xawmesstmp);
#else
  sprintf(xawmesstmp, "recording time = %f seconds,  ", elapsed_time);
  fprintf(stderr,xawmesstmp);
  strcat(xawmessage,xawmesstmp);
  sprintf(xawmesstmp, "fps = %f\n",(double)fps_num/fps_den);
  fprintf(stderr,xawmesstmp);
  strcat(xawmessage,xawmesstmp);
  sprintf(xawmesstmp, "%ld video frames recorded (%f seconds)\n", video_frames,
	 (double)video_frames*fps_den/fps_num);
  fprintf(stderr,xawmesstmp);
  strcat(xawmessage,xawmesstmp);
  sprintf(xawmesstmp, "%ld audio frames recorded (%ld audio buffers, %f seconds)\n", audio_frames,audio_buffers,(double)nbsamplesaudios / DEC_FREQ);
  fprintf(stderr,xawmesstmp);
  strcat(xawmessage,xawmesstmp);
  sprintf(xawmesstmp, "%ld dropped video frames (fpscod=%f fpscapt=%f)\n",
	 dropped_video_frames, (double)video_frames / elapsed_time,
	 ((double)video_frames + dropped_video_frames) / elapsed_time);
  fprintf(stderr,xawmesstmp);
  strcat(xawmessage,xawmesstmp);
  sprintf(xawmesstmp, "%ld dropped audio buffers (%f seconds)\n",
	 dropped_audio_buffers, (double)dropped_audio_samples / DEC_FREQ);
  fprintf(stderr,xawmesstmp);
  strcat(xawmessage,xawmesstmp);

  if(video_frames==0) {
    sprintf(xawmesstmp, "@@@@ No video frames recorded\n");
    fprintf(stderr,xawmesstmp);
    strcat(xawmessage,xawmesstmp);
  }
  else if(dropped_audio_buffers*20>audio_frames) {
    sprintf(xawmesstmp, "@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@\n");
    fprintf(stderr,xawmesstmp);
    strcat(xawmessage,xawmesstmp);
    sprintf(xawmesstmp, "@@ You computer is too slow to do such recording @@\n");
    fprintf(stderr,xawmesstmp);
    strcat(xawmessage,xawmesstmp);
    sprintf(xawmesstmp, "@@ try to decrease image dimension               @@\n");
    fprintf(stderr,xawmesstmp);
    strcat(xawmessage,xawmesstmp);
    sprintf(xawmesstmp, "@@ or the framerate (to %.1ffps ?)               @@\n",
	   (double)video_frames / elapsed_time * 1.05);
    fprintf(stderr,xawmesstmp);
    strcat(xawmessage,xawmesstmp);
    sprintf(xawmesstmp, "@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@\n");
    fprintf(stderr,xawmesstmp);
    strcat(xawmessage,xawmesstmp);
  }
  else if(dropped_video_frames*20>video_frames) {
    sprintf(xawmesstmp, "You could try record to the better fps=%.1f\n",
	   ((double)video_frames + dropped_video_frames)
	   / elapsed_time * 0.95);
    fprintf(stderr,xawmesstmp);
    strcat(xawmessage,xawmesstmp);
  }
#endif
#else
  sprintf(xawmesstmp, "%ld frames recorded in %f seconds => FPS = %f\n", video_frames, elapsed_time,(double)fps_num/fps_den);
  fprintf(stderr,xawmesstmp);
  strcat(xawmessage,xawmesstmp);
#endif


  // Release allocated buffers
  switch(divx.codec)
    {
#ifdef HAVE_DIVX4LINUX
    case DIVX4LINUX:
      if (handle != NULL)
        {
          /* Release the encoder */
          ret = encore(handle, ENC_OPT_RELEASE, 0, 0);
          if (ret != ENC_OK)
            {
              sprintf(xawmesstmp, "divx_stop: divx4linux codec release failed\n");
              fprintf(stderr,xawmesstmp);
              strcat(xawmessage,xawmesstmp);
            }
          else if (debug)
            fprintf(stderr, "divx_stop: divx4linux codec released\n");

          handle = NULL;
        }

      if (enc_frame != NULL)
        {
          free(enc_frame);
          enc_frame = NULL;
          if (debug)
            fprintf(stderr, "divx_stop: divx4linux encoded frame freed\n");
        }
      break;
#endif // HAVE_DIVX4LINUX

#ifdef HAVE_FFMPEG
    case FFMPEG_MPEG4:
	case FFMPEG_MPEG1:
#if LIBAVCODEC_BUILD >= 4734
	case FFMPEG_FFVHUFF:
#endif
	case FFMPEG_HUFFYUV:
      if (codec_context != NULL)
        {
          avcodec_close(codec_context);
          free(codec_context);
          codec_context = NULL;
	  handle = NULL;
          free(picture_420p);
          picture_420p = NULL;
          if (debug)
            fprintf(stderr, "divx_stop: ffmpeg codec released\n");
        }

      if (enc_frame != NULL)
        {
          free(enc_frame);
          enc_frame = NULL;
          if (debug)
            fprintf(stderr, "divx_stop: ffmpeg encoded frame freed\n");
        }
      break;
#endif // HAVE_FFMPEG

#ifdef HAVE_XVID
    case XVID:
      if (handle != NULL)
        {
          /* Release the encoder */
          ret = xvid_encore(handle, XVID_DEC_DESTROY, NULL, NULL);
          if (ret != 0)
            {
              sprintf(xawmesstmp, "divx_stop: XviD codec release failed\n");
              fprintf(stderr,xawmesstmp);
              strcat(xawmessage,xawmesstmp);
            }
          else if (debug)
            fprintf(stderr, "divx_stop: XviD codec released\n");

          handle = NULL;
        }

      if (enc_frame != NULL)
        {
          free(enc_frame);
          enc_frame = NULL;
          if (debug)
            fprintf(stderr, "divx_stop: XviD encoded frame freed\n");
        }
      break;
#endif // HAVE_XVID
    }

  if (avifile)
    {
      sprintf(xawmesstmp, "file size = %.1f Mb,  ",avifile->pos/1024.0/1024.0);
      fprintf(stderr,xawmesstmp);
      strcat(xawmessage,xawmesstmp);
      sprintf(xawmesstmp, "mean bitrate = %.3f kbps\n",avifile->pos*8.0/elapsed_time/1000);
      fprintf(stderr,xawmesstmp);
      strcat(xawmessage,xawmesstmp);
      if ( AVI_close(avifile) != 0)
        AVI_print_error("divx_stop: AVI_close:");

      avifile = NULL;
      if (debug)
        fprintf(stderr, "divx_stop: divx file closed\n");
    }

  if(subfile)
    {
      write_sub(NULL);
      fclose(subfile);
      subfile=NULL;
    }

  video_set_capture_size(prev_w_capture, prev_h_capture);

  if (debug)
    fprintf(stderr, "...Done\n");

  if (xawpopup && divx.stream == 0 && !fs)
    PopupMessage(xawmessage);
}

int
is_divx_in_progress(void)
{
  return divx_in_progress;
}

int
is_http_in_progress(void)
{
  return divx_in_progress && divx.stream;
}

int
divx_display_frames(void)
{
  //when divx_stop is started from divx_encode
  //the frame size changes, assigning xvimage to NULL
  if(!displaynextframe) { displaynextframe=1; return 0; }
  return divx.display_frames;
}

#ifndef NOAUDIO
/*****************************************************************************
 *                            RAW AUDIO FUNCTIONS
 ****************************************************************************/

void
divx_audio_start(void)
{
  int arg = 0;
  int ret;

  divx_audio = 1;
  if(audio_open(AUDIO_RO, AUDIO_S16LE, achans, DEC_FREQ, divx.audio_fragments,
		divx.audio_sizefragment)<0) {
    fprintf(stderr,"audio_open failed\n");
    divx_audio = 0;
    return;
  }
  mixer_set_mode(MIXER_MODE_CAPTURE);
#ifdef _POSIX_THREAD_IS_GNU_PTH
  {
    pthread_attr_t attr;
    pthread_attr_init(&attr);
    /* increase if you have a "**Pth** STACK OVERFLOW" */
    pthread_attr_setstacksize(&attr, 256*1024);
    ret = pthread_create (&audio_thread, &attr, audio_record, (void *) arg);
    pthread_attr_destroy(&attr);
    /* in this case audioframetime must be equal to videoframetime */
    divx.audio_buffer_size=DEC_FREQ*achans*fps_den/fps_num;
  }
#else
  ret = pthread_create (&audio_thread, NULL, audio_record, (void *) arg);
#endif

  if (ret != 0)
  {
    sprintf(xawmesstmp, "Audio record thread failed to start.\n");
    fprintf(stderr,xawmesstmp);
    strcat(xawmessage,xawmesstmp);
  }
  else if (debug)
    fprintf (stderr, "Audio record thread started.\n");

}

inline void
write_audio_to_avi(short int *buffer, int size)
{
#ifdef HAVE_LAME
  int nsamples;
#endif // HAVE_LAME

  if(slw) return;
#ifdef SYNCHRO
  if(((double)nbsamplesaudios / DEC_FREQ - audiostamp > 
      (double)video_frames * fps_den / fps_num
      - videostampmin + divx.delay + divx.maxgap)
     || (audiostamp > videostampmax + 4.0)) { //if video is blocked
    dropped_audio_samples += size / achans;
    dropped_audio_buffers++;
    return;
  }
  nbsamplesaudios += size / achans;
#endif

#ifdef HAVE_LAME
  if (divx.compress_audio)
    {
      if(!divx.mp3_vbr) {
	nsamples = lame_encode(buffer, size / achans);
	slw = AVI_write_audio(avifile, lame_buffer, nsamples);
	audio_frames++;
      } else {
	/* in this case an AVI frame must contain only one MP3 frame */
	int i;
	for(i=0;i<size/achans;i+=avifile->a_framesize) {
	  if(i+avifile->a_framesize<size/achans) {
	    nsamples = lame_encode(buffer+i,avifile->a_framesize);
	    slw = AVI_write_audio(avifile, lame_buffer, nsamples);
	    if(slw) return;
	    audio_frames++;
	  } else {
	    nsamples = lame_encode(buffer+i, size / achans - i);
	    if(nsamples) {
	      slw = AVI_write_audio(avifile, lame_buffer, nsamples);
	      audio_frames++;
	    }
	  }
	}
      }
    }
  else
#endif // HAVE_LAME
    {
      slw = 
	AVI_write_audio(avifile, (char *)buffer, size * sizeof (short int));
      audio_frames++;
    }
    
    audio_buffers++;
}

static void*
audio_record(void *arg)
{
  int error;
  short int *buffer;
  struct timeval tv;

  pthread_setcancelstate(PTHREAD_CANCEL_DISABLE,NULL);
  buffer = (short int *)malloc (/*BUFFER_SIZE*/divx.audio_buffer_size * sizeof (short int));
  if (buffer == NULL)
    {
      error = 1;
      goto error;
    }

  while (1)
    {
      int r;
#ifdef _POSIX_THREAD_IS_GNU_PTH
      pthread_yield_np();
#endif
      r = audio_read(buffer, divx.audio_buffer_size*sizeof (short int));
      if(r<0) {
	fprintf(stderr,"audio_read failed");
	error = 2;
	goto error;
      }
      r /= sizeof(short);
      gettimeofday (&tv, NULL);
      audiostamp=tv.tv_usec/1000000.0+tv.tv_sec;
      write_audio_to_avi(buffer, r);

      pthread_setcancelstate(PTHREAD_CANCEL_ENABLE,NULL);
      pthread_testcancel ();
      pthread_setcancelstate(PTHREAD_CANCEL_DISABLE,NULL);
    }
  return NULL;

 error:
  switch (error)
    {
    case 1:
      sprintf(xawmesstmp, "audio_record: Error allocating audio buffer: exiting audio record thread.\n");
      fprintf(stderr,xawmesstmp);
      strcat(xawmessage,xawmesstmp);
      break;
    case 2:
      sprintf(xawmesstmp, "audio_record: Error reading device: exiting audio record thread.\n");
      fprintf(stderr,xawmesstmp);
      strcat(xawmessage,xawmesstmp);
      break;
    }
  pthread_exit ((void *)&error);
}

void
divx_audio_stop(void)
{
  void *ret;

  if(!divx_audio) return;

  pthread_cancel (audio_thread);
  pthread_join (audio_thread, &ret);

  audio_close();
  mixer_set_mode(MIXER_MODE_TVLINE);
  divx_audio= 0;
  
}


/*****************************************************************************
 *                           MP3 AUDIO FUNCTIONS
 ****************************************************************************/
#ifdef HAVE_LAME

inline static int
lame_encode(short int *raw_buffer, int samples_per_channel)
{
  int mp3_samples;

  if(achans == 1)
    mp3_samples = lame_encode_buffer(gfp, raw_buffer, raw_buffer, samples_per_channel, lame_buffer, lame_buffer_size);
  else
    mp3_samples = lame_encode_buffer_interleaved(gfp, raw_buffer, samples_per_channel, lame_buffer, lame_buffer_size);
  return  mp3_samples;
}

static void
lame_stop(void)
{
  int i;

  if(gfp == NULL) return;

  i = lame_encode_flush(gfp, lame_buffer, lame_buffer_size);

  if (avifile && !slw)
    slw = AVI_write_audio(avifile, lame_buffer, i);

  lame_close(gfp);
  gfp = NULL;

  if (lame_buffer)
    {
      free(lame_buffer);
      lame_buffer = NULL;
      lame_buffer_size = 0;
    }
}
#endif // HAVE_LAME

#else //NOAUDIO

inline void write_audio_to_avi(short int *buffer, int size){}

#endif //NOAUDIO


/* function which should be executed in main.c, or by the interface ?*/
void setfps(double fps) 
{
#ifdef SYNCHRO
  divx.fps= 1000 * fps + 0.5;
#endif
}

void write_sub(char *s)
{
  static char *lastsub = NULL;
  static long lastsubbegin;
  long frame;
  int i,j,k,debutligne,lignevide,l,nlignes;
  char *s2;

  if(!subfile) return;
  frame=video_frames;
  if(lastsub!=NULL) {
    if(debug)
      printf("%ld %ld\n",lastsubbegin-lastsubend,
	     frame-lastsubbegin);
    fprintf(subfile,"%ld %ld\n",lastsubbegin-lastsubend,
	    frame-lastsubbegin);
    fputs(lastsub,subfile);
    fputs("\n\n",subfile);
    fflush(subfile);
    lastsubend=frame;
    free(lastsub);
  }
  if(s==NULL || *s=='\0') {lastsub=NULL;return;}
  l=strlen(s);
  s2=(char *)malloc(l+1);

  debutligne=0;
  lignevide=1;
  for(nlignes=0,i=0,j=0;i<l;i++) {
    if(s[i]!='\n' && i!=l-1) {
      if(s[i]!=' ' && s[i]!='#') lignevide=0;
    } else {
      if(s[i]!='\n' && s[i]!='#' && s[i]!=' ') //just case i==l-1
	lignevide=0;
      if(!lignevide && nlignes<8) {
	for(k=debutligne;k<=i;k++,j++)
	  if(s[k]!='#') s2[j]=s[k]; else s2[j]=' ';
	nlignes++;
      }
      debutligne=i+1;
      lignevide=1;
    }
  }
  if(j==0) 
    {free(s2);lastsub=NULL; return;}
  s2[j]='\0';
  lastsub=s2;
  lastsubbegin=frame;
  if(debug)
    printf("lastsub=@@%s@@\n",lastsub);
}


/* forces the capture height except if we are recording */
void force_capt_height_unless_divx(int height) {
  if(!divx_in_progress)
    video_set_capture_size(-2, height);
  else
    prev_h_capture=height;
}
