/*****************************************************************************
 * commands.c: lirc commands handling
 *****************************************************************************
 * $Id: commands.c,v 1.44 2004/12/05 23:14:14 pingus77 Exp $
 *****************************************************************************
 * Copyright (C) 2001 Keuleu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************
 *
 * This software was based on xawtv. Those portions are
 * Copyright (C) 2000 Gerd Knorr <kraxel@cs.tu-berlin.de>
 *
 *****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <math.h>
#include <stdarg.h>
#include <time.h>

#include <X11/Intrinsic.h>

#include "config.h"
#include "commands.h"
#include "channel.h"

/* ----------------------------------------------------------------------- */
extern int debug;
extern int res_mode;
extern char *res_modes[];
extern Widget app_shell;

// Volume inc, dec, mute, tvmute or numeric value
void VolumeAction (Widget, XEvent *, String *, Cardinal *);

// Show Time
void ShowtimeAction (Widget, XEvent *, String *, Cardinal *);
 
 // Mouse Pointer visible / hidden
void PointerAction (Widget, XEvent *, String *, Cardinal *); 

// Toggle fullscreen mode
void FullScreenAction (Widget, XEvent *, String *, Cardinal *);

// recall the last nxtvepg title info
void CallTitleAction (Widget, XEvent *, String *, Cardinal *);

// Stay on Top Yes / No
void StayOnTopAction (Widget, XEvent *, String *, Cardinal *);

// Window Decoration Yes / No
void DecorationAction (Widget, XEvent *, String *, Cardinal *);

// Record a movie Start / Stop
void RecordMovieAction (Widget, XEvent *, String *, Cardinal *);

// Subtitles management
void SubtitlesPagenoAction (Widget, XEvent *, String *, Cardinal *);
void UnSubtitlesPagenoAction (Widget, XEvent *, String *, Cardinal *);
void ToggleSubAction (Widget, XEvent *, String *, Cardinal *);

// call the Teletext
void AleVTAction (Widget, XEvent *, String *, Cardinal *);

// for vtx...
void osd (int x_osd, int y_osd, char *osd_msg, int osd_time);

// Grab an Image: jpeg | ppm & win | full
void SnapAction (Widget, XEvent *, String *, Cardinal *);

// Channel Hopping slow | fast
void ZapAction (Widget, XEvent *, String *, Cardinal *);

//#ifdef USE_LIBXOSD  <= xosd inside now
// Activate / Deactivate Xosd
void XosdAction();
//#endif <= xosd inside now

// Save all parameters in RC file
void SaveAction (Widget, XEvent *, String *, Cardinal *);

// BlackBorders management: inc, dec, numeric value, 16:9, 4:3, 20:9
void SetBlackborderAction (Widget, XEvent *, String *, Cardinal *);

// Others....
void SetChannelAction (Widget, XEvent *, String *, Cardinal *);
void SetSourceAction (Widget, XEvent *, String *, Cardinal *);
void SetResAction (Widget, XEvent *, String *, Cardinal *);
void TuneAction (Widget, XEvent *, String *, Cardinal *);
void DeinterlaceAction (Widget, XEvent *, String *, Cardinal *);
void ExitCB (Widget widget, XtPointer client_data, XtPointer calldata);


char *commands_list[] = {
"setstation",		"Set the TV station",
"setchannel",		"Tune TV channel",
"deinterlace",		"Select a deinterlace filter",
"zap",			"Channel Hopping Slow / Fast / Off",
"volume",		"Set the volume",
"blackborders",          "Set the BlackBorders",
"fullscreen",		"Toggle fullscreen mode",
"setres",		"Set resource value",
"showtime",		"Display time",
"mouse",		"Mouse Pointer Hidden / Visible",
"title",		"recall nxtvepg title info",
"stayontop",		"Stay on Top Yes / No",
"decoration",		"Window Decoration Yes / No",
"xosd",			"Toggles xosd use for osd display",
"list",			"Lists all xdtv commands",
"quit",			"Quit xdtv",
"keypad",		"Enter digit n",
"record",		"Starts/stops recording",
"subpageno",		"Display subtitles: Choose subtitles page number",
"unsubpageno",		"Display subtitles: Choose subtitles page number",
"togglesub",		"Toggles subtitles: Activate/Deactivate subtitles",
"teletext",		"Starts/stops Videotext/Teletext decoder",
"vtx",			"Display subtitles",
"run",			"Runs an external command",
"setinput",		"Sets the input source (Television, Camera, S-VIDEO)",
"snap",			"capture an image",
"save",			"Save The Configuration",
(char*)-1,		(char*)-1
};

/* ----------------------------------------------------------------------- */

static int get_arg(char **arg, char **s)
{
  int n;
  char *tok;

  if ((tok=strtok_r(NULL, " ",s)) != NULL)
    n = 1;
  else
    n = 0;

  *arg = tok;

  return n;
}

int
do_command_single(char *lirc_cmd)
{
    char *tok;
    int   argc, i=0;
    char *cmd;
    char *s;

    cmd = malloc(strlen(lirc_cmd)+1);
    strcpy(cmd, lirc_cmd);

    tok = strtok_r(cmd, " ", &s);
    if(tok != NULL)
      {
	if (strcmp(tok, "setstation") == 0)
          {
	    argc = get_arg(&tok, &s); // next, prev, back
	    SetChannelAction(NULL, NULL, &tok, &argc);
          }
	else if (strcmp(tok, "setchannel") == 0)
          {
	    argc = get_arg(&tok, &s); // next, prev
	    TuneAction(NULL, NULL, &tok, &argc);
          }
	else if (strcmp(tok, "deinterlace") == 0)
          {
	    argc = 0; // no arg just a list
	    DeinterlaceAction(NULL, NULL, NULL, &argc);
          }
	else if (strcmp(tok, "zap") == 0)
          {
	    argc = get_arg(&tok, &s); // no argument = Slow Hopping / Off; argument = fast => Fast Hopping
	    ZapAction(NULL, NULL, &tok, &argc);
          }
	else if (strcmp(tok, "resmode") == 0)
          {
	    res_mode = (res_mode + 1) % 4;
	    osd(30, 20, res_modes[res_mode], 6000);
          }
	else if (strcmp(tok, "resource") == 0)
          {
            char *argv[2];
            argc = get_arg(&tok, &s); // numeric value i.e.: resource +640 or resource -640 only for LIRC!!
            argv[1] = tok;
            argc += 1;
            argv[0] = res_modes[res_mode];
            SetResAction (NULL, NULL, argv, &argc);
          }
	else if  (strcmp(tok, "volume") == 0)
          {
            argc = get_arg(&tok, &s); // inc, dec, mute, tvmute or numeric value
            VolumeAction(NULL, NULL, &tok, &argc);
          } 
        else if  (strcmp(tok, "blackborders") == 0)
          {
            argc = get_arg(&tok, &s); 	// Set Blackborders inc, dec, numeric value, 16:9, 4:3, 20:9 
	    				// 1.33:1, 1.78:1, 2.35:1, 1.66:1, 1.85:1, 2.20:1, 2.40:1,
					// 2.55:1, 2.65:1, 2.76:1
            SetBlackborderAction(NULL, NULL, &tok, &argc);
          }
        else if  (strcmp(tok, "fullscreen") == 0)
          {
            argc = get_arg(&tok, &s); // Toggle fullscreen mode
            FullScreenAction(NULL, NULL, &tok, &argc);
          }
        else if  (strcmp(tok, "stayontop") == 0)
          {
            // no args
            argc = 0; // Stay on Top Yes / No
            StayOnTopAction(NULL, NULL, NULL, &argc);
          }
        else if  (strcmp(tok, "decoration") == 0)
          {
            // no args
            argc = 0; // Window Decoration Yes / No
            DecorationAction(NULL, NULL, NULL, &argc);
          }	  	  
        else if  (strcmp(tok, "setres") == 0)
          {
            char *argv[2];
            argc = get_arg(&tok, &s); // capture
            argv[0] = tok;
            argc += get_arg(&tok, &s); // toggle
            argv[1] = tok;
            SetResAction (NULL, NULL, argv, &argc);
          }
        else if  (strcmp(tok, "showtime") == 0)
          {
            // no args
            argc = 0; // Show Time
            ShowtimeAction (NULL, NULL, NULL, &argc);
          }
        else if  (strcmp(tok, "mouse") == 0)
          {
            // no args
            argc = 0; // Mouse Pointer visible / hidden
            PointerAction (NULL, NULL, NULL, &argc);
          }
        else if  (strcmp(tok, "title") == 0)
          {
            // no args
            argc = 0;  // recall the last nxtvepg title info
	  CallTitleAction (NULL, NULL, NULL, &argc);
          }
        else if  (strcmp(tok, "save") == 0)
          {
            // no args
            argc = 0;  // Save the Configuration
	  SaveAction (NULL, NULL, NULL, &argc);
          }	  
//#ifdef USE_LIBXOSD <= xosd inside now
	 else if  (strcmp(tok, "xosd") == 0)
          {
            // no args
            argc = 0;
            XosdAction ();
          }
//#endif <= xosd inside now
        else if  (strcmp(tok, "list") == 0)
	 {
	   fprintf(stderr,"\nSupported xdtv commands:\n");
	   while ((char *)-1 != commands_list[i])
	    {
		fprintf(stderr,"%15s: ", commands_list[i]);
		fprintf(stderr,"%s\n", commands_list[++i]);
		i++;
	    }
	 }
	else if  (strcmp(tok, "run") == 0)
	 {
	   char command[255] = "\0";
	   int argc;
	   for (i=0;i < 9;i++)
	    {
		argc += get_arg(&tok, &s);
		if (tok)
		 {
			strcat (command, tok);
			strcat (command, " ");
		}
	    }
	    printf("Running %s\n", command);
	    system(command);
	 }
	else if  (strcmp(tok, "quit") == 0)
          {
            // no args
            argc = 0;
            ExitCB (app_shell, NULL, NULL);
          }
        else if  (strcmp(tok, "keypad") == 0)
          {
            argc = get_arg(&tok, &s); // [0..9]
	    if (NULL != tok)
    		SetChannelAction(NULL, NULL, &tok, &argc);
          }
        else if  (strcmp(tok, "record") == 0)
          {
            // no args
            argc = 0; // record a movie Start / Stop
            RecordMovieAction(NULL, NULL, NULL, &argc);
          }
        else if  (strcmp(tok, "subpageno") == 0)
          {
            // no args
            argc = 0;
            SubtitlesPagenoAction(NULL, NULL, NULL, &argc);
          }
        else if  (strcmp(tok, "unsubpageno") == 0)
          {
            // no args
            argc = 0;
            SubtitlesPagenoAction(NULL, NULL, NULL, &argc);
          }
        else if  (strcmp(tok, "togglesub") == 0)
          {
            // no args
            argc = 0;
            ToggleSubAction(NULL, NULL, NULL, &argc);
          }
        else if  (strcmp(tok, "teletext") == 0)
          {
            // no args
            argc = 0;
            AleVTAction(NULL, NULL, NULL, &argc);
          }
	else if (strcmp(tok, "setinput") == 0)
	  {
	    argc = get_arg(&tok, &s); // next, prev
            SetSourceAction(NULL, NULL, &tok, &argc);
	  }
        else if  (strcmp(tok, "snap") == 0)
          {
            char *argv[2];
            argc = get_arg(&tok, &s); // file type (jpeg | ppm)
            argv[0] = tok;
	    if (debug)
	      fprintf(stderr,"First Argument: %s\n",tok);
            argc += get_arg(&tok, &s); // file format (full | win)
            argv[1] = tok;
	    if (debug)
	      fprintf(stderr,"Second Argument: %s\n",tok);
            SnapAction(NULL, NULL, argv, &argc);
          }
        else if  (strcmp(tok, "vtx") == 0)
          {
	    int i,j,nblines=0;
	    tok += strlen(tok)+1;
	    if(tok==cmd+strlen(lirc_cmd)+1)
	      tok="";
	    else {
	      for(i=0;i<strlen(tok);i++) if(tok[i]=='|') tok[i]='\n';
	      /* remove the first empty lines...*/
	      for(i=0,j=-1;tok[i]==' '||tok[i]=='\n';i++)
		if(tok[i]=='\n') j=i;
	      tok=tok+j+1;
	    }
	    for(i=0;i<strlen(tok);i++) if(tok[i]=='\n') nblines++;
	    if(nblines>8) {
	      if(debug) {
		fprintf(stderr,"teletext page contains %d lines\n"
			" does not seem to be a subtitle\n",nblines);
	      }
	    } else
	      osd(-1, -1, tok, -1);
	    tok=NULL;
          }
        else
          {
	    fprintf (stderr, "lirc: unknown command: %s\n", tok);
          }
      }
    if(tok!=NULL) tok=strtok_r(NULL, " ",&s);
    if(tok!=NULL)
      fprintf (stderr,"lirc: not all the arguments are used\n");

    free(cmd);

    return 0;
}

void do_command(char *c) {
  char *tok,*s;

  if(c==NULL) return;
  c=strdup(c);
  tok= strtok_r(c, ";",&s);
  while(tok!=NULL) {
    do_command_single(tok);
    tok=strtok_r(NULL,";",&s);
  }
  free(c);
}

char**
split_cmdline(char *line, int *count)
{
    static char cmdline[1024];
    static char *argv[32];
    int  argc,i;

    strcpy(cmdline,line);
    for (argc=0, i=0; argc<31;) {
	argv[argc++] = cmdline+i;
	while (cmdline[i] != ' ' &&
	       cmdline[i] != '\t' &&
	       cmdline[i] != '\0')
	    i++;
	if (cmdline[i] == '\0')
	    break;
	cmdline[i++] = '\0';
	while (cmdline[i] == ' ' ||
	       cmdline[i] == '\t')
	    i++;
	if (cmdline[i] == '\0')
	    break;
    }
    argv[argc] = NULL;

    *count = argc;
    return argv;
}
