/*
  qpegps is a program for displaying a map centered at the current longitude/
  latitude as read from a gps receiver.

  Copyright (C) 2002 Ralf Haselmeier <Ralf.Haselmeier@gmx.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/


#include "track.h"

#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

// write config file
#define WRITE_CONFIG	application->settings->writeConfig()

/*
 * move to next comma separated field
 */

char *nextfield(char *s) {
    char *r = strchr(s,',');
    if (r) return r+1;
    else return NULL;
}

/*
 * calculate NMEA checksum
 */

void checksumNMEA(QString *ns) {
    const char *n = ns->latin1();
    unsigned checksum = n[1];
    QString nend;

    for (unsigned i=2; i<strlen(n); i++)
        checksum = checksum ^ (unsigned) n[i];
    nend.sprintf("*%02x\r\n",checksum);
    ns->append(nend);
}

/*
 * convert (char*) "hh:mm:ss" to (double) hhmmss
 */

double tmstr2double(char *str) {
    int h,m,s;
    sscanf(str,"%d:%d:%d",&h,&m,&s);
    return (double) (10000*h + 100*m + s);
}

/*
 * convert (double) hhmmss to (char*) "hh:mm:ss"
 */

void tmdouble2str(double tm,char *str) {
    int h = (int) tm / 10000,
	m = (int) tm / 100 - 100 * h,
	s = (int) tm % 100;
    sprintf(str,"%02d:%02d:%02d",h,m,s);
}

/*********************************************************************
 * TRACKPOINT constructor - create trackpoint from various formats
 *********************************************************************/

TrackPoint::TrackPoint(QString *nmea) {
    const char *str = nmea->latin1();

    switch (*str) {
	case '$':	{ 				// assuming NMEA GPGGA messages
            double lon,lat,hdop;
            char lonsign,latsign,*s = (char *) str;
            int status,sats;

            s = nextfield(s);	time = atof(s);
            s = nextfield(s);	lat = atof(s);
            s = nextfield(s);	latsign = *s;
            s = nextfield(s);	lon = atof(s);
            s = nextfield(s);	lonsign = *s;
            s = nextfield(s);	status = atoi(s);
            s = nextfield(s); sats = atoi(s);
            s = nextfield(s); hdop = atof(s);
            s = nextfield(s); altitude = atof(s);

            latitude = floor(lat/100) + (lat - floor(lat/100)*100) / 60;
            longitude = floor(lon/100) + (lon - floor(lon/100)*100) / 60;
            if (latsign == 'S') latitude = -latitude;
            if (lonsign == 'W') longitude = -longitude;
            break; }

	case 'T': {						// assuming PCX5 format
            char tm[20],dt[20];
            sscanf(str,"T %lf %lf %s %s %lf",&latitude,&longitude,dt,tm,&altitude);
            time = tmstr2double(tm);
            break; }

	case ' ': case '-':
	case '0': case '1': case '2': case '3': case '4':
	case '5': case '6': case '7': case '8': case '9':
        {	// assuming gpsdrive format
            char t[5][20];
            sscanf(str,"%lf %lf %lf %s %s %s %s %s",
                   &latitude,&longitude,&altitude,t[0],t[1],t[2],t[3],t[4]);
            for (int i=0; i<5; i++) if (strchr(t[i],':')) // looking for time
		time = tmstr2double(t[i]);
            break; }

	default:
	    qWarning (tr("unknown trackpoint format"));
	    break;
    }
}

/*
 * create trackpoint from real values
 */

TrackPoint::TrackPoint(QString tim,double lat,double lon,double alt) {
    time = tmstr2double((char *) tim.latin1());
    latitude = lat; longitude = lon; altitude = alt;
}

/*
 * create NMEA GGA message from the trackpoint
 */

QString TrackPoint::toNMEA() {
    QString nmea, checksum;
    double lat,lg,tdeg,tmin,ndeg,nmin;
    char tsign,nsign;

    // latitude
    lat = fabs(latitude);
    if (latitude > 0) tsign = 'N';
    else tsign = 'S';
    tdeg = floor(lat);
    tmin = (lat-tdeg)*60.0;

    // longitude
    lg = fabs(longitude);
    if (longitude > 0) nsign = 'E';
    else nsign = 'W';
    ndeg = floor(lg);
    nmin = (lg-ndeg)*60.0;

    nmea.sprintf(
        "$GPGGA,%010.3f,%02d%07.4f,%c,%02d%07.4f,%c,1,0,0,%d,M,,,,0000",
        time,(int) tdeg,tmin,tsign,(int) ndeg,nmin,nsign,(int) altitude);
    checksumNMEA(&nmea);

    return nmea;
}

/*
 * create Garmin PCX5 message for the trackpoint
 */

QString TrackPoint::toPCX5() {
    QString pcx;
    char tm[20];

    tmdouble2str(time,tm);

    // \TODO: date
    pcx.sprintf(
        "T %+09.6f %+09.6f %s %s %4d\n",
        latitude,longitude,"00-JAN-00",tm,(int) altitude);

    return pcx;
}

/*
 * create gpsdrive trackpoint
 */

QString TrackPoint::toDrive() {
    QString drive;
    char tm[20];

    tmdouble2str(time,tm);

    // \TODO: date
    drive.sprintf(
        "%9.6f %9.6f %4d %s\n",
        latitude,longitude,(int) altitude,tm);

    return drive;
}


/*
 * return distance from given point in a strange metric (degree based)
 */

double TrackPoint::dist(double lat,double lon) {
    return fabs(latitude-lat) + fabs(longitude-lon);
}

/*
 * return time difference from given point in seconds
 */

double TrackPoint::timediff(QString tim) {
    int h,m,s;
    sscanf(tim.latin1(),"%d:%d:%d",&h,&m,&s);
    return fabs(time - (double) (10000*h + 100*m + s));
}

/*********************************************************************
 * TRACK constructor - set default values, create widgets
 *********************************************************************/

Track::Track(Qpegps *appl, QWidget *parent,const char *name,WFlags fl):
    QScrollView(parent, name, fl)
{
    application = appl;
    gpsData = application->gpsData;
    wDo = rDo = false;

    logdir = new QDir();
    logdir->setNameFilter("[^.]*");

    setHScrollBarMode(AlwaysOff);
    setVScrollBarMode(Auto);
    mainBox = new QVBox(this);
    addChild(mainBox);
    setResizePolicy(AutoOneFit);


    instructions = new QLabel(
	tr("\n"
	" - TRACKLOG is saved under filename\n"
	"   selected when unchecking write checkbox\n"
	" - (re)select map to display track in Info\n"),
	mainBox);

    // create field for track directory
    tBox = new QHBox(mainBox); tBox->setMargin(5);
    tLabel = new QLabel(tr("Track dir: "),tBox);
    tLE = new QLineEdit(tBox);
    tButton = new QPushButton(tr("search"),tBox);
    tLE->setText(gpsData->trackPathStr);
    connect(tButton,SIGNAL(pressed()),SLOT(setTrackPath()));
    connect(tLE,SIGNAL(returnPressed()),SLOT(tLEChanged()));

    // create checkboxes and comboboxes for tracklog files
    wBox = new QHBox(mainBox); wBox->setMargin(5);
    wCB = new QCheckBox(wBox); wBox->setMargin(5);
    wLabel = new QLabel(tr("write "),wBox);
    wLog = new QComboBox(true,wBox,tr("write_log_filename"));
    connect(wCB,SIGNAL(toggled(bool)),SLOT(setWriteCB(bool)));

    rBox = new QHBox(mainBox); rBox->setMargin(5);
    rCB = new QCheckBox(rBox);
    rLabel = new QLabel(tr("read  "),rBox);
    rLog = new QComboBox(true,rBox,tr("read_log_filename"));
    connect(rCB,SIGNAL(toggled(bool)),SLOT(setReadCB(bool)));
    connect(rLog,SIGNAL(activated(const QString &)),
            SLOT(setReadName(const QString &)) );

    updateFileList();

    // minimal time difference between 2 positions
    dBox = new QHBox(mainBox); dBox->setMargin(5);
    dLabel = new QLabel(tr("min time difference [s]    "),dBox);
    dLE = new QLineEdit(dBox);
    QString buf; buf.sprintf("%d",gpsData->updt_freq);
    dLE->setText(buf);
    connect(dLE,SIGNAL(returnPressed()),SLOT(dLEChanged()));


#ifndef DESKTOP
    // track line thickness
    QStringList thickList;
    thickList << "1" << "2" << "3" << "4" << "5";
    lBox = new QHBox(mainBox); lBox->setMargin(5);
    lLabel = new QLabel(tr("line thickness"),lBox);
    lMenuB = new MenuButton(thickList,lBox);
    lMenuB->select(gpsData->track_thick-1);
    connect(lMenuB,SIGNAL(selected(int)),SLOT(lMenuBChanged(int)));
#endif

#if 0
    // CF GPS period
    cBox = new QHBox(mainBox); cBox->setMargin(5);
    cLabel = new QLabel(tr("CF GPS period [s] "),cBox);
    cLE = new QLineEdit(cBox);
    connect(cLE,SIGNAL(returnPressed()),SLOT(cLEChanged()));
#endif
}

Track::~Track() {
    if (wDo) Write(wLog->currentText());
}

/*
 * update comboboxes with log names
 */

void Track::updateFileList() {
    // keep old selected filenames
    QString wOld = wLog->currentText(), rOld = rLog->currentText();
    // remove old files
    wLog->clear();
    rLog->clear();

    // if trackPathStr exists, change logdir to it
    if (logdir->exists(gpsData->trackPathStr)) {
        logdir->setPath(gpsData->trackPathStr);
        // fill list with new files
        const QFileInfoList *list = logdir->entryInfoList();
        QFileInfoListIterator it( *list );
        QFileInfo *fi;
        while ( (fi = it.current())) {
            // add file to list
            wLog->insertItem(fi->fileName().latin1());
            rLog->insertItem(fi->fileName().latin1());
            // if it's the same as the old one select it
            if (fi->fileName() == wOld) wLog->setCurrentItem(wLog->count() - 1);
            if (fi->fileName() == rOld) rLog->setCurrentItem(rLog->count() - 1);
            ++it;
        }
    }
}

/*
 * write the tracklog which is in mamory
 */

void Track::Write(QString filename,int format) {
    QString				pathfile = gpsData->trackPathStr;
    QTextStream		*wStream;
    QFile					wFile;
    TrackPoint		*tp;

    pathfile.append("/");
    pathfile.append(filename);
    wFile.setName(pathfile);
    if (wFile.open(IO_WriteOnly|IO_Append)) {
        if ((wStream = new QTextStream(&wFile))) {
            while (!wTrack.isEmpty()) {
                tp = wTrack.first();
                switch (format) {
		    case NMEA:		
			*wStream << tp->toNMEA();
			break;
		    case PCX5:	
			*wStream << tp->toPCX5();	
			break;
		    case GPSDRIVE:	*wStream << tp->toDrive();
			break;
                }
                wTrack.removeFirst();
                delete tp;
            }
            delete wStream;
        }
        wFile.close();
    }
    if (filename == rLog->currentText()) Read(filename);
    // we've just written and cleared track we are displaying ... reread
    updateFileList();	// fill comboboxes with log names
}

/*
 * read tracklog into the list of trackpoints
 */

void Track::Read(QString filename) {
    QString				pathfile = gpsData->trackPathStr,trkp;
    QTextStream		*rStream;
    QFile					rFile;
    TrackPoint		*tp;

    pathfile.append("/");
    pathfile.append(filename);
    rFile.setName(pathfile);

    while (!rTrack.isEmpty()) {			// delete old track if any
        tp = rTrack.first();
        rTrack.removeFirst();
        delete tp;
    }

    if (rFile.open(IO_ReadOnly)) {	// open file with new one
        rStream = new QTextStream(&rFile);

        while (!rStream->eof()) {			// get points
            trkp = rStream->readLine();
            tp = new TrackPoint(&trkp);
            rTrack.append(tp);
        }

        delete rStream;
        rFile.close();
    }
}

/*
 * new gps data, update tracklog
 */

void Track::update() {
    if (wDo && gpsData->status) {
        if (wTrack.isEmpty() ||
	    (wTrack.last()->dist(gpsData->currPos.latitude,
				 gpsData->currPos.longitude) > MINTRACKDIST &&
	     wTrack.last()->timediff(gpsData->ts.time) > gpsData->updt_freq)) {

            TrackPoint *tp = new TrackPoint(
		gpsData->ts.time,
		gpsData->currPos.latitude,
		gpsData->currPos.longitude,
		gpsData->altitude.altitude
		);
            wTrack.append(tp);
        }
    }
}

/*
 * toggled write checkbox
 */

void Track::setWriteCB(bool state) {
    wDo = state;
    if (!wDo) Write(wLog->currentText());
}

/*
 * toggled read checkbox
 */

void Track::setReadCB(bool state) {
    TrackPoint *tp;

    rDo = state;
    if (rDo) Read(rLog->currentText());
    else {
        while (!rTrack.isEmpty()) {
            tp = rTrack.first();
            rTrack.removeFirst();
            delete tp;
        }
    }
}

/*
 * changed read log filename
 */

void Track::setReadName(const QString &) {
    if (rDo) Read(rLog->currentText());
}

/*
 * display track on the screen
 */

void Track::drawTrack(QPainter *painter,MapBase *actmap,
                      int x1,int y1,int mx,int my) {
    bool first = true;

    QList<TrackPoint> *disp[2];
    if (rDo) {
        disp[0] = &rTrack;
        // if read and write tracks are the same display also the part in memory
        if (wDo && (wLog->currentText() == rLog->currentText())) disp[1] = &wTrack;
        else disp[1] = NULL;

        for (int i=0; i<2; i++)	if (disp[i] && !disp[i]->isEmpty()) {
            double lg,lt,xwp=0,ywp=0;
            int xtp,ytp;

            QPen pen = painter->pen();
            pen.setColor(*(gpsData->trackColor));
            pen.setWidth(gpsData->track_thick);
            painter->setPen(pen);
            TrackPoint *tp = disp[i]->first();
            while (tp) {	// go through all points
                lt = tp->latitude * M_PI / 180.0;
                lg = tp->longitude * M_PI / 180.0;
                actmap->calcxy(&xwp, &ywp, lg, lt);
                xtp = (int) xwp - x1;
                ytp = (int) ywp - y1;
                if (xtp >= 0 && ytp >= 0 && (int) xwp < mx && (int) ywp <= my) {
                    if (first) {	// first point in read track begins the track
                        painter->moveTo(xtp,ytp); first = false;
                    } else {			// next are added
                        painter->lineTo(xtp,ytp);
                    }
                }
                else
		{
		    // if one point was outside the screen, start line at the next point within the screen
		    first = true;
		}
                tp = disp[i]->next();
            }
        }
    }
}

/*
 * track path changed
 */

void Track::tLEChanged() {
    gpsData->trackPathStr = tLE->text();
    WRITE_CONFIG;
    updateFileList();
}

/*
 * searching for track directory
 */

void Track::setTrackPath() {
    DirDialog getDirDialog(this, 0, TRUE, 0);
    getDirDialog.setCaption(tr("select track directory"));
    getDirDialog.exec();
    if(getDirDialog.result()==QDialog::Accepted) {
        gpsData->trackPathStr = getDirDialog.selectedPath;
        tLE->setText(gpsData->trackPathStr);
    }
    WRITE_CONFIG;
    updateFileList();
}

/*
 * minimal time difference between 2 positions changed
 */

void Track::dLEChanged() {
    gpsData->updt_freq = dLE->text().toInt();
    WRITE_CONFIG;
}

/*
 * line thickness changes
 */

void Track::lMenuBChanged(int idx) {
    gpsData->track_thick = idx + 1;
    WRITE_CONFIG;
}

/*
 * set rate of message
 */

void Track::setRate(unsigned message,unsigned rate) {
    QString msg;
    int fd;

    if (gpsData->gpsdArgStr.contains("/dev/ttyS3")) {
        msg.sprintf("$PSRF103,%02u,00,%02u,01",message,rate);
        checksumNMEA(&msg);
        if ((fd = open("/dev/ttyS3",O_WRONLY))) {
            write(fd,msg.latin1(),strlen(msg.latin1()));
            close(fd);
            qWarning(tr("changing msg%d rate to %ds"),message,rate);
        }
    }
}

/*
 * set rate of all message
 */

void Track::cLEChanged() {
    int i,val = cLE->text().toInt();

    if (val < 1) val = 1;
    else if (val > 255) val = 255;

    for (i=0; i<6; i++) setRate(i,val);
}
