/*
  qpegps is a program for displaying a map centered at the current longitude/
  latitude as read from a gps receiver.
 
  Copyright (C) 2002 Ralf Haselmeier <Ralf.Haselmeier@gmx.de>
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 
*/


#include "settings.h"
ConfigDialog::ConfigDialog(QStringList profiles, int currItem, QWidget *parent, const char *name,
                           bool modal,WFlags f) :
    QDialog(parent,name,modal,f)
{
  resize(220,220);
  vbox = new QVBox(this);
  info = new QLabel(tr("\nqpeGPS\n\nselect a profile:\n"),vbox);
  info->setAlignment(Qt::AlignCenter);
  profileSelector = new QListBox(vbox);
  profileSelector->insertStringList(profiles);
  profileSelector->setCurrentItem(currItem);
  cbox = new QCheckBox(vbox);
  cbox->setText(tr("do not show this dialog again"));
  cbox->setChecked(FALSE);
  vbox->resize(geometry().size());
}

ConfigDialog::~ConfigDialog(){};


Settings::Settings(GpsData *gData, QWidget *parent, const char *name, WFlags fl):
    QScrollView (parent, name, fl)
{
  gpsData = gData;

  setHScrollBarMode(AlwaysOff);
  setVScrollBarMode(Auto);
  mainBox = new QVBox(this);
  addChild(mainBox);
  setResizePolicy(AutoOneFit);
  mainBox->setSizePolicy(QSizePolicy(QSizePolicy::Maximum,QSizePolicy::Maximum));

  configIdx = 0;
  configFileList += QPEGPS_DEFAULT_CONF;
  vbox1 = new QVBox(mainBox);
  horbox2 = new QHBox(vbox1);
  configName = new QComboBox(horbox2);
  configName->setEditable(TRUE);
  configName->setInsertionPolicy(QComboBox::AtCurrent);
  horbox6 = new QHBox(vbox1);
  configAdd = new QPushButton(tr("Add"),horbox6);
  configDel = new QPushButton(tr("Delete"),horbox6);
  showConfigDialog = new QCheckBox(horbox6);
  showConfigDialog->setText(tr("startup dialog"));

  horbox3 = new QHBox(mainBox);
  mapDir = new QLabel(tr("Map dir"),horbox3);
  mapPath = new QLineEdit(horbox3);
  mapPathB = new QPushButton(tr("search"),horbox3);
  /*
      horbox11 = new QHBox(mainBox);
      proxyUrlL = new QLabel(tr("Use Proxy:"),horbox11);
      proxyUrlLE = new QLineEdit(horbox11);
  */
  //geoDatL = new QLabel(tr("GEO Datum:"),mainBox);
  grid2 = new QGrid(2,mainBox);
  geoDatGpsL = new QLabel(tr("GPS"), grid2);
  geoDatGpsL->setAlignment(AlignCenter | AlignVCenter);
  geoDatGpsCB = new QComboBox(grid2);
  geoDatGpsCB->insertStringList(gpsData->geoDatum.getDatumList());
  geoDatGpsCB->setEditable(FALSE);
  geoDatMapL = new QLabel(tr("Map"), grid2);
  geoDatMapL->setAlignment(AlignCenter | AlignVCenter);
  geoDatMapCB = new QComboBox(grid2);
  geoDatMapCB->insertStringList(gpsData->geoDatum.getDatumList());
  geoDatMapCB->setEditable(FALSE);

  grid1 = new QGrid(4,mainBox);
  altL = new QLabel(tr("Altitude"), grid1);
  altL->setAlignment(AlignCenter | AlignVCenter);
  altCB = new QComboBox(FALSE, grid1, "Altitude");
  altCB->insertItem(tr("none"));
  altCB->insertItem(tr("m"));
  altCB->insertItem(tr("feet"));
  altCB->insertItem(tr("FL"));
  speedL = new QLabel(tr("Speed"), grid1);
  speedL->setAlignment(AlignCenter | AlignVCenter);
  speedCB = new QComboBox(FALSE, grid1, "Speed");
  speedCB->insertItem(tr("none"));
  speedCB->insertItem(tr("kmh"));
  speedCB->insertItem(tr("kn"));
  speedCB->insertItem(tr("mph"));
  distL = new QLabel(tr("Distance"), grid1);
  distL->setAlignment(AlignCenter | AlignVCenter);
  distCB = new QComboBox(FALSE, grid1, "Distance");
  distCB->insertItem(tr("none"));
  distCB->insertItem(tr("km"));
  distCB->insertItem(tr("nmi"));
  distCB->insertItem(tr("mi"));
  posL = new QLabel(tr("Position"), grid1);
  posL->setAlignment(AlignCenter | AlignVCenter);
  posCB = new QComboBox(FALSE, grid1, "Position");
  posCB->insertItem(tr("DD.d"));
  posCB->insertItem(tr("DDMM.m'"));
  posCB->insertItem(tr("DDMM'SS.s''"));
  textSizeL = new QLabel(tr("Text Size"), grid1);
  textSizeL->setAlignment(AlignRight | AlignVCenter);
  textSizeSB = new QSpinBox(4, 30, 1, grid1, "Text Size");
  new QLabel(" ", grid1); // dummy label for grid format
  colorB = new QPushButton(tr("Colors"),grid1);

  horbox1 = new QHBox(mainBox);
  checkLabel = new QLabel(tr("display:"), horbox1 );
  bearTB = new QCheckBox(horbox1);
  bearTB->setText(tr("Bearing"));
  headTB = new QCheckBox(horbox1);
  headTB->setText(tr("Heading"));
  timeTB = new QCheckBox(horbox1);
  timeTB->setText(tr("Time"));

  selectConfigR(TRUE);

  // create config file
  selectConfigW(FALSE);


  connect( altCB, SIGNAL(activated(int)),
           SLOT(setAlt(int)) );
  connect( speedCB, SIGNAL(activated(int)),
           SLOT(setSpeed(int)) );
  connect( distCB, SIGNAL(activated(int)),
           SLOT(setDist(int)) );
  connect( posCB, SIGNAL(activated(int)),
           SLOT(setPos(int)) );
  connect( bearTB, SIGNAL(toggled(bool)),
           SLOT(setBear(bool)) );
  connect( headTB, SIGNAL(toggled(bool)),
           SLOT(setHead(bool)) );
  connect( timeTB, SIGNAL(toggled(bool)),
           SLOT(setTime(bool)) );
  connect( showConfigDialog, SIGNAL(clicked()),
           SLOT(setDialog()) );
  connect( mapPathB, SIGNAL(pressed()),
           SLOT(setMapPath()) );
  connect( colorB, SIGNAL(pressed()),
           SLOT(setColors()) );
  connect( mapPath, SIGNAL(returnPressed()),
           SLOT(mapPathLEChanged()) );
  /*
      connect( proxyUrlLE, SIGNAL(returnPressed()),
               SLOT(proxyUrlLEChanged()) );
  */
  connect( textSizeSB , SIGNAL(valueChanged(int)),
           SLOT(textSizeChanged(int)) );
  connect( geoDatGpsCB , SIGNAL(activated(int)),
           SLOT(geoDatGpsChanged(int)) );
  connect( geoDatMapCB , SIGNAL(activated(int)),
           SLOT(geoDatMapChanged(int)) );
  connect( configAdd, SIGNAL(pressed()),
           SLOT(addConfig()) );
  connect( configDel, SIGNAL(pressed()),
           SLOT(delConfig()) );
  connect( configName, SIGNAL(activated(int)),
           SLOT(selectConfig(int)) );
}

Settings::~Settings()
{}

void Settings::selectConfigR(bool construct)
{
  /* which config ? */
  qpegpsConfigSelect = new Config("qpegps");
  qpegpsConfigSelect->setGroup("main");
  configIdx = qpegpsConfigSelect->readNumEntry("current",0);
  configFileList = qpegpsConfigSelect->readListEntry("profiles",'*');
  if(configFileList.isEmpty())
    configFileList += QPEGPS_DEFAULT_CONF;
  configName->clear();
  configName->insertStringList(configFileList);
  showConfigDialog->setChecked(qpegpsConfigSelect->readBoolEntry("startupDialog", TRUE));

  if(construct && showConfigDialog->isChecked() && configFileList.count() > 1)
  {
    ConfigDialog cDialog(configFileList, configIdx, this, tr("select profile"), TRUE, 0);
    cDialog.setCaption(tr("qpeGPS"));
    cDialog.exec();
    configIdx = cDialog.profileSelector->currentItem();
    showConfigDialog->setChecked(!(cDialog.cbox->isChecked()));
  }

  configName->setCurrentItem(configIdx);
  qpegpsConfig = new Config("qpegps_" + configFileList[configIdx] + ".conf",Config::File);

  // new Settings are valid
  readConfig();
  writeConfig(); //creates file if it is not existing, yet
}

void Settings::selectConfigW(bool add)
{
  /* which config ? */
  qpegpsConfigSelect->setGroup("main");
  qpegpsConfigSelect->writeEntry("current",(int)configIdx);
  qpegpsConfigSelect->writeEntry("profiles",configFileList.join("*"));
  qpegpsConfigSelect->writeEntry("startupDialog",showConfigDialog->isChecked());

  qpegpsConfigSelect->~Config();
  qpegpsConfigSelect = new Config("qpegps");

  qpegpsConfig->~Config();
  qpegpsConfig = new Config("qpegps_" + configFileList[configIdx] + ".conf",Config::File);

  // new Settings are valid
  if(!add)readConfig();
  writeConfig(); //creates file if it is not existing, yet
}


void Settings::readConfig()
{
  qpegpsConfig->setGroup("units");
  gpsData->altitude.altUnit = (Altitude::Alt) qpegpsConfig->readNumEntry("altitude", Altitude::Feet);
  gpsData->speed.speedUnit = (Speed::Sp) qpegpsConfig->readNumEntry("speed", Speed::Knots);
  gpsData->wpDistance.distUnit = (Distance::Dist) qpegpsConfig->readNumEntry("distance", Distance::Naut);
  gpsData->currPos.posUnit = (Position::Pos) qpegpsConfig->readNumEntry("position", Position::DegMin);

  qpegpsConfig->setGroup("show");
  gpsData->bearing.show = qpegpsConfig->readBoolEntry("bearing", TRUE);
  gpsData->heading.show = qpegpsConfig->readBoolEntry("heading", TRUE);
  gpsData->showTime = qpegpsConfig->readBoolEntry("time", TRUE);

  qpegpsConfig->setGroup("gps");
  gpsData->gpsdArgStr = qpegpsConfig->readEntry("gpsd",gpsdDefaultArg);
  gpsData->host = qpegpsConfig->readEntry("host",gpsdDefaultHost);
  gpsData->port = qpegpsConfig->readNumEntry("port",gpsdDefaultPort);

  qpegpsConfig->setGroup("map");
  gpsData->mapPathStr = qpegpsConfig->readEntry("path",gpsData->qpedir+"/qpegps/maps");

  qpegpsConfig->setGroup("download");
  gpsData->proxyUrl = qpegpsConfig->readEntry("proxy","");

  QDir md(gpsData->mapPathStr);

  //gpsData->mapPathStr = md.canonicalPath();

  qpegpsConfig->setGroup("icons");
  gpsData->iconsPathStr = qpegpsConfig->readEntry("path",gpsData->qpedir+"/qpegps/icons");
  QDir md2(gpsData->iconsPathStr);
  //gpsData->iconsPathStr = md2.canonicalPath();

  qpegpsConfig->setGroup("datum");
  gpsData->gpsDatumIdx = qpegpsConfig->readEntry("GpsDatum","1").toInt();
  gpsData->mapDatumIdx = qpegpsConfig->readEntry("MapsDatum","1").toInt();

  qpegpsConfig->setGroup("color");

  okColorName = qpegpsConfig->readEntry("ok","#000000");
  noFixColorName = qpegpsConfig->readEntry("noFix","#ffff00");
  headColorName = qpegpsConfig->readEntry("heading","#00ff00");
  bearColorName = qpegpsConfig->readEntry("bearing","#ff0000");
  trackColorName = qpegpsConfig->readEntry("trackC","#ff0000");

  gpsData->textSize = qpegpsConfig->readEntry("textSize","15").toInt();

  qpegpsConfig->setGroup("track");
  gpsData->trackPathStr = qpegpsConfig->readEntry("path",
                          gpsData->qpedir+"/qpegps/tracks");
  QDir md3(gpsData->trackPathStr);
  //gpsData->trackPathStr = md3.canonicalPath();
  gpsData->updt_freq = qpegpsConfig->readEntry("updt_freq","10").toInt();
  gpsData->track_thick = qpegpsConfig->readEntry("track_thick","2").toInt();

  // set buttons...
  altCB->setCurrentItem((int)gpsData->altitude.altUnit);
  speedCB->setCurrentItem((int)gpsData->speed.speedUnit);
  distCB->setCurrentItem((int)gpsData->wpDistance.distUnit);
  posCB->setCurrentItem((int)gpsData->currPos.posUnit);
  bearTB->setChecked(gpsData->bearing.show);
  headTB->setChecked(gpsData->heading.show);
  timeTB->setChecked(gpsData->showTime);
  mapPath->setText(gpsData->mapPathStr);
  textSizeSB->setValue(gpsData->textSize);
  // proxyUrlLE->setText(gpsData->proxyUrl);
  geoDatGpsCB->setCurrentItem(gpsData->gpsDatumIdx-1);
  geoDatMapCB->setCurrentItem(gpsData->mapDatumIdx-1);

  delete gpsData->statusOkColor;
  delete gpsData->statusNoFixColor;
  delete gpsData->headColor;
  delete gpsData->bearColor;
  delete gpsData->trackColor;

  gpsData->statusOkColor = new QColor(okColorName);
  gpsData->statusNoFixColor = new QColor(noFixColorName);
  gpsData->headColor = new QColor(headColorName);
  gpsData->bearColor = new QColor(bearColorName);
  gpsData->trackColor = new QColor(trackColorName);
}


void Settings::writeConfig()
{
  qpegpsConfig->setGroup("units");
  qpegpsConfig->writeEntry("altitude",(int)gpsData->altitude.altUnit);
  qpegpsConfig->writeEntry("speed",(int)gpsData->speed.speedUnit);
  qpegpsConfig->writeEntry("distance",(int)gpsData->wpDistance.distUnit);
  qpegpsConfig->writeEntry("position",(int)gpsData->currPos.posUnit);
  qpegpsConfig->setGroup("show");
  qpegpsConfig->writeEntry("bearing",gpsData->bearing.show);
  qpegpsConfig->writeEntry("heading",gpsData->heading.show);
  qpegpsConfig->writeEntry("time",gpsData->showTime);
  qpegpsConfig->setGroup("gps");
  qpegpsConfig->writeEntry("gpsd", gpsData->gpsdArgStr);
  qpegpsConfig->writeEntry("host", gpsData->host);
  qpegpsConfig->writeEntry("port", gpsData->port);
  qpegpsConfig->setGroup("map");
  qpegpsConfig->writeEntry("path", gpsData->mapPathStr);
  qpegpsConfig->setGroup("download");
  qpegpsConfig->writeEntry("proxy",gpsData->proxyUrl);
  qpegpsConfig->setGroup("color");

  qpegpsConfig->writeEntry("ok", okColorName);
  qpegpsConfig->writeEntry("noFix", noFixColorName);
  qpegpsConfig->writeEntry("heading", headColorName);
  qpegpsConfig->writeEntry("bearing", bearColorName);
  qpegpsConfig->writeEntry("trackC", trackColorName);

  qpegpsConfig->writeEntry("textSize", gpsData->textSize);
  qpegpsConfig->setGroup("track");
  qpegpsConfig->writeEntry("path", gpsData->trackPathStr);
  qpegpsConfig->writeEntry("updt_freq", gpsData->updt_freq);
  qpegpsConfig->writeEntry("track_thick", gpsData->track_thick);
  qpegpsConfig->setGroup("datum");
  qpegpsConfig->writeEntry("GpsDatum", (int)gpsData->gpsDatumIdx);
  qpegpsConfig->writeEntry("MapsDatum", (int)gpsData->mapDatumIdx);

  qpegpsConfig->~Config();
  qpegpsConfig = new Config("qpegps_" + configFileList[configIdx] + ".conf",Config::File);

}

void Settings::setAlt(int id)
{
  gpsData->altitude.altUnit = (Altitude::Alt)id;
  writeConfig();
}
void Settings::setSpeed(int id)
{
  gpsData->speed.speedUnit = (Speed::Sp)id;
  writeConfig();
}
void Settings::setDist(int id)
{
  gpsData->wpDistance.distUnit = (Distance::Dist)id;
  writeConfig();
}
void Settings::setPos(int id)
{
  gpsData->currPos.posUnit = (Position::Pos)id;
  writeConfig();
}

void Settings::setBear(bool state)
{
  gpsData->bearing.show = state;
  writeConfig();
}
void Settings::setHead(bool state)
{
  gpsData->heading.show = state;
  writeConfig();
}
void Settings::setTime(bool state)
{
  gpsData->showTime = state;
  writeConfig();
}
void Settings::setDialog()
{
  selectConfigW(FALSE);
}

#if 0
void Settings::proxyUrlLEChanged()
{
  gpsData->proxyUrl = proxyUrlLE->text();
  writeConfig();
}
#endif

void Settings::mapPathLEChanged()
{
  gpsData->mapPathStr = mapPath->text();
  QDir md(gpsData->mapPathStr);

  //gpsData->mapPathStr = md.canonicalPath();
  writeConfig();
  emit mapPathChanged();
}

void Settings::setMapPath()
{
  // FIXME *gpsData->mapPathStr = QFileDialog::getExistingDirectory(*gpsData->mapPathStr);
  DirDialog getDirDialog(this, 0, TRUE, 0);
  getDirDialog.setCaption(tr("select map directory"));
  getDirDialog.exec();
  if(getDirDialog.result()==QDialog::Accepted)
  {
    gpsData->mapPathStr = getDirDialog.selectedPath;
    QDir md(gpsData->mapPathStr);

    //gpsData->mapPathStr = md.canonicalPath();
    mapPath->setText(gpsData->mapPathStr);
    writeConfig();
    emit mapPathChanged();
  }

}
void Settings::setColors()
{
  QValueList <QColor>colors;
  colors.append(*gpsData->statusOkColor);
  colors.append(*gpsData->statusNoFixColor);
  colors.append(*gpsData->headColor);
  colors.append(*gpsData->bearColor);
  colors.append(*gpsData->trackColor);
  QStringList items;
  items.append(tr("for status \"GPS OK\""));
  items.append(tr("for status \"no position fix\""));
  items.append(tr("heading"));
  items.append(tr("bearing"));
  items.append(tr("track"));

  ColorDlg setColorDialog(&colors,&items,this, 0, TRUE, 0);
  setColorDialog.setCaption(tr("assign colors"));
  setColorDialog.exec();
  if(setColorDialog.result()==QDialog::Accepted)
  {
    *gpsData->statusOkColor = colors[0];
    *gpsData->statusNoFixColor = colors[1];
    *gpsData->headColor = colors[2];
    *gpsData->bearColor = colors[3];
    *gpsData->trackColor = colors[4];
    okColorName = gpsData->statusOkColor->name();
    noFixColorName = gpsData->statusNoFixColor->name();
    headColorName = gpsData->headColor->name();
    bearColorName = gpsData->bearColor->name();
    trackColorName = gpsData->trackColor->name();
    writeConfig();
  }
}


void Settings::textSizeChanged(int idx)
{
  gpsData->textSize = idx;
  writeConfig();
}

void Settings::geoDatGpsChanged(int idx)
{
  gpsData->gpsDatumIdx = idx+1;
  writeConfig();
}

void Settings::geoDatMapChanged(int idx)
{
  gpsData->mapDatumIdx = idx+1;
  writeConfig();
}

void Settings::addConfig()
{
  int i=1;
  configIdx = configFileList.count();
  while(configFileList.contains(tr("newProfile%1").arg(i)))
    i++;
  configFileList.append(tr("newProfile%1").arg(i));
  configName->clear();
  configName->insertStringList(configFileList);
  configName->setCurrentItem(configIdx);
  selectConfigW(TRUE);
}

void Settings::delConfig()
{
  if(configFileList.count() > 1)
  {
    QString delFile;
    delFile = "rm qpegps_" + configFileList[configIdx] + ".conf";
    configFileList.remove(configFileList[configIdx]);
    configName->clear();
    configName->insertStringList(configFileList);
    configIdx = 0;
    configName->setCurrentItem(configIdx);
    selectConfigW(FALSE);
    system(delFile.latin1());
  }
}


void Settings::selectConfig(int idx)
{
  if(idx != (int)configIdx)
  {
    //select
    configIdx = idx;
    selectConfigW(FALSE);
  }
  else
  {
    // renmae
    QString delFile;
    delFile = "rm qpegps_" + configFileList[configIdx] + ".conf";
    if(QString::compare(configFileList[configIdx], configName->currentText()))
    {
      configFileList[configIdx] = configName->currentText();
      selectConfigW(TRUE);
      system(delFile.latin1());
    }
  }
}



