/*
  qpegps is a program for displaying a map centered at the current longitude/
  latitude as read from a gps receiver.

  Copyright (C) 2002 Ralf Haselmeier <Ralf.Haselmeier@gmx.de>
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 
*/


#include "gpsdata.h"
#include "track.h"

// initialization of static elements:
Altitude::Alt Altitude::altUnit = Altitude::Feet;
Speed::Sp Speed::speedUnit = Speed::Knots;
Distance::Dist Distance::distUnit = Distance::Naut;
Position::Pos Position::posUnit = Position::DegMin;

QString Altitude::toString()
{
    switch(altUnit)
    {
    case None:
        return tr("");
        break;
    case Meter:
        return tr("%1 m").arg(altitude,0,'f',0);
        break;
    case Feet:
        return tr("%1 ft").arg(altitude*3.2808399,0,'f',0);
        break;
    case FL:
        return tr("%1 FL").arg(altitude*0.032808399,0,'f',1);
        break;
    }
    return 0;
}

QString Speed::toString()
{
    switch(speedUnit)
    {
    case None:
        return tr("");
        break;
    case Kmh:
        return tr("%1 kmh").arg(speed*1.852,0,'f',1);
        break;
    case Knots:
        return tr("%1 kn").arg(speed,0,'f',1);
        break;
    case Mph:
        return tr("%1 mph").arg(speed*1.1507794,0,'f',1);
        break;
    }
    return 0;
}

QString Distance::toString()
{
    switch(distUnit)
    {
    case None:
        return tr("");
        break;
    case Km:
        return tr("%1 km").arg(distance*1.852,0,'f',1);
        break;
    case Naut:
        return tr("%1 nmi").arg(distance,0,'f',1);
        break;
    case Statute:
        return tr("%1 mi").arg(distance*1.1507794,0,'f',1);
        break;
    }
    return 0;
}

Position::Position(QWidget *parent, const char *name):QObject(parent,name)
{
    latitude=0;
    longitude=0;
}

QString Position::latToString()
{
    double lat,deg,min,sec;
    char sign;
    lat = fabs(latitude);
    if(latitude > 0)
        sign = 'N';
    else
        sign = 'S';
    switch(posUnit)
    {
    case Degree:
        return tr("%1%2").arg(lat,0,'f',5).arg(sign);
        break;
    case DegMin:
        deg = floor(lat);
        min = (lat-deg)*60.0;
        return tr("%1%2'%3").arg(deg, 0,'f',0)
               .arg(min,0,'f',3).arg(sign);
        break;
    case DegMinSec:
        deg = floor(lat);
        min = floor((lat-deg)*60.0);
        sec = (lat-(deg+min/60.0))*3600.0;
        return tr("%1%2'%3\"%4").arg(deg, 0,'f',0)
               .arg(min, 0,'f',0)
               .arg(sec, 0,'f',2).arg(sign);
        break;
    }
    return 0;
}

QString Position::longToString()
{
    double lg,deg,min,sec;
    char sign;
    lg = fabs(longitude);
    if(longitude > 0)
        sign = 'E';
    else
        sign = 'W';
    switch(posUnit)
    {
    case Degree:
        return tr("%1%2").arg(lg,0,'f',5).arg(sign);
        break;
    case DegMin:
        deg = floor(lg);
        min = (lg-deg)*60.0;
        return tr("%1%2'%3").arg(deg, 0,'f',0)
               .arg(min,0,'f',3).arg(sign);
        break;
    case DegMinSec:
        deg = floor(lg);
        min = floor((lg-deg)*60.0);
        sec = (lg-(deg+min/60.0))*3600.0;
        return tr("%1%2'%3\"%4").arg(deg, 0,'f',0)
               .arg(min, 0,'f',0)
               .arg(sec, 0,'f',2).arg(sign);
        break;
    }
    return 0;
}

double Position::setLong(QString lgString)
{
    int p;
    float deg=0, min=0, sec=0, west=1;
    QString lgStr;
    QRegExp re("[\\s'\"]");

    lgStr = lgString.simplifyWhiteSpace();

    p=lgStr.find('w',0,FALSE);
    if(p>=0)
    {
        west = -1.0;
        lgStr.remove(p,1);
        lgStr = lgStr.simplifyWhiteSpace();
    }
    p=lgStr.find('e',0,FALSE);
    if(p>=0)
    {
        lgStr.remove(p,1);
        lgStr = lgStr.simplifyWhiteSpace();
    }

    deg = lgStr.toFloat();
    p = lgStr.find(re);
    if(p>0)
    {
        lgStr = lgStr.mid(p+1);
        min = lgStr.toFloat();
        p = lgStr.find(re);
        if(p>0)
        {
            lgStr = lgStr.mid(p+1);
            sec = lgStr.toFloat();
        }
    }
    longitude = west * (deg + (min/60.0) + (sec/3600.0));
    return longitude;
}

double Position::setLat(QString ltString)
{
    int p;
    float deg=0, min=0, sec=0, south=1;
    QString ltStr;
    QRegExp re("[\\s'\"]");

    ltStr = ltString.simplifyWhiteSpace();

    p=ltStr.find('s',0,FALSE);
    if(p>=0)
    {
        south = -1.0;
        ltStr.remove(p,1);
        ltStr = ltStr.simplifyWhiteSpace();
    }
    p=ltStr.find('n',0,FALSE);
    if(p>=0)
    {
        ltStr.remove(p,1);
        ltStr = ltStr.simplifyWhiteSpace();
    }

    deg = ltStr.toFloat();
    p = ltStr.find(re);
    if(p>0)
    {
        ltStr = ltStr.mid(p+1);
        min = ltStr.toFloat();
        p = ltStr.find(re);
        if(p>0)
        {
            ltStr = ltStr.mid(p+1);
            sec = ltStr.toFloat();
        }
    }
    latitude = south * (deg + (min/60.0) + (sec/3600.0));
    return latitude;
}

QString Position::setLong(double lg)
{
    longitude = lg;
    return longToString();
}

QString Position::setLat(double lt)
{
    latitude = lt;
    return latToString();
}

QString Angle::toString()
{
    if(show)
        return tr("%1").arg(angle,0,'f',0);
    else
        return tr("");
}

QString TimeStamp::toString()
{
    if (date.isEmpty() || time.isEmpty())
        return "* No GMT Signal rcvd *";

    return date + " " + time;
}


bool SatInfo::operator == (const SatInfo & other) const
{
    return ((d_satName == other.d_satName) &&
            (d_elevation == other.d_elevation) &&
            (d_azimut == other.d_azimut) &&
            (d_snr == other.d_snr));
}

SatInfo & SatInfo::operator =(const class SatInfo & other)
{
    // memberwise copy (needed here because QObject operator = is private)
    d_satName = other.d_satName;
    d_elevation = other.d_elevation;
    d_azimut = other.d_azimut;
    d_snr = other.d_snr;
    d_updated = other.d_updated;

    return *this;
}

GpsData::GpsData(QWidget *parent, const char *name):
        QObject(parent,name)
{
    d_connected = false;
    d_aliveGPS = false;
    ttwph=99; ttwpm=99; ttwps=99;
    showTime = FALSE;
    ts.date = "";
    ts.time = "";
    d_no_of_satellites = 0;
    d_no_of_fix_satellites = 0;
    d_Receiver = "";
}

GpsData::~GpsData()
{
};

QString GpsData::timeToString()
{
    if(avspeed.speed > 0)
    {
        ttwph = wpDistance.distance / avspeed.speed;
        ttwpm = (ttwph - floor(ttwph)) * 60.0;
        ttwph = floor(ttwph);
        ttwps = floor((ttwpm - floor(ttwpm)) * 60.0);
        ttwpm = floor(ttwpm);
    }

    if(showTime)
        return tr("%1:%2:%3").arg(ttwph,2,'f',0).arg(ttwpm,2,'f',0).arg(ttwps,2,'f',0);
    else
        return tr("");
}

