/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2006 NoMachine, http://www.nomachine.com/.         */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rigths reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "NXlib.h"

#include "Mask.h"
#include "Tight.h"

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG

#define RGB24_TO_PIXEL32(r,g,b)                                 \
  (((CARD32)(r) & 0xFF) << format.redShift   |                  \
   ((CARD32)(g) & 0xFF) << format.greenShift |                  \
   ((CARD32)(b) & 0xFF) << format.blueShift)

/*
 * Maximum size of the header of the data
 * part of one image
 * (1 for the filter type       +
 *  4 for the number of entries +
 *  256 * 4 for the colors      )
 *
 */

#define MAX_HEADER_SIZE 1029

/*
 * Filters to improve zlib data compression
 */

#define rfbTightFilterCopy             0x00
#define rfbTightFilterPalette          0x01
#define rfbTightFilterGradient         0x02


/*
 * Compression level stuff. The following array
 * contains various encoder parameters for each
 * of 10 compression levels (0..9). 
 */

typedef struct TIGHT_CONF_s
{
  int maxRectSize, maxRectWidth;
  int monoMinRectSize, gradientMinRectSize;
  int idxZlibLevel, monoZlibLevel, rawZlibLevel, gradientZlibLevel;
  unsigned int gradientThreshold, gradientThreshold24;
  int idxMaxColorsDivisor;
} TIGHT_CONF;


/*
 * Stuff dealing with palettes.
 */

typedef struct COLOR_LIST_s
{
  struct COLOR_LIST_s *next;
  int idx;
  CARD32 rgb;
} COLOR_LIST;

typedef struct PALETTE_ENTRY_s
{
  COLOR_LIST *listNode;
  int numPixels;
} PALETTE_ENTRY;

typedef struct PALETTE_s
{
  PALETTE_ENTRY entry[256];
  COLOR_LIST *hash[256];
  COLOR_LIST list[256];
} PALETTE;

typedef struct
{
  CARD8  bitsPerPixel; 
  CARD8  depth;
  CARD8  bigEndian;
  CARD16 redMax;
  CARD16 greenMax;
  CARD16 blueMax;
  CARD8  redShift;
  CARD8  greenShift;
  CARD8  blueShift;
} tightPixelFormat;


/*
 * Prototypes of functions.
 */

void CopyBuffer                      (char *srcBuf, char *dstBuf, int x, int y, int w,
                                          int h, int bytesPerLine, int bitsPerPixel);

Bool CheckSolidTile8                 (int x, int y, int w, int h, CARD32 *colorPtr,
                                          Bool needSameColor);
Bool CheckSolidTile16                (int x, int y, int w, int h, CARD32 *colorPtr,
                                          Bool needSameColor);
Bool CheckSolidTile24                (int x, int y, int w, int h, CARD32 *colorPtr,
                                          Bool needSameColor);
Bool CheckSolidTile32                (int x, int y, int w, int h, CARD32 *colorPtr,
                                          Bool needSameColor);

Bool SendMonoRect                    (int x, int y, int w, int h);
Bool SendIndexedRect                 (int x, int y, int w, int h);
Bool SendFullColorRect               (int x, int y, int w, int h);
Bool SendGradientRect                (int x, int y, int w, int h);

void FillPalette8                    (int count);
void FillPalette16                   (int count);
void FillPalette24                   (int count);
void FillPalette32                   (int count);

void PaletteReset                    (void);
int  PaletteInsert                   (CARD32 rgb, int numPixels, int bpp);

void Pack24                          (char *buf, tightPixelFormat *fmt, int count);

void EncodeIndexedRect16             (CARD8 *buf, int count);
void EncodeIndexedRect24             (CARD8 *buf, int count);
void EncodeIndexedRect32             (CARD8 *buf, int count);

void EncodeMonoRect8                 (CARD8 *buf, int w, int h);
void EncodeMonoRect16                (CARD8 *buf, int w, int h);
void EncodeMonoRect24                (CARD8 *buf, int w, int h);
void EncodeMonoRect32                (CARD8 *buf, int w, int h);

void FilterGradient24                (char *buf, tightPixelFormat *fmt, int w, int h);
void FilterGradient24bpp             (char *buf, tightPixelFormat *fmt, int w, int h);
void FilterGradient16                (CARD16 *buf, tightPixelFormat *fmt, int w, int h);
void FilterGradient32                (CARD32 *buf, tightPixelFormat *fmt, int w, int h);

int           DetectSmoothImage      (tightPixelFormat *fmt, int w, int h);
unsigned long DetectSmoothImage24    (tightPixelFormat *fmt, int w, int h);
unsigned long DetectSmoothImage24bpp (tightPixelFormat *fmt, int w, int h);
unsigned long DetectSmoothImage16    (tightPixelFormat *fmt, int w, int h);
unsigned long DetectSmoothImage32    (tightPixelFormat *fmt, int w, int h);


static Bool rfbTightDisableGradient = 0;

/*
 * Keep in this variable the depth of
 * execution of the recursive function
 * NXEncodeTight
 */

static int recDepth = 0;

/*
 * Parameters for every level of tight
 * encoding.
 */

static TIGHT_CONF tightConf[10] =
{
    {   512,   32,   6, 65536, 0, 0, 0, 0,   0,   0,   4 },
    {  2048,  128,   6, 65536, 1, 1, 1, 0,   0,   0,   8 },
    {  6144,  256,   8, 65536, 3, 3, 2, 0,   0,   0,  24 },
    { 10240, 1024,  12, 65536, 5, 5, 3, 0,   0,   0,  32 },
    { 16384, 2048,  12, 65536, 6, 6, 4, 0,   0,   0,  32 },
    { 32768, 2048,  12,  4096, 7, 7, 5, 4, 150, 380,  32 },
    { 65280, 2048,  16,  4096, 7, 7, 6, 4, 170, 420,  48 },
    { 65280, 2048,  16,  4096, 8, 8, 7, 5, 180, 450,  64 },
    { 65280, 2048,  32,  8192, 9, 9, 8, 6, 190, 475,  64 },
    { 65280, 2048,  32,  8192, 9, 9, 9, 6, 200, 500,  96 }
};

/*
 * Variable indicating that the image is of
 * depth 24 and redMax, greenMax, blueMax
 * are all 0xFF.
 */

static Bool usePixelFormat24;

/*
 * Variables that are set at the 1st
 * execution of NXEncodeTight.
 */

static int              compressLevel;
static tightPixelFormat format;
static int              packMethod = 0;

static int              globalParamsSet = 0;

/*
 * Variables that are set at every call
 * of NXEncodeTight.
 */

static Display*         display;
static Drawable         dwble;
static Visual*          vis;
static char*            imgDataBuf;
static int              imgBpl, imgFormat;
static void*            gContext;
static XImage*          maskedImage;
static int              maskMethod;
static const ColorMask* colormask;

static int              imgParamsSet = 0;

/*
 * Stuff dealing with palettes.
 */

static int              paletteNumColors, paletteMaxColors;
static CARD32           monoBackground, monoForeground;
static PALETTE          palette;

/*
 * Pointers to dynamically-allocated
 * buffers.
 */

static int              tightBeforeBufSize = 0;
static char*            tightBeforeBuf = NULL;
static char*            tightFilteredBuf = NULL;

static int*             prevRowBuf = NULL;

/*
 * Functions implementation.
 */

int NXEncodeTightInit(Display *dpy, Drawable drawable, Visual *visual, void *gc,
                          XImage *image, unsigned int method, unsigned int mask_method)
{
  /*
   * Setup of parameters that are application
   * specific. Done only once, at the first
   * call of NXEncodeTight.
   */

  if (!globalParamsSet)
  {
    unsigned int linkType;
    unsigned int protoMajor, protoMinor, protoPatch;
    int          frameTimeout, pingTimeout, splitMode, splitThreshold;
    int          dataLevel, streamLevel, deltaLevel;
    int          i;
    unsigned int nxPackMethod;
    unsigned int nxPackQuality;
    unsigned int loadCache, saveCache, startupCache;

    /*
     * Pixel format.
     */

    format.depth        = image->depth;
    format.bitsPerPixel = image->bits_per_pixel;
    format.bigEndian    = image->bitmap_bit_order == MSBFirst;

    format.redShift   = FindLSB(image->red_mask) - 1;
    format.greenShift = FindLSB(image->green_mask) - 1;
    format.blueShift  = FindLSB(image->blue_mask) - 1;

    format.redMax   = image->red_mask >> format.redShift;
    format.greenMax = image->green_mask >> format.greenShift;
    format.blueMax  = image->blue_mask >> format.blueShift;

    if (format.depth == 24 && format.bitsPerPixel == 32 &&
          format.redMax == 0xFF && format.greenMax == 0xFF &&
              format.blueMax == 0xFF)
    {
      usePixelFormat24 = 1;
    }
    else
    {
      usePixelFormat24 = 0;
    }

    /*
     * Calculate the tight compression level
     * according to the zlib compression level
     * used by the proxies.
     */

    NXGetControlParameters(dpy, &linkType, &protoMajor, &protoMinor, &protoPatch,
                               &protoMajor, &protoMinor, &protoPatch, &frameTimeout, &pingTimeout,
                                   &splitMode, &splitThreshold, &nxPackMethod, &nxPackQuality,
                                       &dataLevel, &streamLevel, &deltaLevel, &loadCache,
                                           &saveCache, &startupCache);

    #ifdef DEBUG
    fprintf(stderr, "******NXEncodeTightInit : dataLevel [%d]\n", dataLevel);
    #endif

    for(i = 0; (i < 10) && (dataLevel >= tightConf[i].idxZlibLevel); i++);
    if(i < 10)
      compressLevel = i;
    else
      compressLevel = i -1;

    globalParamsSet = 1;
  }

  /*
   * Setup of parameters that are image specific.
   * Done when there's a new image to handle, but
   * not in the recursive calls.
   */

  if(!imgParamsSet)
  {

    /*
     * Final display.
     */

    display = dpy;
    dwble = drawable;
    gContext = gc;
    vis = visual;

    /*
     * Image stuff.
     */

    imgDataBuf = image->data;
    imgBpl = image->bytes_per_line;
    imgFormat = image->format;

    /*
     * Pack method and Mask method.
     */

    packMethod = method;
    maskMethod = mask_method;
    colormask = MethodColorMask(maskMethod);

    /*
     * Store the image pointer to free at
     * the end in NXENcodeTightUninit
     */

    maskedImage = image;

    imgParamsSet = 1;
  }

  #ifdef DEBUG
  fprintf(stderr, "******NXEncodeTightInit: depth[%d] bpp[%d] rm[%d] gm[%d] bm[%d] rs[%d] gs[%d] bs[%d]\n",
              format.depth, format.bitsPerPixel, format.redMax, format.greenMax, format.blueMax,
                  format.redShift, format.greenShift, format.blueShift);
  #endif

  return 1;
}


void NXEncodeTightUninit()
{
  if(maskedImage != NULL)
  {
    XFree(maskedImage->data);
    maskedImage -> data = NULL;
    XFree(maskedImage);
    maskedImage = NULL;
  }

  imgDataBuf = NULL;
  imgParamsSet = 0;
  recDepth = 0;
}


void IncreaseRecDepth()
{
  recDepth++;
}


void DecreaseRecDepth()
{
  recDepth--;

  if(recDepth <= 0)
  {
    NXEncodeTightUninit();
  }
}


int GetEncodeTightMaxRows(int w)
{
  int maxRectSize, maxRectWidth, nMaxWidth;

  maxRectSize = tightConf[compressLevel].maxRectSize;
  maxRectWidth = tightConf[compressLevel].maxRectWidth;
  nMaxWidth = (w > maxRectWidth) ? maxRectWidth : w;

  return maxRectSize / nMaxWidth;
}


int GetEncodeTightInitStatus()
{
  return imgParamsSet;
}


void FindBestSolidArea(int x, int y, int w, int h, CARD32 colorValue,
                                  int *w_ptr, int *h_ptr)
{
  int dx, dy, dw, dh;
  int w_prev;
  int w_best = 0, h_best = 0;

  #ifdef DEBUG
  fprintf(stderr, "******FindBestSolidArea: x[%d] y[%d] w[%d] h[%d]\n",
             x, y, w, h);
  #endif

  w_prev = w;

  for (dy = y; dy < y + h; dy += MAX_SPLIT_TILE_SIZE)
  {

    dh = (dy + MAX_SPLIT_TILE_SIZE <= y + h) ? MAX_SPLIT_TILE_SIZE : (y + h - dy);
    dw = (w_prev > MAX_SPLIT_TILE_SIZE) ? MAX_SPLIT_TILE_SIZE : w_prev;

    if (!CheckSolidTile(x, dy, dw, dh, &colorValue, 1))
      break;

    for (dx = x + dw; dx < x + w_prev;)
    {
      dw = (dx + MAX_SPLIT_TILE_SIZE <= x + w_prev) ? MAX_SPLIT_TILE_SIZE : (x + w_prev - dx);
      if (!CheckSolidTile(dx, dy, dw, dh, &colorValue, 1))
        break;
      dx += dw;
    }

    w_prev = dx - x;
    if (w_prev * (dy + dh - y) > w_best * h_best)
    {
      w_best = w_prev;
      h_best = dy + dh - y;
    }
  }

  *w_ptr = w_best;
  *h_ptr = h_best;
}


void ExtendSolidArea(int x, int y, int w, int h, CARD32 colorValue,
                                int *x_ptr, int *y_ptr, int *w_ptr, int *h_ptr)
{
  int cx, cy;

  #ifdef DEBUG
  fprintf(stderr, "******ExtendSolidArea: x[%d] y[%d] w[%d] h[%d]\n",
              x, y, w, h);
  #endif

  /*
   * Try to extend the area upwards.
   */

  for ( cy = *y_ptr - 1;
        cy >= y && CheckSolidTile(*x_ptr, cy, *w_ptr, 1, &colorValue, 1);
        cy-- );
  *h_ptr += *y_ptr - (cy + 1);
  *y_ptr = cy + 1;

  /*
   * ... downwards.
   */

  for ( cy = *y_ptr + *h_ptr;
        cy < y + h &&
        CheckSolidTile(*x_ptr, cy, *w_ptr, 1, &colorValue, 1);
        cy++ );
  *h_ptr += cy - (*y_ptr + *h_ptr);

  /*
   * ... to the left.
   */

  for ( cx = *x_ptr - 1;
        cx >= x && CheckSolidTile(cx, *y_ptr, 1, *h_ptr, &colorValue, 1);
        cx-- );
  *w_ptr += *x_ptr - (cx + 1);
  *x_ptr = cx + 1;

  /*
   * ... to the right.
   */

  for ( cx = *x_ptr + *w_ptr;
        cx < x + w &&
        CheckSolidTile(cx, *y_ptr, 1, *h_ptr, &colorValue, 1);
        cx++ );
  *w_ptr += cx - (*x_ptr + *w_ptr);
}


Bool CheckSolidTile(int x, int y, int w, int h,
                               CARD32 *colorPtr, Bool needSameColor)
{
  switch(format.bitsPerPixel)
  {
  case 32:
    return CheckSolidTile32(x, y, w, h, colorPtr, needSameColor);
  case 24:
    return CheckSolidTile24(x, y, w, h, colorPtr, needSameColor);
  case 16:
    return CheckSolidTile16(x, y, w, h, colorPtr, needSameColor);
  default:
    return CheckSolidTile8(x, y, w, h, colorPtr, needSameColor);
  }
}

Bool CheckSolidTile24(int x, int y, int w, int h, 
                          CARD32 *colorPtr, Bool needSameColor)          
{                                                                             
  CARD8 *fbptr;
  CARD32 colorValue, curColor;
  int dx, dy;

  fbptr = (CARD8 *)&imgDataBuf[y * imgBpl + x * 3];

  colorValue = RGB24_TO_PIXEL32(*fbptr, *(fbptr + 1), *(fbptr + 2));

  if (needSameColor && (CARD32)colorValue != *colorPtr)
    return 0;
                                                                              
  for (dy = 0; dy < h; dy++)
  {
    for (dx = 0; dx < w; dx++)
    {
      curColor = RGB24_TO_PIXEL32(*(fbptr + dx*3), *(fbptr + dx*3 + 1), *(fbptr + dx*3 + 2));

      if (colorValue != curColor)
      {
        return 0;
      }
    }

    fbptr += imgBpl;
  }
                                                                              
  *colorPtr = (CARD32)colorValue;
  return 1;
}

#define DEFINE_CHECK_SOLID_FUNCTION(bpp)                                      \
                                                                              \
Bool CheckSolidTile##bpp(int x, int y, int w, int h,                          \
                               CARD32 *colorPtr, Bool needSameColor)          \
{                                                                             \
    CARD##bpp *fbptr;                                                         \
    CARD##bpp colorValue;                                                     \
    int dx, dy;                                                               \
                                                                              \
    fbptr = (CARD##bpp *)&imgDataBuf[y * imgBpl + x * format.bitsPerPixel/8]; \
    colorValue = *fbptr;                                                      \
    if (needSameColor && (CARD32)colorValue != *colorPtr)                     \
        return 0;                                                             \
                                                                              \
    for (dy = 0; dy < h; dy++)                                                \
    {                                                                         \
      for (dx = 0; dx < w; dx++)                                              \
      {                                                                       \
            if (colorValue != fbptr[dx])                                      \
                return 0;                                                     \
      }                                                                       \
      fbptr = (CARD##bpp *)((CARD8 *)fbptr + imgBpl);                         \
    }                                                                         \
                                                                              \
    *colorPtr = (CARD32)colorValue;                                           \
    return 1;                                                                 \
}                                                                             \

DEFINE_CHECK_SOLID_FUNCTION(8)
DEFINE_CHECK_SOLID_FUNCTION(16)
DEFINE_CHECK_SOLID_FUNCTION(32)


Bool SendRectSimple(int src_x, int src_y, int dst_x, int dst_y, int w, int h)
{
  int maxBeforeSize, maxRectSize, maxRectWidth;
  int subrectMaxWidth, subrectMaxHeight;
  int dx, dy;
  int rw, rh;

  maxRectSize = tightConf[compressLevel].maxRectSize;
  maxRectWidth = tightConf[compressLevel].maxRectWidth;

  maxBeforeSize = maxRectSize * (format.bitsPerPixel / 8) + MAX_HEADER_SIZE;

  #ifdef DEBUG
  fprintf(stderr, "******SendRectSimple: src_x[%d] src_y[%d] dst_x[%d] dst_y[%d] w[%d] h[%d]\n",
              src_x, src_y, dst_x, dst_y, w, h);
  #endif

  if (tightBeforeBufSize < maxBeforeSize)
  {
    tightBeforeBufSize = maxBeforeSize;

    if(tightFilteredBuf == NULL)
      tightFilteredBuf = (char *)malloc(tightBeforeBufSize);
    else
      tightFilteredBuf = (char *)realloc(tightFilteredBuf, tightBeforeBufSize);

    /*
     * tightBeforeBuf points just to the data part
     * tightFilteredBuf points to all the buffer
     */

    tightBeforeBuf = tightFilteredBuf + MAX_HEADER_SIZE;
  }

  if (w > maxRectWidth || w * h > maxRectSize)
  {
    subrectMaxWidth = (w > maxRectWidth) ? maxRectWidth : w;
    subrectMaxHeight = maxRectSize / subrectMaxWidth;

    for (dy = 0; dy < h; dy += subrectMaxHeight)
    {
      for (dx = 0; dx < w; dx += maxRectWidth)
      {
        rw = (dx + maxRectWidth < w) ? maxRectWidth : w - dx;
        rh = (dy + subrectMaxHeight < h) ? subrectMaxHeight : h - dy;

        if (!SendSubrect(src_x + dx, src_y + dy, dst_x + dx, dst_y + dy, rw, rh))
          return 0;
      }
    }
  }
  else
  {
    if (!SendSubrect(src_x, src_y, dst_x, dst_y, w, h))
      return 0;
  }

  return 1;
}


Bool SendSubrect(int src_x, int src_y, int dst_x, int dst_y, int w, int h)
{
  Bool success = 0;

  #ifdef DEBUG
  fprintf(stderr, "******SendSubRect: src_x[%d] src_y[%d] dst_x[%d] dst_y[%d] w[%d] h[%d]\n",
              src_x, src_y, dst_x, dst_y, w, h);
  #endif

  CopyBuffer(imgDataBuf, tightBeforeBuf, src_x, src_y, w, h, imgBpl, format.bitsPerPixel);

  #ifdef DEBUG
  fprintf(stderr, "******SendSubrect: buffer copied\n");
  #endif

  paletteMaxColors = w * h / tightConf[compressLevel].idxMaxColorsDivisor;

  if ( paletteMaxColors < 2 && w * h >= tightConf[compressLevel].monoMinRectSize )
  {
    paletteMaxColors = 2;
  }

  switch (format.bitsPerPixel)
  {
  case 8:
    FillPalette8(w * h);
    break;
  case 16:
    FillPalette16(w * h);
    break;
  case 24:
    FillPalette24(w * h);
    break; 
  default:
    FillPalette32(w * h);
  }

  #ifdef DEBUG
  fprintf(stderr, "******SendSubrect: number of colors [%d]\n",
              paletteNumColors);
  #endif

  switch (paletteNumColors)
  {
    case 0:
    {

      /*
       * Truecolor image.
       */

      if (DetectSmoothImage(&format, w, h))
        success = SendGradientRect(dst_x, dst_y, w, h);
      else
        success = SendFullColorRect(dst_x, dst_y, w, h);
      break;
    }
    case 1:
    {

      /*
       *Solid rectangle.
       */

      success = SendSolidRect(src_x, src_y, dst_x, dst_y, w, h);
      break;
    }
    case 2:
    {

      /*
       * Two-color rectangle.
       */

      success = SendMonoRect(dst_x, dst_y, w, h);
      break;
    }
    default:
    {

      /*
       * Up to 256 different colors.
       */

      success = SendIndexedRect(dst_x, dst_y, w, h);
    }

  }

  return success;
}


/*
 * Subencoding implementations.
 */

Bool SendSolidRect(int src_x, int src_y, int dst_x, int dst_y, int w, int h)
{
  XGCValues gcv;
  char *colorBuf;

  #ifdef DEBUG
  fprintf(stderr, "******SendSolidRect: src_x[%d] src_y[%d] dst_x[%d] dst_y[%d] w[%d] h[%d]\n",
              src_x, src_y, dst_x, dst_y, w, h);
  #endif

  colorBuf = imgDataBuf + src_y * imgBpl + src_x * (int)(format.bitsPerPixel / 8);

  switch(format.bitsPerPixel)
  {
    case 8:
    {
      gcv.foreground = *(CARD8 *)colorBuf;
      break;
    }
    case 16:
    {
      CARD16 colorValue;

      colorValue = *(CARD16 *)colorBuf;

      /*
       * Apply the brightness correction, as done
       * in Unpack16To16 inside Unpack.cpp in
       * nxcomp
       */

      if(colormask -> correction_mask &&
             colorValue != 0x0000 && colorValue != 0xFFFF)
      {
        gcv.foreground = (((((colorValue & 0xf100) >> 8) | colormask -> correction_mask) << 8) & 0xf800) |
                             (((((colorValue & 0x7c0) >> 3) | colormask -> correction_mask) << 3) & 0x7e0) |
                                 (((((colorValue & 0x1f) << 3) | colormask -> correction_mask) >> 3) & 0x1f);
      }
      else
      {
        gcv.foreground = colorValue;
      }

      break;
    }
    case 24:
    {
      CARD32 colorValue;

      colorValue = RGB24_TO_PIXEL32(*(colorBuf), *(colorBuf + 1), *(colorBuf + 2));

      gcv.foreground = colorValue;

      break;

    }
    case 32:
    {
      gcv.foreground = *(CARD32 *)colorBuf;

      /*
       * Apply the brightness correction, as done
       * in Unpack32To32 inside Unpack.cpp in
       * nxcomp
       */

      if(colormask -> correction_mask &&
             gcv.foreground != 0x00000000 &&
                 gcv.foreground != 0xFFFFFFFF)
      {
        gcv.foreground |= ( (colormask -> correction_mask << 16) |
                                 (colormask -> correction_mask << 8)  |
                                    (colormask -> correction_mask) );
      }
      break;
    }
    default:
    {
      #ifdef PANIC
      fprintf(stderr, "******SendSolidRect: PANIC! unsupported Bpp[%d]\n",
                  format.bitsPerPixel);
      #endif

      return 0;
    }
  }

  /*
   * A simple FillRectangle is used
   */

  XChangeGC(display, gContext, GCForeground, &gcv);
  XFillRectangle(display, dwble, gContext, dst_x, dst_y, w, h);

  return 1;
}


Bool SendMonoRect(int x, int y, int w, int h)
{
  int dataLen;
  XImage *image;
  char *finalDataBuf;

  #ifdef DEBUG
  fprintf(stderr, "******SendMonoRect: x[%d] y[%d] w[%d] h[%d]\n",
              x, y, w, h);
  #endif


  /*
   * Put the filter as header of the
   * data part of the packed image.
   */

  if(format.bitsPerPixel == 24)
  {
    finalDataBuf = tightBeforeBuf - 5 - 2 * 4;
  }
  else
  {
    finalDataBuf = tightBeforeBuf - 5 - 2 * format.bitsPerPixel / 8;
  }
  finalDataBuf = tightBeforeBuf - 5 - 2 * format.bitsPerPixel / 8;
  *(CARD8 *)finalDataBuf = (CARD8)rfbTightFilterPalette;
  *(CARD32 *)(finalDataBuf + 1) = 2;

  /*
   * Convert image.
   */

  switch (format.bitsPerPixel)
  {
    case 8:
    {
      EncodeMonoRect8((CARD8 *)tightBeforeBuf, w, h);
      *(finalDataBuf + 5) = (CARD8)monoBackground;
      *(finalDataBuf + 6) = (CARD8)monoForeground;
    }
    case 16:
    {
      EncodeMonoRect16((CARD8 *)tightBeforeBuf, w, h);
      *(CARD16 *)(finalDataBuf + 5) = (CARD16)monoBackground;
      *(CARD16 *)(finalDataBuf + 7) = (CARD16)monoForeground;
      break;
    }
    case 24:
    {
      EncodeMonoRect24((CARD8 *)tightBeforeBuf, w, h);
      *(CARD32 *)(finalDataBuf + 5) = monoBackground;
      *(CARD32 *)(finalDataBuf + 9) = monoForeground;
      break;
    }
    default:
    {
      EncodeMonoRect32((CARD8 *)tightBeforeBuf, w, h);
      *(CARD32 *)(finalDataBuf + 5) = monoBackground;
      *(CARD32 *)(finalDataBuf + 9) = monoForeground;
      break;
    }
  }

  dataLen = (w + 7) / 8;
  dataLen *= h;

  if(format.bitsPerPixel == 24)
  {
    dataLen += 5 + 2 * 4;
  }
  else
  {
    dataLen += 5 + 2 * format.bitsPerPixel / 8;
  }
  
  image = NXCreatePackedImage(display, vis, packMethod, format.depth, imgFormat, NULL,
                                  dataLen, w, h, BitmapPad(display), 0);
  image->data = finalDataBuf;

  NXPutPackedImage(display, 0, dwble, gContext, image, packMethod,
                       format.depth, 0, 0, x, y, w, h);
  XFree(image);

  return 1;
}

Bool SendIndexedRect(int x, int y, int w, int h)
{
  int i, dataLen;
  XImage *image;
  char *finalDataBuf;

  #ifdef DEBUG
  fprintf(stderr, "******SendIndexedRect: x[%d] y[%d] w[%d] h[%d]\n",
              x, y, w, h);
  #endif

  /*
   * Convert image.
   */

  switch (format.bitsPerPixel)
  {
    case 32:
    {
      EncodeIndexedRect32((CARD8 *)tightBeforeBuf, w * h);
      break;
    }
    case 24:
    {
      EncodeIndexedRect24((CARD8 *)tightBeforeBuf, w * h);
      break;
    }
    case 16:
    {
      EncodeIndexedRect16((CARD8 *)tightBeforeBuf, w * h);
      break;
    }
    default:
    {

      /*
       * Should never happen.
       */

      return 0;
    }
  }

  /*
   * Put the filter as header of the
   * data part of the packed image.
   */

  if(format.bitsPerPixel == 24)
  {
    finalDataBuf = tightBeforeBuf - 5 - paletteNumColors * 4;
  }
  else
  {
    finalDataBuf = tightBeforeBuf - 5 - paletteNumColors * format.bitsPerPixel / 8;
  }
  *(CARD8 *)finalDataBuf = (CARD8)rfbTightFilterPalette;
  *(CARD32 *)(finalDataBuf + 1) = paletteNumColors;


  switch(format.bitsPerPixel)
  {
    case 8:
    {
      for (i = 0; i < paletteNumColors; i++)
      {
        *(finalDataBuf + 5 + i) = (CARD8)palette.entry[i].listNode->rgb & 0x000000FF;
      }
      break;
    }
    case 16:
    {
      for (i = 0; i < paletteNumColors; i++)
      {
        *(CARD16 *)(finalDataBuf + 5 + i*2) = (CARD16)palette.entry[i].listNode->rgb & 0x0000FFFF;
      }
      break;
    }
    default:
    {
      for (i = 0; i < paletteNumColors; i++)
      {
        if (usePixelFormat24)
        {
          *(CARD32 *)(finalDataBuf + 5 + i*4) = (CARD32)palette.entry[i].listNode->rgb & 0x00FFFFFF;
        }
        else
        {
          *(CARD32 *)(finalDataBuf + 5 + i*4) = (CARD32)palette.entry[i].listNode->rgb;
        }
      }
      break;
    }
  }

  if(format.bitsPerPixel == 24)
  {
    dataLen = w * h + 5 + paletteNumColors * 4;
  }
  else
  {
    dataLen = w * h + 5 + paletteNumColors * format.bitsPerPixel / 8;
  }

  image = NXCreatePackedImage(display, vis, packMethod, format.depth, imgFormat, NULL,
                                  dataLen, w, h, BitmapPad(display), 0);
  image->data = finalDataBuf;

  NXPutPackedImage(display, 0, dwble, gContext, image, packMethod,
                       format.depth, 0, 0, x, y, w, h);
  XFree(image);

  return 1;
}

Bool SendFullColorRect(int x, int y, int w, int h)
{
  int len, dataLen;
  XImage *image;
  char *finalDataBuf;

  #ifdef DEBUG
  fprintf(stderr, "******SendFullColorRect: x[%d] y[%d] w[%d] h[%d]\n",
              x, y, w, h);
  #endif

  /*
   * Put the filter as header of the
   * data part of the packed image.
   */

  finalDataBuf = tightBeforeBuf - 1;

  *finalDataBuf = (CARD8)rfbTightFilterCopy;

  /*
   * Convert image if needed
   */

  if (usePixelFormat24)
  {
    Pack24(tightBeforeBuf, &format, w * h);
    len = 3;
  }
  else
    len = format.bitsPerPixel / 8;

  dataLen = w * h * len + 1;

  image = NXCreatePackedImage(display, vis, packMethod, format.depth, imgFormat, NULL,
                                  dataLen, w, h, BitmapPad(display), 0);
  image->data = finalDataBuf;

  NXPutPackedImage(display, 0, dwble, gContext, image, packMethod,
                       format.depth, 0, 0, x, y, w, h);
  XFree(image);

  return 1;
}

Bool SendGradientRect(int x, int y, int w, int h)
{
  int len, dataLen;
  XImage *image;
  char *finalDataBuf;

  #ifdef DEBUG
  fprintf(stderr, "******SendGradientRect: x[%d] y[%d] w[%d] h[%d]\n",
              x, y, w, h);
  #endif

  if (format.bitsPerPixel == 8)
    return SendFullColorRect(x, y, w, h);

  /*
   * Put the filter as header of the
   * data part of the packed image.
   */

  finalDataBuf = tightBeforeBuf - 1;
  *finalDataBuf = (CARD8)rfbTightFilterGradient;


  /*
   * Convert image.
   */

  if (prevRowBuf == NULL)
    prevRowBuf = (int *)malloc(2048 * 3 * sizeof(int));

  if (usePixelFormat24)
  {
    FilterGradient24(tightBeforeBuf, &format, w, h);
    len = 3;
  }
  else if (format.bitsPerPixel == 32)
  {
    FilterGradient32((CARD32 *)tightBeforeBuf, &format, w, h);
    len = 4;
  }
  else if (format.bitsPerPixel == 24)
  {
    FilterGradient24bpp(tightBeforeBuf, &format, w, h);
    len = 3;
  }
  else
  {
    FilterGradient16((CARD16 *)tightBeforeBuf, &format, w, h);
    len = 2;
  }

  dataLen = w * h * len + 1;

  image = NXCreatePackedImage(display, vis, packMethod, format.depth, imgFormat, NULL,
                                  dataLen, w, h, BitmapPad(display), 0);
  image->data = finalDataBuf;

  NXPutPackedImage(display, 0, dwble, gContext, image, packMethod,
                       format.depth, 0, 0, x, y, w, h);
  XFree(image);

  return 1;
}

/*
 * Determine how many different colors
 * are used in a rectangle.
 */

void FillPalette8(int count)
{
  CARD8 *data = (CARD8 *)tightBeforeBuf;
  CARD8 c0, c1;
  int i, n0, n1;

  paletteNumColors = 0;

  c0 = data[0];
  for (i = 1; i < count && data[i] == c0; i++);
  if (i == count)
  {
    /*
     * Solid rectangle.
     */

    paletteNumColors = 1;
    return;
  }

  if (paletteMaxColors < 2)
    return;

  n0 = i;
  c1 = data[i];
  n1 = 0;
  for (i++; i < count; i++)
  {
    if (data[i] == c0)
    {
      n0++;
    }
    else if (data[i] == c1)
    {
      n1++;
    }
    else
      break;
  }
  if (i == count)
  {
    if (n0 > n1)
    {
      monoBackground = (CARD32)c0;
      monoForeground = (CARD32)c1;
    }
    else
    {
      monoBackground = (CARD32)c1;
      monoForeground = (CARD32)c0;
    }

    /*
     * Two colors
     */

    paletteNumColors = 2;   
  }
}

void FillPalette24(int count)                                        
{                                                                       
  CARD8 *data = (CARD8 *)tightBeforeBuf;                        
  CARD32 c0, c1, ci, color;                                                 
  int i, n0, n1, ni;                                                    
                                                                        
  color = c0 = RGB24_TO_PIXEL32(*data, 
                                *(data + 1),
                                *(data + 2));
  ci = 0;                                                               
  for (i = 1; i < count && color == c0; i++)
  {
    color = RGB24_TO_PIXEL32(*(data + i*3),
                             *(data + i*3 + 1),
                             *(data + i*3 + 2));
  }
  
  if (i >= count)                                                       
  {                                                                     
                                                                        
    /*                                                                  
     * Solid rectangle.                                                 
     */                                                                 
                                                                        
    paletteNumColors = 1;                                               
    return;                                                             
  }                                                                     
                                                                        
  if (paletteMaxColors < 2)                                             
  {                                                                     
                                                                        
    /*                                                                  
     * Full-color encoding preferred.                                   
     */                                                                 
                                                                        
    paletteNumColors = 0;                                               
    return;                                                             
  }                                                                     
                                                                        
  n0 = i;                                                               
  c1 = color;
  n1 = 0;
  for (i++; i < count; i++)                                             
  {                                                                         
    ci = RGB24_TO_PIXEL32(*(data + i*3),
                          *(data + i*3 + 1),
                          *(data + i*3 + 2));

    if (ci == c0)                                   
    {                                                                   
      n0++;                                                             
    }                                                                   
    else if (ci == c1)                                                  
    {                                                                   
      n1++;                                                             
    }                                                                   
    else                                                                
      break;                                                            
  }                                                                     
  if (i >= count)                                                       
  {                                                                     
    if (n0 > n1)                                                        
    {                                                                   
      monoBackground = (CARD32)c0;                                      
      monoForeground = (CARD32)c1;                                      
    }                                                                   
    else                                                                
    {                                                                   
      monoBackground = (CARD32)c1;                                      
      monoForeground = (CARD32)c0;                                      
    }                                                                   
                                                                        
    /*                                                                  
     * Two colors                                                       
     */                                                                 
                                                                        
    paletteNumColors = 2;                                               
    return;                                                             
  }                                                                     
                                                                        
  PaletteReset();                                                       
  PaletteInsert (c0, (CARD32)n0, 32);                                  
  PaletteInsert (c1, (CARD32)n1, 32);                                  
                                                                        
  ni = 1;                                                               

  for (i++; i < count; i++)                                             
  {                                                                     
    color = RGB24_TO_PIXEL32(*(data + i*3), *(data + i*3 + 1), *(data + i*3 + 2)); 

    if (color == ci)
    {
      ni++;
    }
    else                                                                
    {                                                                   
      if (!PaletteInsert (ci, (CARD32)ni, 32))                         
        return;                                                         
      ci = color;                                                     
      ni = 1;                                                           
    }                                                                   
  }                                                                     

  PaletteInsert (ci, (CARD32)ni, 32);                                  
}


#define DEFINE_FILL_PALETTE_FUNCTION(bpp)                               \
                                                                        \
void FillPalette##bpp(int count)                                        \
{                                                                       \
  CARD##bpp *data = (CARD##bpp *)tightBeforeBuf;                        \
  CARD##bpp c0, c1, ci;                                                 \
  int i, n0, n1, ni;                                                    \
                                                                        \
  c0 = data[0];                                                         \
  ci = 0;                                                               \
  for (i = 1; i < count && data[i] == c0; i++);                         \
  if (i >= count)                                                       \
  {                                                                     \
                                                                        \
    /*                                                                  \
     * Solid rectangle.                                                 \
     */                                                                 \
                                                                        \
    paletteNumColors = 1;                                               \
    return;                                                             \
  }                                                                     \
                                                                        \
  if (paletteMaxColors < 2)                                             \
  {                                                                     \
                                                                        \
    /*                                                                  \
     * Full-color encoding preferred.                                   \
     */                                                                 \
                                                                        \
    paletteNumColors = 0;                                               \
    return;                                                             \
  }                                                                     \
                                                                        \
  n0 = i;                                                               \
  c1 = data[i];                                                         \
  n1 = 0;                                                               \
  for (i++; i < count; i++)                                             \
  {                                                                     \
    ci = data[i];                                                       \
    if (ci == c0)                                                       \
    {                                                                   \
      n0++;                                                             \
    }                                                                   \
    else if (ci == c1)                                                  \
    {                                                                   \
      n1++;                                                             \
    }                                                                   \
    else                                                                \
      break;                                                            \
  }                                                                     \
  if (i >= count)                                                       \
  {                                                                     \
    if (n0 > n1)                                                        \
    {                                                                   \
      monoBackground = (CARD32)c0;                                      \
      monoForeground = (CARD32)c1;                                      \
    }                                                                   \
    else                                                                \
    {                                                                   \
      monoBackground = (CARD32)c1;                                      \
      monoForeground = (CARD32)c0;                                      \
    }                                                                   \
                                                                        \
    /*                                                                  \
     * Two colors                                                       \
     */                                                                 \
                                                                        \
    paletteNumColors = 2;                                               \
    return;                                                             \
  }                                                                     \
                                                                        \
  PaletteReset();                                                       \
  PaletteInsert (c0, (CARD32)n0, bpp);                                  \
  PaletteInsert (c1, (CARD32)n1, bpp);                                  \
                                                                        \
  ni = 1;                                                               \
  for (i++; i < count; i++)                                             \
  {                                                                     \
    if (data[i] == ci)                                                  \
    {                                                                   \
      ni++;                                                             \
    }                                                                   \
    else                                                                \
    {                                                                   \
      if (!PaletteInsert (ci, (CARD32)ni, bpp))                         \
        return;                                                         \
      ci = data[i];                                                     \
      ni = 1;                                                           \
    }                                                                   \
  }                                                                     \
  PaletteInsert (ci, (CARD32)ni, bpp);                                  \
}

DEFINE_FILL_PALETTE_FUNCTION(16)
DEFINE_FILL_PALETTE_FUNCTION(32)


/*
 * Functions to operate with palette structures.
 */

#define HASH_FUNC16(rgb) ((int)((((rgb) >> 8) + (rgb)) & 0xFF))
#define HASH_FUNC32(rgb) ((int)((((rgb) >> 16) + ((rgb) >> 8)) & 0xFF))

void PaletteReset(void)
{
  paletteNumColors = 0;
  memset(palette.hash, 0, 256 * sizeof(COLOR_LIST *));
}

int PaletteInsert(CARD32 rgb, int numPixels, int bpp)
{
  COLOR_LIST *pnode;
  COLOR_LIST *prev_pnode = NULL;
  int hash_key, idx, new_idx, count;

  hash_key = (bpp == 16) ? HASH_FUNC16(rgb) : HASH_FUNC32(rgb);

  pnode = palette.hash[hash_key];

  while (pnode != NULL)
  {
    if (pnode->rgb == rgb)
    {

      /*
       * Such palette entry already exists.
       */

      new_idx = idx = pnode->idx;
      count = palette.entry[idx].numPixels + numPixels;
      if (new_idx && palette.entry[new_idx-1].numPixels < count)
      {
        do
        {
          palette.entry[new_idx] = palette.entry[new_idx-1];
          palette.entry[new_idx].listNode->idx = new_idx;
          new_idx--;
        }
        while (new_idx && palette.entry[new_idx-1].numPixels < count);
        palette.entry[new_idx].listNode = pnode;
        pnode->idx = new_idx;
      }
      palette.entry[new_idx].numPixels = count;
      return paletteNumColors;
    }
    prev_pnode = pnode;
    pnode = pnode->next;
  }

  /*
   * Check if palette is full.
   */

  if (paletteNumColors == 256 || paletteNumColors == paletteMaxColors)
  {
    paletteNumColors = 0;
    return 0;
  }

  /*
   * Move palette entries with lesser pixel
   * counts.
   */

  for ( idx = paletteNumColors;
            idx > 0 && palette.entry[idx-1].numPixels < numPixels;
                idx-- )
  {
    palette.entry[idx] = palette.entry[idx-1];
    palette.entry[idx].listNode->idx = idx;
  }

  /* Add new palette entry into the freed
   * slot.
   */

  pnode = &palette.list[paletteNumColors];
  if (prev_pnode != NULL)
  {
    prev_pnode->next = pnode;
  }
  else
  {
    palette.hash[hash_key] = pnode;
  }
  pnode->next = NULL;
  pnode->idx = idx;
  pnode->rgb = rgb;
  palette.entry[idx].listNode = pnode;
  palette.entry[idx].numPixels = numPixels;

  return (++paletteNumColors);
}


/*
 * Converting 32-bit color samples into 24-bit
 * colors. Should be called only when redMax,
 * greenMax and blueMax are 255. Color components
 * assumed to be byte-aligned.
 */


void Pack24(char *buf, tightPixelFormat *fmt, int count)
{
  CARD32 *buf32;
  CARD32 pix;
  int r_shift, g_shift, b_shift;

  buf32 = (CARD32 *)buf;

  r_shift = fmt->redShift;
  g_shift = fmt->greenShift;
  b_shift = fmt->blueShift;

  while (count--)
  {
    pix = *buf32++;
    *buf++ = (char)(pix >> r_shift);
    *buf++ = (char)(pix >> g_shift);
    *buf++ = (char)(pix >> b_shift);
  }
}


/*
 * Converting truecolor samples into palette
 * indices.
 */

void EncodeIndexedRect24(CARD8 *buf, int count)                      
{                                                                       
  COLOR_LIST *pnode;                                                    
  CARD8 *src;                                                       
  CARD32 rgb;                                                        
  int rep = 0;                                                          
                                                                        
  src = (CARD8 *) buf;                                              
                                                                        
  while (count--)                                                       
  {                                                                     
    rgb = RGB24_TO_PIXEL32(*(src), *(src + 1), *(src + 2));

    src += 3;

    while (count && rgb == RGB24_TO_PIXEL32(*(src), *(src + 1), *(src + 2)))
    {                                                                   
      rep++; 
      src += 3;
      count--;
    }                                                                   
    
    pnode = palette.hash[HASH_FUNC32(rgb)];
    
    while (pnode != NULL)
    {                                                                   
      if ((CARD32)pnode->rgb == rgb)                                 
      {                                                                 
        *buf++ = (CARD8)pnode->idx;                                     
        while (rep)                                                     
        {                                                               
          *buf++ = (CARD8)pnode->idx;                                   
          rep--;                                                        
        }                                                               
        break;                                                          
      }                                                                 
      pnode = pnode->next;                                              
    }                                                                   
  }                                                                     
}


#define DEFINE_IDX_ENCODE_FUNCTION(bpp)                                 \
                                                                        \
void EncodeIndexedRect##bpp(CARD8 *buf, int count)                      \
{                                                                       \
  COLOR_LIST *pnode;                                                    \
  CARD##bpp *src;                                                       \
  CARD##bpp rgb;                                                        \
  int rep = 0;                                                          \
                                                                        \
  src = (CARD##bpp *) buf;                                              \
                                                                        \
  while (count--)                                                       \
  {                                                                     \
    rgb = *src++;                                                       \
    while (count && *src == rgb)                                        \
    {                                                                   \
      rep++, src++, count--;                                            \
    }                                                                   \
    pnode = palette.hash[HASH_FUNC##bpp(rgb)];                          \
    while (pnode != NULL)                                               \
    {                                                                   \
      if ((CARD##bpp)pnode->rgb == rgb)                                 \
      {                                                                 \
        *buf++ = (CARD8)pnode->idx;                                     \
        while (rep)                                                     \
        {                                                               \
          *buf++ = (CARD8)pnode->idx;                                   \
          rep--;                                                        \
        }                                                               \
        break;                                                          \
      }                                                                 \
      pnode = pnode->next;                                              \
    }                                                                   \
  }                                                                     \
}

DEFINE_IDX_ENCODE_FUNCTION(16)
DEFINE_IDX_ENCODE_FUNCTION(32)

void EncodeMonoRect24(CARD8 *buf, int w, int h)                      
{                                                                       
  CARD8 *ptr;                                                       
  CARD32 bg, color;                                                         
  unsigned int value, mask;                                             
  int aligned_width;                                                    
  int x, y, bg_bits;                                                    
                                                                        
  ptr = (CARD8*) buf;                                              
  bg =  (CARD32) monoBackground;                                      
  aligned_width = w - w % 8;                                            
                                                                        
  for (y = 0; y < h; y++)                                               
  {                                                                     
    for (x = 0; x < aligned_width; x += 8)                              
    {                                                                   
      for (bg_bits = 0; bg_bits < 8; bg_bits++)                         
      { 
        color = RGB24_TO_PIXEL32(*(ptr), *(ptr + 1), *(ptr + 2));

        ptr += 3;

        if (color != bg)
        {
          break;
        }
      }                                                                 

      if (bg_bits == 8)                                                 
      {                                                                 
        *buf++ = 0;

        continue;
      }                                                                 

      mask = 0x80 >> bg_bits;                                           
      value = mask;                                                     
      for (bg_bits++; bg_bits < 8; bg_bits++)                           
      {                                                                 
        mask >>= 1;                                                     
        
        color = RGB24_TO_PIXEL32(*(ptr), *(ptr + 1), *(ptr + 2));

        ptr += 3;

        if (color != bg)
        {
          value |= mask;
        }
      }
      *buf++ = (CARD8)value;
    }                                                                   
                                                                        
    mask = 0x80;                                                        
    value = 0;                                                          
    if (x >= w)                                                         
      continue;                                                         
                                                                        
    for (; x < w; x++)                                                  
    {            
      color = RGB24_TO_PIXEL32(*(ptr),
                               *(ptr + 1),
                               *(ptr + 2));
      ptr += 3;

      if (color != bg)                                                 
      {                                                                 
        value |= mask;                                                  
      }                                                                 
      mask >>= 1;                                                       
    }                                                                  
    *buf++ = (CARD8)value;                                              
  }                                                                     
}

#define DEFINE_MONO_ENCODE_FUNCTION(bpp)                                \
                                                                        \
void EncodeMonoRect##bpp(CARD8 *buf, int w, int h)                      \
{                                                                       \
  CARD##bpp *ptr;                                                       \
  CARD##bpp bg;                                                         \
  unsigned int value, mask;                                             \
  int aligned_width;                                                    \
  int x, y, bg_bits;                                                    \
                                                                        \
  ptr = (CARD##bpp *) buf;                                              \
  bg = (CARD##bpp) monoBackground;                                      \
  aligned_width = w - w % 8;                                            \
                                                                        \
  for (y = 0; y < h; y++)                                               \
  {                                                                     \
    for (x = 0; x < aligned_width; x += 8)                              \
    {                                                                   \
      for (bg_bits = 0; bg_bits < 8; bg_bits++)                         \
      {                                                                 \
        if (*ptr++ != bg)                                               \
          break;                                                        \
      }                                                                 \
      if (bg_bits == 8)                                                 \
      {                                                                 \
        *buf++ = 0;                                                     \
        continue;                                                       \
      }                                                                 \
      mask = 0x80 >> bg_bits;                                           \
      value = mask;                                                     \
      for (bg_bits++; bg_bits < 8; bg_bits++)                           \
      {                                                                 \
        mask >>= 1;                                                     \
        if (*ptr++ != bg)                                               \
        {                                                               \
          value |= mask;                                                \
        }                                                               \
      }                                                                 \
      *buf++ = (CARD8)value;                                            \
    }                                                                   \
                                                                        \
    mask = 0x80;                                                        \
    value = 0;                                                          \
    if (x >= w)                                                         \
      continue;                                                         \
                                                                        \
    for (; x < w; x++)                                                  \
    {                                                                   \
      if (*ptr++ != bg)                                                 \
      {                                                                 \
        value |= mask;                                                  \
      }                                                                 \
      mask >>= 1;                                                       \
    }                                                                   \
    *buf++ = (CARD8)value;                                              \
  }                                                                     \
}

DEFINE_MONO_ENCODE_FUNCTION(8)
DEFINE_MONO_ENCODE_FUNCTION(16)
DEFINE_MONO_ENCODE_FUNCTION(32)


/*
 * "Gradient" filter for 24-bit color samples.
 * Should be called only when redMax, greenMax
 * and blueMax are 255. Color components assumed
 * to be byte-aligned.
 */

void FilterGradient24(char *buf, tightPixelFormat *fmt, int w, int h)
{
  CARD32 *buf32;
  CARD32 pix32;
  int *prevRowPtr;
  int shiftBits[3];
  int pixHere[3], pixUpper[3], pixLeft[3], pixUpperLeft[3];
  int prediction;
  int x, y, c;

  buf32 = (CARD32 *)buf;
  memset (prevRowBuf, 0, w * 3 * sizeof(int));

  shiftBits[0] = fmt->redShift;
  shiftBits[1] = fmt->greenShift;
  shiftBits[2] = fmt->blueShift;

  for (y = 0; y < h; y++)
  {
    for (c = 0; c < 3; c++)
    {
      pixUpper[c] = 0;
      pixHere[c] = 0;
    }
    prevRowPtr = prevRowBuf;
    for (x = 0; x < w; x++)
    {
      pix32 = *buf32++;
      for (c = 0; c < 3; c++)
      {
        pixUpperLeft[c] = pixUpper[c];
        pixLeft[c] = pixHere[c];
        pixUpper[c] = *prevRowPtr;
        pixHere[c] = (int)(pix32 >> shiftBits[c] & 0xFF);
        *prevRowPtr++ = pixHere[c];

        prediction = pixLeft[c] + pixUpper[c] - pixUpperLeft[c];
        if (prediction < 0)
        {
          prediction = 0;
        }
        else if (prediction > 0xFF)
        {
          prediction = 0xFF;
        }
        *buf++ = (char)(pixHere[c] - prediction);
      }
    }
  }
}

void FilterGradient24bpp(char *buf, tightPixelFormat *fmt, int w, int h)
{
  CARD8 *buf8;
  CARD32 pix32;
  int *prevRowPtr;
  int shiftBits[3];
  int pixHere[3], pixUpper[3], pixLeft[3], pixUpperLeft[3];
  int prediction;
  int x, y, c;

  buf8 = (CARD8 *) buf;
  memset (prevRowBuf, 0, w * 3 * sizeof(int));

  shiftBits[0] = fmt->redShift;
  shiftBits[1] = fmt->greenShift;
  shiftBits[2] = fmt->blueShift;

  for (y = 0; y < h; y++)
  {
    for (c = 0; c < 3; c++)
    {
      pixUpper[c] = 0;
      pixHere[c] = 0;
    }
    prevRowPtr = prevRowBuf;
    for (x = 0; x < w; x++)
    {
      pix32 = RGB24_TO_PIXEL32(*(buf8),
                               *(buf8 + 1),
                               *(buf8 + 2));
       
      buf8 += 3;

      for (c = 0; c < 3; c++)
      {
        pixUpperLeft[c] = pixUpper[c];
        pixLeft[c] = pixHere[c];
        pixUpper[c] = *prevRowPtr;
        pixHere[c] = (int)(pix32 >> shiftBits[c] & 0xFF);
        *prevRowPtr++ = pixHere[c];

        prediction = pixLeft[c] + pixUpper[c] - pixUpperLeft[c];
        if (prediction < 0)
        {
          prediction = 0;
        }
        else if (prediction > 0xFF)
        {
          prediction = 0xFF;
        }
        
        *buf++ = (char)(pixHere[c] - prediction);
      }
    }
  }
}

/*
 * "Gradient" filter for color depths 16 & 32.
 */

#define DEFINE_GRADIENT_FILTER_FUNCTION(bpp)                             \
                                                                         \
void FilterGradient##bpp(CARD##bpp *buf, tightPixelFormat *fmt,   \
                                    int w, int h)                        \
{                                                                        \
  CARD##bpp pix, diff;                                                   \
  Bool endianMismatch;                                                   \
  int *prevRowPtr;                                                       \
  int maxColor[3], shiftBits[3];                                         \
  int pixHere[3], pixUpper[3], pixLeft[3], pixUpperLeft[3];              \
  int prediction;                                                        \
  int x, y, c;                                                           \
                                                                         \
  memset (prevRowBuf, 0, w * 3 * sizeof(int));                           \
                                                                         \
                                                                         \
  endianMismatch = 0;                                                    \
                                                                         \
  maxColor[0] = fmt->redMax;                                             \
  maxColor[1] = fmt->greenMax;                                           \
  maxColor[2] = fmt->blueMax;                                            \
  shiftBits[0] = fmt->redShift;                                          \
  shiftBits[1] = fmt->greenShift;                                        \
  shiftBits[2] = fmt->blueShift;                                         \
                                                                         \
  for (y = 0; y < h; y++)                                                \
  {                                                                      \
    for (c = 0; c < 3; c++)                                              \
    {                                                                    \
      pixUpper[c] = 0;                                                   \
      pixHere[c] = 0;                                                    \
    }                                                                    \
    prevRowPtr = prevRowBuf;                                             \
    for (x = 0; x < w; x++)                                              \
    {                                                                    \
      pix = *buf;                                                        \
                                                                         \
      diff = 0;                                                          \
      for (c = 0; c < 3; c++)                                            \
      {                                                                  \
        pixUpperLeft[c] = pixUpper[c];                                   \
        pixLeft[c] = pixHere[c];                                         \
        pixUpper[c] = *prevRowPtr;                                       \
        pixHere[c] = (int)(pix >> shiftBits[c] & maxColor[c]);           \
        *prevRowPtr++ = pixHere[c];                                      \
                                                                         \
        prediction = pixLeft[c] + pixUpper[c] - pixUpperLeft[c];         \
        if (prediction < 0)                                              \
        {                                                                \
          prediction = 0;                                                \
        }                                                                \
        else if (prediction > maxColor[c])                               \
        {                                                                \
          prediction = maxColor[c];                                      \
        }                                                                \
        diff |= ((pixHere[c]-prediction) & maxColor[c])<< shiftBits[c];  \
      }                                                                  \
                                                                         \
      *buf++ = diff;                                                     \
    }                                                                    \
  }                                                                      \
}

DEFINE_GRADIENT_FILTER_FUNCTION(16)
DEFINE_GRADIENT_FILTER_FUNCTION(32)


/*
 * Code to guess if given rectangle is suitable
 * for smooth image compression (by applying
 * "gradient" filter).
 */

#define DETECT_SUBROW_WIDTH    7
#define DETECT_MIN_WIDTH       8
#define DETECT_MIN_HEIGHT      8

int DetectSmoothImage(tightPixelFormat *fmt, int w, int h)
{
  unsigned long avgError;

  if (fmt->bitsPerPixel == 8 || w < DETECT_MIN_WIDTH || h < DETECT_MIN_HEIGHT)
  {
    return 0;
  }

  if (rfbTightDisableGradient || w * h < tightConf[compressLevel].gradientMinRectSize)
  {
    return 0;
  }

  if (fmt->bitsPerPixel == 32)
  {
    if (usePixelFormat24)
    {
      avgError = DetectSmoothImage24(fmt, w, h);
      return (avgError < tightConf[compressLevel].gradientThreshold24);
    }
    else
    {
      avgError = DetectSmoothImage32(fmt, w, h);
    }
  }
  else if(fmt->bitsPerPixel == 24)
  {
    avgError = DetectSmoothImage24bpp(fmt, w, h);
  }
  else
  {
    avgError = DetectSmoothImage16(fmt, w, h);
  }

  return (avgError < tightConf[compressLevel].gradientThreshold);
}

unsigned long DetectSmoothImage24(tightPixelFormat *fmt, int w, int h)
{
  int off;
  int x, y, d, dx, c;
  int diffStat[256];
  int pixelCount = 0;
  int pix, left[3];
  unsigned long avgError;

  /* 
   *  If client is big-endian, color samples 
   *  begin from the second byte (offset 1) 
   *  of a 32-bit pixel value. 
   */

  off = (fmt->bigEndian != 0);

  memset(diffStat, 0, 256*sizeof(int));

  y = 0, x = 0;
  while (y < h && x < w)
  {
    for (d = 0; d < h - y && d < w - x - DETECT_SUBROW_WIDTH; d++)
    {
      for (c = 0; c < 3; c++)
      {
        left[c] = (int)tightBeforeBuf[((y+d)*w+x+d)*4+off+c] & 0xFF;
      }
      for (dx = 1; dx <= DETECT_SUBROW_WIDTH; dx++)
      {
        for (c = 0; c < 3; c++)
        {
          pix = (int)tightBeforeBuf[((y+d)*w+x+d+dx)*4+off+c] & 0xFF;
          diffStat[abs(pix - left[c])]++;
          left[c] = pix;
        }
        pixelCount++;
      }
    }
    if (w > h)
    {
      x += h;
      y = 0;
    }
    else
    {
      x = 0;
      y += w;
    }
  }

  if (diffStat[0] * 33 / pixelCount >= 95)
    return 0;

  avgError = 0;
  for (c = 1; c < 8; c++)
  {
    avgError += (unsigned long)diffStat[c] * (unsigned long)(c * c);
    if (diffStat[c] == 0 || diffStat[c] > diffStat[c-1] * 2)
      return 0;
  }
  for (; c < 256; c++)
  {
    avgError += (unsigned long)diffStat[c] * (unsigned long)(c * c);
  }
  avgError /= (pixelCount * 3 - diffStat[0]);

  return avgError;
}

unsigned long DetectSmoothImage24bpp(tightPixelFormat *fmt, int w, int h)
{
  int x, y, d, dx, c;
  int diffStat[256];
  int pixelCount = 0;
  int pix, left[3];
  unsigned long avgError;

  #ifdef DEBUG
  fprintf(stderr, "******DetectSmoothImage24: Start.\n");
  #endif

  memset(diffStat, 0, 256 * sizeof(int));

  y = 0, x = 0;

  while (y < h && x < w)
  {
    for (d = 0; d < h - y && d < w - x - DETECT_SUBROW_WIDTH; d++)
    {
      for (c = 0; c < 3; c++)
      {
        left[c] = *(tightBeforeBuf + ((y+d)*w+x+d)*3+c) & 0xFF;
      }

      for (dx = 1; dx <= DETECT_SUBROW_WIDTH; dx++)
      {
        for (c = 0; c < 3; c++)
        {
          pix = *(tightBeforeBuf + ((y+d)*w+x+d+dx)*3+c) & 0xFF;
          diffStat[abs(pix - left[c])]++;
          left[c] = pix;
        }

        pixelCount++;
      }
    }

    if (w > h)
    {
      x += h;
      y = 0;
    }
    else
    {
      x = 0;
      y += w;
    }
  }

  if (diffStat[0] * 33 / pixelCount >= 95)
    return 0;

  avgError = 0;
  for (c = 1; c < 8; c++)
  {
    avgError += (unsigned long)diffStat[c] * (unsigned long)(c * c);
    if (diffStat[c] == 0 || diffStat[c] > diffStat[c-1] * 2)
      return 0;
  }
  for (; c < 256; c++)
  {
    avgError += (unsigned long)diffStat[c] * (unsigned long)(c * c);
  }
  avgError /= (pixelCount * 3 - diffStat[0]);

  #ifdef DEBUG
  fprintf(stderr, "******DetectSmoothImage24: End.\n");
  #endif

  return avgError;
}

#define DEFINE_DETECT_FUNCTION(bpp)                                          \
                                                                             \
unsigned long DetectSmoothImage##bpp (tightPixelFormat *fmt, int w, int h)   \
{                                                                            \
  Bool endianMismatch;                                                       \
  CARD##bpp pix;                                                             \
  int maxColor[3], shiftBits[3];                                             \
  int x, y, d, dx, c;                                                        \
  int diffStat[256];                                                         \
  int pixelCount = 0;                                                        \
  int sample, sum, left[3];                                                  \
  unsigned long avgError;                                                    \
                                                                             \
  endianMismatch = 0;                                                        \
                                                                             \
  maxColor[0] = fmt->redMax;                                                 \
  maxColor[1] = fmt->greenMax;                                               \
  maxColor[2] = fmt->blueMax;                                                \
  shiftBits[0] = fmt->redShift;                                              \
  shiftBits[1] = fmt->greenShift;                                            \
  shiftBits[2] = fmt->blueShift;                                             \
                                                                             \
  memset(diffStat, 0, 256*sizeof(int));                                      \
                                                                             \
  y = 0, x = 0;                                                              \
  while (y < h && x < w)                                                     \
  {                                                                          \
    for (d = 0; d < h - y && d < w - x - DETECT_SUBROW_WIDTH; d++)           \
    {                                                                        \
      pix = ((CARD##bpp *)tightBeforeBuf)[(y+d)*w+x+d];                      \
                                                                             \
      for (c = 0; c < 3; c++)                                                \
      {                                                                      \
        left[c] = (int)(pix >> shiftBits[c] & maxColor[c]);                  \
      }                                                                      \
      for (dx = 1; dx <= DETECT_SUBROW_WIDTH; dx++)                          \
      {                                                                      \
        pix = ((CARD##bpp *)tightBeforeBuf)[(y+d)*w+x+d+dx];                 \
                                                                             \
        sum = 0;                                                             \
        for (c = 0; c < 3; c++)                                              \
        {                                                                    \
          sample = (int)(pix >> shiftBits[c] & maxColor[c]);                 \
          sum += abs(sample - left[c]);                                      \
          left[c] = sample;                                                  \
        }                                                                    \
        if (sum > 255)                                                       \
          sum = 255;                                                         \
        diffStat[sum]++;                                                     \
          pixelCount++;                                                      \
      }                                                                      \
    }                                                                        \
    if (w > h)                                                               \
    {                                                                        \
      x += h;                                                                \
      y = 0;                                                                 \
    }                                                                        \
    else                                                                     \
    {                                                                        \
      x = 0;                                                                 \
    y += w;                                                                  \
    }                                                                        \
  }                                                                          \
                                                                             \
  if ((diffStat[0] + diffStat[1]) * 100 / pixelCount >= 90)                  \
    return 0;                                                                \
                                                                             \
  avgError = 0;                                                              \
  for (c = 1; c < 8; c++)                                                    \
  {                                                                          \
    avgError += (unsigned long)diffStat[c] * (unsigned long)(c * c);         \
    if (diffStat[c] == 0 || diffStat[c] > diffStat[c-1] * 2)                 \
      return 0;                                                              \
  }                                                                          \
  for (; c < 256; c++)                                                       \
  {                                                                          \
    avgError += (unsigned long)diffStat[c] * (unsigned long)(c * c);         \
  }                                                                          \
  avgError /= (pixelCount - diffStat[0]);                                    \
                                                                             \
  return avgError;                                                           \
}                                                                            \

DEFINE_DETECT_FUNCTION(16)
DEFINE_DETECT_FUNCTION(32)


void CopyBuffer(char *srcBuf, char *dstBuf, int x, int y, int w, int h,
                           int bytesPerLine, int bitsPerPixel)
{
  int j;
  char *srcData, *dstData;

  /*
   * There is enough place in the dstBuff for sure
   */

  srcData = srcBuf + y * bytesPerLine + x * bitsPerPixel / 8;
  dstData = dstBuf;

  for(j = 0; j < h; j++)
  {
    memcpy(dstData, srcData, w * bitsPerPixel / 8);
    dstData += w * bitsPerPixel / 8;
    srcData += bytesPerLine;
  }
}

