/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2003 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#ifndef Split_H
#define Split_H

#include "Types.h"
#include "Timestamp.h"

#include "Message.h"

#include "EncodeBuffer.h"
#include "DecodeBuffer.h"

//
// Set the verbosity level.
//

#define PANIC
#undef  TEST

//
// Define this to know how many splits
// are allocated and deallocated.
//

#undef  REFERENCES

//
// Maximum amount of messages that can
// be in split store at any given time.
//

#define SPLIT_STORE_SIZE_LIMIT       40

//
// Size of header of messages saved on
// disk.
//

#define SPLIT_HEADER_SIZE            12

//
// Divide big pieces of data in smaller
// chunks and send them at idle time.
//

class SplitStore;

class Split
{
  friend class SplitStore;

  public:

  Split()
  {
    client_   = nothing;
    position_ = nothing;

    completion_ = 0;

    store_ = NULL;

    size_   = 0;
    c_size_ = 0;
    next_   = 0;

    checksum_ = NULL;
    abort_    = 0;

    #ifdef REFERENCES

    references_++;

    *logofs << "Split: Created new Split at " 
            << this << " out of " << references_ 
            << " allocated references.\n" << logofs_flush;

    #endif
  }

  ~Split()
  {
    #ifdef REFERENCES

    references_--;

    *logofs << "Split: Deleted Split at " 
            << this << " out of " << references_ 
            << " allocated references.\n" << logofs_flush;
    #endif

    delete [] checksum_;
  }

  md5_byte_t *getChecksum()
  {
    return checksum_;
  }

  Message *getMessage()
  {
    return store_ -> get(position_);
  }

  MessageStore *getStore()
  {
    return store_;
  }
  
  int getAbort()
  {
    return abort_;
  }

  void setAbort()
  {
    abort_ = 1;
  }

  void resetAbort()
  {
    abort_ = 0;
  }

  private:

  //
  // The agent's client which
  // is splitting the message.
  //

  int client_;

  //
  // Where to find the message
  // in the message store.
  //

  int position_;

  //
  // Is this the last message
  // in a start-split/end-split
  // sequence.

  int completion_;

  //
  // Which store is involved.
  //

  MessageStore *store_;

  //
  // This is the uncompressed data
  // size of original message.
  //

  int size_;

  //
  // This is the size of compressed
  // data, if in this form.
  //

  int c_size_;

  //
  // This is the current byte in request's
  // data where to read and write the next
  // block.
  //

  int next_;

  //
  // Keep checksum in case we were able to
  // load message data from disk.
  //

  md5_byte_t *checksum_;

  //
  // Was this split aborted?
  //

  int abort_;

  //
  // Container for the data part.
  //

  T_data data_;

  #ifdef REFERENCES

  static int references_;

  #endif
};

class SplitStore
{
  public:

  SplitStore(Compressor *compressor, Decompressor *decompressor)

    : compressor_(compressor), decompressor_(decompressor)
  {
    lastCommit_ = 0;

    lastClient_   = nothing;
    lastRequest_  = nothing;
    lastPosition_ = nothing;

    lastCompletion_ = 0;

    splits_ = new T_splits();

    current_ = splits_ -> end();

    abort_ = 0;

    splitStorageSize_ = 0;

    #ifdef TEST
    *logofs << "SplitStore: Size of split store after constructor is "
            << splitStorageSize_ << "/" << totalSplitStorageSize_
            << ".\n" << logofs_flush;
    #endif
  }

  ~SplitStore()
  {
    for (T_splits::iterator i = splits_ -> begin();
             i != splits_ -> end(); i++)
    {
      delete *i;
    }

    delete splits_;

    totalSplitStorageSize_ -= splitStorageSize_;

    #ifdef TEST
    *logofs << "SplitStore: Size of split store after destructor is "
            << splitStorageSize_ << "/" << totalSplitStorageSize_
            << ".\n" << logofs_flush;
    #endif
  }

  //
  // Mark the split element at the tail of the queue with the
  // id of client. Not all splits added to the split store
  // carry a completion id. Completion id is only set at the
  // time an end-split operation is received by channel. The
  // method returns nothing if the queue is empty or tail of
  // the queue already contains a split marked as completion.
  // This implicitly means that all messages were immediately
  // sent and no split was added to the queue across the last
  // start-split/end-split sequence.
  //

  int markCompletion() const
  {
    if (splits_ -> size() > 0)
    {
      Split *split = (*(--(splits_ -> end())));

      if (split -> completion_ == 1)
      {
        return 0;
      }
      else
      {
        split -> completion_ = 1;

        return 1;
      }
    }

    return 0;
  }

  int getLastClient() const
  {
    return lastClient_;
  }

  int getLastRequest() const
  {
    return lastRequest_;
  }

  int getLastPosition() const
  {
    return lastPosition_;
  }

  int getLastCompletion() const
  {
    return lastCompletion_;
  }

  int setLastCommit(int position)
  {
    return (lastCommit_ = position);
  }

  int getLastCommit() const
  {
    return lastCommit_;
  }

  int getNodeSize(const Split *split) const
  {
    //
    // Take in account 32 bytes
    // of node overhead.
    //

    return sizeof(class Split) + 32 + split -> size_;
  }

  int getStorageSize()
  {
    return splitStorageSize_;
  }

  int getTotalStorageSize()
  {
    return totalSplitStorageSize_;
  }

  int getSize()
  {
    return splits_ -> size();
  }

  int getAbort()
  {
    return abort_;
  }

  void resetAbort()
  {
    abort_ = 0;
  }

  T_splits *getSplits()
  {
    return splits_;
  }

  MessageStore *getStore()
  {
    if (splits_ -> size() == 0)
    {
      return NULL;
    }

    return (*splits_ -> begin()) -> store_;
  }

  int add(MessageStore *store, int client, int position,
              const unsigned char *buffer, const int size);

  int add(MessageStore *store, int position, md5_byte_t *checksum,
              unsigned char *buffer, const int size);

  int send(EncodeBuffer &encodeBuffer, int packetSize);

  int receive(DecodeBuffer &decodeBuffer);

  //
  // Find message on disk and update its last
  // modification time. If successful, channel
  // can send an abort message to the remote
  // side.
  //

  int find(Split *split);

  //
  // Remove the element on top of the queue and discard
  // any split data that still needed to be transferred.
  // Channel uses this to restart clients that were
  // streaming a request at the time of a reset.
  //

  int pop();

  private:

  const char *name(const md5_byte_t *checksum);

  Split *push(MessageStore *store, int client, int position,
                  md5_byte_t *checksum, const unsigned char *buffer,
                      const int size);

  int start(EncodeBuffer &encodeBuffer, int packetSize);

  int start(DecodeBuffer &decodeBuffer);

  int finish(Split *split);

  //
  // Load data from disk as soon as the unsplit
  // procedure is successfully aborted by the
  // remote side.
  //

  int load(Split *split);

  //
  // Save data once message is recomposed
  // at local side.
  //

  int save(Split *split);

  //
  // Amount of data in the repository.
  //

  int splitStorageSize_;

  static int totalSplitStorageSize_;

  //
  // Repository where to save splits.
  //

  T_splits *splits_;

  //
  // Current element being transferred.
  //

  T_splits::iterator current_;

  //
  // Is there any split loaded from disk
  // for which we still need to send an
  // abort message?
  //

  int abort_;

  //
  // Set to agent's client which originally
  // added the message to the split store if
  // last send operation reached the end of
  // a split.
  //

  int lastClient_;

  //
  // Set to opcode associated to message store
  // if last send operation reached the end of
  // a split.
  //

  int lastRequest_;

  //
  // Set if last send reached the end of a split.
  // Position is set by channel at the time split
  // is added to the split store.
  //

  int lastPosition_;

  //
  // Set if last send operation reached both the
  // end of a split and split was carrying the
  // information about completion. When a split
  // carrying a completion id is completed, the
  // channel can send an appropriate split notify
  // event used by agent to restart its client.
  //

  int lastCompletion_;

  //
  // Position of last commit operation performed
  // by channel. Used to differentially encode
  // the position of next message to commit.
  //

  int lastCommit_;

  //
  // Compress and decompress data payload.
  //

  Compressor   *compressor_;
  Decompressor *decompressor_;

  #ifdef REFERENCES

  static int references_;

  #endif
};

#endif /* Split_H */

