/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2003 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#ifndef ServerChannel_H
#define ServerChannel_H

#include "Channel.h"

#include "SequenceQueue.h"

#include "ServerReadBuffer.h"

//
// Available unpack methods.
//

#include "Unpack.h"
#include "Tight.h"
#include "Jpeg.h"
#include "Png.h"

//
// Set the verbosity level.
//

#define PANIC
#undef  TEST
#undef  DEBUG

//
// How many sequence numbers of split commit
// requests we are going to save in order to
// mask errors.
//

#define MAX_COMMIT_SEQUENCE_QUEUE         16

//
// Define this to know when a channel
// is created or destroyed.
//

#undef  REFERENCES

//
// This class implements the X server
// side compression of X protocol.
//

class ServerChannel : public Channel
{
  public:

  ServerChannel(Transport *transport, Compressor *compressor,
                    Decompressor *decompressor);

  virtual ~ServerChannel();

  virtual int handleRead(EncodeBuffer &encodeBuffer);

  virtual int handleWrite(const unsigned char *message, unsigned int length);

  virtual int handleSplit(EncodeBuffer &encodeBuffer, MessageStore *store, 
                              const unsigned char *buffer, const unsigned int size);

  virtual int handleSplit(EncodeBuffer &encodeBuffer, int packetLimit);

  virtual int handleUnsplit(DecodeBuffer &decodeBuffer, MessageStore *store,
                                unsigned char *buffer, const unsigned int size);

  virtual int handleUnsplit(DecodeBuffer &decodeBuffer);

  virtual int handleAbortSplit(EncodeBuffer &encodeBuffer);

  virtual int handleAbortSplit(DecodeBuffer &decodeBuffer);

  //
  // Send the last motion notify event
  // received from the X server to the
  // remote proxy.
  //

  virtual int handleMotion(EncodeBuffer &encodeBuffer, int forceMotion = 0);

  virtual int handleWakeup(EncodeBuffer &encodeBuffer)
  {
    return 0;
  }

  virtual int handleConfiguration();

  virtual int handleFinish();

  virtual int handleNotify(T_notification_type type)
  {
    return 0;
  }
    
  virtual int handleReset();

  virtual int needSplit() const
  {
    return 0;
  }

  virtual int needMotion() const
  {
    return (finish_ == 0 && lastMotion_[0] != '\0');
  }

  virtual int needWakeup() const
  {
    return 0;
  }

  virtual int updateWakeup()
  {
    return 0;
  }

  virtual int needLimit()
  {
    return (transport_ -> length() >=
                control -> TransportXBufferLimit);
  }

  virtual int getLimit()
  {
    return (control -> TransportXBufferLimit);
  }

  virtual T_channel_type getType()
  {
    return CHANNEL_X;
  }

  void setBigEndian(int flag);

  protected:

  int handleFastReadReply(EncodeBuffer &encodeBuffer, const unsigned char &opcode,
                              const unsigned char *&buffer, const unsigned int &size);

  int handleFastReadEvent(EncodeBuffer &encodeBuffer, const unsigned char &opcode,
                              const unsigned char *&buffer, const unsigned int &size);

  int handleFastWriteRequest(DecodeBuffer &decodeBuffer, unsigned char &opcode,
                                 unsigned char *&buffer, unsigned int &size);

  //
  // Synchronize last client's sequence
  // number with last request actually
  // received by X server.
  //

  int handleResetSequence(unsigned short int sequenceNum);

  //
  // Finalize the split operation after
  // the complete request has been
  // recomposed.
  //

  int handleCommit(DecodeBuffer &decodeBuffer, unsigned char &opcode,
                       unsigned char *&buffer, unsigned int &size);

  //
  // Set the unpack colormap and the alpha
  // blending data to be used to unpack
  // images.
  //

  int handleGeometry(unsigned char &opcode, unsigned char *&buffer,
                         unsigned int &size);

  int handleColormap(unsigned char &opcode, unsigned char *&buffer,
                         unsigned int &size);

  int handleAlpha(unsigned char &opcode, unsigned char *&buffer,
                      unsigned int &size);

  //
  // Manage the decoded buffer to unpack
  // the image and move the data to the
  // shared memory segment.
  //

  int handleImage(unsigned char &opcode, unsigned char *&buffer,
                      unsigned int &size, unsigned int &flush);

  //
  // Uncompress a packed image in one
  // or more graphic X requests.
  //

  int handleUnpack(unsigned char &opcode, unsigned char *&buffer,
                       unsigned int &size);

  //
  // Move the image to the shared
  // memory buffer.
  //

  int handleShmem(unsigned char &opcode, unsigned char *&buffer,
                      unsigned int &size, unsigned int &flush);

  //
  // Model notification of congestion events.
  //

  virtual int isCongested()
  {
    int state = (transport_ -> blocked() == 1);

    if (state == 0 && control -> AgentSyncPropagate == 1)
    {
      if (syncCounter_ > 0 &&
              diffTimestamp(firstSyncTs_, getTimestamp()) >=
                  control -> CongestionTimeout)
      {
        #if defined(INFO) || defined(TEST)
        if (congestion_ == 0)
        {
          *logofs << "ServerChannel: Forcing channel for FD#" << fd_
                  << " in congestion with sync delay of "
                  << diffTimestamp(firstSyncTs_, getTimestamp())
                  << " Ms.\n" << logofs_flush;
        }
        #endif

        state = 1;
      }
    }

    return state;
  }

  virtual int isReliable()
  {
    return 1;
  }

  void handleEncodeCharInfo(const unsigned char *nextSrc, EncodeBuffer &encodeBuffer);

  //
  // Handle the MIT-SHM initialization
  // messages exchanged with the remote
  // proxy.
  //

  int handleShmemInit(EncodeBuffer &encodeBuffer, const unsigned char opcode,
                          const unsigned int stage, const unsigned char *buffer,
                              const unsigned int size);

  int handleShmemInit(DecodeBuffer &decodeBuffer, unsigned char &opcode,
                          unsigned char *&buffer, unsigned int &size);

  //
  // Wait for a MIT-SHM image completion
  // event from the real X server.
  //

  int handleShmemEvent();

  int checkShmemEvent(unsigned char event, unsigned short sequence,
                          const unsigned char *buffer);

  int checkShmemError(unsigned char error, unsigned short sequence,
                          const unsigned char *buffer);

  //
  // Allocate and free the shared memory
  // support resurces.
  //

  int handleShmemStateAlloc();
  int handleShmemStateRemove();

  //
  // Temporary storage for image info.
  //

  int handleImageStateAlloc(unsigned char opcode)
  {
    if (imageState_ == NULL)
    {
      imageState_ = new T_image_state();
    }

    imageState_ -> opcode = opcode;

    return 1;
  }

  int handleImageStateRemove()
  {
    if (imageState_ != NULL)
    {
      delete imageState_;

      imageState_ = NULL;
    }

    return 1;
  }

  //
  // Store the information needed to unpack
  // images per each known agent's client.
  //

  int handleUnpackStateInit(int client);

  int handleUnpackAllocGeometry(int client);
  int handleUnpackAllocColormap(int client);
  int handleUnpackAllocAlpha(int client);

  int handleUnpackStateRemove(int client);

  typedef struct
  {
    T_geometry *geometry;
    T_colormap *colormap;
    T_alpha    *alpha;

  } T_unpack_state;

  T_unpack_state *unpackState_[256];

  //
  // Handle creation of the special
  // unpacker for tight images.
  //

  int handleTightStateAlloc()
  {
    if (tightState_ == NULL)
    {
      tightState_ = new Tight();
    }

    return 1;
  }

  int handleTightStateRemove()
  {
    delete tightState_;

    return 1;
  }

  //
  // Own read buffer. It is able to identify
  // full messages read from X descriptor.
  //

  ServerReadBuffer readBuffer_;

  //
  // Opcode of last request coming from X
  // client.
  //

  unsigned char clientOpcode_;
  unsigned char serverOpcode_;

  //
  // Sequence number of last request coming
  // from X client or X server.
  //

  unsigned int clientSequence_;
  unsigned int serverSequence_;

  //
  // Used to identify replies based on sequence
  // number of original request.
  //

  SequenceQueue sequenceQueue_;

  //
  // Last motion notify read from X server.
  //

  unsigned char lastMotion_[32];
  T_timestamp   lastMotionTs_;

  //
  // Sequence numbers of last auto-generated
  // put image requests. Needed to intercept
  // and suppress errors generated by such
  // requests.
  //

  unsigned int commitSequenceQueue_[MAX_COMMIT_SEQUENCE_QUEUE];

  //
  // Let agent select which expose
  // events is going to receive.
  //

  unsigned int enableExpose_;
  unsigned int enableGraphicsExpose_;
  unsigned int enableNoExpose_;

  //
  // Monitor sync replies coming from X server
  // in order to slow down the client peer in
  // case of congestion.
  //

  int syncCounter_;
  T_timestamp firstSyncTs_;

  //
  // Unpack tight images.
  //

  Tight *tightState_;

  //
  // Used in initialization and handling
  // of MIT-SHM shared memory put images.
  //

  typedef struct
  {
    int           stage;
    int           present;
    int           enabled;
    int           segment;
    int           id;
    void          *address;
    unsigned int  size;

    unsigned char opcode;
    unsigned char event;
    unsigned char error;

    unsigned int  sequence;
    unsigned int  offset;
    T_timestamp   last;

    unsigned int  checked;

  } T_shmem_state;

  T_shmem_state *shmemState_;

  //
  // Used to pass current image data between
  // the different decompression stages.
  //

  typedef struct
  {
    unsigned char  opcode;

    unsigned int   drawable;
    unsigned int   gcontext;

    unsigned char  method;

    unsigned char  format;
    unsigned char  srcDepth;
    unsigned char  dstDepth;

    unsigned int   srcLength;
    unsigned int   dstLength;
    unsigned int   dstLines;

    short int      srcX;
    short int      srcY;
    unsigned short srcWidth;
    unsigned short srcHeight;

    short int      dstX;
    short int      dstY;
    unsigned short dstWidth;
    unsigned short dstHeight;

    unsigned char  leftPad;

  } T_image_state;

  T_image_state *imageState_;

  //
  // Keep track of object creation and
  // deletion.
  //

  private:

  #ifdef REFERENCES

  static int references_;

  #endif
};

#endif /* ServerChannel_H */
