#!/bin/sh
#
# Patch mtd into kernel
#
# usage:patch [-j] kernelpath
# 	kernelpath must be given
#	-j includes filesystems (jffs, jffs2)
#	
# Works for Kernels >= 2.4.11 full functional
# Works for Kernels >= 2.4 and <= 2.4.10 partly (JFFS2 support is missing)
# For 2.2 Kernels it's actually disabled, as I have none to test it.
#
# Your can use it for pristine kernels and for already patches kernels too.
#
# Detects Kernelversion and applies neccecary modifications
# For Kernelversions < 2.4.20 ZLIB-Patch is applied, if 
# filesystem option is set and ZLIB-Patch is not already there 
#
# Maybe some sed/awk experts would make it better, but I'm not
# one of them. Feel free to make it better
# 
# Thomas (tgxl@linutronix.de)
#
# $Id: patchin.sh,v 1.10 2003/02/10 10:55:55 gleixner Exp $
#
# 02-10-2003 tglx replaced grep -m by head -n 1, as older grep versions don't support -m	
#
# Keep usage here, easier to find :)
usage () {
	echo "usage:  $0 [-j] kernelpath"
	echo "-j include jffs(2) filesystems" 
	exit 1
}

# Preset variables
LINUXDIR=
FILESYSTEMS=no
VERSION=0
PATCHLEVEL=0
SUBLEVEL=0
ZLIBPATCH=no
CONFIG=Config.in

# Get commandline options
while getopts j opt
do
    case "$opt" in
      j)  FILESYSTEMS=yes;;
      \?)
	  usage;
    esac
done
shift `expr $OPTIND - 1`
LINUXDIR=$1

# Check if kerneldir contains a Makefile
if [ ! -f $LINUXDIR/Makefile ] 
then 
	echo "Directory $LINUXDIR does not exist or is not a kernel source directory";
	exit 1;
fi

# Get kernel version
VERSION=`grep -s VERSION <$LINUXDIR/Makefile | head -n 1 | sed s/'VERSION = '//`
PATCHLEVEL=`grep -s PATCHLEVEL <$LINUXDIR/Makefile | head -n 1 | sed s/'PATCHLEVEL = '//`
SUBLEVEL=`grep -s SUBLEVEL <$LINUXDIR/Makefile | head -n 1 | sed s/'SUBLEVEL = '//`

# Can we handle this ?
if test $VERSION -ne 2 -o $PATCHLEVEL -lt 4
then 
	echo "Cannot patch kernel version $VERSION.$PATCHLEVEL.$SUBLEVEL";
	exit 1;
fi

# Use Kconfig instead of Config.in for Kernels >= 2.5
if test $PATCHLEVEL -gt 4
then
	CONFIG=Kconfig;
fi

# Have we to use ZLIB PATCH ? 
if [ "$FILESYSTEMS" = "yes" ]
then
	PATCHDONE=`grep -s zlib_deflate $LINUXDIR/lib/Makefile | head -n 1`
	if test $PATCHLEVEL -eq 4 -a $SUBLEVEL -lt 20 
	then
		if [ "$PATCHDONE" = "" ] 
		then
			ZLIBPATCH=yes;
		fi
	fi
fi

# Here we go
cd `dirname $0`
THISDIR=`pwd`
TOPDIR=`dirname $THISDIR`

# Check which header files we need depending on kernel version
HDIR="$TOPDIR/include/linux"
if test $PATCHLEVEL -eq 4 
then	
	# 2.4 below 2.4.20 zlib headers are neccecary
	if test $SUBLEVEL -lt 20
	then
		JFFS2_H="$HDIR/jffs2*.h $HDIR/workqueue.h $HDIR/z*.h $HDIR/rb*.h"
	else
		JFFS2_H="$HDIR/jffs2*.h $HDIR/workqueue.h $HDIR/rb*.h"
	fi
else
	#	>= 2.5
	JFFS2_H="$HDIR/jffs2*.h "
fi

echo Patching $LINUXDIR 
echo Include Filesytems: $FILESYSTEMS
echo Zlib-Patch needed: $ZLIBPATCH
read -p "Can we start now ? [y/N]" ANSWER
echo ""

if [ "$ANSWER" != "y" ]
then
	echo Patching Kernel canceled
	exit 1;
fi

cd $LINUXDIR

#	make symlinks
mkdir -p drivers/mtd
mkdir -p drivers/mtd/chips
mkdir -p drivers/mtd/devices
mkdir -p drivers/mtd/maps
mkdir -p drivers/mtd/nand
mkdir -p include/linux/mtd
ln -sf $TOPDIR/drivers/mtd/*.[ch] drivers/mtd
ln -sf $TOPDIR/drivers/mtd/Makefile $TOPDIR/drivers/mtd/Rules.make $TOPDIR/drivers/mtd/$CONFIG drivers/mtd
ln -sf $TOPDIR/drivers/mtd/chips/*.[ch] drivers/mtd/chips
ln -sf $TOPDIR/drivers/mtd/chips/Makefile $TOPDIR/drivers/mtd/chips/$CONFIG drivers/mtd/chips
ln -sf $TOPDIR/drivers/mtd/devices/*.[ch] drivers/mtd/devices
ln -sf $TOPDIR/drivers/mtd/devices/Makefile $TOPDIR/drivers/mtd/devices/$CONFIG drivers/mtd/devices
ln -sf $TOPDIR/drivers/mtd/maps/*.[ch] drivers/mtd/maps
ln -sf $TOPDIR/drivers/mtd/maps/Makefile $TOPDIR/drivers/mtd/maps/$CONFIG drivers/mtd/maps
ln -sf $TOPDIR/drivers/mtd/nand/*.[ch] drivers/mtd/nand
ln -sf $TOPDIR/drivers/mtd/nand/Makefile $TOPDIR/drivers/mtd/nand/$CONFIG drivers/mtd/nand
ln -sf $TOPDIR/include/linux/mtd/*.h include/linux/mtd

if [ "$FILESYSTEMS" = "yes" ]
then
	mkdir -p fs/jffs
	mkdir -p fs/jffs2
	ln -sf $TOPDIR/fs/jffs/*.[ch] fs/jffs
	ln -sf $TOPDIR/fs/jffs/Makefile fs/jffs
	ln -sf $TOPDIR/fs/jffs2/*.[ch] fs/jffs2
	ln -sf $TOPDIR/fs/jffs2/Makefile fs/jffs2
	ln -sf $TOPDIR/include/linux/jffs.h include/linux
	ln -sf $JFFS2_H include/linux
	
	PATCHDONE=`grep -s jffs2 fs/Makefile | head -n 1`
	if [ "$PATCHDONE" = "" ]
	then
		echo "Add JFFS2 to Makefile and Config.in manually. JFFS2 is included as of 2.4.12"	
	else
		if test $PATCHLEVEL -lt 5
		then
			JFFS=`grep -n JFFS fs/Config.in | head -n 1 | sed s/:.*//`
			CRAMFS=`grep -n CRAMFS fs/Config.in | head -n 1 | sed s/:.*//`
			let JFFS=JFFS-1
			let CRAMFS=CRAMFS-1
			sed "$JFFS"q fs/Config.in >Config.tmp
			cat $TOPDIR/fs/Config.in >>Config.tmp
			sed 1,"$CRAMFS"d fs/Config.in >>Config.tmp
			mv -f Config.tmp fs/Config.in
			
			if [ -f include/linux/crc32.h ] 
			then
				# check, if it is already defined there
				CRC32=`grep -s 'crc32(' include/linux/crc32.h | head -n 1`
				if [ "$CRC32" = "" ]
				then
					# patch in header form fs/jffs2
					LASTLINE=`grep -n '#endif' include/linux/crc32.h | head -n 1 | sed s/:.*//`
					let LASTLINE=LASTLINE-1
					sed "$LASTLINE"q include/linux/crc32.h >Crc32.tmp
					cat fs/jffs2/crc32.h >>Crc32.tmp
					echo "#endif" >>Crc32.tmp
					mv -f Crc32.tmp include/linux/crc32.h
				fi
			else
				ln -sf $TOPDIR/fs/jffs2/crc32.h include/linux
			fi
		fi
	fi
fi

if [ "$ZLIBPATCH" = "yes" ]
then
	mkdir -p lib/zlib_deflate
	mkdir -p lib/zlib_inflate
	ln -sf $TOPDIR/lib/zlib_deflate/*.[ch] lib/zlib_deflate
	ln -sf $TOPDIR/lib/zlib_deflate/Makefile lib/zlib_deflate
	ln -sf $TOPDIR/lib/zlib_inflate/*.[ch] lib/zlib_inflate
	ln -sf $TOPDIR/lib/zlib_inflate/Makefile lib/zlib_inflate
	patch -p1 -i $TOPDIR/lib/patch-Makefile
fi

echo "Patching done"

if test $PATCHLEVEL -lt 5
then 
	# FIXME: SED/AWK experts should know how to do it automagic
	echo "Please update Documentation/Configure.help from $TOPDIR/Documentation/Configure.help"
fi

