/*

  Copyright (C) 2000, The MITRE Corporation

  Use of this software is subject to the terms of the GNU General
  Public License version 2.

  Please read the file LICENSE for the exact terms.

*/

/*
 * Classes for handling Udp sockets.
 *
 * Author: Kevin H. Grace, kgrace@mitre.org
 *         Mike Butler,    mgb@mitre.org
 *         The MITRE Corporation
 *         202 Burlington Rd
 *         Bedford, MA  01730
 *
 * $Id: UtUdpSocket.h,v 1.9 2004/01/02 12:18:47 br1 Exp $
 */

#ifndef __UdpSocket_h
#define __UdpSocket_h

#include <UtReport.h>
#include <UtString.h>

#include <iostream>
#include <cerrno>
#include <cstdio>
#include <cstring>
#include <strstream>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/ioctl.h>
//#include <net/if.h>
#include <linux/if.h>
#include <net/ethernet.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <netdb.h>

#include "NeighbourEncryption.h"

static bool isEncrypted;

/* A facade class to facilitate converting between 'sockaddr' and 'sockaddr_in' structs
 * Note: Assumes addresses in sockaddr and sockaddr_in are in NETWORK order.
 */
class InetAddr {
  friend ostream &operator<<(ostream &os, const InetAddr &rhs) {
    unsigned long tmp = ntohl(rhs.cAddr.sin_addr.s_addr);

    os << (0xff &(tmp>>24)) << "."
       << (0xff &(tmp>>16)) << "."
       << (0xff &(tmp>> 8)) << "."
       << (0xff &(tmp>> 0)) << ":"
       << ntohs(rhs.cAddr.sin_port);
    return(os);
  }
private:
  sockaddr_in cAddr;
public:
  enum { BCAST = 0xffffffff, ANY=0x0 };
  InetAddr(const String &s ) { SetAddr(s); }
  InetAddr(const sockaddr& sa)  { cAddr = (*((sockaddr_in*)&sa)); }
  InetAddr(const sockaddr_in &sa) { cAddr = sa; }
  // Assumes addresses are in HOST order!
  InetAddr(unsigned int addr = ANY, unsigned short port = 0) {
    memset(&cAddr,0,sizeof(cAddr));
    cAddr.sin_family = AF_INET;
    cAddr.sin_port   = htons(port);
    cAddr.sin_addr.s_addr = htonl(addr);
  }
  operator String() const {
	  std::ostrstream tmp;
    tmp << (*this) << ends;
    char *s = tmp.str();
    String result(s);
    delete [] s;
    return(result);
  }
  unsigned int Addr() const {
    return(ntohl(cAddr.sin_addr.s_addr));
  }
  int Port() const {
    return(ntohs(cAddr.sin_port));
  }
  operator sockaddr() const {
    return(*((sockaddr *)&cAddr));
  };
  operator sockaddr_in() const { return(cAddr); }

  struct sockaddr_in getSockaddr_in() {
  	return cAddr;
  }

  const size_t Size() const { return(sizeof(cAddr)); }
  bool SetAddr(const String &s) {
    int a, b, c, d, p;
    sockaddr_in *sa = (sockaddr_in *)&cAddr;
    memset(sa, 0, sizeof(cAddr));
    sa->sin_family = AF_INET;
    sa->sin_addr.s_addr = htonl(INADDR_ANY);

    /* Empty address is ok... */
    if(!s.length()) return(true);

    /* Just port spec? */
    if(1 == sscanf(s.c_str(), ":%d", &p)) {
      sa->sin_port = htons(p);
      return(true);
    }

    /* Host and port? Snarf and look for host ID too... */
    if(1 == sscanf(s.c_str(), "%*[^:]:%d", &p)) {
      sa->sin_port = htons(p);
    }

    /* Host as IP address? */
    if(4 == sscanf(s.c_str(), "%d.%d.%d.%d", &a, &b, &c, &d)) {
      sa->sin_addr.s_addr = htonl((((((a<<8)|b)<<8)|c)<<8)|d);
      return(true);
    }

    /* Just host name? Attempt to resolve to IP... */
    char name[s.length()+1];
    if(2 != sscanf(s.c_str(), "%[^:]:%d", name, &p)) s.copy(name);
    struct hostent *he = gethostbyname(name);

    /* Can't figure it out, looks bad... */
    if(!he) return(false);
    sa->sin_family = he->h_addrtype;
    memcpy(&sa->sin_addr.s_addr, he->h_addr_list[0], he->h_length);
    return(true);
  }
};

class UdpDatagram {
  friend ostream &operator<<(ostream &os, const UdpDatagram &rhs) {
    os << "(" << rhs.cAddr << ") \"" << rhs.cData << "\"";
    return(os);
  }
private:
  InetAddr cAddr;
  String cData;
public:
  UdpDatagram(const String &s, InetAddr addr) : cAddr(addr), cData(s) {}
  UdpDatagram(const String &s = "") : cData(s) {}
  const String &GetData() const { return(cData); }
  const String &SetData(const String &d) { return(cData = d); }
  const InetAddr &GetAddr() const { return(cAddr); }
  void  SetAddr(const InetAddr &addr) { cAddr = addr; }
  size_t Length() const { return(cData.length()); }

};

class UdpSocket {
  int cFd;
  InetAddr cListenAddr;
  NeighbourEncryption* cEnc;

public:
  UdpSocket(const InetAddr& listen, int broadcast = 1, int mcLoop = 0) {
    if ((cFd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
      Report::Error("UdpSocket::UdpSocket() - socket() failed.");
    }

    cListenAddr = listen;
    sockaddr addr = cListenAddr;

    // If this is multicast address, prep data structure...
    bool mcast = IN_MULTICAST(ntohl(sockaddr_in(cListenAddr).sin_addr.s_addr)) ? 1 : 0;
    if(mcast) {
      struct ip_mreq mreq;
      memset(&mreq, 0, sizeof(mreq));
      mreq.imr_multiaddr = sockaddr_in(cListenAddr).sin_addr;
      mreq.imr_interface.s_addr = INADDR_ANY;

      if(setsockopt(cFd, IPPROTO_IP, IP_ADD_MEMBERSHIP, &mreq, sizeof(mreq))) {
	Report::Error(String("UdpSocket::UdpSocket() - setsockopt(IP_ADD_MEMBERSHIP) failed: ") +
		      strerror(errno));
      }
      if(setsockopt(cFd, IPPROTO_IP, IP_MULTICAST_LOOP, &mcLoop, sizeof(mcLoop))) {
	Report::Error(String("UdpSocket::UdpSocket() - setsockopt(IP_MULTICAST_LOOP) failed: ") +
		      strerror(errno));
      }
    }

    // Allow us to reuse this address in case we restart
    int reuse = 1;
    if(setsockopt(cFd, SOL_SOCKET, SO_REUSEADDR, &reuse, sizeof(reuse)) < 0) {
      close(cFd);
      char buf[1024];
      sprintf(buf,"UdpSocket::UdpSocket() - setsockopt(SO_REUSEADDR) failed: %s",strerror(errno));
      Report::Error(buf);
    }

    // Allow us to broadcast if desired
    if(broadcast) {
      if(setsockopt(cFd, SOL_SOCKET, SO_BROADCAST, &broadcast, sizeof(broadcast)) < 0) {
	close(cFd);
	char buf[1024];
	sprintf(buf,"UdpSocket::UdpSocket() - setsockopt(SO_BROADCAST) failed: %s",strerror(errno));
	Report::Error(buf);
      }
    }

    /* If receive port given, bind that port... */
    if(cListenAddr.Port()) {
      if(bind(cFd, &addr, sizeof(addr))) {
	close(cFd);
	char buf[1024];
	sprintf(buf,"UdpSocket::UdpSocket() - bind() failed: %s",strerror(errno));
	Report::Error(buf);
      }
    }

    cEnc = new NeighbourEncryption();
  };

  ~UdpSocket() {
  	close(cFd);
	delete( cEnc );
  };

  int Fd() { return cFd; }

  /* Binds the UdpSocket to a device interface
   * User may supply an optional filter to be attached to the interface
   */
  void BindDevice(const String& ifname) {
    struct ifreq ifr;
    memset(&ifr.ifr_name,0,IFNAMSIZ);
    int len = ((ifname.length() > IFNAMSIZ) ? IFNAMSIZ : ifname.length());
    strncpy(ifr.ifr_name, ifname.c_str(), len);
if (setsockopt(cFd, SOL_SOCKET, SO_BINDTODEVICE, (char *)&ifr, sizeof(ifr)) < 0) {
char buf[1024];
sprintf(buf,"UdpSocket::BindDevice() - setsockopt(SO_BINDTODEVICE) on '%s' failed.",ifname.c_str());
Report::Error(buf);
}
};


#define BUF_LEN 2048

UdpDatagram Recv() {
	char buf[BUF_LEN];
	int len;
	sockaddr addr = cListenAddr;
	socklen_t aLen = sizeof(addr);

	len = recvfrom(cFd, buf, sizeof(buf), 0, &addr, &aLen);

	if(len < 0) {
		char buf[1024];
		sprintf(buf,"UdpSocket::Recv() - recvfrom() failed: %s",strerror(errno));
		Report::Error(buf);
	}
	else {
		//* this should be somewhere else!
		if(isEncrypted) {
			//printf("* socket recv len: %d\n", len );
			//printf("* socket recv: %s\n", buf );
			// only if version is 1 and msgtype set to 50
			if (buf[0] == 1 && buf[1] == 50) {
				// encrypted data starts at 2
				len = cEnc->decrypt(buf+2, len-2, BUF_LEN-2, addr);
				if (len <= 0) {
					// decrypt failed -> discard
					printf("decrypt failed\n");
	    				len = 0;
				}
				//printf("* socket revc clear len: %d\n", len );
				//printf("* socket revc clear: %s\n", buf+2 );
				return(UdpDatagram(String(buf+2, 0, len), addr));
			}
			else {
				// we are in encrypted mode and receiving
				// unencrypted mobilemesh packages
				// we want to discard them
				printf("discarding unenc packet\n");
	    			len = 0;
    			}
		}
	}
	return(UdpDatagram(String(buf, 0, len), addr));
};


bool Send(const UdpDatagram &data) {
	int len;
	sockaddr addr = data.GetAddr();

	//* this should be somewhere else!
	if(isEncrypted) {
		char buf[BUF_LEN];
		//printf("* socket send clear len: %d\n", data.Length() );
		//printf("* socket send clear data: %s\n", data.GetData().c_str() );

		len = cEnc->encrypt(data.GetData().c_str(), buf+2, data.Length(), BUF_LEN-2);
		len = len+2;
		// set version to 1
		buf[0] = 1;
		// set Message Type to 50
		buf[1] = 50;

		//printf("* socket send enc len: %d\n", len );
		//printf("* socket send enc data: %s\n", buf );
		len = sendto(cFd, (void *)buf, len,
		 	0, &addr, sizeof(addr));
	}
	else {
		// normal operation
		len = sendto(cFd, (void *)data.GetData().c_str(), data.Length(),
			 0, &addr, sizeof(addr));
	}

	if(len <= 0) {
		char buf[1024];
		sprintf(buf,"UdpSocket::Send() - sendto() failed: %s",strerror(errno));
		Report::Error(buf);
		return false;
	}
	return true;
  };
};


#endif
