
/*
 * MIXMOS - an oss mixer
 *
 * Copyright (C) 2003 Tomasz Mka
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include "mixer.h"
#include "i18n.h"

static gint mixer_handler;
static gint devmask, stereodevs, recordsrc, recordmask;
static guchar temp[4];

static struct struct_mixer mixer[SOUND_MIXER_NRDEVICES];
static gchar *channels_lab[SOUND_MIXER_NRDEVICES] = SOUND_DEVICE_LABELS;
static struct mixer_info oss_info;


/***********************************************************************/

gint mixer_open(gchar *mixer_device)
{
	mixer_handler = open (mixer_device, O_RDWR, 0);
	if(mixer_handler < 0) 
		return ERR_OPEN;
    if(ioctl(mixer_handler, SOUND_MIXER_READ_DEVMASK, &devmask) == -1)
		return ERR_DEVMASK;
    if(ioctl(mixer_handler, SOUND_MIXER_READ_STEREODEVS, &stereodevs) == -1)
		return ERR_STEREODEVS;

	return 0;
}


/***********************************************************************/

gchar* mixer_get_name(void)
{
	ioctl(mixer_handler, SOUND_MIXER_INFO, &oss_info);
	return oss_info.name;
}

/***********************************************************************/

gint mixer_close(void)
{
	return (close(mixer_handler));
}

/***********************************************************************/

gint mixer_get_real_channel(gint channel)
{
gint i, j;
	
	for (i=0, j=-1; i<SOUND_MIXER_NRDEVICES; i++) {
		if (mixer[i].present == 1) j++;
		if (j==channel) break;
	}

	return i;
}

/***********************************************************************/

gint mixer_set_volume_mono(gint channel, gint volume)
{
gint chn;

	chn = mixer_get_real_channel(channel);

	if(mixer[chn].stereo == 1)
		return ERR_ISNOTMONO;

	if(volume < 0 || volume > 100) 
		return ERR_SETVOLUME;

    if(ioctl(mixer_handler, MIXER_WRITE(chn), &volume) == -1)
		return ERR_SETVOLUME;

	return 0;
}

/***********************************************************************/

gint mixer_set_volume_stereo(gint channel, gint volume_left, gint volume_right)
{
gint chn, volume;

	chn = mixer_get_real_channel(channel);

	if(mixer[chn].stereo != 1)
		return ERR_ISNOTSTEREO;

	if(volume_left < 0 || volume_left > 100) 
		return ERR_SETVOLUME;
	if(volume_right < 0 || volume_right > 100) 
		return ERR_SETVOLUME;

    volume = (volume_right << 8) | volume_left;

    if(ioctl(mixer_handler, MIXER_WRITE(chn), &volume) == -1)
		return ERR_SETVOLUME;

	return 0;
}

/***********************************************************************/

gint mixer_set_recordsrc(gint channel)
{
gint chn;

	chn = mixer_get_real_channel(channel);

	if(mixer[chn].recordable != 1)
		return ERR_ISNOTRECORDABLE;

    if (((recordsrc >> chn) & 1) == 0) recordsrc |= (1 << chn);

    if (ioctl(mixer_handler, SOUND_MIXER_WRITE_RECSRC, &recordsrc) == -1)
		return ERR_RECMASK;

	return 0;
}

/***********************************************************************/

gint mixer_state_read(void)
{
gint i, j;

	for (i=j=0; i<SOUND_MIXER_NRDEVICES; i++) {

		mixer[i].present = 0;

		if((devmask >> i) & 1) {
	
			mixer[i].channel = j++;
			mixer[i].r_channel = i;
			mixer[i].present = 1;

			if(ioctl(mixer_handler, MIXER_READ(i), &temp) == -1) 
				return ERR_STEREODEVS;
			if(ioctl(mixer_handler, SOUND_MIXER_READ_RECSRC,&recordsrc) == -1)
				return ERR_RECSRC;
			if(ioctl(mixer_handler,SOUND_MIXER_READ_RECMASK,&recordmask) == -1)
				return ERR_RECMASK;

			mixer[i].volume_left = temp[0];
			mixer[i].volume_right = temp[1];
			mixer[i].stereo = (stereodevs >> i) & 1;
			mixer[i].recordable = (recordmask >> i) & 1;
			mixer[i].record_selected = (recordsrc >> i) & 1;

		}
	}

	return 0;
}

/***********************************************************************/

gint mixer_get_channels(void)
{
gint i, j;

	for (i=j=0; i<SOUND_MIXER_NRDEVICES; i++)
		if (mixer[i].present == 1) j++;

	return j;
}

/***********************************************************************/

gchar* mixer_get_channel_name(gint channel)
{
	return channels_lab[mixer_get_real_channel(channel)];
}

/***********************************************************************/

gint mixer_get_channel_stereo(gint channel)
{
	return mixer[mixer_get_real_channel(channel)].stereo;
}

/***********************************************************************/

gint mixer_get_channel_recordable(gint channel)
{
	return mixer[mixer_get_real_channel(channel)].recordable;
}

/***********************************************************************/

gint mixer_get_recordsrc(gint channel)
{
	return mixer[mixer_get_real_channel(channel)].record_selected;
}

/***********************************************************************/

gint mixer_get_volume_l(gint channel)
{
	return mixer[mixer_get_real_channel(channel)].volume_left;
}

gint mixer_get_volume_r(gint channel)
{
	return mixer[mixer_get_real_channel(channel)].volume_right;
}

/***********************************************************************/

void mixer_state_print(void)
{
gint i;

	printf(_("\nSoundcard: %s\n\n"), mixer_get_name());

	for (i=0; i<SOUND_MIXER_NRDEVICES; i++) {

		if(mixer[i].present) {

			printf(_("%2d/%2d | %8s : %02d / %02d (stereo: %d, recordable: %d, record_select: %d)\n"), 
					mixer[i].channel, mixer[i].r_channel, channels_lab[i], mixer[i].volume_left, 
					mixer[i].volume_right, mixer[i].stereo, mixer[i].recordable, mixer[i].record_selected);
		}

	}

	printf("\n");
}

/***********************************************************************/

void mixer_check_error(gint err_code)
{
	if(err_code) {

		switch(err_code) {
			case ERR_OPEN:
				fprintf(stderr, _("Cannot open mixer device.\n"));
				break;
			case ERR_DEVMASK:
				fprintf(stderr, _("Cannot read SOUND_MIXER_DEVMASK.\n"));
				break;
			case ERR_STEREODEVS:
				fprintf(stderr, _("Cannot read SOUND_MIXER_STEREODEVS.\n"));
				break;
			case ERR_RECSRC:
				fprintf(stderr, _("Cannot read SOUND_MIXER_RECSRC.\n"));
				break;
			case ERR_RECMASK:
				fprintf(stderr, _("Cannot read SOUND_MIXER_RECMASK.\n"));
				break;
			case ERR_SETVOLUME:
				fprintf(stderr, _("Cannot set volume.\n"));
				break;
			case ERR_ISNOTMONO:
				fprintf(stderr, _("Selected channel is not mono channel.\n"));
				break;
			case ERR_ISNOTSTEREO:
				fprintf(stderr, _("Selected channel is not stereo channel.\n"));
				break;
			case ERR_ISNOTRECORDABLE:
				fprintf(stderr, _("Selected channel is not recordable.\n"));
				break;
			case ERR_NOCHANNEL:
				fprintf(stderr, _("Channel not found.\n"));
				break;

			default:
				fprintf(stderr, _("Error.\n"));
				break;
		}

		exit(-1);
	}
}

/***********************************************************************/


