/* miniapm - A tiny battery monitor

   Copyright 2002 Matthew Allum

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/

#include <libmb/mb.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef ENABLE_NLS
# include <libintl.h>
# define _(text) gettext(text)
#else
# define _(text) (text)
#endif

#ifdef USE_LIBSN
#define SN_API_NOT_YET_FROZEN 1
#include <libsn/sn.h>
#endif 

#define KEYRUS_IMG	"keyrus.png"
#define KEYENG_IMG	"keyeng.png"

static int keybstatus = 0;

static MBPixbuf *pb;
static MBPixbufImage *rus_icon = NULL, *eng_icon = NULL,
  *rus_icon_scaled = NULL, *eng_icon_scaled = NULL; 


static char* ThemeName = NULL;

#ifdef USE_LIBSN

static SnDisplay *sn_dpy;

static void 
sn_activate(char *name, char *exec_str)
{
  SnLauncherContext *context;
  pid_t child_pid = 0;

  context = sn_launcher_context_new (sn_dpy, 0);
  
  sn_launcher_context_set_name (context, name);
  sn_launcher_context_set_binary_name (context, exec_str);
  
  sn_launcher_context_initiate (context, "monoluanch launch", exec_str,
				CurrentTime);

  switch ((child_pid = fork ()))
    {
    case -1:
      fprintf (stderr, "Fork failed\n" );
      break;
    case 0:
      sn_launcher_context_setup_child_process (context);
      mb_exec(exec_str);
      fprintf (stderr, "Failed to exec %s \n", exec_str);
      _exit (1);
      break;
    }
}

#endif


void 
fork_exec(char *cmd)
{
  switch (fork())
    {
    case 0:
      setpgid(0, 0); /* Stop us killing child */
      mb_exec(cmd);
      fprintf(stderr, "minitime: Failed to Launch '%s'", cmd);
      exit(1);
    case -1:
      fprintf(stderr, "minitime: Failed to Launch '%s'", cmd);
      break;
    }
}

void
paint_callback (MBTrayApp *app, Drawable drw )
{


  int      power_pixels = 0;
  unsigned char r = 0, g = 0, b = 0;
  int      x, y;
  int      bar_width, bar_height, bar_x, bar_y;

  MBPixbufImage *img_backing = NULL;
  
  img_backing = mb_tray_app_get_background (app, pb);

  mb_pixbuf_img_composite(pb, img_backing, keybstatus?rus_icon_scaled:eng_icon_scaled, 0, 0);

  mb_pixbuf_img_render_to_drawable(pb, img_backing, drw, 0, 0);

  mb_pixbuf_img_free( pb, img_backing );
}

void
resize_callback (MBTrayApp *app, int w, int h )
{

 if (rus_icon_scaled) mb_pixbuf_img_free(pb, rus_icon_scaled);
 if (eng_icon_scaled) mb_pixbuf_img_free(pb, eng_icon_scaled);

 rus_icon_scaled = mb_pixbuf_img_scale(pb, rus_icon, w, h);
 eng_icon_scaled = mb_pixbuf_img_scale(pb, eng_icon, w, h);

}

void 
load_icon(void)
{
 char *icon_path = NULL;
 
 printf("%s() called", __func__);

 if (rus_icon) mb_pixbuf_img_free(pb, rus_icon);
 if (eng_icon) mb_pixbuf_img_free(pb, eng_icon);

 icon_path = mb_dot_desktop_icon_get_full_path (ThemeName, 
						32, 
						KEYRUS_IMG );

 if (icon_path == NULL 
     || !(rus_icon = mb_pixbuf_img_new_from_file(pb, icon_path)))
    {
      fprintf(stderr, "mb-applet-keyboard: failed to load icon %s\n", KEYRUS_IMG);
      exit(1);
    }

  free(icon_path);

  icon_path = mb_dot_desktop_icon_get_full_path (ThemeName, 
						 32, 
						 KEYENG_IMG );

 if (icon_path == NULL 
     || !(eng_icon = mb_pixbuf_img_new_from_file(pb, icon_path)))
    {
      fprintf(stderr, "mb-applet-battery: failed to load icon %s\n", KEYENG_IMG);
      exit(1);
    }

 free(icon_path);

 return;
}

void 
theme_callback (MBTrayApp *app, char *theme_name)
{
  if (!theme_name) return;
  if (ThemeName) free(ThemeName);
  ThemeName = strdup(theme_name);
  load_icon();
  resize_callback (app, mb_tray_app_width(app), mb_tray_app_width(app) );
}

void
button_callback (MBTrayApp *app, int x, int y, Bool is_released )
{
  char tray_msg[256];

  if (!is_released) return;

  if (keybstatus) {
	keybstatus = 0;
	system("xmodmap /etc/X11/kb/corgi.xmodmap");
  } else {
	keybstatus = 1;	
	system("xmodmap /etc/X11/kb/corgi.xmodmap_ru");
  }

//  sprintf(tray_msg, _("Status HBZ.\n"));
//  mb_tray_app_tray_send_message(app, tray_msg, 5000);

  mb_tray_app_repaint(app);
}

/*
void
context_callback ( MBTrayApp *app )
{
#ifdef USE_LIBSN
  if (CONTEXT_APP_WANT_SN)
    {
      sn_activate(CONTEXT_APP, CONTEXT_APP " " CONTEXT_APP_ARGS);      
      return;
    }
#endif

  fork_exec(CONTEXT_APP " " CONTEXT_APP_ARGS);
}
 */

static Bool 
file_exists(char *filename)
{
  struct stat st; 		/* XXX Should probably check if exe too */
  if (stat(filename, &st)) return False;
  return True;
}

int
main( int argc, char *argv[])
{
  MBTrayApp *app = NULL;
  struct timeval tv;

#if ENABLE_NLS
  setlocale (LC_ALL, "");
  bindtextdomain (PACKAGE, DATADIR "/locale");
  bind_textdomain_codeset (PACKAGE, "UTF-8"); 
  textdomain (PACKAGE);
#endif


  app = mb_tray_app_new ( _("Keyboard Switcher"),
			  resize_callback,
			   paint_callback,
			  &argc,
			  &argv );  
  
   pb = mb_pixbuf_new(mb_tray_app_xdisplay(app), 
		      mb_tray_app_xscreen(app));
   
   mb_tray_app_set_button_callback (app, button_callback );
   
   mb_tray_app_set_theme_change_callback (app, theme_callback );
   
   load_icon();

   keybstatus = 0;
   mb_tray_app_set_icon(app, pb, eng_icon);

#ifdef USE_LIBSN
  sn_dpy = sn_display_new (mb_tray_app_xdisplay(app), NULL, NULL);
#endif
   
   mb_tray_app_main (app);

   return 1;
}


