/* H5400 Fingerprint Sensor Interface driver 
 * Copyright (c) 2004 Jrgen Andreas Michaelsen <jorgenam@ifi.uio.no> 
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/ioctl.h>
#include <linux/fs.h>
#include <linux/delay.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/miscdevice.h>
#include <linux/interrupt.h>

#include <asm/uaccess.h>
#include <asm/atomic.h>

#include <asm/mach-types.h>
#include <asm/arch/hardware.h>
#include <asm/arch/irqs.h>
#include <asm/arch/h5400-asic.h>
#include <asm/arch/h5400-irqs.h>
#include <asm/arch/h5400-gpio.h>

#include <asm/arch/h5400-fsi.h>

atomic_t h5400_fsi_in_use;
DECLARE_WAIT_QUEUE_HEAD(h5400_fsi_rqueue);
struct timer_list h5400_fsi_temp_timer;

unsigned int h5400_fsi_prescale = 19;
unsigned int h5400_fsi_dmi = 1124;
unsigned int h5400_fsi_treshold_on = 20;
unsigned int h5400_fsi_treshold_off = 4;
unsigned int h5400_fsi_buffer_size = H5400_FSI_FRAME_SIZE * (sizeof(unsigned long)) * 3;

volatile unsigned char h5400_fsi_lfrm = 0;

void h5400_fsi_set_mode(unsigned int cmd)
{
	static unsigned int current_mode = 0;
	static unsigned long control_register = 0;
	static unsigned int temp_on = 0;

	switch(cmd) {
	case H5400_FSI_CMD_MODE_STOP:
		if ((control_register & 0x3) == 0)
			break;
		control_register &= ~0x3;
		h5400_asic_write_register(H5400_ASIC_FSI_Control, control_register);
		current_mode = cmd;
		break;
	case H5400_FSI_CMD_MODE_START:
		if (current_mode == cmd)
			control_register |= 0x2;
		else
			control_register = 0xF9 | temp_on;

		h5400_asic_write_register(H5400_ASIC_FSI_Control, control_register);
		control_register |= 0x2;
		current_mode = cmd;
		h5400_fsi_lfrm = 0;
		break;
	case H5400_FSI_CMD_MODE_NAP:
		control_register = 0x2 | temp_on;
		h5400_asic_write_register(H5400_ASIC_FSI_Control, control_register);
		current_mode = cmd;
		break;
	case H5400_FSI_CMD_STOP_ACQ_INT:
		control_register &= 0x7F;
		h5400_asic_write_register(H5400_ASIC_FSI_Control, control_register);
		break;
	case H5400_FSI_CMD_START_ACQ_INT:
		control_register |= 0x80;
		h5400_asic_write_register(H5400_ASIC_FSI_Control, control_register);
		break;
	case H5400_FSI_CMD_STOP_INFO_INT:
		control_register &= 0x8F;
		h5400_asic_write_register(H5400_ASIC_FSI_Control, control_register);
		break;
	case H5400_FSI_CMD_START_INFO_INT:
		control_register |= 0x70;
		h5400_asic_write_register(H5400_ASIC_FSI_Control, control_register);
		break;
	case H5400_FSI_CMD_RESET:
		control_register = 0 | temp_on;
		current_mode = H5400_FSI_CMD_MODE_STOP;
		h5400_asic_write_register(H5400_ASIC_FSI_Control, control_register);
		break;
	case H5400_FSI_CMD_START_TEMP:
		control_register |= H5400_FSI_CONTROL_TEMP;
		h5400_asic_write_register(H5400_ASIC_FSI_Control, control_register);
		temp_on = H5400_FSI_CONTROL_TEMP;
		break;
	case H5400_FSI_CMD_STOP_TEMP:
		control_register &= ~H5400_FSI_CONTROL_TEMP;
		h5400_asic_write_register(H5400_ASIC_FSI_Control, control_register);
		temp_on = 0;
		break;
	}
}

void h5400_fsi_timer_temp_callback(unsigned long input)
{
	printk(KERN_DEBUG "%s: stopping temperature increase (status=%ld)\n", __FUNCTION__, h5400_asic_read_register(H5400_ASIC_FSI_Control) & H5400_FSI_CONTROL_TEMP);
	h5400_fsi_set_mode(H5400_FSI_CMD_STOP_TEMP);
}

inline int h5400_fsi_analyze_column(struct h5400_fsi_read_state *read_state, unsigned int data)
{
	if ((data & 0xEFEF) == 0xE0E0) { /* sync column */
		read_state->frame_number++;
		read_state->frame_offset = 0;
		read_state->detect_value = 0;

		return 1;

	} else if (read_state->frame_number < 1) { /* skip first frame */
		return 0;

	} else if (read_state->frame_offset > (((H5400_FSI_FRAME_SIZE-1)/2)-8) &&
		   read_state->frame_offset < ((H5400_FSI_FRAME_SIZE-1)/2)) {
		/* FIXME: how do we solve this? */

	} else if (read_state->frame_offset == ((H5400_FSI_FRAME_SIZE-1)/2)) {
		read_state->detect_value += abs(H5400_FSI_DB1_EVEN(data) - H5400_FSI_DB1_ODD(data));
		read_state->detect_value += abs(H5400_FSI_DB1_ODD(data) - H5400_FSI_DB2_EVEN(data));
		read_state->detect_value += abs(H5400_FSI_DB2_EVEN(data) - H5400_FSI_DB2_ODD(data));
		read_state->detect_value += abs(H5400_FSI_DB2_ODD(data) - H5400_FSI_DB3_EVEN(data));
		read_state->detect_value += abs(H5400_FSI_DB3_EVEN(data) - H5400_FSI_DB3_ODD(data));
		read_state->detect_value += abs(H5400_FSI_DB3_ODD(data) - H5400_FSI_DB4_EVEN(data));
		read_state->detect_value += abs(H5400_FSI_DB4_EVEN(data) - H5400_FSI_DB4_ODD(data));

		if (read_state->finger_present == 0 && read_state->detect_value > read_state->treshold_on)
			read_state->detect_count++;
		else if (read_state->finger_present && read_state->detect_value < read_state->treshold_off)
			read_state->detect_count++;
		else
			read_state->detect_count = 0;

		if (read_state->detect_count > 2) {
			read_state->finger_present = !read_state->finger_present;
#if 1
			printk(KERN_DEBUG "%s: finger_present=%d at frame=%d [value=%d]\n", __FUNCTION__, read_state->finger_present, read_state->frame_number, read_state->detect_value);
#endif
		}
	}

	read_state->frame_offset++;
	return 0;
}

void h5400_fsi_run_tasklet(unsigned long state_ptr)
{
	unsigned int i;
	unsigned long data;
	unsigned int words_in_fifo;
	int sync;
	struct h5400_fsi_read_state *read_state;


	read_state = (struct h5400_fsi_read_state *)state_ptr;	
	words_in_fifo = (h5400_asic_read_register(H5400_ASIC_FSI_Status)>>1) & 0x1FF;

	for (i = 0; i < words_in_fifo; i++) {
		data = h5400_asic_read_register(H5400_ASIC_FSI_Data32);
		sync = h5400_fsi_analyze_column(read_state, data);

#if 0
		if (read_state->finger_present == 0 && sync == 0)
			data = 0;
#endif

		if (read_state->do_read) {
			read_state->buffer[read_state->write_pos] = data;
			read_state->write_pos++;
			if (read_state->write_pos >= read_state->buffer_size)
				read_state->write_pos = 0;

			if (sync && read_state->finger_present == 0)
				read_state->finished = 1;

			if (read_state->write_pos == read_state->read_pos)
				printk(KERN_WARNING "%s: DCOL frame=%d offset=%d\n", __FUNCTION__,
				       read_state->frame_number, read_state->frame_offset);
		}
		else if (sync && read_state->finger_present)
			read_state->do_read = 1;
	}

	h5400_fsi_set_mode(H5400_FSI_CMD_START_ACQ_INT);
	wake_up_interruptible (&h5400_fsi_rqueue);
}

DECLARE_TASKLET_DISABLED(h5400_fsi_tasklet,h5400_fsi_run_tasklet,0);

void h5400_fsi_irq_handler(int irq, void *dev_id, struct pt_regs *regs)
{
	unsigned int fifo_status;


	fifo_status = h5400_asic_read_register(H5400_ASIC_FSI_Status);

	if (fifo_status & 1) {
		h5400_fsi_set_mode(H5400_FSI_CMD_STOP_ACQ_INT);
		tasklet_schedule(&h5400_fsi_tasklet);
	}
	if (fifo_status & (1<<10)) {
		printk( KERN_DEBUG "%s: DCOL\n", __FUNCTION__ );
		h5400_asic_write_register(H5400_ASIC_FSI_Status, 1<<10);
	}
	if (fifo_status & (1<<11)) {
		printk( KERN_DEBUG "%s: INVL\n", __FUNCTION__ );
		h5400_asic_write_register(H5400_ASIC_FSI_Status, 1<<11);
	}
	if (fifo_status & (1<<12)) { /* LFRM */
		h5400_asic_write_register(H5400_ASIC_FSI_Status, 1<<12);
		h5400_fsi_lfrm = 1;
		tasklet_schedule(&h5400_fsi_tasklet);
	}
}

inline int h5400_fsi_copy_to_user(struct h5400_fsi_read_state *read_state, char *buf)
{
	unsigned int avail_words;

	if (read_state->read_pos == read_state->write_pos)
		return 0;
	else if (read_state->write_pos < read_state->read_pos)
		avail_words = read_state->buffer_size - read_state->read_pos;
	else
		avail_words = read_state->write_pos - read_state->read_pos;
	
	if (copy_to_user(buf+read_state->word_count*4,read_state->buffer+read_state->read_pos,avail_words*4))
		return -EFAULT;

	read_state->word_count += avail_words;
	if ((read_state->word_count+H5400_FSI_FRAME_SIZE) > read_state->word_dest ||
	    read_state->finished)
		h5400_fsi_set_mode(H5400_FSI_CMD_MODE_STOP);		

	read_state->read_pos += avail_words;
	if (read_state->read_pos >= read_state->buffer_size)
		read_state->read_pos = 0;
		
	if (read_state->read_pos < read_state->write_pos)
		return h5400_fsi_copy_to_user(read_state, buf);

	return 0;
}

int h5400_fsi_open(struct inode *inode, struct file *filp)
{
	struct h5400_fsi_read_state *read_state;

	/* we allow only one user at a time */
	if (atomic_dec_and_test(&h5400_fsi_in_use)) {
		atomic_inc(&h5400_fsi_in_use);
		return -EBUSY;
	}

	/* init state */
	read_state = kmalloc(sizeof *read_state,GFP_KERNEL);
	if (read_state == NULL) {
		atomic_inc(&h5400_fsi_in_use);
		return -ENOMEM;
	}

	read_state->buffer = NULL;
	read_state->word_count = 0;
	read_state->word_dest = 0;
	read_state->detect_count = 0;
	read_state->frame_number = 0;
	read_state->frame_offset = 0;

	filp->private_data = read_state;

#if 0
	/* GPIO61 turns out to be serial power, not fingerchip power.
	   Any other guesses?  */
	SET_H5400_GPIO(POWER_FP_N, 0); /* power on */
#endif
	h5400_asic_write_register(H5400_ASIC_GPIO_GPD_CON, 0xaaaa); /* connect the wires */
	h5400_asic_write_register(H5400_ASIC_GPIO_GPE_CON, 0x7);

	H5400_ASIC_SET_BIT(H5400_ASIC_CPM_ClockControl, H5400_ASIC_CPM_CLKCON_FCD_CLKEN); /* start PCLK */

	/* init hardware */
	h5400_fsi_set_mode(H5400_FSI_CMD_RESET);
	h5400_asic_write_register(H5400_ASIC_FSI_Prescaler, h5400_fsi_prescale); 
	h5400_asic_write_register(H5400_ASIC_FSI_DMI, h5400_fsi_dmi);

	h5400_asic_write_register(H5400_ASIC_FSI_FIFO, 1); /* FIFO reset */

	return 0;
}

int h5400_fsi_release(struct inode *inode, struct file *filp)
{
	/* put scanner to sleep before we exit */
	h5400_fsi_set_mode(H5400_FSI_CMD_STOP_TEMP);
#if 0
	SET_H5400_GPIO(POWER_FP_N, 1); /* power off */
#endif
	H5400_ASIC_CLEAR_BIT(H5400_ASIC_CPM_ClockControl, H5400_ASIC_CPM_CLKCON_FCD_CLKEN); /* Stop PCLK */

	kfree(filp->private_data);
	atomic_inc(&h5400_fsi_in_use);

	return 0;
}

ssize_t h5400_fsi_read(struct file *filp, char *buf, size_t count, loff_t *offp)
{
	int errval = 0;
	struct h5400_fsi_read_state *read_state;

	read_state = (struct h5400_fsi_read_state *)filp->private_data;

	/* round count request down to a whole frame */
	read_state->word_dest = ((count/(sizeof (unsigned long)))/H5400_FSI_FRAME_SIZE) * H5400_FSI_FRAME_SIZE;
	if (read_state->word_dest == 0)
		return 0;

	/* allocate buffer and initialize state */
	read_state->buffer_size = h5400_fsi_buffer_size;
	read_state->buffer = kmalloc(read_state->buffer_size,GFP_KERNEL);
	if (read_state->buffer == NULL)
		return -ENOMEM;

	read_state->word_count = 0;
	read_state->write_pos = 0;
	read_state->read_pos = 0;
	read_state->detect_count = 0;
	read_state->frame_number = 0;
	read_state->frame_offset = 0;
	read_state->do_read = 0;
	read_state->finished = 0;
	read_state->finger_present = 0;
	read_state->treshold_on = h5400_fsi_treshold_on;
	read_state->treshold_off = h5400_fsi_treshold_off;

	h5400_fsi_lfrm = 0;

	h5400_fsi_tasklet.data = (unsigned long)read_state;
	tasklet_enable(&h5400_fsi_tasklet);

	/* start clock, reset and frame receive start */
	h5400_asic_write_register(H5400_ASIC_FSI_FIFO, 0x3);
	h5400_fsi_set_mode(H5400_FSI_CMD_MODE_START);

	/* gather data */
	while (read_state->word_count < read_state->word_dest) {
		interruptible_sleep_on(&h5400_fsi_rqueue);
		if (signal_pending(current)) {
			errval = -ERESTARTSYS;
			goto h5400_fsi_read_done;
		}

		errval = h5400_fsi_copy_to_user(read_state, buf);
		if (errval)
			goto h5400_fsi_read_done;

		if (h5400_fsi_lfrm)
			break;
	}

 h5400_fsi_read_done:

	/* stop clock and set stop mode */
	h5400_fsi_set_mode(H5400_FSI_CMD_MODE_STOP);

	tasklet_disable(&h5400_fsi_tasklet);
	h5400_fsi_tasklet.data = 0;

	kfree(read_state->buffer);

	if (errval)
		return errval;

	count = read_state->word_count * (sizeof (unsigned long));

	*offp += count;
	return count;
}

int h5400_fsi_ioctl(struct inode *inode, struct file *filp, unsigned int cmd, unsigned long arg)
{
	struct h5400_fsi_read_state *read_state;

	read_state = (struct h5400_fsi_read_state *)filp->private_data;

	switch(cmd)
	{
	case H5400_FSI_IOCSTOPTEMP: /* unconditional stop temperature increase */
		h5400_fsi_set_mode(H5400_FSI_CMD_STOP_TEMP);
		printk(KERN_DEBUG "%s: H5400_FSI_IOCSTOPTEMP\n", __FUNCTION__);
		break;
	case H5400_FSI_IOCSTARTTEMP: /* start temperature increase */
		h5400_fsi_set_mode(H5400_FSI_CMD_START_TEMP);
		printk(KERN_DEBUG "%s: H5400_FSI_IOCSTARTTEMP\n", __FUNCTION__);
		break;
	case H5400_FSI_IOCINCTEMP: /* increase temperature [arg] steps */
		h5400_fsi_set_mode(H5400_FSI_CMD_START_TEMP);
		mod_timer(&h5400_fsi_temp_timer, jiffies + 150 * arg);
		printk(KERN_DEBUG "%s: H5400_FSI_IOCINCTEMP %ld steps\n", __FUNCTION__, arg);
		break;
	case H5400_FSI_IOCSETPRESCALE:
		h5400_asic_write_register(H5400_ASIC_FSI_Prescaler, arg); /* FIXME: check range? */
		printk(KERN_DEBUG "%s: H5400_FSI_IOCSETPRESCALE %ld\n", __FUNCTION__, arg);
		break;
	case H5400_FSI_IOCSETDMI:
		h5400_asic_write_register(H5400_ASIC_FSI_DMI, arg); /* FIXME: check range? */
		printk(KERN_DEBUG "%s: H5400_FSI_IOCSETDMI %ld\n", __FUNCTION__, arg);
		break;
	case H5400_FSI_IOCSETTRESHOLDON:
		read_state->treshold_on = arg;
		printk(KERN_DEBUG "%s: H5400_FSI_IOCSETTRESHOLDON %ld\n", __FUNCTION__, arg);
		break;
	case H5400_FSI_IOCSETTRESHOLDOFF:
		read_state->treshold_off = arg;
		printk(KERN_DEBUG "%s: H5400_FSI_IOCSETTRESHOLDOFF %ld\n", __FUNCTION__, arg);
		break;
	default:
		return -ENOTTY;
	}

	return 0;
}

static struct file_operations h5400_fsi_fops =
{
	owner:    THIS_MODULE,
	open:     h5400_fsi_open,
	release:  h5400_fsi_release,
	read:     h5400_fsi_read,
	ioctl:    h5400_fsi_ioctl,
};

static struct miscdevice h5400_fsi_miscdev = 
{
	MISC_DYNAMIC_MINOR,
	"fsi",
	&h5400_fsi_fops
};

static int __init
h5400_fsi_init(void)
{ 
	int irq_result;

	if (!machine_is_h5400())
		return -ENODEV;

	misc_register(&h5400_fsi_miscdev);

	irq_result = request_irq(IRQ_H5400_FCD, h5400_fsi_irq_handler, 0, "h5400-fsi", NULL);
	if (irq_result) {
		misc_deregister(&h5400_fsi_miscdev);
		return irq_result;
	}

	init_timer(&h5400_fsi_temp_timer);
	h5400_fsi_temp_timer.function = h5400_fsi_timer_temp_callback;

	atomic_set(&h5400_fsi_in_use,0);

	return 0; 
}

static void __exit
h5400_fsi_exit(void)
{
	h5400_fsi_set_mode(H5400_FSI_CMD_STOP_TEMP);
	del_timer(&h5400_fsi_temp_timer);
	H5400_ASIC_CLEAR_BIT(H5400_ASIC_CPM_ClockControl, H5400_ASIC_CPM_CLKCON_FCD_CLKEN); /* Stop PCLK */
	free_irq(IRQ_H5400_FCD, NULL);
	misc_deregister(&h5400_fsi_miscdev);
}

EXPORT_NO_SYMBOLS;

MODULE_AUTHOR("Jrgen Andreas Michaelsen <jorgenam@ifi.uio.no>");
MODULE_DESCRIPTION("H5400 Fingerprint Scanner Interface driver");
MODULE_LICENSE("GPL");

MODULE_PARM(h5400_fsi_prescale,"i");
MODULE_PARM(h5400_fsi_dmi,"i");
MODULE_PARM(h5400_fsi_treshold_on,"i");
MODULE_PARM(h5400_fsi_treshold_off,"i");
MODULE_PARM(h5400_fsi_buffer_size,"i");

module_init(h5400_fsi_init);
module_exit(h5400_fsi_exit);
