/*
 * This is a kernel driver for a non volatile ramdisk
 *
 * originally created 22 Jan 2002
 *
 * author: Erez Doron (erez@savan.com)
 *
 * licence: GPL
 *
 * Jamey Hicks JAN-30-2002 
 * Added nvrd_size cmdline option, made all routines static, reindented to
 * linux style, and added module_init and module_cleanup.  
 *
 * Erez Doron FEB-18-2002
 * Fixed offset problem.
 * IMPORTANT: the driver is not backward compatible
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/init.h>
#ifndef I386
#include <asm/arch/memory.h>
#endif
#include <asm/io.h>
#include <asm/uaccess.h>
#include <linux/devfs_fs_kernel.h>
#include <linux/blk.h>
#include <linux/blkpg.h>


#define NVRD_MAJOR	0               /* auto detect */
#define NVRD_NAME	"nvrd"
#define BLOCKSIZE	1024
#define SECTSIZE	512
#define SECTSIZE_SHIFT	9	/* log 2 of SECTSIZE */

static int nvrd_major;
static unsigned int nvrd_nbytes=0;      /* size in bytes */
static int nvrd_nblocks;		/* size in blocks */
static int nvrd_blocksize;
static int nvrd_sectsize;
static int nvrd_sectsize_shift;	/* log 2 of nvrd_sectsize */
static long nvrd_nsectors;
static int nvrd_bytesize;	/* size in bytes */
static char *nvrd_dptr;		/* data pointer */
static devfs_handle_t devfs_handle;

MODULE_PARM(nvrd_nbytes, "i");
MODULE_PARM_DESC(nvrd_nbytes, "Physical Memory Size (i.e. real 'mem=')");

int __init setup_nvrd_size(char *cmdline)
{
        char *aftersizeptr;
        nvrd_nbytes = memparse(cmdline, &aftersizeptr);
        printk("%s: nvrd_size=%x\n", __FUNCTION__, nvrd_nbytes);
        return 1;
} 

__setup("nvrd_size=", setup_nvrd_size);

static int nvrd_open(struct inode *inode, struct file *file)
{
        if (MINOR(inode->i_rdev) > 0)
                return -ENXIO;

        MOD_INC_USE_COUNT;

        return 0;
}

static int nvrd_release(struct inode *inode, struct file *file)
{
        MOD_DEC_USE_COUNT;

        return 0;
}

static int nvrd_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg)
{

        if (MINOR(inode->i_rdev)) return -EINVAL;
        switch (cmd) {
        case BLKFLSBUF :
                if (!capable(CAP_SYS_ADMIN))
                        return -EACCES;
                destroy_buffers(inode->i_rdev);
                break;

        case BLKGETSIZE :
                return put_user(nvrd_nsectors, (long *) arg);

        case BLKROSET :
        case BLKROGET :
        case BLKSSZGET :
                return blk_ioctl(inode->i_rdev, cmd, arg);    

        default :
                return -EINVAL;
        }

        return 0;
}

static struct block_device_operations nvrd_bd_op = {
        open:	        nvrd_open,
        release:	nvrd_release,
        ioctl:	        nvrd_ioctl,
};

static int nvrd_make_request(request_queue_t *q, int rw, struct buffer_head *sbh)
{
        char *ptr, *bdata;
        long offset, len ,len1;


        if ((MINOR(sbh->b_rdev)) > 0) goto fail;


        len = sbh->b_size;

        /* i put the first block in end of mem and last block
         * in the beginning so when enlarging the ramdisk by 
         * starting it previuosly in the memory, will really
         * enlarge the end of the disk so one could use
         * programs like resize2fs
         */

           offset = (nvrd_nsectors-(sbh->b_rsector)-1l) << nvrd_sectsize_shift;
           //offset = sbh->b_rsector << nvrd_sectsize_shift;
           bdata = bh_kmap(sbh);

	for (;len;len-=len1,offset-=nvrd_sectsize,bdata+=nvrd_sectsize)
	{
		//len1=min(len,nvrd_sectsize)
	   len1=len<nvrd_sectsize ? len:nvrd_sectsize;


           if (((offset+len1)>nvrd_bytesize)||(offset<0))
           {
#ifdef DEBUG
                   printk(KERN_INFO NVRD_NAME ": attempt to access beyond device, b_rsector=%lu\n",(unsigned long)(sbh->b_rsector));
		   printk(KERN_INFO NVRD_NAME ": offset=%li, len1=%li, nvrd_bytesize=%li\n",offset,len1,(long)nvrd_bytesize);
#else
                   printk(KERN_INFO NVRD_NAME ": attempt to access beyond device\n");
#endif
                   goto fail;
           }

           ptr = nvrd_dptr + offset;


           switch (rw) {
           case READ :
           case READA : 
                   memcpy(bdata, ptr, len1);
                   break;
           case WRITE :
                   memcpy(ptr, bdata, len1);
                   break;
           default :
                   printk(KERN_INFO KERN_INFO NVRD_NAME ": bad command: %d\n", rw);
                   goto fail;
           }
	}
   
        sbh->b_end_io(sbh, 1);
        return 0;
    
 fail:
        sbh->b_end_io(sbh, 0);
        return 0;
}

static int __init init_nvrd(void)
{
        unsigned long MemEnd,UsedMemEnd;

        UsedMemEnd=(num_physpages<<PAGE_SHIFT) + PHYS_OFFSET;  /* PHYS_OFFSET is physical address of start of DRAM */
        MemEnd=(((unsigned long)nvrd_nbytes)<<20) + PHYS_OFFSET;

#ifdef DEBUG
        printk(KERN_INFO NVRD_NAME ": num_physpages= %lu\n",num_physpages);
        printk(KERN_INFO NVRD_NAME ": max_mapnr= %lu\n",max_mapnr);
        printk(KERN_INFO NVRD_NAME ": PAGE_SIZE= %lu\n",PAGE_SIZE);
        printk(KERN_INFO NVRD_NAME ": PAGE_SHIFT= %u\n",PAGE_SHIFT);
        printk(KERN_INFO NVRD_NAME ": using adresses: 0x%lX..0x%lX\n",UsedMemEnd,MemEnd-1);
#endif
        if (MemEnd<=(UsedMemEnd-PAGE_SIZE))
        {
                printk(KERN_INFO NVRD_NAME ": kernel must be loaded with 'mem=' and " NVRD_NAME " must be inserted with 'nvrd_nbytes=' and size of physical memory in Mb\n");
                return -ENOMEM;
        }
	
        nvrd_sectsize = SECTSIZE;
        nvrd_sectsize_shift = SECTSIZE_SHIFT;
        nvrd_blocksize = BLOCKSIZE;
        nvrd_nblocks=(MemEnd-UsedMemEnd)/nvrd_blocksize; // in blocks
        nvrd_bytesize = (nvrd_nblocks*nvrd_blocksize);
        nvrd_nsectors=nvrd_bytesize >> nvrd_sectsize_shift;
#ifdef DEBUG
        printk(KERN_INFO NVRD_NAME ": nvrd_bytesize= %u\n",nvrd_bytesize);
        printk(KERN_INFO NVRD_NAME ": nvrd_nsectors= %lu\n",nvrd_nsectors);
#endif

        // remap physical memory to virtual one
        nvrd_dptr=__ioremap(UsedMemEnd,MemEnd-UsedMemEnd,0);
        if (!nvrd_dptr)
        {
                printk(KERN_INFO NVRD_NAME ": __ioremap failed\n");
                return -EIO;
        }
#ifdef DEBUG
        printk(KERN_INFO NVRD_NAME ": virtual mem pointer= 0x%lX\n",(unsigned long)nvrd_dptr);
#endif
	

        if ((nvrd_major = devfs_register_blkdev(NVRD_MAJOR, NVRD_NAME, &nvrd_bd_op)) < 0) {
                printk(KERN_INFO NVRD_NAME ": Device registration failed (%d)\n", nvrd_major);
                return -EIO;
        }

        printk(KERN_INFO NVRD_NAME ": major number %d assigned\n", nvrd_major);
    
        blk_queue_make_request(BLK_DEFAULT_QUEUE(nvrd_major), &nvrd_make_request);

        devfs_handle = devfs_mk_dir(NULL, NVRD_NAME, NULL);
        devfs_register_series(devfs_handle, "%u", 1,
                              DEVFS_FL_DEFAULT, nvrd_major, 0,
                              S_IFBLK | S_IRUSR | S_IWUSR,
                              &nvrd_bd_op, NULL);

    
        hardsect_size[nvrd_major] = &nvrd_sectsize;
        blksize_size[nvrd_major] = &nvrd_blocksize;
        blk_size[nvrd_major] = &nvrd_nblocks;

        return 0;
}

static void __exit cleanup_nvrd(void)
{

        destroy_buffers(MKDEV(nvrd_major, 0));
        devfs_unregister(devfs_handle);
        devfs_unregister_blkdev(nvrd_major, NVRD_NAME);
        hardsect_size[nvrd_major] = NULL;
        blksize_size[nvrd_major] = NULL;
        blk_size[nvrd_major] = NULL;

        printk(KERN_INFO NVRD_NAME " unloaded\n");
}

module_init(init_nvrd);
module_exit(cleanup_nvrd);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Erez Doron <erez@savan.com>");
MODULE_DESCRIPTION("Non volatile Ram Disk");
