/****************************************************************************/
/* Copyright 2000 Compaq Computer Corporation.                              */
/*                                           .                              */
/* Copying or modifying this code for any purpose is permitted,             */
/* provided that this copyright notice is preserved in its entirety         */
/* in all copies or modifications.  COMPAQ COMPUTER CORPORATION             */
/* MAKES NO WARRANTIES, EXPRESSED OR IMPLIED, AS TO THE USEFULNESS          */
/* OR CORRECTNESS OF THIS CODE OR ITS FITNESS FOR ANY PARTICULAR            */
/* PURPOSE.                                                                 */
/****************************************************************************/
/*
 * bootldr file for Compaq Personal Server Bootloader
 *
 */

#include <linux/types.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include "architecture.h"
#include <linux/bootldr/bootldr.h>
#include "serial.h"
#include <linux/bootldr/commands.h>
#include <linux/vmalloc.h>

static void msleep(unsigned int msec)
{
        current->state = TASK_INTERRUPTIBLE;
        schedule_timeout( (msec * HZ + 999) / 1000);
}


#define VERSION_MAJOR 3
#define VERSION_MINOR 0
#define VERSION_MICRO 0
#define VERSION_SPECIAL "LAB"

static char bootprog_name[] = "LAB - Linux As Bootldr";

#define	DEB(s)	putstr(s)

#define ROUNDUP(v,szpwr2) (((v)+((szpwr2)-1))&~(szpwr2-1))

int ack_commands = 0;
int no_cmd_echo = 0;
long reboot_button_enabled = 0;

unsigned long last_ram_load_address = 0;
unsigned long last_ram_load_size = 0;

extern void *SerBase, *Ser1Base, *Ser3Base;

extern void getcmd_ex(char*, size_t);


void discover_machine_type(long boot_flags);

void parseargs(char *argstr, int *argc_p, char **argv, char** resid);
void unparseargs(char *argstr, int argc, const char **argv);

#define ERASE_PATTERN1 0x55555555
#define ERASE_PATTERN2 0xAAAAAAAA

void eraseRAM( unsigned long pattern1, unsigned long pattern2 );

void bootmenu(void);

#define	VER_SEP	    "-"

void print_bootldr_version2(
    char*   prefix)
{
  char vbuf[32];

  if (prefix)
      putstr(prefix);
  putstr(bootprog_name);
  putstr(", Rev ");
  dwordtodecimal(vbuf, VERSION_MAJOR); putstr(vbuf); putstr(VER_SEP);
  dwordtodecimal(vbuf, VERSION_MINOR); putstr(vbuf); putstr(VER_SEP);
  dwordtodecimal(vbuf, VERSION_MICRO); putstr(vbuf);
  if(strlen(VERSION_SPECIAL) > 0){
    putstr("-");
    putstr(VERSION_SPECIAL);
  }
  putstr("\r\n");
}

void print_bootldr_version(void)
{
    print_bootldr_version2(">> ");
}

void
print_version_verbose(
    char*   prefix)
{
    print_bootldr_version2(prefix);

    if (prefix)
	putstr(prefix);

    putstr("Contact: bootldr@handhelds.org\r\n");
}

void print_banner(void)
{
  long armrev = 0;
  long armcomp;
  long armarch;
  long armpart;
  long cpsr = 0;
  __asm__("mrc p15, 0, %0, c0, c0, 0" : "=r" (armrev));
  __asm__("mrs %0, cpsr" : "=r" (cpsr));

  armcomp = (armrev & ARM_COMP_MASK) >> ARM_COMP_SHIFT;
  armarch = (armrev & ARM_ARCH_MASK) >> ARM_ARCH_SHIFT;
  armpart = (armrev & ARM_PART_MASK) >> ARM_PART_SHIFT;

  print_bootldr_version();

  putstr("> Contact: bootldr@handhelds.org\r\n");
  
  putstr(">> ");

  switch(armcomp){
      case ARM_ID_COMP_DEC:
	  putstr("DEC ");	  
	  switch (armarch){
	      case ARM_ID_ARCH_SA:
		  putstr("StrongArm ");
		  switch (armpart){
		      case ARM_ID_PART_SA110:
			  putstr("SA110 ");
			  break;
		      case ARM_ID_PART_SA1100:
			  putstr("Cpu Part: SA1100 ");
			  break;
		      default:
			  putLabeledWord("Unknown Part: ", armpart);
		  }		  
		  break;
	      default:
		  putLabeledWord("Unknown Architecture: ", armarch);
	  }	  
	  break;	  
      case ARM_ID_COMP_INTEL:
	  putstr("Intel ");
	  switch (armarch){
	      case ARM_ID_ARCH_SA:
		  putstr("StrongArm ");
		  switch (armpart){
		      case ARM_ID_PART_SA1110:
			  putstr("SA1110 ");
			  break;
		      case ARM_ID_PART_SA1100:
			  putstr("SA1100 ");
			  break;
		      default:
			  putLabeledWord("Unknown Part: ", armpart);
		  }
		  break;
	      case ARM_ID_ARCH_XSCALE:
		  putstr("XScale ");
		  switch (armpart){
		      case ARM_ID_PART_PXA250:
			  putstr("PXA250 (Cotulla) ");
			  break;
		      default:
			  putLabeledWord("Unknown Part: ", armpart);
		  }

		  break;
	      default:
		  putLabeledWord("Unknown Architecture: ", armarch);
	  }	  
	  break;
      default:
	  putLabeledWord("CPU made by unknown Company: ", armcomp);
	  
  }
  
  
  switch(armpart){
      case ARM_ID_PART_SA1110:
	  switch(armrev & ARM_REVISION_MASK){
	      case  0: putstr("revision A0\r\n"); break;
	      case  4: putstr("revision B0\r\n"); break;
	      case  5: putstr("revision B1\r\n"); break;
	      case  6: putstr("revision B2\r\n"); break;
	      case  8: putstr("revision B4\r\n"); break;
	      default: putLabeledWord("revision ", armrev);
	  }
	  break;

      default:
	  putLabeledWord("revision", armrev & ARM_REVISION_MASK);

  }
  


#if CONFIG_MACH_SKIFF
  putLabeledWord(">>  Corelogic Rev=", *(volatile unsigned long *)(DC21285_ARMCSR_BASE + PCI_VENDOR_ID));
  putLabeledWord(">>  Corelogic fclk=", param_mem_fclk_21285.value;
#endif

  putstr(">> (c) 2000-2003 Compaq Computer Corporation, provided with NO WARRANTY under the terms of the GNU General Public License.\r\n");
  putstr(">> See http://www.handhelds.org/bootldr/ for full license and sources\r\n");
}

int	pushed_chars_len = 0;
int	pushed_chars_idx = 0;
char	pushed_chars[128];


int
push_cmd_chars(
    char*   chars,
    int	    len)
{
    if (pushed_chars_len + len > sizeof(pushed_chars))
	return (-1);

    memcpy(&pushed_chars[pushed_chars_len], chars, len);
    pushed_chars_len += len;
    return 0;
}

int
get_pushed_char(
    void)
{
    int	    ret;

    if (pushed_chars_idx >= pushed_chars_len) {
	pushed_chars_idx = 0;
	pushed_chars_len = 0;
	ret = -1;
    }
    else
	ret = pushed_chars[pushed_chars_idx++] & 0xff;

    /*  putLabeledWord("gpc, ret: 0x", ret); */
    
    return(ret);
}

/* Iterations for delay loop, assumes caches on: */
#define DELAY_UNIT    (15000000)
#define AWAITKEY_UNIT (1000000)

static void msleep(unsigned int msec);

void delay_seconds(unsigned long units)
{
  /* wtf? */
  msleep(units*1000);
}

#define CMDBUFSIZE 256
char cmdBuffer[CMDBUFSIZE];

int argc;

enum ParseState {
   PS_WHITESPACE,
   PS_TOKEN,
   PS_STRING,
   PS_ESCAPE
};

enum ParseState stackedState;

void parseargs(char *argstr, int *argc_p, char **argv, char** resid)
{
  // const char *whitespace = " \t";
  int argc = 0;
  char c;
  enum ParseState lastState = PS_WHITESPACE;
   
  /* tokenize the argstr */
  while ((c = *argstr) != 0) {
    enum ParseState newState;

    if (c == ';' && lastState != PS_STRING && lastState != PS_ESCAPE)
	break;
    
    if (lastState == PS_ESCAPE) {
      newState = stackedState;
    } else if (lastState == PS_STRING) {
      if (c == '"') {
        newState = PS_WHITESPACE;
        *argstr = 0;
      } else {
        newState = PS_STRING;
      }
    } else if ((c == ' ') || (c == '\t')) {
      /* whitespace character */
      *argstr = 0;
      newState = PS_WHITESPACE;
    } else if (c == '"') {
      newState = PS_STRING;
      *argstr = 0;
      argv[argc++] = argstr + 1;
    } else if (c == '\\') {
      stackedState = lastState;
      newState = PS_ESCAPE;
    } else {
      /* token */
      if (lastState == PS_WHITESPACE) {
        argv[argc++] = argstr;
      }      
      newState = PS_TOKEN;
    }

    lastState = newState;
    argstr++;
  }

  argv[argc] = NULL;
  if (argc_p != NULL)
    *argc_p = argc;

  if (*argstr == ';') {
      *argstr++ = '\0';
  }
  *resid = argstr;
}



// this is getting more compliacated,  this function will averride any of the
// args in argstr with tthe args from argv.  this will allow you to override the
// param linuxargs from the commandline. e.g. init=/myRC will override
// init=linuxRC from the params.
void unparseargs(char *argstr, int argc, const char **argv)
{
   int i;
   char *cutStart;
   char *cutEnd;
   char *paramEnd;
   int delta;   
   int j;
   
   
   for (i = 0; i < argc; i++) {
      if (argv[i] != NULL) {
         if ((paramEnd = strchr(argv[i],'=')))// we have an a=b arg
            {
	      int argstrlen = strlen(argstr);
	      int needlelen = 0;
	      char needle[256];
	       paramEnd++;
	       putstr("haystack = <");
	       putstr(argstr);
	       putstr(">\r\n");
   
	       needlelen = (int)(paramEnd - argv[i]);
	       strncpy(needle, argv[i], needlelen);
	       needle[needlelen] = 0;
	       putstr("needle = <");
	       putnstr(needle,needlelen);
	       putstr(">\r\n");

	       if ((cutStart = (char *)strstr(argstr, needle)) != NULL){
                  // found a match
                  if (!(cutEnd = strchr(cutStart,' '))) {
                     cutEnd = argstr + argstrlen;
		  } else {
                     cutEnd++; // skip the space
                  }
                  delta = (int)(cutEnd - cutStart);		   
                  for (j=(int) (cutEnd - argstr); j < argstrlen; j++) 
                     argstr[j-delta] = argstr[j];
                  // new end of argstr
                  argstr[argstrlen - delta] = '\0';
		   
	       }
            }	   
         strcat(argstr, " ");
         strcat(argstr, argv[i]);
      }
   }
}

int parsecmd(int argc, const char **argv)
{
   /* find the command name */
   const char *cmdname = argv[0];
   int cmdnum = 0;

#if 0
   if (argc < 1)
      return -1;
   /* do a string compare for the first offset characters of cmdstr
      against each member of the cmdlist */
   while (cmdlist[cmdnum].cmdstr != NULL) {
      if (strcmp(cmdlist[cmdnum].cmdstr, cmdname) == 0)
         return(cmdnum);
     cmdnum++;
   }
#endif
   return(-1);
}

void
exec_string(
    char*   buf)
{
  int argc;
  char *argv[256];
  char*	resid;

  while (*buf) {
      memset(argv, 0, sizeof(argv));
      parseargs(buf, &argc, argv, &resid);
      if (argc > 0) {
		lab_execcommand(argc, (const char**) argv);
      } else {
		exec_string("help");
      }
      buf = resid;
  }
}

void bootmenu(void)
{
  serial->enabled = 1;


  while (1) {
/* the following line will kill off the system in the following circumstance:
 * 1) boot lab.
 * 2) copy ymodem: fs:/foo.txt
 * 3) ymodem any file down.
 * 4) copy fs:/foo.txt fs:/bar.txt
 * 5) for some reason, there will be a null pointer exception in fput.
 * to be worked on tomorrow.
 */
    putstr("boot> ");
    getcmd_ex(cmdBuffer, CMDBUFSIZE);
    
    if (cmdBuffer[0]) {
        /*
	 * useful if calling a modified bootldr's bootmenu
	 * function from another place.
	 */
	if (strcmp("quit", cmdBuffer) == 0)
	    return;
	
	exec_string(cmdBuffer);
    }
  }
}

void bootldr_main()
{
     print_banner();
     
     bootmenu(); /* does not return */
     return;
}

void  hex_dump(
    unsigned char   *data,
    size_t	    num)
{
    int     i;
    long    oldNum;
    char    buf[90];
    char*   bufp;
    int	    line_resid;
    
    while (num)
    {
	bufp = buf;
	binarytohex(bufp, (unsigned long)data, 4);
	bufp += 8;
	*bufp++ = ':';
	*bufp++ = ' ';
	
	oldNum = num;
	
	for (i = 0; i < 16 && num; i++, num--) {
	    binarytohex(bufp, (unsigned long)data[i], 1);
	    bufp += 2;
	    *bufp++ = (i == 7) ? '-' : ' ';
	}

	line_resid = (16 - i) * 3;
	if (line_resid) {
	    memset(bufp, ' ', line_resid);
	    bufp += line_resid;
	}
	
	memcpy(bufp, "| ", 2);
	bufp += 2;
	
	for (i = 0; i < 16 && oldNum; i++, oldNum--)
	    *bufp++ = BL_ISPRINT(data[i]) ? data[i] : '.';

	line_resid = 16 - i;
	if (line_resid) {
	    memset(bufp, ' ', 16-i);
	    bufp += 16 - i;
	}
	
	*bufp++ = '\r';
	*bufp++ = '\n';
	*bufp++ = '\0';
	putstr(buf);
	data += 16;
    }
}


void parseParamName(int argc,const char **argv,char *pName,char *pValue)
{
    int i;
    int len = 0;
    char *pAll;
    char *p;
    unsigned char canKill = 0;
    unsigned char valueSearch = 0;
    

    for (i=1; i < argc; i++){
#if 0
	putLabeledWord("i = ",i);
	putstr("argv = <");
	putstr(argv[i]);
	putstr(">\r\n");
#endif
	len+= strlen(argv[i]);
    }


    if ((pAll = (char *) vmalloc((len+1) * sizeof(char))) == NULL){
	putstr("out of room to build params list!\r\n");
	return;
    }
    pAll[0] = '\0';
    p = pAll;    
    for (i=1; i < argc; i++){
	strcpy(p,argv[i]);
	p += strlen(argv[i]);
	strcpy(p," ");
	p += 1;
    }
    // kill the last unneeded space
    *--p ='\0';
    

#if 0
    putstr("pAll = <");
    putstr(pAll);
    putstr(">\r\n");
#endif

    // eliminate = if it's there
    // we only kill the first one and only if its 
    p = pAll;
    while((p - pAll) < (strlen(argv[1]) + strlen(argv[2])+1)){

	if (canKill && (isblank(*p) || (*p == '=')))
	    valueSearch = 1;
	if (!canKill && isalnum(*p))
	    canKill = 1;
	else if (valueSearch && isalnum(*p))
	    break;
	if (canKill && (*p == '='))
	    *p = ' ';
	p++;
    }
    
#if 0
    putstr("pAll_post = <");
    putstr(pAll);
    putstr(">\r\n");
#endif

    p = pAll;
    while ((*p != ' ') && (*p != '\t'))
	p++;
    *p++ = '\0';
    
    while ((*p == ' ') || (*p == '\t'))
        p++;
    strcpy(pName,pAll);
    strcpy(pValue,p);
    
    vfree(pAll);
    return;
    
}

// this builds up the command buffer and either processes or prints it.
void parseParamFile(unsigned char *src,unsigned long size,int just_show)
{
    unsigned char *p = src;
    char cmdbuf[1024];
    char* cmdp;

    while ((p-src) < size){
	cmdp = cmdbuf;
	while (((p-src) < size) && (*p != '\r') && (*p != '\n'))
	    *cmdp++ = *p++;
	*cmdp = '\0';
	putstr("+ ");
	putstr(cmdbuf);
	putstr("\r\n");
	while ((*p == '\r') || (*p == '\n'))
	    p++;
	if (!just_show)
	    exec_string(cmdbuf);
    }
}


