/*
* Driver interface to the ASIC Complasion chip on the iPAQ H3800
*
* Copyright 2001 Compaq Computer Corporation.
*
* Use consistent with the GNU GPL is permitted,
* provided that this copyright notice is
* preserved in its entirety in all copies and derived works.
*
* COMPAQ COMPUTER CORPORATION MAKES NO WARRANTIES, EXPRESSED OR IMPLIED,
* AS TO THE USEFULNESS OR CORRECTNESS OF THIS CODE OR ITS
* FITNESS FOR ANY PARTICULAR PURPOSE.
*
* Author:  Andrew Christian
*          <Andrew.Christian@compaq.com>
*          October 2001
*/

#include <linux/module.h>
#include <linux/version.h>

#include <linux/init.h>
#include <linux/fs.h>
#include <linux/interrupt.h>
#include <linux/sched.h>
#include <linux/pm.h>
#include <linux/sysctl.h>
#include <linux/proc_fs.h>
#include <linux/ctype.h>
#include <linux/delay.h>

#include <asm/irq.h>
#include <asm/uaccess.h>   /* for copy to/from user space */
#include <asm/arch/hardware.h>
#include <asm/arch/h3900_asic.h>

#define H3600_ASIC_PROC_DIR     "asic_debug"
#define REG_DIRNAME             "registers"

#define PDEBUG(format,arg...) printk(KERN_DEBUG __FILE__ ":%s - " format "\n", __FUNCTION__, ## arg)
#define PALERT(format,arg...) printk(KERN_ALERT __FILE__ ":%s - " format "\n", __FUNCTION__, ## arg)
#define PERROR(format,arg...) printk(KERN_ERR __FILE__ ":%s - " format "\n", __FUNCTION__, ## arg)

/* Coded lifted from "registers.c" */

static ssize_t proc_read_reg(struct file * file, char * buf,
		size_t nbytes, loff_t *ppos);
static ssize_t proc_write_reg(struct file * file, const char * buffer,
		size_t count, loff_t *ppos);

static struct file_operations proc_reg_operations = {
	read:	proc_read_reg,
	write:	proc_write_reg
};

typedef struct asic_reg_entry {
	u16   bytes;
	u32   phyaddr;
	char* name;
	char* description;
	unsigned short low_ino;
} asic_reg_entry_t;

static struct proc_dir_entry   *asic_proc_dir;
static struct proc_dir_entry   *reg_proc_dir;

static asic_reg_entry_t asic_regs[] =
{
/*	{ bytes, virt_addr,   name,     description } */
	{ 4, _H3800_ASIC2_Base + 0x0000, "GPIODIR",    "GPIO Input/Output direction register" },
	{ 4, _H3800_ASIC2_Base + 0x0004, "GPIINTTYPE", "GPI Interrupt Type (Edge/Level)"},
	{ 4, _H3800_ASIC2_Base + 0x0008, "GPIINTESEL", "GPI Interrupt active edge select"},
	{ 4, _H3800_ASIC2_Base + 0x000c, "GPIINTALSEL","GPI Interrupt active level select"},
	{ 4, _H3800_ASIC2_Base + 0x0010, "GPIINTFLAG", "GPI Interrupt active flag" },
	{ 4, _H3800_ASIC2_Base + 0x0014, "GPIOPIOD",   "GPIO Port input/output data" },
	{ 4, _H3800_ASIC2_Base + 0x0018, "GPOBFSTAT",  "GPO output data in batt_fault" },
	{ 4, _H3800_ASIC2_Base + 0x001c, "GPIINTSTAT", "GPI Interrupt status" },
	{ 4, _H3800_ASIC2_Base + 0x003c, "GPIOALT",    "GPIO ALTernate function" },

	{ 4, _H3800_ASIC2_Base + 0x0200, "KPIODIR",    "KPIO Input/output direction"},
	{ 4, _H3800_ASIC2_Base + 0x0204, "KPIINTTYP",  "KPI interrupt type (edge/level)" },
	{ 4, _H3800_ASIC2_Base + 0x0208, "KPIINTESEL", "KPI Interrupt active edge select" },
	{ 4, _H3800_ASIC2_Base + 0x020c, "KPIINTALSEL","KPI Interrupt active level select" },
	{ 4, _H3800_ASIC2_Base + 0x0210, "KPIINTFLAG", "KPI Interrupt active flag" },
	{ 4, _H3800_ASIC2_Base + 0x0214, "KPIOPIOD",   "KPIO Port input/output data" },
	{ 4, _H3800_ASIC2_Base + 0x0218, "KPOBFSTAT",  "KPO Ouput data in batt_fault status" },
	{ 4, _H3800_ASIC2_Base + 0x021c, "KPIINTSTAT", "KPI Interrupt status" },
	{ 4, _H3800_ASIC2_Base + 0x023c, "KALT",       "KIU alternate function" },

	{ 4, _H3800_ASIC2_Base + 0x0400, "SPICR",      "SPI control register" },
	{ 4, _H3800_ASIC2_Base + 0x0404, "SPIDR",      "SPI data register" },
	{ 4, _H3800_ASIC2_Base + 0x0408, "SPIDCS",     "SPI chip select disabled register" },

	{ 4, _H3800_ASIC2_Base + 0x0600, "PWM0_TBS",   "PWM Time base set register" },
	{ 4, _H3800_ASIC2_Base + 0x0604, "PWM0_PTS",   "PWM Period time set register" },
	{ 4, _H3800_ASIC2_Base + 0x0608, "PWM0_DTS",   "PWM Duty time set register" },

	{ 4, _H3800_ASIC2_Base + 0x0700, "PWM1_TBS",   "PWM Time base set register" },
	{ 4, _H3800_ASIC2_Base + 0x0704, "PWM1_PTS",   "PWM Period time set register" },
	{ 4, _H3800_ASIC2_Base + 0x0708, "PWM1_DTS",   "PWM Duty time set register" },

	{ 4, _H3800_ASIC2_Base + 0x0800, "LED0_TBS",   "LED time base set register" },
	{ 4, _H3800_ASIC2_Base + 0x0804, "LED0_PTS",   "LED period time set register" },
	{ 4, _H3800_ASIC2_Base + 0x0808, "LED0_DTS",   "LED duty time set register" },
	{ 4, _H3800_ASIC2_Base + 0x080c, "LED0_ASTC",  "LED auto stop counter register" },

	{ 4, _H3800_ASIC2_Base + 0x0880, "LED1_TBS",   "LED time base set register" },
	{ 4, _H3800_ASIC2_Base + 0x0884, "LED1_PTS",   "LED period time set register" },
	{ 4, _H3800_ASIC2_Base + 0x0888, "LED1_DTS",   "LED duty time set register" },
	{ 4, _H3800_ASIC2_Base + 0x088c, "LED1_ASTC",  "LED auto stop counter register" },

	{ 4, _H3800_ASIC2_Base + 0x0900, "LED2_TBS",   "LED time base set register" },
	{ 4, _H3800_ASIC2_Base + 0x0904, "LED2_PTS",   "LED period time set register" },
	{ 4, _H3800_ASIC2_Base + 0x0908, "LED2_DTS",   "LED duty time set register" },
	{ 4, _H3800_ASIC2_Base + 0x090c, "LED2_ASTC",  "LED auto stop counter register" },

	{ 4, _H3800_ASIC2_Base + 0x0a00, "UART0_BUF",  "Receive/transmit buffer"},
	{ 4, _H3800_ASIC2_Base + 0x0a04, "UART0_IER",  "Interrupt enable" },
	{ 4, _H3800_ASIC2_Base + 0x0a08, "UART0_IIR",  "Interrupt identify" },
	{ 4, _H3800_ASIC2_Base + 0x0a08, "UART0_FCR",  "Fifo control" },
	{ 4, _H3800_ASIC2_Base + 0x0a0c, "UART0_LCR",  "Line control" },
	{ 4, _H3800_ASIC2_Base + 0x0a10, "UART0_MCR",  "Modem control" },
	{ 4, _H3800_ASIC2_Base + 0x0a14, "UART0_LSR",  "Line status" },
	{ 4, _H3800_ASIC2_Base + 0x0a18, "UART0_MSR",  "Modem status" },
	{ 4, _H3800_ASIC2_Base + 0x0a1c, "UART0_SCR",  "Scratch pad" },

	{ 4, _H3800_ASIC2_Base + 0x0c00, "UART1_BUF",  "Receive/transmit buffer"},
	{ 4, _H3800_ASIC2_Base + 0x0c04, "UART1_IER",  "Interrupt enable" },
	{ 4, _H3800_ASIC2_Base + 0x0c08, "UART1_IIR",  "Interrupt identify" },
	{ 4, _H3800_ASIC2_Base + 0x0c08, "UART1_FCR",  "Fifo control" },
	{ 4, _H3800_ASIC2_Base + 0x0c0c, "UART1_LCR",  "Line control" },
	{ 4, _H3800_ASIC2_Base + 0x0c10, "UART1_MCR",  "Modem control" },
	{ 4, _H3800_ASIC2_Base + 0x0c14, "UART1_LSR",  "Line status" },
	{ 4, _H3800_ASIC2_Base + 0x0c18, "UART1_MSR",  "Modem status" },
	{ 4, _H3800_ASIC2_Base + 0x0c1c, "UART1_SCR",  "Scratch pad" },

	{ 4, _H3800_ASIC2_Base + 0x0e00, "TIMER_0",    "Timer counter 0 register" },
	{ 4, _H3800_ASIC2_Base + 0x0e04, "TIMER_1",    "Timer counter 1 register" },
	{ 4, _H3800_ASIC2_Base + 0x0e08, "TIMER_2",    "Timer counter 2 register" },
	{ 4, _H3800_ASIC2_Base + 0x0e0a, "TIMER_CNTL", "Timer control register (write only)" },
	{ 4, _H3800_ASIC2_Base + 0x0e10, "TIMER_CMD",  "Timer command register" },

	{ 4, _H3800_ASIC2_Base + 0x1000, "CDEX",       "Crystal source, control clock" },

	{ 4, _H3800_ASIC2_Base + 0x1200, "ADMUX",      "ADC multiplixer select register" },
	{ 4, _H3800_ASIC2_Base + 0x1204, "ADCSR",      "ADC control and status register" },
	{ 4, _H3800_ASIC2_Base + 0x1208, "ADCDR",      "ADC data register" },
	
	{ 4, _H3800_ASIC2_Base + 0x1600, "INTMASK",    "Interrupt mask control & cold boot flag" },
	{ 4, _H3800_ASIC2_Base + 0x1604, "INTCPS",     "Interrupt timer clock pre-scale" },
	{ 4, _H3800_ASIC2_Base + 0x1608, "INTTBS",     "Interrupt timer set" },

	{ 4, _H3800_ASIC2_Base + 0x1800, "OWM_CMD",    "OWM command register" },
	{ 4, _H3800_ASIC2_Base + 0x1804, "OWM_DATA",   "OWM transmit/receive buffer" },
	{ 4, _H3800_ASIC2_Base + 0x1808, "OWM_INT",    "OWM interrupt register" },
	{ 4, _H3800_ASIC2_Base + 0x180c, "OWM_INTEN",  "OWM interrupt enable register" },
	{ 4, _H3800_ASIC2_Base + 0x1810, "OWM_CLKDIV", "OWM clock divisor register" },

	{ 4, _H3800_ASIC2_Base + 0x1a00, "SSETR",      "Size of flash memory setting register" },

	{ 2, _H3900_ASIC3_Base + 0x0100, "GPIO_B_Mask", "" },
	{ 2, _H3900_ASIC3_Base + 0x0104, "GPIO_B_Direction", "" },
	{ 2, _H3900_ASIC3_Base + 0x0108, "GPIO_B_Out", "" },
	{ 2, _H3900_ASIC3_Base + 0x010c, "GPIO_B_TriggerType", "" },
	{ 2, _H3900_ASIC3_Base + 0x0110, "GPIO_B_EdgeTrigger", "" },
	{ 2, _H3900_ASIC3_Base + 0x0114, "GPIO_B_LevelTrigger", "" },
	{ 2, _H3900_ASIC3_Base + 0x0118, "GPIO_B_SleepMask", "" },
	{ 2, _H3900_ASIC3_Base + 0x011c, "GPIO_B_SleepOut", "" },
	{ 2, _H3900_ASIC3_Base + 0x0120, "GPIO_B_BattFaultOut", "" },
	{ 2, _H3900_ASIC3_Base + 0x0124, "GPIO_B_IntStatus", "" },
	{ 2, _H3900_ASIC3_Base + 0x012c, "GPIO_B_SleepConf", "" },
	{ 2, _H3900_ASIC3_Base + 0x0130, "GPIO_B_Status", "" },

	{ 2, _H3900_ASIC3_Base + 0x0a00, "ASIC3_CLOCK_CDEX", "" },
	{ 2, _H3900_ASIC3_Base + 0x0a04, "ASIC3_CLOCK_SEL", "" },
	
	{ 2, _H3900_ASIC3_Base + 0x0404, "SD_CONFIG_Command", "" },         
	{ 2, _H3900_ASIC3_Base + 0x0410, "SD_CONFIG_Addr0", "" },           
	{ 2, _H3900_ASIC3_Base + 0x0412, "SD_CONFIG_Addr1", "" },           
	{ 2, _H3900_ASIC3_Base + 0x043c, "SD_CONFIG_IntPin", "" },          
	{ 2, _H3900_ASIC3_Base + 0x0440, "SD_CONFIG_ClkStop", "" },         
	{ 2, _H3900_ASIC3_Base + 0x0442, "SD_CONFIG_ClockMode", "" },       
	{ 2, _H3900_ASIC3_Base + 0x0444, "SD_CONFIG_SDHC_PinStatus", "" },  
	{ 2, _H3900_ASIC3_Base + 0x0448, "SD_CONFIG_SDHC_Power1", "" },     
	{ 2, _H3900_ASIC3_Base + 0x044a, "SD_CONFIG_SDHC_Power3", "" },     
	{ 2, _H3900_ASIC3_Base + 0x044c, "SD_CONFIG_SDHC_CardDetect", "" }, 
	{ 2, _H3900_ASIC3_Base + 0x0450, "SD_CONFIG_SDHC_Slot", "" },       
	{ 2, _H3900_ASIC3_Base + 0x04f0, "SD_CONFIG_SDHC_ExtGateClk1", "" },
	{ 2, _H3900_ASIC3_Base + 0x04f8, "SD_CONFIG_SDHC_ExtGateClk3", "" },

	{ 2, _H3900_ASIC3_Base + 0x1000, "SD_CTRL_Cmd", "" },
	{ 2, _H3900_ASIC3_Base + 0x1004, "SD_CTRL_Arg0", "" },
	{ 2, _H3900_ASIC3_Base + 0x1006, "SD_CTRL_Arg1", "" },
	{ 2, _H3900_ASIC3_Base + 0x1008, "SD_CTRL_StopInternal", "" },
	{ 2, _H3900_ASIC3_Base + 0x100a, "SD_CTRL_TransferSectorCount", "" },
	{ 2, _H3900_ASIC3_Base + 0x100c, "SD_CTRL_Response0", "" },
	{ 2, _H3900_ASIC3_Base + 0x100e, "SD_CTRL_Response1", "" },
	{ 2, _H3900_ASIC3_Base + 0x1010, "SD_CTRL_Response2", "" },
	{ 2, _H3900_ASIC3_Base + 0x1012, "SD_CTRL_Response3", "" },
	{ 2, _H3900_ASIC3_Base + 0x1014, "SD_CTRL_Response4", "" },
	{ 2, _H3900_ASIC3_Base + 0x1016, "SD_CTRL_Response5", "" },
	{ 2, _H3900_ASIC3_Base + 0x1018, "SD_CTRL_Response6", "" },
	{ 2, _H3900_ASIC3_Base + 0x101a, "SD_CTRL_Response7", "" },
	{ 2, _H3900_ASIC3_Base + 0x101c, "SD_CTRL_CardStatus", "" },
	{ 2, _H3900_ASIC3_Base + 0x101e, "SD_CTRL_BufferCtrl", "" },
	{ 2, _H3900_ASIC3_Base + 0x1020, "SD_CTRL_IntMask0", "" },  
	{ 2, _H3900_ASIC3_Base + 0x1022, "SD_CTRL_IntMask1", "" },  
	{ 2, _H3900_ASIC3_Base + 0x1024, "SD_CTRL_CardClockCtrl", "" },      
	{ 2, _H3900_ASIC3_Base + 0x1026, "SD_CTRL_MemCardXferDataLen", "" }, 
	{ 2, _H3900_ASIC3_Base + 0x1028, "SD_CTRL_MemCardOptionSetup", "" }, 
	{ 2, _H3900_ASIC3_Base + 0x102c, "SD_CTRL_ErrorStatus0", "" },       
	{ 2, _H3900_ASIC3_Base + 0x102e, "SD_CTRL_ErrorStatus1", "" },       
	{ 2, _H3900_ASIC3_Base + 0x1030, "SD_CTRL_DataPort", "" },           
	{ 2, _H3900_ASIC3_Base + 0x1034, "SD_CTRL_TransactionCtrl", "" },    
	{ 2, _H3900_ASIC3_Base + 0x10e0, "SD_CTRL_SoftwareReset", "" },      

	{ 2, _H3900_ASIC3_Base + 0x0a00, "CLOCK_CDEX", "" },      
	{ 2, _H3900_ASIC3_Base + 0x0a04, "CLOCK_SEL", "" },      
	
	{ 2, _H3900_ASIC3_Base + 0x0b00, "INTR_IntMask", "" },      
	{ 2, _H3900_ASIC3_Base + 0x0b04, "INTR_PIntStat", "" },      
	{ 2, _H3900_ASIC3_Base + 0x0b08, "INTR_IntCPS", "" },      
	{ 2, _H3900_ASIC3_Base + 0x0b0c, "INTR_IntTBS", "" },      

	{ 2, _H3900_ASIC3_Base + 0x0e00, "SDHWCTRL_SDConf", "" },      

	{ 4, _H3800_ASIC2_Base + 0x1f00, "FLASHWP",    "Flash write protect" },
};

#define NUM_OF_ASIC_REG_ENTRY	(sizeof(asic_regs)/sizeof(asic_reg_entry_t))

static int proc_read_reg(struct file * file, char * buf,
		size_t nbytes, loff_t *ppos)
{
	int i_ino = (file->f_dentry->d_inode)->i_ino;
	char outputbuf[15];
	int count;
	int i;
	asic_reg_entry_t* current_reg=NULL;
	if (*ppos>0) /* Assume reading completed in previous read*/
		return 0;
	for (i=0;i<NUM_OF_ASIC_REG_ENTRY;i++) {
		if (asic_regs[i].low_ino==i_ino) {
			current_reg = &asic_regs[i];
			break;
		}
	}
	if (current_reg==NULL)
		return -EINVAL;

	switch (current_reg->bytes) {
	case 1:
		count = sprintf(outputbuf, "0x%02X\n", *((volatile u8 *)(current_reg->phyaddr)));
		break;
	case 2:
		count = sprintf(outputbuf, "0x%04X\n", *((volatile u16 *)(current_reg->phyaddr)));
		break;
	case 4:
	default:
		count = sprintf(outputbuf, "0x%08X\n", *((volatile u32 *)(current_reg->phyaddr)));
		break;
	}
	*ppos+=count;
	if (count>nbytes)  /* Assume output can be read at one time */
		return -EINVAL;
	if (copy_to_user(buf, outputbuf, count))
		return -EFAULT;
	return count;
}

static ssize_t proc_write_reg(struct file * file, const char * buffer,
		size_t count, loff_t *ppos)
{
	int i_ino = (file->f_dentry->d_inode)->i_ino;
	asic_reg_entry_t* current_reg=NULL;
	int i;
	unsigned long newRegValue;
	char *endp;

	for (i=0;i<NUM_OF_ASIC_REG_ENTRY;i++) {
		if (asic_regs[i].low_ino==i_ino) {
			current_reg = &asic_regs[i];
			break;
		}
	}
	if (current_reg==NULL)
		return -EINVAL;

	newRegValue = simple_strtoul(buffer,&endp,0);
	switch (current_reg->phyaddr) {
	case 1:
		*((volatile u8 *)(current_reg->phyaddr))=newRegValue;
		break;
	case 2:
		*((volatile u16 *)(current_reg->phyaddr))=newRegValue;
		break;
	case 4:
	default:
		*((volatile u32 *)(current_reg->phyaddr))=newRegValue;
		break;
	}
	return (count+endp-buffer);
}

static int h3900_asic_debug_init (void)
{
	int i;

	asic_proc_dir = proc_mkdir(H3600_ASIC_PROC_DIR, NULL);
	if ( !asic_proc_dir ) {
		PERROR("unable to create proc entry %s", H3600_ASIC_PROC_DIR);
		return -ENOMEM;
	}

	reg_proc_dir = proc_mkdir(REG_DIRNAME, asic_proc_dir);
	if (reg_proc_dir == NULL) {
		PERROR("can't create /proc/%s", REG_DIRNAME);
		return(-ENOMEM);
	}

	for(i=0;i<NUM_OF_ASIC_REG_ENTRY;i++) {
		struct proc_dir_entry *entry = create_proc_entry(asic_regs[i].name,
								 S_IWUSR |S_IRUSR | S_IRGRP | S_IROTH,
								 reg_proc_dir);
		
		if ( !entry ) {
			PERROR("can't create /proc/%s/%s", REG_DIRNAME, asic_regs[i].name);
			return(-ENOMEM);
		}

		asic_regs[i].low_ino = entry->low_ino;
		entry->proc_fops = &proc_reg_operations;
	}
	
	return 0;
}

static void h3900_asic_debug_exit (void)
{
	int i;

	if (asic_proc_dir) {
		for(i=0;i<NUM_OF_ASIC_REG_ENTRY;i++)
			remove_proc_entry(asic_regs[i].name,reg_proc_dir);
		
		remove_proc_entry(REG_DIRNAME, reg_proc_dir);
		remove_proc_entry(REG_DIRNAME, asic_proc_dir);
	}
}

module_init (h3900_asic_debug_init)
module_exit (h3900_asic_debug_exit)
